package com.kidoz.sdk.api.players.video_player.ui.custom_views;

import android.animation.Animator;
import android.app.Activity;
import android.content.Context;
import android.content.pm.ActivityInfo;
import android.content.res.Configuration;
import android.graphics.Color;
import android.graphics.PixelFormat;
import android.os.Handler;
import android.view.Gravity;
import android.view.LayoutInflater;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.view.WindowManager;
import android.widget.FrameLayout;
import android.widget.RelativeLayout;
import android.widget.SeekBar;

import com.google.android.youtube.player.YouTubePlayer;
import com.kidoz.sdk.api.R;
import com.kidoz.sdk.api.general.utils.SDKLogger;
import com.kidoz.sdk.api.players.video_player.logics.VideoPlayerListener;
import com.kidoz.sdk.api.players.video_player.ui.fragments.VideoFragment;
import com.kidoz.sdk.api.players.video_player.util.AnimationsUtils;
import com.kidoz.sdk.api.players.video_player.util.ScreenUtils;
import com.kidoz.sdk.api.structure.ContentItem;
import com.kidoz.sdk.api.ui_views.ContentPanelView;

import java.util.ArrayList;

/**
 * Created by roni on 7/9/2015.
 */
public class VideoPlayerController
{
    private final String TAG = VideoPlayerController.class.getSimpleName();
    private static final float VIDEO_WINDOW_RATIO = 0.315f;
    private Context mContext;
    private View mRootView;
    private RelativeLayout.LayoutParams mVideoContainerLayoutParams;
    private InterceptableTouchEventRelativeLayout mVideoControllerContainer;
    private PlayAndPauseButton mPlayAndPauseButton;
    private SeekBarController mSeekBarController;
    private VideoPlayerTopBar mVideoPlayerTopBar;
    private RelatedContentPortraitContainer mRelatedContentPortraitContainer;
    private RelatedContentLandscapeContainer mRelatedContentLandscapeContainer;
    private VideoPlayerListener mVideoPlayerListener;
    private VideoFragment mVideoFragment;
    private FrameLayout mVideoPlayerContainerFrameLayout;
    private int mLandscapeRelatedContainerHideTranslateValue;
    private Handler mHandler;
    private boolean mIsUserPauseVideo;

    public VideoPlayerController(Context context)
    {
        mHandler = new Handler();
        mContext = context;
        initView();
        fixVideoLayout();
    }

    private void initView()
    {
        initRootView();
        initVideoContainerLayout();
        initTopBar();
        initSeekBarController();
        initRelatedContentPortraitContainer();
        initRelatedContentLandscapeContainer();
        initPlayAndPauseButton();
    }

    private void initRelatedContentPortraitContainer()
    {
        mRelatedContentPortraitContainer = (RelatedContentPortraitContainer) mRootView.findViewById(R.id.RelatedContentPortraitContainer);
    }

    private void initRelatedContentLandscapeContainer()
    {
        mRelatedContentLandscapeContainer = (RelatedContentLandscapeContainer) mRootView.findViewById(R.id.RelatedContentLandscapeContainer);
        int widthMeasureSpec = View.MeasureSpec.makeMeasureSpec(ViewGroup.LayoutParams.MATCH_PARENT, View.MeasureSpec.UNSPECIFIED);
        int heightMeasureSpec = View.MeasureSpec.makeMeasureSpec(ViewGroup.LayoutParams.WRAP_CONTENT, View.MeasureSpec.UNSPECIFIED);
        mRelatedContentLandscapeContainer.measure(widthMeasureSpec, heightMeasureSpec);
        mSeekBarController.measure(widthMeasureSpec, heightMeasureSpec);
        mLandscapeRelatedContainerHideTranslateValue = mRelatedContentLandscapeContainer.getMeasuredHeight() + mSeekBarController.getMeasuredHeight();
        mRelatedContentLandscapeContainer.setTranslateDistance(mLandscapeRelatedContainerHideTranslateValue);

        View relatedContainerAnchor = mRootView.findViewById(R.id.LandscapeRelatedAnchorView);
        RelativeLayout.LayoutParams anchorParams = (RelativeLayout.LayoutParams) relatedContainerAnchor.getLayoutParams();
        anchorParams.height = mSeekBarController.getMeasuredHeight();
        relatedContainerAnchor.setLayoutParams(anchorParams);

        if (mContext.getResources().getConfiguration().orientation == Configuration.ORIENTATION_PORTRAIT)
        {
            setLandscapeRelatedContentContainerVisibility(false);
        }
        else
        {
            setLandscapeRelatedContentContainerVisibility(true);
        }

        mRelatedContentLandscapeContainer.setRelatedContentLandscapeContainerListener(new RelatedContentLandscapeContainer.RelatedContentLandscapeContainerListener()
        {
            @Override public void onInterceptTouchRequest()
            {
                mHandler.removeCallbacksAndMessages(null);
                animateControlsEnter();
                mHandler.postDelayed(new Runnable()
                {
                    @Override public void run()
                    {
                        animateControlsExit();
                    }
                }, AnimationsUtils.TOUCH_DELAY_ANIMATION_TIME);
            }

            @Override public void onRelatedPanelStateChanged(boolean isOpened)
            {
                if (mVideoPlayerListener != null)
                {
                    mVideoPlayerListener.onRelatedPanelStateChanged(isOpened);
                }
            }
        });
    }

    private void initTopBar()
    {
        mVideoPlayerTopBar = (VideoPlayerTopBar) mRootView.findViewById(R.id.VideoPlayerTopBar);
        mVideoPlayerTopBar.setVideoPlayerTopBarListener(new VideoPlayerTopBar.VideoPlayerTopBarListener()
        {
            @Override public void onBackPressed()
            {
                if (mVideoPlayerListener != null)
                {
                    mRootView.setBackgroundColor(Color.parseColor("#000000"));
                    mVideoPlayerContainerFrameLayout.setVisibility(View.INVISIBLE);
                    mRootView.postDelayed(new Runnable()
                    {
                        @Override public void run()
                        {
                            mVideoFragment.onDestroy();
                            mVideoPlayerListener.onBackPressed();
                        }
                    }, 300);
                }
            }

            @Override public void onLikeClicked()
            {
                if (mVideoPlayerListener != null)
                {
                    mVideoPlayerListener.onLikeClicked();
                }
            }

            @Override public void onInfoClicked()
            {
                if (mVideoPlayerListener != null)
                {
                    mVideoPlayerListener.onInfoClicked();
                }
            }
        });
    }

    private void setTopBarVisibility(boolean isVisible)
    {
        if (isVisible == true)
        {
            mVideoPlayerTopBar.setTranslationY(0);
        }
        else
        {
            mVideoPlayerTopBar.setTranslationY(-1 * mVideoPlayerTopBar.getMeasuredHeight());
        }
    }

    private void setLandscapeRelatedContentContainerVisibility(boolean isVisible)
    {
        if (isVisible == true)
        {
            mRelatedContentLandscapeContainer.setTranslationY(0);
        }
        else
        {
            mRelatedContentLandscapeContainer.setTranslationY(mLandscapeRelatedContainerHideTranslateValue);
        }
    }

    private void initRootView()
    {
        LayoutInflater layoutInflater = (LayoutInflater) mContext.getSystemService(Context.LAYOUT_INFLATER_SERVICE);
        mRootView = layoutInflater.inflate(R.layout.video_player_controller, null, false);
        mRootView.setDuplicateParentStateEnabled(true);
    }

    public void initWindowParameters()
    {
        WindowManager.LayoutParams params = new WindowManager.LayoutParams();
        params.gravity = Gravity.TOP;
        params.x = 0;
        params.y = 0;
        params.height = WindowManager.LayoutParams.MATCH_PARENT;
        params.width = WindowManager.LayoutParams.MATCH_PARENT;
        params.type = WindowManager.LayoutParams.TYPE_SYSTEM_ALERT;
        params.flags = WindowManager.LayoutParams.FLAG_FULLSCREEN | WindowManager.LayoutParams.FLAG_WATCH_OUTSIDE_TOUCH | WindowManager.LayoutParams.FLAG_NOT_FOCUSABLE | WindowManager.LayoutParams.FLAG_LAYOUT_NO_LIMITS | WindowManager.LayoutParams.FLAG_LAYOUT_IN_SCREEN | WindowManager.LayoutParams.FLAG_NOT_TOUCH_MODAL;
        params.format = PixelFormat.TRANSLUCENT;
        params.windowAnimations = R.style.VideoPlayerAnimation;

        WindowManager windowManager = (WindowManager) mContext.getSystemService(Context.WINDOW_SERVICE);
        windowManager.addView(mRootView, params);
    }

    private void initSeekBarController()
    {
        mSeekBarController = (SeekBarController) mRootView.findViewById(R.id.SeekBarController);
        mSeekBarController.setOnSeekBarChangeListener(new SeekBar.OnSeekBarChangeListener()
        {
            @Override public void onProgressChanged(SeekBar seekBar, int progress, boolean fromUser)
            {
                if (fromUser == true && mVideoFragment != null)
                {
                    mHandler.removeCallbacksAndMessages(null);
                    animateControlsEnter();
                    int newProgress = (int) (mVideoFragment.getVideoTotalTime() * progress) / SeekBarController.SEEK_BAR_MAX_PROGRESS_VALUE;
                    mVideoFragment.seekVideo(newProgress);
                    mSeekBarController.updateDuration(newProgress, mVideoFragment.getVideoTotalTime());
                }
            }

            @Override public void onStartTrackingTouch(SeekBar seekBar)
            {
                mHandler.removeCallbacksAndMessages(null);
                animateControlsEnter();
                mSeekBarController.stopProgressUpdate();
            }

            @Override public void onStopTrackingTouch(SeekBar seekBar)
            {
                mSeekBarController.updateDuration(mVideoFragment.getVideoCurrentTime(), mVideoFragment.getVideoTotalTime());
                mHandler.removeCallbacksAndMessages(null);
                animateControlsExit();
                mVideoFragment.playVideo();
            }
        });
        mSeekBarController.setSeekBarControllerListener(new SeekBarController.SeekBarControllerListener()
        {
            @Override public void onRequestProgressUpdate()
            {
                if (mSeekBarController != null)
                {
                    mSeekBarController.updateDuration(mVideoFragment.getVideoCurrentTime(), mVideoFragment.getVideoTotalTime());
                }
            }
        });
        if (mContext != null && mContext instanceof Activity)
        {
            mSeekBarController.setActivity((Activity) mContext);
        }
    }

    private void initPlayAndPauseButton()
    {
        mPlayAndPauseButton = (PlayAndPauseButton) mRootView.findViewById(R.id.PlayAndPauseButton);
        mPlayAndPauseButton.setOnClickListener(new View.OnClickListener()
        {
            @Override public void onClick(View view)
            {
                AnimationsUtils.animateClick(view, new Animator.AnimatorListener()
                {
                    @Override public void onAnimationStart(Animator animation)
                    {

                    }

                    @Override public void onAnimationEnd(Animator animation)
                    {
                        if (mVideoFragment != null)
                        {
                            if (mVideoFragment.getIsPlaying() == true)
                            {
                                mVideoFragment.pauseVideo();
                                mIsUserPauseVideo = true;
                            }
                            else
                            {
                                mIsUserPauseVideo = false;
                                if (mPlayAndPauseButton.getPlayAndPauseStatus() == PlayAndPauseButton.PLAY_AND_PAUSE_STATUS.ERROR)
                                {
                                    mVideoFragment.refreshVideo();
                                }
                                else
                                {
                                    mVideoFragment.playVideo();
                                }
                            }
                        }
                    }

                    @Override public void onAnimationCancel(Animator animation)
                    {

                    }

                    @Override public void onAnimationRepeat(Animator animation)
                    {

                    }
                });
            }
        });
        mPlayAndPauseButton.setInterceptTouchEventListener(new PlayAndPauseButton.InterceptTouchEventListener()
        {
            @Override public void setIsCanInterceptTouchEvent(boolean isCanInterceptTouchEvent)
            {
                if (mVideoControllerContainer != null)
                {
                    mVideoControllerContainer.setIsInterceptTouchEvent(isCanInterceptTouchEvent);
                }
            }
        });
    }

    public boolean getIsUserPausedVideo()
    {
        return mIsUserPauseVideo;
    }


    private void initVideoContainerLayout()
    {
        mVideoControllerContainer = (InterceptableTouchEventRelativeLayout) mRootView.findViewById(R.id.VideoControllerContainer);
        mVideoContainerLayoutParams = (RelativeLayout.LayoutParams) mVideoControllerContainer.getLayoutParams();
        mVideoControllerContainer.setOnTouchListener(new View.OnTouchListener()
        {
            @Override public boolean onTouch(View v, MotionEvent event)
            {
                boolean result = true;
                switch (event.getAction())
                {
                    case MotionEvent.ACTION_DOWN:
                    {
                        mHandler.removeCallbacksAndMessages(null);
                        if (mVideoControllerContainer.getIsInterceptTouchEvent() == true)
                        {
                            animateControlsEnter();
                        }
                        else
                        {
                            animateControlsExit();
                        }
                        break;
                    }

                    case MotionEvent.ACTION_UP:
                    {
                        result = false;
                        if (mPlayAndPauseButton.getPlayAndPauseStatus() == PlayAndPauseButton.PLAY_AND_PAUSE_STATUS.PLAYING)
                        {
                            mHandler.removeCallbacksAndMessages(null);
                            mHandler.postDelayed(new Runnable()
                            {
                                @Override public void run()
                                {
                                    animateControlsExit();
                                }
                            }, AnimationsUtils.TOUCH_DELAY_ANIMATION_TIME);
                        }
                        break;
                    }
                }
                return result;
            }
        });
    }

    public void fixVideoLayout()
    {
        mRelatedContentPortraitContainer.refresh();
        mRelatedContentLandscapeContainer.refresh();
        if (mContext.getResources().getConfiguration().orientation == Configuration.ORIENTATION_PORTRAIT)
        {
            setLandscapeRelatedContentContainerVisibility(false);
            setTopBarVisibility(true);
            mVideoContainerLayoutParams = new RelativeLayout.LayoutParams(0, 0);
            mVideoContainerLayoutParams.width = ScreenUtils.getScreenSize(mContext, true);
            mVideoContainerLayoutParams.height = (int) (ScreenUtils.getScreenSize(mContext, false) * VIDEO_WINDOW_RATIO);
            mVideoContainerLayoutParams.addRule(RelativeLayout.BELOW, R.id.VideoPlayerTopBar);
            mVideoControllerContainer.setLayoutParams(mVideoContainerLayoutParams);
            if (mVideoPlayerContainerFrameLayout != null)
            {
                RelativeLayout.LayoutParams params = (RelativeLayout.LayoutParams) mVideoPlayerContainerFrameLayout.getLayoutParams();
                params.addRule(RelativeLayout.BELOW, R.id.AnchorView);
                params.width = ScreenUtils.getScreenSize(mContext, true);
                params.height = (int) (ScreenUtils.getScreenSize(mContext, false) * VIDEO_WINDOW_RATIO);
                mVideoPlayerContainerFrameLayout.setLayoutParams(params);
            }
        }
        else
        {
            if (mVideoFragment != null && mVideoFragment.getIsPlaying() == false)
            {
                setTopBarVisibility(true);
                setLandscapeRelatedContentContainerVisibility(true);
            }
            else
            {
                setTopBarVisibility(false);
                setLandscapeRelatedContentContainerVisibility(false);
            }
            mVideoContainerLayoutParams = new RelativeLayout.LayoutParams(0, 0);
            mVideoContainerLayoutParams.width = ScreenUtils.getScreenSize(mContext, true);
            mVideoContainerLayoutParams.height = ScreenUtils.getScreenSize(mContext, false);
            mVideoControllerContainer.setLayoutParams(mVideoContainerLayoutParams);
            if (mVideoPlayerContainerFrameLayout != null)
            {
                RelativeLayout.LayoutParams params = new RelativeLayout.LayoutParams(RelativeLayout.LayoutParams.MATCH_PARENT, RelativeLayout.LayoutParams.MATCH_PARENT);
                mVideoPlayerContainerFrameLayout.setLayoutParams(params);
            }
            if (mPlayAndPauseButton.getPlayAndPauseStatus() == PlayAndPauseButton.PLAY_AND_PAUSE_STATUS.ERROR || mPlayAndPauseButton.getPlayAndPauseStatus() == PlayAndPauseButton.PLAY_AND_PAUSE_STATUS.LOADING)
            {
                mRelatedContentLandscapeContainer.animateExit();
                mSeekBarController.animateExit();
            }
        }
    }

    public void closeView()
    {
        mSeekBarController.stopProgressUpdate();
        WindowManager windowManager = (WindowManager) mContext.getSystemService(Context.WINDOW_SERVICE);
        windowManager.removeView(mRootView);
    }

    public void setKidAvatar(String kidAvatarURL)
    {
        mRelatedContentPortraitContainer.setKidAvatarURL(kidAvatarURL);
    }

    public void setVideoName(String videoName)
    {
        mRelatedContentPortraitContainer.setVideoName(videoName);
    }

    public void setRelatedContent(ArrayList<ContentItem> content)
    {
        mRelatedContentPortraitContainer.setContent(content);
        mRelatedContentLandscapeContainer.setContent(content);
    }

    public void setRelatedContentItemListener(ContentPanelView.IOnContentItemClickListener listener)
    {
        mRelatedContentPortraitContainer.setRelatedContentListener(listener);
        mRelatedContentLandscapeContainer.setRelatedContentListener(listener);
    }

    public void setVideoPlayerListener(VideoPlayerListener videoPlayerListener)
    {
        mVideoPlayerListener = videoPlayerListener;
    }

    public void setVideoFragment(VideoFragment videoFragment, FrameLayout videoPlayerContainerFrameLayout)
    {
        mVideoPlayerContainerFrameLayout = videoPlayerContainerFrameLayout;
        mVideoFragment = videoFragment;
        mVideoFragment.setVideoPlayerListener(new VideoPlayerListener()
        {
            @Override public void onBuffering(boolean isBuffering)
            {
                SDKLogger.printErrorLog(TAG, ">>>>onBuffering = " + Boolean.toString(isBuffering));
            }

            @Override public void reloadRelatedContent()
            {

            }

            @Override public void onPaused()
            {
                SDKLogger.printErrorLog(TAG, ">>>>onPaused");

                mHandler.removeCallbacksAndMessages(null);
                mPlayAndPauseButton.setStatus(PlayAndPauseButton.PLAY_AND_PAUSE_STATUS.PAUSING);
                animateControlsEnter();
                mSeekBarController.stopProgressUpdate();
                if (mVideoPlayerListener != null)
                {
                    mVideoPlayerListener.onPaused();
                }
            }

            @Override public void onPlaying()
            {
                SDKLogger.printErrorLog(TAG, ">>>>onPlaying");
                mVideoPlayerContainerFrameLayout.setVisibility(View.VISIBLE);
                mRootView.postDelayed(new Runnable()
                {
                    @Override public void run()
                    {
                        mRootView.setBackgroundColor(Color.TRANSPARENT);
                    }
                }, 400);
                mIsUserPauseVideo = false;
                if (mContext != null && mContext instanceof Activity)
                {
                    ScreenUtils.setScreenOrientation((Activity) mContext, ActivityInfo.SCREEN_ORIENTATION_SENSOR);
                    ScreenUtils.hideNavigationAndStatusBar(((Activity) mContext).getWindow());
                }
                mVideoPlayerListener.reloadRelatedContent();
                mHandler.removeCallbacksAndMessages(null);
                mPlayAndPauseButton.setStatus(PlayAndPauseButton.PLAY_AND_PAUSE_STATUS.PLAYING);
                animateControlsExit();
                if (mVideoPlayerListener != null)
                {
                    mVideoPlayerListener.onPlaying();
                }
            }

            @Override public void onSeekTo(int newPosition)
            {

            }

            @Override public void onStopped()
            {
                SDKLogger.printErrorLog(TAG, ">>>>onStopped");
            }

            @Override public void onError(String reason)
            {
                SDKLogger.printErrorLog(TAG, ">>>>Can't load Video: " + reason);
                mHandler.removeCallbacksAndMessages(null);
                mPlayAndPauseButton.setStatus(PlayAndPauseButton.PLAY_AND_PAUSE_STATUS.ERROR);
                animateControlsEnter();
                mSeekBarController.stopProgressUpdate();
            }

            @Override public void onFinishedLoading()
            {
                SDKLogger.printErrorLog(TAG, ">>>>onFinishedLoading");

                mVideoFragment.playVideo();
                mSeekBarController.updateDuration(mVideoFragment.getVideoCurrentTime(), mVideoFragment.getVideoTotalTime());
            }

            @Override public void onLoading()
            {
                SDKLogger.printErrorLog(TAG, ">>>>onLoading");
                mHandler.removeCallbacksAndMessages(null);
                mPlayAndPauseButton.setStatus(PlayAndPauseButton.PLAY_AND_PAUSE_STATUS.LOADING);
                animateControlsEnter();
                mPlayAndPauseButton.animateLoading();
            }

            @Override public void onVideoEnded()
            {
                SDKLogger.printErrorLog(TAG, ">>>>onVideoEnded");

                mHandler.removeCallbacksAndMessages(null);
                mPlayAndPauseButton.setStatus(PlayAndPauseButton.PLAY_AND_PAUSE_STATUS.PAUSING);
                animateControlsEnter();
                mVideoPlayerListener.onVideoEnded();
            }

            @Override public void onVideoStarted()
            {
                SDKLogger.printErrorLog(TAG, ">>>>onVideoStarted");
                mVideoPlayerListener.onVideoStarted();
            }

            @Override public void onNext()
            {

            }

            @Override public void onPlaylistEnded()
            {

            }

            @Override public void onPrevious()
            {

            }

            @Override public void onBackPressed()
            {

            }

            @Override public void onLikeClicked()
            {

            }

            @Override public void onInfoClicked()
            {

            }

            @Override public void onRelatedPanelStateChanged(boolean isOpened)
            {

            }
        });
        mVideoFragment.setOnFullscreenListener(new YouTubePlayer.OnFullscreenListener()
        {
            @Override public void onFullscreen(boolean isFullScreen)
            {
                fixVideoLayout();
                mSeekBarController.refreshRotateOnClickStatus();
            }
        });

        fixVideoLayout();
    }

    public void setNumberOfLikes(String numberOfLikes)
    {
        if (mVideoPlayerTopBar != null)
        {
            mVideoPlayerTopBar.setNumberOfLikesTextView(numberOfLikes);
        }
    }

    public void setLikeIndication(boolean isLiked)
    {
        if (mVideoPlayerTopBar != null)
        {
            mVideoPlayerTopBar.setLikeIndicationStatus(isLiked);
        }
    }


    private void animateControlsEnter()
    {
        // 1. Animate top bar enter
        mVideoPlayerTopBar.animateEnter();

        // 2. Animate play and pause button enter
        mPlayAndPauseButton.animateEnter();

        // 3. Animate seek bar enter
        // 4. Animate related content container enter
        if (mPlayAndPauseButton.getPlayAndPauseStatus() != PlayAndPauseButton.PLAY_AND_PAUSE_STATUS.ERROR && mPlayAndPauseButton.getPlayAndPauseStatus() != PlayAndPauseButton.PLAY_AND_PAUSE_STATUS.LOADING)
        {
            mRelatedContentLandscapeContainer.animateEnter();
            mSeekBarController.animateEnter();
        }
        else
        {
            mRelatedContentLandscapeContainer.animateExit();
            mSeekBarController.animateExit();
        }

        // 5. Enable controls clicks
        mVideoControllerContainer.setIsInterceptTouchEvent(false);
    }

    private void animateControlsExit()
    {
        // 1. Animate top bar exit
        mVideoPlayerTopBar.animateExit();

        // 2. Animate play and pause button exit
        mPlayAndPauseButton.animateExit();

        // 3. Animate seek bar exit
        mSeekBarController.animateExit();

        // 4. Animate related content container exit
        mRelatedContentLandscapeContainer.animateExit();

        // 5. Disable controls clicks
        mVideoControllerContainer.setIsInterceptTouchEvent(true);
    }

    public void performClickIfNeeded(int x, int y)
    {
        mVideoPlayerTopBar.performClickIfNeeded(x, y);
    }
}
