package com.kidoz.sdk.api.players.video_player.adapters;

import android.animation.Animator;
import android.animation.ObjectAnimator;
import android.content.Context;
import android.content.res.Configuration;
import android.support.v4.view.ViewCompat;
import android.support.v7.widget.CardView;
import android.support.v7.widget.RecyclerView;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.view.animation.LinearInterpolator;
import android.widget.ImageView;
import android.widget.RelativeLayout;
import android.widget.TextView;

import com.kidoz.sdk.api.R;
import com.kidoz.sdk.api.players.video_player.infra_structure.BaseContentItem;
import com.kidoz.sdk.api.players.video_player.logics.RELATED_CONTENT_CONTAINER_TYPE;
import com.kidoz.sdk.api.players.video_player.logics.RelatedContentListener;
import com.kidoz.sdk.api.players.video_player.util.AnimationsUtils;
import com.kidoz.sdk.api.players.video_player.util.FontUtils;
import com.kidoz.sdk.api.players.video_player.util.ImageUtil;
import com.kidoz.sdk.api.players.video_player.util.ScreenUtils;
import com.squareup.imagelib.Picasso;

import java.util.ArrayList;


/**
 * Created by roni on 7/8/2015.
 */
public class RelatedContentAdapter extends RecyclerView.Adapter<RelatedContentAdapter.ViewHolder> implements View.OnClickListener
{
    private final String TAG = RelatedContentAdapter.class.getSimpleName();

    public static final float ITEM_RATIO_TABLET_LANDSCAPE = 0.65f;
    public static final float ITEM_RATIO_TABLET_PORTRAIT = 0.8f;
    public static final int ROWS_COUNT_TABLET_LANDSCAPE = 2;
    public static final int ROWS_COUNT_TABLET_PORTRAIT = 2;

    public static final float ITEM_RATIO_PHONE_LANDSCAPE = 0.8f;
    public static final float ITEM_RATIO_PHONE_PORTRAIT = 0.8f;
    public static final int ROWS_COUNT_PHONE_LANDSCAPE = 3;
    public static final int ROWS_COUNT_PHONE_PORTRAIT = 2;

    private Context mContext;
    private ArrayList<BaseContentItem> mContent;
    private int mLastPosition = -1;
    private LinearInterpolator mLinearInterpolator;
    private RelatedContentListener mRelatedContentListener;
    private ArrayList<BaseContentItem> mImpresseionsArrayList;
    private RELATED_CONTENT_CONTAINER_TYPE mContainerType;
    private int mContentTypeIconSize;

    private Animator[] getAnimators(View view)
    {
        ObjectAnimator alpha = ObjectAnimator.ofFloat(view, "alpha", 0, 1f);
        ObjectAnimator scaleX = ObjectAnimator.ofFloat(view, "scaleX", 0.5f, 1f);
        ObjectAnimator scaleY = ObjectAnimator.ofFloat(view, "scaleY", 0.5f, 1f);
        return new ObjectAnimator[]{scaleX, scaleY, alpha};
    }

    private void clearAnimation(View view)
    {
        ViewCompat.setAlpha(view, 1);
        ViewCompat.setScaleY(view, 1);
        ViewCompat.setScaleX(view, 1);
        ViewCompat.setTranslationY(view, 0);
        ViewCompat.setTranslationX(view, 0);
        ViewCompat.setRotation(view, 0);
        ViewCompat.setRotationY(view, 0);
        ViewCompat.setRotationX(view, 0);
        view.setPivotY(view.getMeasuredHeight() / 2);
        ViewCompat.setPivotX(view, view.getMeasuredWidth() / 2);
        ViewCompat.animate(view).setInterpolator(null);
    }

    public RelatedContentAdapter(Context context, RELATED_CONTENT_CONTAINER_TYPE containerType)
    {
        mContainerType = containerType;
        mContext = context;
        mContent = new ArrayList<BaseContentItem>();
        mImpresseionsArrayList = new ArrayList<BaseContentItem>();
        mLinearInterpolator = new LinearInterpolator();
    }

    public void setContent(ArrayList<BaseContentItem> content)
    {
        synchronized (this)
        {
            if (content != null)
            {
                mImpresseionsArrayList.clear();
                if (mContent.isEmpty() == true)
                {
                    mContent.clear();
                    mContent.addAll(content);
                    notifyItemRangeInserted(0, getItemCount());
                    notifyItemRangeChanged(0, getItemCount());
                }
                else
                {
                    mContent.clear();
                    mContent.addAll(content);
                    notifyDataSetChanged();
                }
            }
        }
    }

    public static class ViewHolder extends RecyclerView.ViewHolder
    {
        private CardView mRootView;
        private ImageView mThumbnailSimpleDraweeView;
        private ImageView mContentTypeSimpleDraweeView;
        private TextView mDescriptionTextView;

        public ViewHolder(View rootView)
        {
            super(rootView);
            mRootView = (CardView) rootView;
            mThumbnailSimpleDraweeView = (ImageView) mRootView.findViewById(R.id.ThumbnailSimpleDraweeView);
            mContentTypeSimpleDraweeView = (ImageView) mRootView.findViewById(R.id.ContentTypeIconSimpleDraweeView);
            mDescriptionTextView = (TextView) mRootView.findViewById(R.id.RelatedContentDescriptionTextView);
            mDescriptionTextView.setTypeface(FontUtils.getFont(mDescriptionTextView.getContext(), FontUtils.FONT_TYPE.DOSIS_BOLD.getValue()));
        }
    }

    @Override public int getItemCount()
    {
        int result = 0;
        if (mContent != null)
        {
            result = mContent.size();
        }
        return result;
    }

    @Override public ViewHolder onCreateViewHolder(ViewGroup parent, int arg1)
    {
        View convertView = LayoutInflater.from(parent.getContext()).inflate(R.layout.related_content_item_layout, parent, false);
        ViewHolder viewHolder = new ViewHolder(convertView);
        viewHolder.mRootView.setOnClickListener(this);
        viewHolder.mRootView.setTag(viewHolder);
        return viewHolder;
    }

    @Override public void onBindViewHolder(ViewHolder viewHolder, int position)
    {
        BaseContentItem baseContentItem = mContent.get(position);
        if (baseContentItem != null)
        {
            handleImpresseion(baseContentItem, position, mContainerType);
            fixItemSize(viewHolder);
            loadThumbnailAndSetBackgroundAndTextColor(viewHolder, baseContentItem);
            loadContentTypeIcon(viewHolder, baseContentItem);
            setItemDescription(viewHolder, baseContentItem);
        }
        if (position > mLastPosition)
        {
            for (Animator anim : getAnimators(viewHolder.mRootView))
            {
                anim.setInterpolator(mLinearInterpolator);
                anim.setDuration(300).start();
            }
            mLastPosition = position;
        }
        else
        {
            clearAnimation(viewHolder.mRootView);
        }
    }

    @Override public void onClick(final View view)
    {
        final ViewHolder viewHolder = (ViewHolder) view.getTag();
        final int position = getItemPosition(viewHolder);
        if (position != RecyclerView.NO_POSITION)
        {
            if (mContent.size() > position)
            {
                final BaseContentItem baseContentItem = mContent.get(position);
                AnimationsUtils.animateClick(view, new Animator.AnimatorListener()
                {
                    @Override public void onAnimationStart(Animator animation)
                    {

                    }

                    @Override public void onAnimationEnd(Animator animation)
                    {
                        if (mRelatedContentListener != null)
                        {
                            mRelatedContentListener.onRelatedContentItemClick(position);
                        }
                    }

                    @Override public void onAnimationCancel(Animator animation)
                    {

                    }

                    @Override public void onAnimationRepeat(Animator animation)
                    {

                    }
                });
            }
        }
    }

    private void handleImpresseion(BaseContentItem baseContentItem, int position, RELATED_CONTENT_CONTAINER_TYPE mContainerType)
    {
        if (mImpresseionsArrayList.contains(baseContentItem) == false)
        {
            if (mRelatedContentListener != null)
            {
                int orientation = mContext.getResources().getConfiguration().orientation;
                switch (mContainerType)
                {
                    case VIDEO_GRID:
                    {
                        if (orientation == Configuration.ORIENTATION_PORTRAIT)
                        {
                            mRelatedContentListener.onFirstTimeItemVisible(position, mContainerType);
                            mImpresseionsArrayList.add(baseContentItem);
                        }
                        break;
                    }
                    case VIDEO_HORIZONTAL_SCROLL:
                    {
                        if (orientation == Configuration.ORIENTATION_LANDSCAPE)
                        {
                            mRelatedContentListener.onFirstTimeItemVisible(position, mContainerType);
                            mImpresseionsArrayList.add(baseContentItem);
                        }
                        break;
                    }
                }
            }
        }
    }

    private int getItemPosition(ViewHolder viewHolder)
    {
        int result = viewHolder.getAdapterPosition();
        if (result == RecyclerView.NO_POSITION)
        {
            result = viewHolder.getPosition();
        }
        return result;
    }

    private void loadThumbnailAndSetBackgroundAndTextColor(final ViewHolder viewHolder, final BaseContentItem baseContentItem)
    {
        //        if (baseContentItem.getBackgroundColor() == Color.WHITE)
        //        {
        //            postprocessor = new BasePostprocessor()
        //            {
        //                @Override public String getName()
        //                {
        //                    return "palettePostprocessor";
        //                }
        //
        //                @Override public void process(Bitmap bitmap)
        //                {
        //                    if (bitmap != null && bitmap.isRecycled() == false && baseContentItem.getBackgroundColor() == Color.WHITE)
        //                    {
        //                        // Asynchronous
        //                        Bitmap newBitmap = Bitmap.createBitmap(bitmap);
        //                        Palette.from(newBitmap).generate(new Palette.PaletteAsyncListener()
        //                        {
        //                            public void onGenerated(Palette palette)
        //                            {
        //                                Palette.Swatch swatch = palette.getLightVibrantSwatch();
        //                                if (swatch != null)
        //                                {
        //                                    baseContentItem.setDescriptionColor(swatch.getBodyTextColor());
        //                                    viewHolder.mDescriptionTextView.setTextColor(baseContentItem.getDescriptionColor());
        //                                }
        //
        //                                int paletteColor = palette.getLightVibrantColor(Color.WHITE);
        //                                baseContentItem.setBackgroundColor(paletteColor);
        //                                viewHolder.mRootView.setCardBackgroundColor(baseContentItem.getBackgroundColor());
        //                            }
        //                        });
        //                    }
        //                    else
        //                    {
        //                        viewHolder.mDescriptionTextView.setTextColor(baseContentItem.getDescriptionColor());
        //                        viewHolder.mRootView.setCardBackgroundColor(baseContentItem.getBackgroundColor());
        //                    }
        //                }
        //            };
        //        }
        Picasso.with(mContext).load(baseContentItem.getThumbnailURL()).into(viewHolder.mThumbnailSimpleDraweeView);

        viewHolder.mDescriptionTextView.setTextColor(baseContentItem.getDescriptionColor());
        viewHolder.mRootView.setCardBackgroundColor(baseContentItem.getBackgroundColor());
    }

    private void loadContentTypeIcon(ViewHolder viewHolder, BaseContentItem baseContentItem)
    {
        viewHolder.mContentTypeSimpleDraweeView.setImageURI(ImageUtil.prepareCorrectUri(baseContentItem.getIconResourceID()));
    }

    private void setItemDescription(ViewHolder viewHolder, BaseContentItem baseContentItem)
    {
        if (viewHolder != null && baseContentItem != null)
        {
            viewHolder.mDescriptionTextView.setText(baseContentItem.getDescription());
        }
    }

    private void fixItemSize(ViewHolder viewHolder)
    {
        int numberOfItemsInARow;
        float itemRatio;
        viewHolder.mDescriptionTextView.setLines(2);
        viewHolder.mDescriptionTextView.setMaxLines(2);
        if (mContext.getResources().getConfiguration().orientation == Configuration.ORIENTATION_LANDSCAPE)
        {
            if (ScreenUtils.getIsTablet(mContext) == true)
            {
                numberOfItemsInARow = ROWS_COUNT_TABLET_LANDSCAPE;
                itemRatio = ITEM_RATIO_TABLET_LANDSCAPE;
            }
            else
            {
                viewHolder.mDescriptionTextView.setLines(1);
                viewHolder.mDescriptionTextView.setMaxLines(1);
                numberOfItemsInARow = ROWS_COUNT_PHONE_LANDSCAPE;
                itemRatio = ITEM_RATIO_PHONE_LANDSCAPE;
            }
        }
        else
        {
            if (ScreenUtils.getIsTablet(mContext) == true)
            {
                numberOfItemsInARow = ROWS_COUNT_TABLET_PORTRAIT;
                itemRatio = ITEM_RATIO_TABLET_PORTRAIT;
            }
            else
            {
                numberOfItemsInARow = ROWS_COUNT_PHONE_PORTRAIT;
                itemRatio = ITEM_RATIO_PHONE_PORTRAIT;
            }
        }
        int screenWidth = Math.min(ScreenUtils.getScreenSize(mContext, true), ScreenUtils.getScreenSize(mContext, false));
        int itemWidth = screenWidth / numberOfItemsInARow;
        int itemHeight = (int) (itemWidth * itemRatio);

        RecyclerView.LayoutParams rootViewParams = (RecyclerView.LayoutParams) viewHolder.mRootView.getLayoutParams();
        rootViewParams.width = itemWidth;
        rootViewParams.height = itemHeight;
        viewHolder.mRootView.setLayoutParams(rootViewParams);

        mContentTypeIconSize = (int) (itemWidth * 0.2);
        RelativeLayout.LayoutParams contentTypeIconParams = (RelativeLayout.LayoutParams) viewHolder.mContentTypeSimpleDraweeView.getLayoutParams();
        contentTypeIconParams.width = mContentTypeIconSize;
        contentTypeIconParams.height = mContentTypeIconSize;
        contentTypeIconParams.rightMargin = (int) (mContentTypeIconSize * 0.25);
        contentTypeIconParams.bottomMargin = (int) (mContentTypeIconSize * 0.6) * -1;
        viewHolder.mContentTypeSimpleDraweeView.setLayoutParams(contentTypeIconParams);
    }

    public void setRelatedContentListener(RelatedContentListener relatedContentListener)
    {
        mRelatedContentListener = relatedContentListener;
    }

}
