package com.kidoz.sdk.api.analytics;

import android.app.AlarmManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageInfo;
import android.content.res.Configuration;
import android.net.ConnectivityManager;
import android.os.Build;
import android.provider.Settings.Secure;
import android.util.DisplayMetrics;
import android.view.WindowManager;

import com.google.android.gms.analytics.HitBuilders.EventBuilder;
import com.google.android.gms.analytics.Tracker;
import com.kidoz.sdk.api.general.utils.SDKLogger;
import com.kidoz.sdk.api.general.utils.Utils;
import com.kidoz.sdk.api.players.video_player.util.ScreenUtils;
import com.kidoz.sdk.api.server_connect.BaseConnectionClient;

import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Locale;
import java.util.Random;
import java.util.zip.CRC32;

public class KidozEventManager
{
    private static final String TAG = KidozEventManager.class.getSimpleName();
    public static final String SYNC_ALARM_INTENT = "kidoz.sync.alarm.event.action";
    public static final String STOP_SERVICE_EXTRA = "STOP_SERVICE_EXTRA";
    public static final int SYNC_ALARM_INTENT_REQUEST_CODE = 1982;

    private static final int SYNC_ALARM_INTERVAL_IN_MILLISECONDS = 1 * 5 * 60 * 1000;
    private static final int NIGHT_TIME_HOUR_VALUE_START = 21;
    private static final int NIGHT_TIME_HOUR_VALUE_END = 8;
    private static final int DAY_TIME_AMOUNT = 10;
    private static final int NIGHT_TIME_AMOUNT_EXTRA = 50;
    private static final int THREE_G_CONNECTON_TYPE_AMOUNT = 10;
    private static final int WIFI_CONNECTON_TYPE_AMOUNT_EXTRA = 20;
    private static final int SCREEN_STATE_ON_AMOUNT = 10;
    private static final int SCREEN_STATE_OFF_AMOUNT_EXTRA = 20;
    private static final String OS_TYPE = "Android";

    private static EventLogDatabaseManager sDatabaseManager;
    private static Utils.StaticHandler sStaticHandler;
    private static ArrayList<TimedEventParams> sTimedEventsArrayList;
    private static ArrayList<EventBuilder> sTimedEventBuilderArrayList;

    private static boolean isDisableEventLogging = false;

    public static void setDisableLoggingEvents(boolean disable)
    {
        isDisableEventLogging = disable;
    }

    private KidozEventManager()
    {
    }

    public static void initKidozEventManager(Context context)
    {
        if (sTimedEventsArrayList == null)
        {
            sTimedEventsArrayList = new ArrayList<TimedEventParams>();
        }
        if (sTimedEventBuilderArrayList == null)
        {
            sTimedEventBuilderArrayList = new ArrayList<EventBuilder>();
        }

        if (sDatabaseManager == null)
        {
            sDatabaseManager = new EventLogDatabaseManager(context);
        }

        if (sStaticHandler == null)
        {
            sStaticHandler = new Utils.StaticHandler();
        }
    }

    /**
     * This method randomly set the next sync alarm fire time.
     *
     * @param context
     */
    public static void setSyncAlarm(Context context)
    {
        AlarmManager alarmManager = (AlarmManager) context.getSystemService(Context.ALARM_SERVICE);
        Intent intent = new Intent(context, SyncReciever.class);
        intent.setAction(SYNC_ALARM_INTENT);
        PendingIntent pendingIntent = PendingIntent.getBroadcast(context, SYNC_ALARM_INTENT_REQUEST_CODE, intent, PendingIntent.FLAG_UPDATE_CURRENT);

        // Randomly trigger the next alarm. the minimum trigger time can be 3
        // hours from now, the maximum can be 6.
        Random random = new Random();
        long currentTimeInMilliseconds = Calendar.getInstance().getTimeInMillis();
        long triggerAtMillis = currentTimeInMilliseconds + random.nextInt(SYNC_ALARM_INTERVAL_IN_MILLISECONDS) + SYNC_ALARM_INTERVAL_IN_MILLISECONDS;
        alarmManager.set(AlarmManager.RTC_WAKEUP, triggerAtMillis, pendingIntent);
    }

    /**
     * This method cancel an active alarm if exist.
     *
     * @param context
     */
    public static void cancelSyncAlarm(Context context)
    {
        Intent intent = new Intent(context, SyncReciever.class);
        PendingIntent pendingIntent = PendingIntent.getBroadcast(context, SYNC_ALARM_INTENT_REQUEST_CODE, intent, PendingIntent.FLAG_NO_CREATE);
        AlarmManager alarmManager = (AlarmManager) context.getSystemService(Context.ALARM_SERVICE);
        if (intent != null)
        {
            alarmManager.cancel(pendingIntent);
        }
    }

    /**
     * This Method store the timed event for future "endTimedEvent". The event
     * isn't saved to DB until "endTimedEvent" will be called. specialKey should
     * be a combination of the relevant fragment and the event action so it will
     * be unique.
     *
     * @param context
     * @param kidozEvent
     */
    public static void logTimedEventView(final Context context, final KidozEvent kidozEvent, EventBuilder eventBuilder)
    {
        initKidozEventManager(context);

        String callingScreen = endTimedEventAndGetLastScreenLabel(context);
        if (callingScreen != null)// kidozEvent.getJSONObject().has(LogParameters.CALLING_SCREEN)
        // == false &&
        {
            kidozEvent.addParameterToJsonObject(LogParameters.CALLING_SCREEN, callingScreen);
           // eventBuilder = eventBuilder.setCustomDimension(GoogleAnalyticsTracker.CALLING_SCREEN, callingScreen);
        }
        TimedEventParams timedEventParams = new TimedEventParams();
        timedEventParams.setKidozEvent(kidozEvent);
        timedEventParams.setEventStartTimeInMilliseconds(System.currentTimeMillis());
        sTimedEventsArrayList.add(timedEventParams);
        //sTimedEventBuilderArrayList.add(eventBuilder);
    }

    /**
     * This method log a timed event and return this screen as last screen
     * value.
     *
     * @param context
     */
    public static String endTimedEventAndGetLastScreenLabel(Context context)
    {
        String result = null;
        String duration = null;
        if (sTimedEventsArrayList != null && sTimedEventsArrayList.isEmpty() == false)
        {
            TimedEventParams timedEventParams = sTimedEventsArrayList.get(sTimedEventsArrayList.size() - 1);
            duration = timedEventParams.getDuration();
            timedEventParams.calculateAndAddEventDuration();
            logEvent(context, timedEventParams.getKidozEvent());
            sTimedEventsArrayList.remove(sTimedEventsArrayList.size() - 1);

            try
            {
                KidozEvent kidozEvent = timedEventParams.getKidozEvent();
                result = kidozEvent.getJSONObject().getString(LogParameters.ACTION);
            }
            catch (Exception ex)
            {
                SDKLogger.printErrorLog(TAG, "Error when trying to get last screen value: " + ex.getMessage());
            }
        }
        if (sTimedEventBuilderArrayList != null && sTimedEventBuilderArrayList.isEmpty() == false)
        {
            EventBuilder eventBuilder = sTimedEventBuilderArrayList.get(sTimedEventBuilderArrayList.size() - 1);
            if (eventBuilder != null)
            {
                eventBuilder = eventBuilder.setCustomDimension(GoogleAnalyticsTracker.DURATION, duration);
                Tracker tracker = GoogleAnalyticsTracker.getGoogleAnalyticsTracker(context).getTracker();
                if (tracker != null)
                {
                    tracker.send(eventBuilder.build());
                }
            }

            sTimedEventBuilderArrayList.remove(sTimedEventBuilderArrayList.size() - 1);
        }

        return result;
    }

    /**
     * This method return JsonArray as string. the first object will be the
     * device params and then each JSONObject in a given arraylist (of events).
     *
     * @param context
     * @param kidozEvent
     */
    public static void logEvent(final Context context, final KidozEvent kidozEvent)
    {
        initKidozEventManager(context);

        if (isDisableEventLogging == false)
        {
            sStaticHandler.post(new Runnable()
            {
                @Override public void run()
                {
                    new Thread(new Runnable()
                    {
                        @Override public void run()
                        {
                            SDKLogger.printInfoLog(TAG, "*** KidozEventManager ***");
                            SDKLogger.printInfoLog(TAG, ">>Insert New Log:");
                            kidozEvent.printData();
                            sDatabaseManager.getKidozEventsTable().inserLogEventToDB(kidozEvent);
                        }
                    }).start();
                }
            });
        }
    }

    private static JSONObject getDeviceParamsJsonObject(Context context)
    {
        JSONObject result = new JSONObject();
        try
        {
            String deviceID = Secure.getString(context.getContentResolver(), Secure.ANDROID_ID);
            CRC32 crc = new CRC32();
            crc.update(deviceID.getBytes());
            // AppLogger.printDebbugLog(TAG, ">>>>>ANDROID_ID = " + deviceID);
            // AppLogger.printDebbugLog(TAG, ">>>>>ANDROID_ID size = " +
            // deviceID.length());
            // AppLogger.printDebbugLog(TAG, ">>>>>ANDROID_ID by CRC = " +
            // String.valueOf(crc.getValue()));
            // AppLogger.printDebbugLog(TAG, ">>>>>ANDROID_ID by CRC length = "
            // + String.valueOf(crc.getValue()).length());
            result.put(LogParameters.DEVICE_HASH, String.valueOf(crc.getValue()));
            result.put(LogParameters.USER_REFFERAL, Utils.getDeviceReferral(context));
            result.put(LogParameters.KIDOZ_APP_PACKAGE_NAME, context.getPackageName());
            PackageInfo pInfo = context.getPackageManager().getPackageInfo(context.getPackageName(), 0);
            result.put(LogParameters.APP_VERSION, String.valueOf(pInfo.versionName));
            result.put(LogParameters.DEVICE_TYPE, String.valueOf(Build.DEVICE));
            result.put(LogParameters.DEVICE_BRAND, String.valueOf(Build.MANUFACTURER));
            result.put(LogParameters.DEVICE_MODEL, String.valueOf(Build.MODEL));
            result.put(LogParameters.SCREEN_TYPE, String.valueOf((context.getResources().getConfiguration().screenLayout & Configuration.SCREENLAYOUT_SIZE_MASK)));
            result.put(LogParameters.SCREEN_SIZE, String.valueOf(ScreenUtils.getDeviceScreenSizeInInches(context)));
            WindowManager windowManager = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
            DisplayMetrics metrics = new DisplayMetrics();
            windowManager.getDefaultDisplay().getMetrics(metrics);
            result.put(LogParameters.DPI_FACTOR, String.valueOf(metrics.densityDpi));
            result.put(LogParameters.SCREEN_WIDTH, String.valueOf(Utils.getScreenSize(context, true)));
            result.put(LogParameters.SCREEN_HEIGHT, String.valueOf(Utils.getScreenSize(context, false)));
            result.put(LogParameters.OS_TYPE, String.valueOf(OS_TYPE));
            result.put(LogParameters.OS_VERSION, String.valueOf(Build.VERSION.SDK_INT));
            result.put(LogParameters.CONNECTION_TYPE, LogParameters.convertConnectionTypeToString(BaseConnectionClient.getConnectionType(context)));
            result.put(LogParameters.DEVICE_LANGUAGE, String.valueOf(Locale.getDefault().getLanguage()));
            result.put(LogParameters.TIME_ZONE, LogParameters.getCurrentTimezoneOffset());
            result.put(LogParameters.COUNTRY, String.valueOf(Locale.getDefault().getCountry()));

            // result.put("DeviceProduct", String.valueOf(Build.PRODUCT));
            // TelephonyManager telephonyManager = ((TelephonyManager)
            // context.getSystemService(Context.TELEPHONY_SERVICE));
            // result.put("NetworkOperatorName",
            // String.valueOf(telephonyManager.getNetworkOperatorName()));
            // result.put("DeviceId",
            // String.valueOf(telephonyManager.getDeviceId()));
            // result.put("NetworkOperator",
            // String.valueOf(telephonyManager.getNetworkOperator()));
            // result.put("NetworkCountryIso",
            // String.valueOf(telephonyManager.getNetworkCountryIso()));
            // result.put("SimOperator",
            // String.valueOf(telephonyManager.getSimOperator()));
            // result.put("SimOperatorName",
            // String.valueOf(telephonyManager.getSimOperatorName()));
            // result.put("SimSerialNumber",
            // String.valueOf(telephonyManager.getSimSerialNumber()));
            // result.put("NumberOfCameras",
            // String.valueOf(Camera.getNumberOfCameras()));
        }
        catch (Exception ex)
        {
            SDKLogger.printErrorLog(TAG, "Error when trying to create device params: " + ex.getMessage());
        }
        return result;
    }

    /**
     * This method return wherever a log can be sent. A log can be sent if the
     * following rules are valid: 1. Is connected to the internet? 2. If Kidoz
     * is not running or if it does the screen is off. 3. If there is still
     * events waiting in DB.
     *
     * @return is can send log.
     */
    public static boolean getIsCanSendLog(Context context)
    {
        initKidozEventManager(context);
        boolean result = false;
        // 1. Check if network connection available?
        if (BaseConnectionClient.isNetworkAvailable(context) == true)
        {
            // 2. Check if there is still events waiting in DB?
            if (sDatabaseManager.getKidozEventsTable().isDBempty() == false)
            {
                result = true;
            }
        }
        return result;
    }

    /**
     * This method simply start the SyncEventService.
     *
     * @param context
     */
    public static void startSync(Context context)
    {
        Intent intent = new Intent(context, SyncEventService.class);
        context.startService(intent);
    }

    /**
     * This method create a sync service intent and return it. * IMPORTANTE-
     * This method should only be called from SyncReciever onReceive because
     * this intent will hold a wake lock.
     *
     * @param context
     * @return
     */
    public static Intent getSyncSericeIntent(Context context)
    {
        return new Intent(context, SyncEventService.class);
    }

    /**
     * This method simply stop the SyncEventService by adding extra flag.
     *
     * @param context
     */
    public static void stopSync(Context context)
    {
        Intent intent = new Intent(context, SyncEventService.class);
        intent.putExtra(STOP_SERVICE_EXTRA, true);
        context.stopService(intent);
    }

    public static EventsBulk getNextBulk(Context context)
    {
        initKidozEventManager(context);
        EventsBulk result = new EventsBulk();
        result.setDeviceParamsJSONObject(getDeviceParamsJsonObject(context));
        result.setKidozEventArrayList(sDatabaseManager.getKidozEventsTable().getKidozEvents(calculateBulkSize(context)));
        return result;
    }

    public static void removeBulkFromDB(Context context, EventsBulk eventsBulk)
    {
        initKidozEventManager(context);
        sDatabaseManager.getKidozEventsTable().removeLogEventsFromDB(eventsBulk.getKidozEventArrayList());
    }

    private static int calculateBulkSize(Context context)
    {
        // The minimum bulk is 30, the maximum is 120.
        int result = DAY_TIME_AMOUNT + THREE_G_CONNECTON_TYPE_AMOUNT + SCREEN_STATE_ON_AMOUNT;
        Calendar calendar = Calendar.getInstance();
        int currentHour = calendar.get(Calendar.HOUR_OF_DAY);
        if (currentHour >= NIGHT_TIME_HOUR_VALUE_START || currentHour <= NIGHT_TIME_HOUR_VALUE_END)
        {
            result += NIGHT_TIME_AMOUNT_EXTRA;
        }

        if (BaseConnectionClient.getConnectionType(context) == ConnectivityManager.TYPE_WIFI)
        {
            result += WIFI_CONNECTON_TYPE_AMOUNT_EXTRA;
        }

        if (ScreenUtils.getIsScreenOff(context) == true)
        {
            result += SCREEN_STATE_OFF_AMOUNT_EXTRA;
        }

        return result;
    }

    public static void resetEvents()
    {
        if (sTimedEventsArrayList != null)
        {
            sTimedEventsArrayList.clear();
            sTimedEventsArrayList = null;
            sTimedEventBuilderArrayList.clear();
            sTimedEventBuilderArrayList = null;
        }
    }
}
