package com.kidoz.sdk.api;

import android.animation.Animator;
import android.animation.Animator.AnimatorListener;
import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Color;
import android.graphics.Point;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.util.AttributeSet;
import android.view.Gravity;
import android.view.View;
import android.view.animation.AccelerateInterpolator;
import android.view.animation.DecelerateInterpolator;
import android.widget.Button;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;

import com.kidoz.sdk.api.general.animations.GenAnimator;
import com.kidoz.sdk.api.general.enums.UiStyleType;
import com.kidoz.sdk.api.general.utils.Utils;
import com.kidoz.sdk.api.structure.ContentData;
import com.kidoz.sdk.api.ui_views.ContentHorizontalPanelView;

public class SliderPanelView extends RelativeLayout {

    private static final String TAG = SliderPanelView.class.getSimpleName();

    private static final float CONTENT_PANEL_DEFAULT_RATION = 0.25f;

    private Utils.StaticHandler mStaticHandler = new com.kidoz.sdk.api.general.utils.Utils.StaticHandler();

    private RelativeLayout mRootView;
    private RelativeLayout mHandleVisibleContainer;
    private LinearLayout mContentView;
    private LinearLayout mHandleView;

    private boolean mPanelShown = false;
    private boolean mIsAnimating = false;
    private boolean mIsHandleVisible = true;

    private ContentHorizontalPanelView mContentHorizontalPanelView;
    private ContentLogicLoader mContentLogicLoader;

    /**
     * Handle horizontal gravity , Allowed values <Code> Gravity.LEFT | Gravity.RIGHT </Code>
     */
    private int mHandleGravity = Gravity.LEFT;

    /**
     * Content container background color
     */
    private int mContentBackgroundColor = Color.TRANSPARENT;

    /**
     * Content container background drawable resource
     */
    private int mContentBackgroundResource = -1;

    /**
     * Content view container height
     */
    private int mContentViewHeight = -1;

    /**
     * HandleView ResourceId
     */
    private int mHandleViewResourceId = -1;

    public SliderPanelView(Context context, AttributeSet attrs) {
        super(context, attrs);
        initPanelView(attrs);
    }

    public SliderPanelView(Context context) {
        super(context);
        initPanelView(null);
    }

    public SliderPanelView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        initPanelView(attrs);
    }

    public void initPanelView(AttributeSet attrs) {

        // Check and set xml attributes
        checkAndSetAttributesIfExists(attrs);

        // Initiate content loader manager
        initContentLogicLoader();

        // Create view structure
        createSliderViewStructure();

        // Init created views
        mContentHorizontalPanelView = new ContentHorizontalPanelView(getContext());
        mContentView.addView(mContentHorizontalPanelView, new RelativeLayout.LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT));

        // Apply attributes and init values
        applyAttributesAndInitValues();

        Utils.setOnGlobalLayoutFinishListener(getRootView(), new com.kidoz.sdk.api.general.utils.Utils.OnGlobalLayoutFinishedListener() {
            @Override
            public void onLayoutFinished() {
                GenAnimator.playSlideDownAnimation(mRootView, mContentViewHeight, 0, null, null);
                mPanelShown = false;
            }
        });

        LayoutParams params = new LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT);
        setLayoutParams(params);
    }

    /**
     * Check and set xml attributes if exists and the view is added athrue xml editor
     */
    private void checkAndSetAttributesIfExists(AttributeSet attrs) {
        if (isInEditMode() || attrs == null) {
            return;
        }

        if (attrs != null) {

            TypedArray ta = getContext().obtainStyledAttributes(attrs, R.styleable.SlidePanelView);

            if (ta != null) {
                mContentViewHeight = ta.getDimensionPixelSize(R.styleable.SlidePanelView_panelHeight, -1);
                mContentBackgroundColor = ta.getInt(R.styleable.SlidePanelView_panelBackgroundColor, -1);
                mContentBackgroundResource = ta.getResourceId(R.styleable.SlidePanelView_panelBackgroundResource, -1);
                mHandleViewResourceId = ta.getResourceId(R.styleable.SlidePanelView_panelHandleView, -1);

                String gavityStr = ta.getString(R.styleable.SlidePanelView_panelHandleGravity);
                if (gavityStr != null) {
                    if (gavityStr.equals("left")) {
                        mHandleGravity = Gravity.LEFT;
                    } else if (gavityStr.equals("right")) {
                        mHandleGravity = Gravity.RIGHT;
                    }
                }
            }
            ta.recycle();
        }

        final float density = getContext().getResources().getDisplayMetrics().density;
        if (mContentViewHeight != -1) {
            mContentViewHeight = (int) (mContentViewHeight * density + 0.5f);
        }
    }

    /**
     * apply attributes and init values
     */
    private void applyAttributesAndInitValues() {

        if (mContentBackgroundColor != -1) {
            setSliderContentBackgroundColor(mContentBackgroundColor);
        }

        if (mHandleGravity != -1) {
            setGravity(mHandleGravity);
        }

        if (mContentBackgroundResource != -1) {
            setSliderContentBackgroundResource(mContentBackgroundResource);
        }

        if(mHandleViewResourceId != -1) {
            setCustomHandleView(findViewById(mHandleViewResourceId),mHandleGravity);
        }else {
            // TODO Add default handle view

            //Apply ettributes
            // TODO REMOVE THIS CODE LATER
            Button button = new Button(getContext());
            button.setText("TAP");
            setCustomHandleView(button,mHandleGravity);
            initTapHandleClickListener(button);
            // TODO REMOVE THIS CODE LATER
        }
    }

    /**
     * Create slider view structure
     */
    private void createSliderViewStructure() {

        Point point = Utils.getScreenSize(getContext());
        int size = Math.max(point.x, point.y);

        if (mContentViewHeight == -1) { // If noy attibute was set apply the default value
            mContentViewHeight = (int) (size * CONTENT_PANEL_DEFAULT_RATION);
        }

        // Initi RootView
        mRootView = new RelativeLayout(getContext());

        //Create content container
        mContentView = new LinearLayout(getContext());
        mContentView.setId(com.kidoz.sdk.api.general.utils.Utils.generateViewId());
        RelativeLayout.LayoutParams params = new RelativeLayout.LayoutParams(LayoutParams.MATCH_PARENT, mContentViewHeight);
        params.addRule(RelativeLayout.ALIGN_PARENT_BOTTOM);
        mRootView.addView(mContentView, params);
        mContentView.setVisibility(View.INVISIBLE);


        // Create handle Container
        mHandleVisibleContainer = new RelativeLayout(getContext());
        mHandleVisibleContainer.setId(com.kidoz.sdk.api.general.utils.Utils.generateViewId());
        params = new RelativeLayout.LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.WRAP_CONTENT);
        params.addRule(RelativeLayout.ABOVE, mContentView.getId());
        mRootView.addView(mHandleVisibleContainer, params);

        // Create handle
        mHandleView = new LinearLayout(getContext());
        params = new RelativeLayout.LayoutParams(LayoutParams.WRAP_CONTENT, LayoutParams.WRAP_CONTENT);
        if (mHandleGravity == Gravity.LEFT) {
            params.addRule(RelativeLayout.ALIGN_PARENT_LEFT);
        } else {
            params.addRule(RelativeLayout.ALIGN_PARENT_RIGHT);
        }
        params.addRule(RelativeLayout.CENTER_VERTICAL);
        mHandleVisibleContainer.addView(mHandleView, params);

        // Set root view
        params = new RelativeLayout.LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.WRAP_CONTENT);
        params.addRule(RelativeLayout.ALIGN_PARENT_BOTTOM);
        addView(mRootView, params);
    }

    /**
     * Initiate handle and content views
     */
    private void initTapHandleClickListener(View view) {

        view.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View v) {
                if (mIsAnimating == false) {
                    if (mPanelShown) {
                        collapseView();
                    } else {
                        loadContent();
                        expandView();
                    }
                }
            }
        });
    }

    /**
     * Initiate content loader
     */
    private void initContentLogicLoader() {
        mContentLogicLoader = new ContentLogicLoader(new ContentLogicLoader.IOnContentDataReadyCallback() {
            @Override
            public void onDataReady(ContentData contentData) {
                if (contentData.isHasContentToShow()) {
                    mContentHorizontalPanelView.setContent(contentData.getContentDataItems());
                }
            }
        });
    }

    /**
     * Load recommended content
     */
    private void loadContent() {
        if (mContentHorizontalPanelView != null && mContentHorizontalPanelView.isEmpty() == true) {
            mContentLogicLoader.loadContent(getContext(), UiStyleType.RELATED_PANEL_STYLE);
        }
    }

    /**
     * Expand slider view
     */
    public void expandView() {
        if (mContentViewHeight != 0 && mPanelShown == false) {
            GenAnimator.playSlideUpAnimation(mRootView, mContentViewHeight, 240, new DecelerateInterpolator(), new AnimatorListener() {
                @Override
                public void onAnimationStart(Animator arg0) {
                    mIsAnimating = true;
                    mContentView.setVisibility(View.VISIBLE);
                }

                @Override
                public void onAnimationRepeat(Animator arg0) {
                }

                @Override
                public void onAnimationEnd(Animator arg0) {
                    mPanelShown = true;
                    mIsAnimating = false;
                }

                @Override
                public void onAnimationCancel(Animator arg0) {
                }
            });
        }
    }

    /**
     * Collapse slider view
     */
    private void collapseView() {
        if (mContentViewHeight != 0 && mPanelShown == true) {
            GenAnimator.playSlideDownAnimation(mRootView, mContentViewHeight, 210, new AccelerateInterpolator(), new AnimatorListener() {
                @Override
                public void onAnimationStart(Animator arg0) {
                    mIsAnimating = true;
                }

                @Override
                public void onAnimationRepeat(Animator arg0) {
                }

                @Override
                public void onAnimationEnd(Animator arg0) {
                    mPanelShown = false;
                    mIsAnimating = false;
                    mContentView.setVisibility(View.INVISIBLE);
                }

                @Override
                public void onAnimationCancel(Animator arg0) {
                }
            });
        }
    }


    /**
     * Hide handle view
     */
    public void hideHandleView() {
        if (mIsHandleVisible == true) {
            if (mHandleGravity == Gravity.LEFT) {
                GenAnimator.playSlideOutLeftAnimation(mHandleVisibleContainer, new AnimatorListener() {
                    @Override
                    public void onAnimationStart(Animator arg0) {
                    }

                    @Override
                    public void onAnimationRepeat(Animator arg0) {
                    }

                    @Override
                    public void onAnimationEnd(Animator arg0) {
                        mHandleView.setVisibility(View.INVISIBLE);
                        mIsHandleVisible = false;
                    }

                    @Override
                    public void onAnimationCancel(Animator arg0) {
                    }
                }, 500);
            }
            if (mHandleGravity == Gravity.RIGHT) {
                GenAnimator.playSlideOutRightAnimation(mHandleVisibleContainer, new AnimatorListener() {
                    @Override
                    public void onAnimationStart(Animator arg0) {
                    }

                    @Override
                    public void onAnimationRepeat(Animator arg0) {
                    }

                    @Override
                    public void onAnimationEnd(Animator arg0) {
                        mHandleView.setVisibility(View.INVISIBLE);
                        mIsHandleVisible = false;
                    }

                    @Override
                    public void onAnimationCancel(Animator arg0) {
                    }
                }, 500);
            }
        }
    }

    /**
     * Show handle view
     */
    public void showHandleView() {
        if (mIsHandleVisible == false) {
            if (mHandleGravity == Gravity.LEFT) {
                GenAnimator.playBounceInRightAnimation(mHandleVisibleContainer, new AnimatorListener() {
                    @Override
                    public void onAnimationStart(Animator arg0) {
                        mHandleView.setVisibility(View.VISIBLE);
                    }

                    @Override
                    public void onAnimationRepeat(Animator arg0) {
                    }

                    @Override
                    public void onAnimationEnd(Animator arg0) {
                        mIsHandleVisible = true;
                    }

                    @Override
                    public void onAnimationCancel(Animator arg0) {
                    }
                }, 500);
            } else if (mHandleGravity == Gravity.LEFT) {
                GenAnimator.playBounceInLeftAnimation(mHandleVisibleContainer, new AnimatorListener() {
                    @Override
                    public void onAnimationStart(Animator arg0) {
                    }

                    @Override
                    public void onAnimationRepeat(Animator arg0) {
                    }

                    @Override
                    public void onAnimationEnd(Animator arg0) {
                        mHandleView.setVisibility(View.VISIBLE);
                        mIsHandleVisible = true;
                    }

                    @Override
                    public void onAnimationCancel(Animator arg0) {
                    }
                }, 500);
            }
        }
    }


    /**
     * Show handle view
     *
     * @param view    custom handle view
     * @param gravity horizontal align gravity (Gravity.LEFT | Gravity.RIGHT)
     */
    public void setCustomHandleView(View view, int gravity) {
        if (view != null) {
            mHandleView.removeAllViews();
            mHandleView.addView(view);

            RelativeLayout.LayoutParams params = new RelativeLayout.LayoutParams(LayoutParams.WRAP_CONTENT, LayoutParams.WRAP_CONTENT);
            if (gravity == Gravity.RIGHT) {
                params.addRule(RelativeLayout.ALIGN_PARENT_RIGHT);
            } else {
                params.addRule(RelativeLayout.ALIGN_PARENT_LEFT);
            }
            params.addRule(RelativeLayout.ALIGN_BOTTOM);
            mHandleView.setLayoutParams(params);
            initTapHandleClickListener(view);
        }
    }

    /**
     * Set slider content background color
     *
     * @param color new backgound color
     */
    public void setSliderContentBackgroundColor(int color) {
        if (mContentView != null) {
            mContentView.setBackgroundColor(color);
        }
    }

    /**
     * Set slider content background drawable
     *
     * @param drawable new background drawable
     */
    public void setSliderContentBackgroundDrawable(Drawable drawable) {
        if (mContentView != null) {
            if (Build.VERSION.SDK_INT < Build.VERSION_CODES.JELLY_BEAN) {
                mContentView.setBackgroundDrawable(drawable);
            } else {
                mContentView.setBackground(drawable);
            }
        }
    }


    /**
     * Set slider content background resource
     *
     * @param resource background resource
     */
    public void setSliderContentBackgroundResource(int resource) {
        if (mContentView != null) {
            mContentView.setBackgroundResource(resource);
        }
    }
}
