package com.kidoz.sdk.api;


import android.content.Context;
import android.util.Log;
import android.view.WindowManager;

import com.kidoz.events.DeviceUtils;
import com.kidoz.events.EventManager;
import com.kidoz.events.EventParameters;
import com.kidoz.sdk.api.analytics.GoogleAnalyticsTracker;
import com.kidoz.sdk.api.general.utils.SDKLogger;
import com.kidoz.sdk.api.general.utils.SharedPreferencesUtils;
import com.kidoz.sdk.api.server_connect.ApiResultCallback;
import com.kidoz.sdk.api.server_connect.ResultData;
import com.kidoz.sdk.api.server_connect.SdkAPIManager;
import com.kidoz.sdk.api.structure.CreativeToolsData;
import com.kidoz.sdk.api.structure.SDKConfigurationData;

import org.json.JSONObject;

public class KidozSDK
{
    private static final String TAG = KidozSDK.class.getSimpleName();
    public static final String SERVER_CONNECTION_URL = "https://kidoz.net";
    private static SdkAPIManager sdkAPIManager;
    private static CreativeToolsData sCreativeToolsData;

    private static boolean isInitialized = false;
    private static String mPublicherKey = null;
    private static String mSecurityKey = null;
    private static GoogleAnalyticsTracker mGoogleAnalyticsTracker;

    /**
     * Initialize Kidoz SDK
     *
     * @param context      activity context
     * @param publicher_id valid puclisher id
     * @param token        valid security token
     * @throws RuntimeException in case of invalid or missing publisher_id or security token
     */
    public static void initialize(Context context, String publicher_id, String token)
    {
        try
        {
            DeviceUtils.getGoogleAdvertisingID(context);
            SharedPreferencesUtils.saveSharedPreferencesData(context, "PUBLISHER_ID", publicher_id);
            JSONObject eventParams = new JSONObject();
            eventParams.put(EventParameters.DEVELOPER_ID, publicher_id);
            eventParams.put(EventManager.LOG_LEVEL_KEY, EventManager.LOG_NORMAL_LEVEL);
            EventManager.getInstance().init(context, eventParams);
            EventManager.getInstance().setSyncAlarm(context);
            EventManager.getInstance().startSync(context);
            EventManager.getInstance().logEvent(context, EventManager.LOG_CRITICAL_LEVEL, EventParameters.CATEGORY_SDK, EventParameters.ACTION_SDK_INIT, publicher_id);
            initFrameRate(context);
        }
        catch (Exception ex)
        {
            SDKLogger.printErrorLog(TAG, "Error when trying to create device params: " + ex.getMessage());
        }
        if (publicher_id == null || token == null)
        {
            throw new RuntimeException("Missing valid publisher id or security token!");
        }
        else
        {
            mPublicherKey = publicher_id;
            mSecurityKey = token;
            sdkAPIManager = new SdkAPIManager(context, publicher_id, token);
            isInitialized = true;
        }
        mGoogleAnalyticsTracker = new GoogleAnalyticsTracker(context);
        validateSDKConfiguration(context);
    }

    /**
     * Set creative tools Data
     *
     * @param creativeToolsData creative tools data object
     */
    public static void setCreativeToolsData(CreativeToolsData creativeToolsData)
    {
        sCreativeToolsData = creativeToolsData;
    }

    /**
     * Get creative tools Data
     *
     * @return CreativeToolsData
     */
    public static CreativeToolsData getCreativeToolsData()
    {
        if (sCreativeToolsData == null)
        {
            sCreativeToolsData = new CreativeToolsData();
        }
        return sCreativeToolsData;
    }

    public static SdkAPIManager getApiManager(Context context)
    {
        if (sdkAPIManager == null)
        {
            sdkAPIManager = new SdkAPIManager(context, mPublicherKey, mSecurityKey);
        }
        return sdkAPIManager;
    }

    /**
     * Check if Kidoz SDK has been initialised
     *
     * @return initialized
     */
    static boolean isInitialised()
    {
        if (isInitialized == false)
        {
            Log.e("[KIDOZ SDK]", "Kidoz SDK need to be initialized !");
        }
        return isInitialized;
    }

    /**
     * Validate SDK configuration (Configuration retrieved from remote server)
     */
    private static void validateSDKConfiguration(final Context context)
    {
        if (context != null)
        {
            getApiManager(context).validateSDKConfiguration(context, new ApiResultCallback<SDKConfigurationData>()
            {
                @Override public void onServerResult(ResultData<?> result)
                {
                    if (result != null && result.getResponseStatus() != null && result.getResponseStatus().getIsSuccessful() == true)
                    {
                        if (context != null)
                        {
                            SDKConfigurationData.prepareLockIcons(context);
                        }
                    }
                }

                @Override public void onFailed()
                {

                }
            });
        }
    }

    private static void initFrameRate(Context context)
    {
        float frameRate = ((WindowManager) context.getSystemService(Context.WINDOW_SERVICE)).getDefaultDisplay().getRefreshRate();
        SDKLogger.printDebbugLog(TAG, ">>>>frameRate = " + String.valueOf(frameRate));
    }
}
