/*
 * This file is part of Siebe Projects samples.
 *
 * Siebe Projects samples is free software: you can redistribute it and/or modify
 * it under the terms of the Lesser GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Siebe Projects samples is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * Lesser GNU General Public License for more details.
 *
 * You should have received a copy of the Lesser GNU General Public License
 * along with Siebe Projects samples.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.sakura.momo.comon.utils;


import android.app.Activity;
import android.content.res.Configuration;
import android.graphics.Point;
import android.graphics.Rect;
import android.graphics.drawable.ColorDrawable;
import android.util.Log;
import android.view.Gravity;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewTreeObserver.OnGlobalLayoutListener;
import android.view.WindowManager.LayoutParams;
import android.widget.PopupWindow;

import com.quby.R;


/**
 * The keyboard height provider, this class uses a PopupWindow
 * to calculate the window height when the floating keyboard is opened and closed.
 */
public class KeyboardHeightProvider extends PopupWindow {

    /**
     * The tag for logging purposes
     */
    private final static String TAG = "sample_KeyboardHeightProvider";

    /**
     * The minimum height of the navigation bar
     */
    private final static int NAVIGATION_BAR_MIN_HEIGHT = 100;

    /**
     * The keyboard height observer
     */
    private KeyboardHeightObserver observer;

    /**
     * The cached landscape height of the keyboard
     */
    private int keyboardLandscapeHeight;

    /**
     * The cached portrait height of the keyboard
     */
    private int keyboardPortraitHeight;

    /**
     * The view that is used to calculate the keyboard height
     */
    private View popupView;

    /**
     * The parent view
     */
    private View parentView;

    /**
     * The root activity that uses this KeyboardHeightProvider
     */
    private Activity activity;

    /**
     * Indicates of the navigation is visible or not
     */
    private boolean navigationBarVisible;

    /**
     * Construct a new KeyboardHeightProvider
     *
     * @param activity The parent activity
     */
    public KeyboardHeightProvider(Activity activity) {
        super(activity);
        this.activity = activity;

        try {
            LayoutInflater inflator = (LayoutInflater) activity.getSystemService(Activity.LAYOUT_INFLATER_SERVICE);
            this.popupView = inflator.inflate(R.layout.popupwindow, null, false);
            setContentView(popupView);

            setSoftInputMode(LayoutParams.SOFT_INPUT_ADJUST_RESIZE | LayoutParams.SOFT_INPUT_STATE_ALWAYS_VISIBLE);
            setInputMethodMode(PopupWindow.INPUT_METHOD_NEEDED);

            parentView = activity.findViewById(android.R.id.content);

            setWidth(0);
            setHeight(LayoutParams.MATCH_PARENT);

            popupView.getViewTreeObserver().addOnGlobalLayoutListener(new OnGlobalLayoutListener() {

                @Override
                public void onGlobalLayout() {
                    if (popupView != null) {
                        handleOnGlobalLayout();
                    }
                }
            });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * Start the KeyboardHeightProvider, this must be called after the onResume of the Activity.
     * PopupWindows are not allowed to be registered before the onResume has finished
     * of the Activity.
     */
    public void start() {

        if (!isShowing() && parentView.getWindowToken() != null) {
            setBackgroundDrawable(new ColorDrawable(0));
            showAtLocation(parentView, Gravity.NO_GRAVITY, 0, 0);
        }
    }

    /**
     * Close the keyboard height provider,
     * this provider will not be used anymore.
     */
    public void close() {
        this.observer = null;
        dismiss();
    }

    /**
     * Set the keyboard height observer to this provider. The
     * observer will be notified when the keyboard height has changed.
     * For example when the keyboard is opened or closed.
     *
     * @param observer The observer to be added to this provider.
     */
    public void setKeyboardHeightObserver(KeyboardHeightObserver observer) {
        this.observer = observer;
    }

    /**
     * Get the screen orientation
     *
     * @return the screen orientation
     */
    private int getScreenOrientation() {
        return activity.getResources().getConfiguration().orientation;
    }

    /**
     * Popup window itself is as big as the window of the Activity.
     * The keyboard can then be calculated by extracting the popup view bottom
     * from the activity window height.
     */
    private void handleOnGlobalLayout() {
        try {
            Point screenSize = new Point();
            activity.getWindowManager().getDefaultDisplay().getSize(screenSize);

            Rect rect = new Rect();
            popupView.getWindowVisibleDisplayFrame(rect);

            int orientation = getScreenOrientation();
            int keyboardHeight = 0;
            Log.d("TAG", "keyboardHeight : " + keyboardHeight + "; screenSize.y : " + screenSize.y + "; rect.bottom : "
                    + rect.bottom + "; screenSize.x : " + screenSize.x + "; rect.top : " + rect.top);
            if (rect.bottom < screenSize.y && rect.bottom >= 0) {
                keyboardHeight = rect.bottom;
            }
            if (rect.bottom > screenSize.y / 2 && rect.bottom < (screenSize.y * 3 / 4))
                keyboardHeight = Math.round(screenSize.y / 2);
            else if (rect.bottom > (screenSize.y * 3 / 4))
                keyboardHeight = 0;
            if (orientation == Configuration.ORIENTATION_LANDSCAPE) {
                this.keyboardPortraitHeight = keyboardHeight;
                Log.d("TAG", "keyboardHeight 2 :" + keyboardHeight);
                notifyKeyboardHeightChanged(keyboardPortraitHeight, orientation);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     *
     */
    private boolean isChangedKeyboard;

    private void notifyKeyboardHeightChanged(int height, int orientation) {
        if (observer != null) {
            Log.d("TAG", "MO BAN PHIM");
            observer.onKeyboardHeightChanged(height, orientation);
//            isChangedKeyboard = true;
        } else {
            Log.d("TAG", "TAT BAN PHIM");
//            isChangedKeyboard = false;
        }
    }

    public boolean checkKeyboardChanged() {
        return isChangedKeyboard;
    }
}
