package com.juphoon.cloud;

import android.text.TextUtils;

import com.juphoon.cloud.MtcEngine.MtcNotifyListener;
import com.justalk.cloud.lemon.MtcCallConstants;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

class JCCallImpl extends JCCall implements MtcNotifyListener, JCClientCallback, JCMediaDeviceCallback {

    private static final String COMMANDKEY = "CommandKey";
    private static final String COMMAND_HOLDKEY = "CommandType_HoldKey";

    private static final String COMMAND_HOLD = "CommandType_Hold";

    private JCClient mClient;
    private JCMediaDevice mMediaDevice;
    private List<JCCallCallback> mCallbacks = new ArrayList<>();

    private boolean mConference;
    private List<JCCallItem> mCallItems = new ArrayList<>();

    JCCallImpl(JCClient client, JCMediaDevice mediaDevice, JCCallCallback callback) {
        if (callback == null) {
            throw new RuntimeException("JCCallCallback cannot be null!");
        }
        mCallbacks.add(callback);
        mClient = client;
        mMediaDevice = mediaDevice;
        mConference = false;
        mClient.addCallback(this);
        mMediaDevice.addCallback(this);
        MtcEngine.getInstance().addMtcNotifyListener(this);
    }

    @Override
    public void onNotify(JCNotify notify) {
        if (notify.type == JCNotify.CALL) {
            JCNotify.Call callNotify = notify.callNotify;
            if (callNotify.type == JCNotify.CALL_DIDTERM) {
                notifyCallItemRemove(callNotify.didTerm.callId, REASON_TERM_BY_SELF, "");
            } else if (callNotify.type == JCNotify.CALL_TERMED) {
                notifyCallItemRemove(callNotify.termed.callId, translateFromMtcReason(callNotify.termed.statusCode), callNotify.termed.desc);
            } else if (callNotify.type == JCNotify.CALL_INCOMING) {
                boolean canAnswer = true;
                int reason = REASON_NONE;
                JCCallItem sameCallItem = getCallByUserId(callNotify.incoming.userId);
                if (sameCallItem != null) {
                    JCLog.info(TAG, "The same user calls in");
                    sameCallItem.setSecondCallId(callNotify.incoming.callId); // 加入一个callId
                    return;
                }
                if (mCallItems.size() > 0 && callNotify.incoming.video) {
                    canAnswer = false;
                    JCLog.info(TAG, "Already have a call, %s video call is rejected", callNotify.incoming.userId);
                    reason = REASON_REJECT_VIDEO_WHEN_HAS_CALL;
                } else if (mCallItems.size() > 0 && hasVideo()) {
                    canAnswer = false;
                    JCLog.info(TAG, "Already have a call，%s call rejection", callNotify.incoming.userId);
                    reason = REASON_REJECT_WHEN_HAS_VIDEO_CALL;
                } else if (mCallItems.size() >= maxCallNum) {
                    canAnswer = false;
                    JCLog.info(TAG, "Call exceeds limit，%s call rejection", callNotify.incoming.userId);
                    reason = REASON_OVER_LIMIT;
                }
                JCCallItem item = new JCCallItem();
                item.callId = callNotify.incoming.callId;
                item.userId = callNotify.incoming.userId;
                item.displayName = callNotify.incoming.displayName;
                item.renderId = callNotify.incoming.uri;
                item.video = callNotify.incoming.video;
                item.renderId = callNotify.incoming.renderId;
                item.direction = JCCall.DIRECTION_IN;
                item.active = mCallItems.size() == 0;
                item.uploadVideoStreamSelf = true;
                item.uploadVideoStreamOther = callNotify.incoming.video;
                item.serverCallId = callNotify.incoming.serverCallId;
                item.extraParam = callNotify.incoming.extraParam;
                mCallItems.add(item);
                if (!canAnswer) {
                    JCParam.Term param = new JCParam.Term();
                    param.callId = callNotify.incoming.callId;
                    param.reason = translateToMtcReason(JCCall.REASON_BUSY);
                    if (MtcEngine.getInstance().term(param).succ) {
                        JCLog.info(TAG, "term %s", callNotify.incoming.userId);
                    }
                    item.state = JCCall.STATE_MISSED;
                    notifyCallItemRemove(item.callId, reason, "");
                } else {
                    item.state = JCCall.STATE_PENDING;
                    notifyCallItemAdd(item);
                }

            } else if (callNotify.type == JCNotify.CALL_TRYING) {
                JCCallItem item = getCallItem(callNotify.trying.callId);
                if (item != null) {
                    item.serverCallId = callNotify.trying.serverCallId;
                    notifyCallItemUpdate(item);
                }
            } else if (callNotify.type == JCNotify.CALL_TALKING) {
                JCCallItem item = getCallItem(callNotify.talking.callId);
                if (item != null) {
                    item.state = STATE_TALKING;
                    item.talkingBeginTime = System.currentTimeMillis() / 1000;
                    notifyCallItemUpdate(item);
                    if (mConference) {
                        updateVoiceMix(item);
                    }
                }
            } else if (callNotify.type == JCNotify.CALL_CONNECTING) {
                JCCallItem item = getCallItem(callNotify.connecting.callId);
                if (item != null) {
                    mMediaDevice.startAudio();
                    item.state = STATE_CONNECTING;
                    item.video = callNotify.connecting.video;
                    notifyCallItemUpdate(item);
                    startSendMediaIfNeed();
                }
            } else if (callNotify.type == JCNotify.CALL_ALERTED) {
                JCCallItem item = getCallItem(callNotify.alerted.callId);
                if (item != null) {
                    item.state = STATE_PENDING;
                    notifyCallItemUpdate(item);
                }
            } else if (callNotify.type == JCNotify.CALL_MESSAGE) {
                dealMessage(callNotify.message);
            } else if (callNotify.type == JCNotify.CALL_OTHER_VIDEO_STATUS) {
                JCCallItem item = getCallItem(callNotify.otherVideoStatus.callId);
                if (item != null) {
                    item.uploadVideoStreamOther = isOtherUploadVideo(callNotify.otherVideoStatus.status);
                    notifyCallItemUpdate(item);
                }
            } else if (callNotify.type == JCNotify.CALL_MISS_CALL) {
                JCCallItem item = new JCCallItem();
                item.userId = callNotify.missCall.userId;
                item.state = STATE_MISSED;
                item.video = callNotify.missCall.video;
                item.displayName = callNotify.missCall.displayName;
                item.direction = DIRECTION_IN;
                item.beginTime = callNotify.missCall.time;
                item.serverCallId = callNotify.missCall.serverCallId;
                item.callId = System.currentTimeMillis();
                mCallItems.add(item);
                notifyCallItemRemove(item.callId, REASON_NONE, "");
            } else if (callNotify.type == JCNotify.CALL_NET_STATUS) {
                JCCallItem item = getCallItem(callNotify.netStatus.callId);
                if (item != null) {
                    if (!callNotify.netStatus.send && !callNotify.netStatus.video) {
                        item.netStatus = callNotify.netStatus.status;
                        item.netStatusUpdateTime = System.currentTimeMillis();
                        notifyCallItemUpdate(item);
                        if (callNotify.netStatus.status == JCCall.NET_STATUS_DISCONNECTED && item.getTermWhenNetDisconnected()) {
                            term(item, JCCall.REASON_NETWORK, "");
                        }
                    }
                }
            }
        }
    }

    @Override
    protected void destroyObj() {
        mCallbacks.clear();
        mClient.removeCallback(this);
        mMediaDevice.removeCallback(this);
        MtcEngine.getInstance().removeMtcNotifyListener(this);
        mMediaDevice = null;
        mClient = null;
    }

    @Override
    public boolean call(String userId, boolean video, String extraParam) {
        if (mClient.getState() != JCClient.STATE_LOGINED) {
            JCLog.error(TAG, "call Not logged in");
            return false;
        }
        if (TextUtils.isEmpty(userId)) {
            JCLog.error(TAG, "userId empty");
            return false;
        }
        if (mCallItems.size() > 0) {
            if (video) {
                JCLog.error(TAG, "call already call,cannot initiate video call");
                return false;
            } else if (mCallItems.get(0).getVideo()) {
                JCLog.error(TAG, "call already call,cannot initiate video call");
                return false;
            }
        }
        if (getCallByUserId(userId) != null) {
            JCLog.error(TAG, "call user has already called");
            return false;
        }
        if (userId.equals(mClient.getUserId())) {
            JCLog.error(TAG, "call Can't call yourself");
            return false;
        }
        if (mCallItems.size() >= maxCallNum) {
            JCLog.error(TAG, "call Over the maximum number of calls");
            return false;
        }
        final JCCallItem item = new JCCallItem();
        item.userId = userId;
        item.displayName = userId;
        item.video = video;
        item.direction = JCCall.DIRECTION_OUT;
        item.uploadVideoStreamSelf = video;
        item.uploadVideoStreamOther = video;
        item.callId = System.currentTimeMillis();
        mCallItems.add(item);
        if (mCallItems.size() > 1) {
            becomeActive(item);
        } else {
            item.active = true;
        }
        notifyCallItemAdd(item);

        JCParam.Call param = new JCParam.Call();
        param.userId = userId;
        param.video = video;
        param.displayName = TextUtils.isEmpty(mClient.displayName) ? mClient.getUserId() : mClient.displayName;
        param.extraParam = extraParam;
        JCResult result = MtcEngine.getInstance().call(param);
        if (result.succ) {
            item.callId = result.longValue;
            item.renderId = result.strValue;
            JCLog.info(TAG, "call %s", userId);
        } else {
            item.state = JCCall.STATE_ERROR;
            JCLog.error(TAG, "call Call fails");
            final int finalReason = JCCall.REASON_CALL_FUNCTION_ERROR;
            JCClientThreadImpl.getInstance().postDelayed(new Runnable() {
                @Override
                public void run() {
                    notifyCallItemRemove(item.callId, finalReason, "");
                }
            }, 1000);
        }
        return true;
    }

    @Override
    public boolean term(JCCallItem item, @CallReason int reason, String description) {
        if (item == null) {
            return false;
        }
        if (!mCallItems.contains(item)) {
            JCLog.error(TAG, "term Call object not found");
            return false;
        }
        JCParam.Term param = new JCParam.Term();
        param.callId = item.callId;
        param.reason = translateToMtcReason(reason);
        param.desc = description == null ? "" : description;
        if (MtcEngine.getInstance().term(param).succ) {
            JCLog.info(TAG, "term %s", item.getUserId());
        } else {
            JCLog.info(TAG, "term %s Call failed forced term", item.getUserId());
        }
        if (item.getSecondCallId() != -1) {
            param.callId = item.getSecondCallId();
            if (MtcEngine.getInstance().term(param).succ) {
                JCLog.info(TAG, "term %s", item.getUserId());
            } else {
                JCLog.info(TAG, "term %s 调用失败强行term", item.getUserId());
            }
            item.setSecondCallId(-1);
        }
        notifyCallItemRemove(item.callId, JCCall.REASON_TERM_BY_SELF, "");
        return true;
    }

    @Override
    public boolean answer(JCCallItem item, boolean video) {
        if (item == null) {
            return false;
        }
        if (!mCallItems.contains(item)) {
            JCLog.error(TAG, "answer %s Call object not found", item.getUserId());
            return false;
        }
        item.uploadVideoStreamSelf = video;
        JCParam.Answer param = new JCParam.Answer();
        param.callId = item.callId;
        param.video = video;
        if (MtcEngine.getInstance().answer(param).succ) {
            if (!item.getActive() && !mConference) {
                becomeActive(item);
            }
            JCLog.info(TAG, "answer %s", item.getUserId());
            return true;
        } else {
            JCLog.error(TAG, "answer %s fail", item.getUserId());
            notifyCallItemRemove(item.callId, JCCall.REASON_ANSWER_FAIL, "");
        }
        return false;
    }

    @Override
    public boolean mute(JCCallItem item) {
        if (item == null) {
            return false;
        }
        if (!mCallItems.contains(item)) {
            JCLog.error(TAG, "mute %s Call object not found", item.getUserId());
            return false;
        }
        if (item.getState() != JCCall.STATE_TALKING) {
            JCLog.error(TAG, "mute Only during call");
            return false;
        }
        if (mConference) {
            for (JCCallItem iteSubm : mCallItems) {
                iteSubm.mute = !iteSubm.getMute();
                internalMute(iteSubm);
                notifyCallItemUpdate(iteSubm);
            }
        } else {
            item.mute = !item.getMute();
            internalMute(item);
            notifyCallItemUpdate(item);
        }
        return true;
    }

    @Override
    public boolean hold(JCCallItem item) {
        if (item == null) {
            return false;
        }
        if (!mCallItems.contains(item)) {
            JCLog.error(TAG, "hold Call object not found");
            return false;
        }
        if (item.getHeld()) {
            JCLog.error(TAG, "hold，Unable to operate");
            return false;
        }
        if (item.getState() != JCCall.STATE_TALKING) {
            JCLog.error(TAG, "Hold only on the phone");
            return false;
        }
        if (mConference && !item.getHold()) {
            JCLog.error(TAG, "Only unhold in the meeting");
            return false;
        }
        item.hold = !item.getHold();

        internalMute(item);
        sendHoldCommand(item);
        notifyCallItemUpdate(item);

        JCLog.info(TAG, "hold %s %b", item.getUserId(), item.getHold());

        return true;
    }

    @Override
    public boolean audioRecord(JCCallItem item, boolean enable, String filePath) {
        if (item == null) {
            return false;
        }
        if (!mCallItems.contains(item)) {
            JCLog.error(TAG, "mute %s Call object not found", item.getUserId());
            return false;
        }
        if (enable) {
            if (item.audioRecord) {
                JCLog.error(TAG, "with %s being recorded", item.getUserId());
                return false;
            }
            for (JCCallItem items : mCallItems) {
                if (items.getAudioRecord()) {
                    audioRecord(item, false, "");
                }
            }
        } else {
            if (!item.audioRecord) {
                JCLog.error(TAG, "Not recording with %s ", item.getUserId());
                return false;
            }
        }
        JCParam.CallMedia param = new JCParam.CallMedia();
        param.type = JCParam.CallMedia.AUDIO_RECORD;
        param.callId = item.callId;
        param.enable = enable;
        param.optionalValue = filePath;
        if (MtcEngine.getInstance().callMedia(param).succ) {
            JCLog.info(TAG, "audioRecord");
            item.audioRecord = enable;
            item.audioRecordFilePath = enable ? filePath : "";
            notifyCallItemUpdate(item);
            return true;
        } else {
            JCLog.error(TAG, "audioRecord fail");
        }
        return false;
    }

    @Override
    public boolean videoRecord(JCCallItem item, boolean enable, boolean remote, int width, int height, String filePath) {
        if (item == null) {
            return false;
        }
        if (!mCallItems.contains(item)) {
            JCLog.error(TAG, "No call object found with %s video recording", item.getUserId());
            return false;
        }
        if (enable) {
            if (remote && item.getRemoteVideoRecord()) {
                JCLog.error(TAG, "Recording %s video", item.getUserId());
                return false;
            }
            if (!remote && item.getLocalVideoRecord()) {
                JCLog.error(TAG, "Recording local video");
                return false;
            }
            for (JCCallItem items : mCallItems) {
                if ((remote && items.getRemoteVideoRecord())
                        || (!remote && items.getLocalVideoRecord())) {
                    videoRecord(item, false, remote, 0, 0, "");
                }
            }
        } else {
            if (remote && !item.getRemoteVideoRecord()) {
                JCLog.error(TAG, "Video recording failed with %s not recording video ", item.getUserId());
                return false;
            }
            if (!remote && !item.getLocalVideoRecord()) {
                JCLog.error(TAG, "The local video is not being recorded. Video recording failure has failed. ");
                return false;
            }
        }
        JCParam.CallMedia param = new JCParam.CallMedia();
        param.callId = item.callId;
        param.type = JCParam.CallMedia.VIDEO_RECORD;
        param.remote = remote;
        param.enable = enable;
        param.optionalValue = filePath;
        param.videoRecordWidth = width;
        param.videoRecordHeight = height;
        if (MtcEngine.getInstance().callMedia(param).succ) {
            JCLog.info(TAG, "videoRecord");
            if (remote) {
                item.remoteVideoRecord = enable;
                item.remoteVideoRecordFilePath = enable ? filePath : "";
            } else {
                item.localVideoRecord = enable;
                item.localVideoRecordFilePath = enable ? filePath : "";
            }
            notifyCallItemUpdate(item);
            return true;
        }
        return false;
    }

    @Override
    public boolean becomeActive(JCCallItem item) {
        if (item == null) {
            return false;
        }
        if (mConference) {
            JCLog.info(TAG, "switchActive Conference mode cannot be switched");
            return false;
        }
        if (item.getActive()) {
            JCLog.info(TAG, "switchActive Already activated");
            return true;
        }
        JCCallItem activeItem = getActiveCallItem();
        activeItem.active = false;
        if (!activeItem.hold) {
            hold(activeItem);
        }
        notifyCallItemUpdate(activeItem);

        item.active = true;
        if (item.hold) {
            hold(item);
        }
        notifyCallItemUpdate(item);
        return true;
    }

    @Override
    public boolean enableUploadVideoStream(JCCallItem item) {
        if (item == null) {
            return false;
        }
        if (!item.getVideo()) {
            JCLog.error(TAG, "enableUploadVideoStream Voice call cannot be operated");
            return false;
        }
        JCParam.CallMedia param = new JCParam.CallMedia();
        param.type = JCParam.CallMedia.UPLOAD_VIDEO_STREAM;
        param.callId = item.callId;
        param.enable = !item.getUploadVideoStreamSelf();
        if (MtcEngine.getInstance().callMedia(param).succ) {
            item.uploadVideoStreamSelf = !item.getUploadVideoStreamSelf();
            JCLog.info(TAG, "enableUploadVideoStream");
            notifyCallItemUpdate(item);
            return true;
        } else {
            JCLog.error(TAG, "enableUploadVideoStream fail");
        }
        return false;
    }

    @Override
    public void setConference(boolean conference) {
        mConference = conference;
        for (JCCallItem item : mCallItems) {
            if (mConference) {
                if (item.getHold()) {
                    hold(item);
                }
                item.mute = false;
            }
            updateVoiceMix(item);
            internalMute(item);
            notifyCallItemUpdate(item);
        }
    }

    @Override
    public boolean getConference() {
        return mConference;
    }

    @Override
    public List<JCCallItem> getCallItems() {
        return mCallItems;
    }

    @Override
    public boolean sendMessage(JCCallItem item, String type, String content) {
        if (item == null) {
            return false;
        }
        if (!mCallItems.contains(item)) {
            JCLog.error(TAG, "sendMessage fail：No such call");
        }
        JCParam.CallMessage param = new JCParam.CallMessage();
        param.callId = item.callId;
        param.messageType = type;
        param.content = content;
        return MtcEngine.getInstance().callStreamData(param).succ;
    }

    @Override
    public String getStatistics() {
        JCCallItem item = getActiveCallItem();
        if (item != null) {
            JCParam.CallStatistics param = new JCParam.CallStatistics();
            param.callId = item.callId;
            return MtcEngine.getInstance().callStatistics(param).strValue;

        }
        return "";
    }

    @Override
    protected void addCallback(JCCallCallback callback) {
        mCallbacks.add(callback);
    }

    @Override
    protected void removeCallback(JCCallCallback callback) {
        mCallbacks.remove(callback);
    }

    private JCCallItem getCallByUserId(String userId) {
        for (JCCallItem item : mCallItems) {
            if (item.getUserId().equals(userId)) {
                return item;
            }
        }
        return null;
    }

    private JCCallItem getCallItem(long callId) {
        for (JCCallItem item : mCallItems) {
            if (item.getCallId() == callId || item.getSecondCallId() == callId) {
                return item;
            }
        }
        JCLog.error(TAG, "No call found");
        return null;
    }

    private void autoChooseActive() {
        /** 默认选最早的一个 */
        for (JCCallItem item : mCallItems) {
            item.active = true;
            if (item.getHold()) {
                hold(item);
            } else {
                notifyCallItemUpdate(item);
            }
            return;
        }
    }

    void internalMute(JCCallItem item) {
        JCParam.CallMedia param = new JCParam.CallMedia();
        param.callId = item.callId;

        param.type = JCParam.CallMedia.MUTE;
        param.enable = item.getHeld() || item.getHold() || item.getMute();
        MtcEngine.getInstance().callMedia(param);

        param.type = JCParam.CallMedia.SPEAKER_MUTE;
        param.enable = item.getHeld() || item.getHold();
        MtcEngine.getInstance().callMedia(param);
    }

    boolean hasVideo() {
        for (JCCallItem call : mCallItems) {
            if (call.video) {
                return true;
            }
        }
        return false;
    }

    void sendHoldCommand(JCCallItem item) {
        try {
            JSONObject object = new JSONObject();
            object.put(COMMANDKEY, COMMAND_HOLD);
            object.put(COMMAND_HOLDKEY, item.getHold());
            JCParam.CallMessage param = new JCParam.CallMessage();
            param.callId = item.callId;
            param.content = object.toString();
            if (MtcEngine.getInstance().callMessage(param).succ) {
                JCLog.info(TAG, "Send hold command");
            } else {
                JCLog.error(TAG, "Send hold command fail");
            }
        } catch (JSONException e) {
            e.printStackTrace();
        }
    }

    private JCCallItem getActiveCallItem() {
        for (JCCallItem item : mCallItems) {
            if (item.getActive()) {
                return item;
            }
        }
        JCLog.error(TAG, "Active caller not found");
        return null;
    }

    private boolean updateVoiceMix(JCCallItem item) {
        JCParam.CallMedia param = new JCParam.CallMedia();
        param.type = JCParam.CallMedia.MIX;
        param.callId = item.callId;
        param.enable = mConference;
        if (MtcEngine.getInstance().callMedia(param).succ) {
            JCLog.info(TAG, "%s Mixing", item.getUserId());
            return true;
        } else {
            JCLog.info(TAG, "%s Mixing fail", item.getUserId());
            return false;
        }
    }

    private void dealMessage(JCNotify.Call.Message message) {
        JCCallItem item = getCallItem(message.callId);
        if (item != null) {
            try {
                if (message.tunnel == JCNotify.CALL_MESSAGE_TYPE_INFO) {
                    JSONObject object = new JSONObject(message.content);
                    String command = object.optString(COMMANDKEY);
                    if (TextUtils.equals(command, COMMAND_HOLD)) {
                        JCLog.info(TAG, "Handling the hold command");
                        item.held = object.optBoolean(COMMAND_HOLDKEY);
                        internalMute(item);
                        notifyCallItemUpdate(item);
                    }
                } else {
                    notifyCallMessageReceived(item, message.type, message.content);
                }
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
    }

    private void notifyCallItemAdd(final JCCallItem item) {
        JCClientThreadImpl.getInstance().post(new Runnable() {
            @Override
            public void run() {
                JCLog.info(TAG, "notifyCallItemAdd, userId:%s", item.getUserId());
                for (JCCallCallback callback : mCallbacks) {
                    callback.onCallItemAdd(item);
                }
            }
        });
    }

    private void notifyCallItemRemove(long callId, final int reason, final String description) {
        final JCCallItem item = getCallItem(callId);
        if (item != null) {
            // 处理有两个callId的情况，本方呼出，又收到对方incoming
            if (item.getCallId() == callId && item.getSecondCallId() != -1) {
                item.setCallId(item.getSecondCallId());
                item.setSecondCallId(-1);
                return;
            } else if (item.getSecondCallId() == callId) {
                item.setSecondCallId(-1);
                return;
            }
            mCallItems.remove(item);
            if (item.getActive()) {
                autoChooseActive();
            }
            int state = item.getState();
            if (state <= JCCall.STATE_PENDING && item.getDirection() == JCCall.DIRECTION_IN) {
                item.state = JCCall.STATE_MISSED;
            } else if (state == JCCall.STATE_ERROR || state == JCCall.STATE_MISSED) {
            } else if (state >= JCCall.STATE_CONNECTING) {
                item.state = JCCall.STATE_OK;
            } else {
                if (reason == JCCall.REASON_TERM_BY_SELF) {
                    item.state = JCCall.STATE_CANCEL;
                } else {
                    item.state = JCCall.STATE_CANCELED;
                }
            }
            item.reason = reason;
            JCClientThreadImpl.getInstance().post(new Runnable() {
                @Override
                public void run() {
                    JCLog.info(TAG, "call ended %s reason:%d", item.getUserId(), reason);
                    for (JCCallCallback callback : mCallbacks) {
                        callback.onCallItemRemove(item, reason, description);
                    }
                    if (mCallItems.size() == 0) {
                        // 可能销毁后又回来触发引起空指针
                        if (mMediaDevice != null) {
                            mMediaDevice.stopAudio();
                            mMediaDevice.enableSpeaker(false);
                        } else {
                            JCLog.info(TAG, "dealloc");
                        }
                        MtcEngine.getInstance().hasCalls = false;
                    }
                }
            });
        } else {
            JCLog.info(TAG, "notifyCallItemRemove, Not found item");
        }
    }

    private void notifyCallItemUpdate(final JCCallItem item) {
        JCClientThreadImpl.getInstance().post(new Runnable() {
            @Override
            public void run() {
                JCLog.info(TAG, "notifyCallItemUpdate, userId:%s", item.getUserId());
                for (JCCallCallback callback : mCallbacks) {
                    callback.onCallItemUpdate(item);
                }
            }
        });
    }

    private void notifyCallMessageReceived(final JCCallItem item, final String type, final String content) {
        JCClientThreadImpl.getInstance().post(new Runnable() {

            @Override
            public void run() {
                JCLog.info(TAG, "notifyCallMessageReceived, from:%s, type:%s, content:%s", item.getUserId(), type, content);
                for (JCCallCallback callback : mCallbacks) {
                    callback.onMessageReceive(type, content, item);
                }
            }
        });
    }

    private int translateToMtcReason(int reason) {
        switch (reason) {
            case JCCall.REASON_BUSY:
                return MtcCallConstants.EN_MTC_CALL_TERM_STATUS_BUSY;
            case JCCall.REASON_DECLINE:
                return MtcCallConstants.EN_MTC_CALL_TERM_STATUS_DECLINE;
            default:
                return MtcCallConstants.EN_MTC_CALL_TERM_STATUS_NORMAL;
        }
    }

    private int translateFromMtcReason(int reason) {
        switch (reason) {
            case MtcCallConstants.EN_MTC_CALL_TERM_STATUS_NORMAL:
                return JCCall.REASON_NONE;
            case MtcCallConstants.EN_MTC_CALL_TERM_STATUS_BUSY:
                return JCCall.REASON_BUSY;
            case MtcCallConstants.EN_MTC_CALL_TERM_STATUS_DECLINE:
                return JCCall.REASON_DECLINE;
            case MtcCallConstants.EN_MTC_CALL_TERM_STATUS_TIMEOUT:
                return JCCall.REASON_TIMEOUT;
            case MtcCallConstants.EN_MTC_CALL_TERM_STATUS_USER_OFFLINE:
                return JCCall.REASON_USER_OFFLINE;
            case MtcCallConstants.EN_MTC_CALL_TERM_STATUS_NOT_FOUND:
                return JCCall.REASON_NOT_FOUND;
            default:
                return JCCall.REASON_OTHER;
        }
    }

    private boolean isOtherUploadVideo(int status) {
        return status != MtcCallConstants.EN_MTC_CALL_TRANSMISSION_CAMOFF;
    }

    @Override
    public void onLogin(boolean result, @JCClient.ClientReason int reason) {
        if (result && fetchMissedCallWhenLogin) {
            JCLog.info(TAG, "Pull missed calls");
            MtcEngine.getInstance().fetchCalls(new JCParam.CallFetch());
        }
    }

    @Override
    public void onLogout(@JCClient.ClientReason int reason) {
        for (JCCallItem item : mCallItems) {
            notifyCallItemRemove(item.callId, JCCall.REASON_NOT_LOGIN, "");
        }
    }

    @Override
    public void onClientStateChange(@JCClient.ClientState int state, @JCClient.ClientState int oldState) {

    }

    @Override
    public void onCameraUpdate() {
        startSendMediaIfNeed();
    }

    @Override
    public void onAudioOutputTypeChange(boolean speaker) {

    }

    @Override
    public void onRenderReceived(JCMediaDeviceVideoCanvas canvas) {

    }

    @Override
    public void onRenderStart(JCMediaDeviceVideoCanvas canvas) {

    }

    private void startSendMediaIfNeed() {
        JCCallItem item = getActiveCallItem();
        if (item != null) {
            if (item.getVideo() && item.getUploadVideoStreamSelf()
                    && item.getState() >= STATE_CONNECTING && item.getState() <= STATE_TALKING
                    && (mMediaDevice.isCameraOpen() || mMediaDevice.isVideoFileOpen())) {
                JCParam.CallMedia param = new JCParam.CallMedia();
                param.type = JCParam.CallMedia.ATTACH_CAMERA;
                param.callId = item.getCallId();
                param.enable = true;
                if (mMediaDevice.isVideoFileOpen()) {
                    param.optionalValue = mMediaDevice.getVideoFileId();
                } else {
                    param.optionalValue = mMediaDevice.getCamera();
                }
                MtcEngine.getInstance().callMedia(param);
            }
        }
    }

}
