package com.juphoon.cloud;

import android.util.SparseArray;

import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

class JCStorageImpl extends JCStorage implements JCClientCallback, MtcEngine.MtcNotifyListener {

    static final String TAG = JCStorageImpl.class.getSimpleName();

    private JCClient mClient;
    private List<JCStorageCallback> mCallbacks = new ArrayList<>();
    private SparseArray<JCStorageItem> mItems = new SparseArray<>();

    JCStorageImpl(JCClient client, JCStorageCallback callback) {
        if (callback == null) {
            throw new RuntimeException("JCStorage callback cannot be null!");
        }
        mCallbacks.add(callback);
        mClient = client;
        mClient.addCallback(this);
        MtcEngine.getInstance().addMtcNotifyListener(this);
    }

    @Override
    protected void destroyObj() {
        mCallbacks.clear();
        mClient.removeCallback(this);
        MtcEngine.getInstance().removeMtcNotifyListener(this);
        mClient = null;
    }

    @Override
    public JCStorageItem uploadFile(String path) {
        if (mClient.getState() != JCClient.STATE_LOGINED) {
            JCLog.error(TAG, "uploadFile not login");
            return null;
        }
        if (!JCUtils.isFileExist(path)) {
            JCLog.error(TAG, "uploadFile file does not exist");
            return null;
        }
        if (JCUtils.getFileSize(path) > MAX_FILE_SIZE) {
            JCLog.error(TAG, "uploadFile File size exceeds limit");
            return null;
        }
        JCStorageItem item = new JCStorageItem();
        item.fileId = UUID.randomUUID().toString();
        item.direction = DIRECTION_UPLOAD;
        item.path = path;

        JCParam.FileDeal param = new JCParam.FileDeal();
        param.type = JCParam.FileDeal.UPLOAD;
        param.path = path;
        param.expireTime = FILE_EXIPIRE;

        JCResult result = MtcEngine.getInstance().dealFile(param);
        if (result.succ) {
            JCLog.info(TAG, "uploadFile");
            item.state = ITEM_STATE_TRANSFERRING;
            item.sessId = (int) result.longValue;
            item.cookie = result.cookie;
            mItems.put(result.cookie, item);
        } else {
            item.state = ITEM_STATE_FAIL;
        }
        notifyFileUpdate(item);
        return item;
    }

    @Override
    public JCStorageItem downloadFile(String uri, String savePath) {
        if (mClient.getState() != JCClient.STATE_LOGINED) {
            JCLog.error(TAG, "uploadFile not login");
            return null;
        }
        JCStorageItem item = new JCStorageItem();
        item.fileId = UUID.randomUUID().toString();
        item.direction = DIRECTION_DOWNLOAD;
        item.uri = uri;
        item.path = savePath;

        JCParam.FileDeal param = new JCParam.FileDeal();
        param.type = JCParam.FileDeal.DOWNLOAD;
        param.uri = uri;
        param.path = savePath;

        JCResult result = MtcEngine.getInstance().dealFile(param);
        if (result.succ) {
            JCLog.info(TAG, "downloadFile");
            item.state = ITEM_STATE_TRANSFERRING;
            item.sessId = (int) result.longValue;
            item.cookie = result.cookie;
            mItems.put(result.cookie, item);
        } else {
            item.state = ITEM_STATE_FAIL;
        }
        notifyFileUpdate(item);
        return item;

    }

    @Override
    public boolean cancelFile(JCStorageItem item) {
        if (item.state <= JCStorage.ITEM_STATE_TRANSFERRING) {
            JCParam.FileDeal param = new JCParam.FileDeal();
            param.type = JCParam.FileDeal.CANCEL;
            param.sessId = item.sessId;
            JCResult result = MtcEngine.getInstance().dealFile(param);
            if (result.succ) {
                JCLog.info(TAG, "File canceled successfully");
                item.state = JCStorage.ITEM_STATE_CANCEL;
                notifyFileUpdate(item);
                return true;
            } else {
                JCLog.info(TAG, "File cancellation failed");
            }
        } else {
            JCLog.error(TAG, "File cannot be cancelled");
            removeFileItem(item.cookie);
        }
        return false;
    }

    @Override
    public void onLogin(boolean result, @JCClient.ClientReason int reason) {

    }

    @Override
    public void onLogout(@JCClient.ClientReason int reason) {

    }

    @Override
    public void onClientStateChange(@JCClient.ClientState int state, @JCClient.ClientState int oldState) {

    }

    @Override
    public void onNotify(JCNotify notify) {
        if (notify.type == JCNotify.STORAGE) {
            if (notify.storageNotify.type == JCNotify.STORAGE_SEND_OK
                    || notify.storageNotify.type == JCNotify.STORAGE_RECV_OK) {
                JCStorageItem item = getStorageItem(notify.cookie);
                if (item != null) {
                    item.state = ITEM_STATE_OK;
                    if (notify.storageNotify.type == JCNotify.STORAGE_SEND_OK) {
                        item.uri = notify.storageNotify.sendOk.uri;
                    }
                    notifyFileUpdate(item);
                    removeFileItem(notify.cookie);
                }
            } else if (notify.storageNotify.type == JCNotify.STORAGE_SEND_FAIL
                    || notify.storageNotify.type == JCNotify.STORAGE_RECV_FAIL) {
                JCStorageItem item = getStorageItem(notify.cookie);
                if (item != null) {
                    item.state = ITEM_STATE_FAIL;
                    notifyFileUpdate(item);
                    removeFileItem(notify.cookie);
                }
            } else if (notify.storageNotify.type == JCNotify.STORAGE_PROGRESS) {
                JCStorageItem item = getStorageItem(notify.cookie);
                if (item != null) {
                    item.state = ITEM_STATE_TRANSFERRING;
                    item.progress = notify.storageNotify.storageProgress.progress;
                    notifyFileUpdate(item);
                }
            }
        }
    }

    private void notifyFileUpdate(final JCStorageItem item) {
        JCClientThreadImpl.getInstance().post(new Runnable() {
            @Override
            public void run() {
                JCLog.info(TAG, "File status update");
                for (JCStorageCallback callback : mCallbacks) {
                    callback.onFileUpdate(item);
                }
            }
        });
    }

    private JCStorageItem getStorageItem(int cookie) {
        JCStorageItem item = mItems.get(cookie);
        if (item == null) {
            JCLog.error(TAG, "The file transfer was not found");
        }
        return item;
    }

    private void removeFileItem(int cookie) {
        mItems.remove(cookie);
    }
}
