package com.juphoon.cloud;

import android.text.TextUtils;
import android.util.Base64;
import android.util.SparseArray;

import com.justalk.cloud.lemon.MtcImConstants;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.UUID;

class JCMessageChannelImpl extends JCMessageChannel implements MtcEngine.MtcNotifyListener, JCClientCallback {

    static final String TAG = JCMessageChannelImpl.class.getSimpleName();

    private JCClient mClient;
    private List<JCMessageChannelCallback> mCallbacks = new ArrayList<>();
    private SparseArray<JCMessageChannelItem> mMessages = new SparseArray<>();

    public JCMessageChannelImpl(JCClient client, JCMessageChannelCallback callback) {
        if (callback == null) {
            throw new RuntimeException("JCMessageChannel callback cannot be null!");
        }
        thumbDir = JCUtils.getSdkInfoDir(client.getContext()) + "thumb";
        new File(thumbDir).mkdirs();
        mCallbacks.add(callback);
        mClient = client;
        mClient.addCallback(this);
        MtcEngine.getInstance().addMtcNotifyListener(this);
    }

    @Override
    protected void destroyObj() {
        mCallbacks.clear();
        mClient.removeCallback(this);
        MtcEngine.getInstance().removeMtcNotifyListener(this);
        mClient = null;
    }

    @Override
    public JCMessageChannelItem sendMessage(@Type int type, String keyId, String messageType, String text,
                                            Map<String, String> extraParams) {
        if (text == null || text.length() == 0) {
            JCLog.error(TAG, "Text content cannot be empty");
            return null;
        }
        if (text.length() > ITEM_MAX_TEXT_SIZE) {
            JCLog.error(TAG, "Text content exceeds 10k");
            return null;
        }
        JCMessageChannelItem item = new JCMessageChannelItem();
        item.messageId = UUID.randomUUID().toString();
        item.messageType = messageType;
        item.text = text;
        item.extraParams = extraParams;
        item.userId = type == JCMessageChannel.TYPE_1TO1 ? keyId : mClient.getUserId();
        item.groupId = type == TYPE_GROUP ? keyId : null;
        item.time = System.currentTimeMillis();
        item.direction = DIRECTION_SEND;
        item.type = type;
        item.displayName = TextUtils.isEmpty(mClient.displayName) ? mClient.getUserId() : mClient.displayName;

        JCParam.Message param = new JCParam.Message();
        param.userId = item.userId;
        param.messageType = messageType;
        param.messageContent = text;
        param.displayName = item.displayName;
        param.groupId = item.groupId;

        try {
            JSONObject object = new JSONObject();
            object.put(ITEM_MESSAGE_ID, item.messageId);
            if (extraParams != null) {
                JSONObject extra = new JSONObject();
                for (Map.Entry<String, String> entry : extraParams.entrySet()) {
                    extra.put(entry.getKey(), entry.getValue());
                }
                object.put(ITEM_EXTRA_DATA, extra);
            }
            param.messageInfo = object.toString();
        } catch (JSONException e) {
            e.printStackTrace();
        }

        JCResult result = MtcEngine.getInstance().sendMessage(param);
        if (result.succ) {
            mMessages.put(result.cookie, item);
            item.state = ITEM_STATE_SENDING;
            notifySendUpdate(item);
        } else {
            JCLog.error(TAG, "Call failed to send");
            item.state = ITEM_STATE_SENDFAIL;
            notifySendUpdate(item);
        }

        return item;

    }

    @Override
    public JCMessageChannelItem sendFile(@Type int type, String keyId, String messageType, String fileUri,
                                         String thumbPath, int size, int duration, Map<String, String> extraParams) {
        if (TextUtils.isEmpty(fileUri)) {
            JCLog.error(TAG, "sendMessage File address cannot be empty");
            return null;
        }
        byte[] thumbData = null;
        if (!TextUtils.isEmpty(thumbPath)) {
            if (JCUtils.isFileExist(thumbPath) && JCUtils.getFileSize(thumbPath) <= ITEM_MAX_THUMB_SIZE) {
                thumbData = JCUtils.getBytes(thumbPath);
            } else {
                JCLog.error(TAG, "sendMessage Thumbnail does not exist or exceeds 6k limit");
                return null;
            }
        }
        JCMessageChannelItem item = new JCMessageChannelItem();
        item.messageId = UUID.randomUUID().toString();
        item.messageType = messageType;
        item.fileUri = fileUri;
        item.thumbPath = thumbPath;
        item.fileSize = size;
        item.duration = duration;
        item.extraParams = extraParams;
        item.userId = type == TYPE_1TO1 ? keyId : null;
        item.groupId = type == TYPE_GROUP ? keyId : null;
        item.time = System.currentTimeMillis();
        item.direction = DIRECTION_SEND;
        item.type = type;
        item.displayName = TextUtils.isEmpty(mClient.displayName) ? mClient.getUserId() : mClient.displayName;

        JCParam.Message param = new JCParam.Message();
        param.userId = keyId;
        param.messageType = messageType;
        param.displayName = item.displayName;
        param.messageContent = "file";
        param.groupId = item.groupId;

        try {
            JSONObject object = new JSONObject();
            object.put(ITEM_MESSAGE_ID, item.messageId);
            object.put(ITEM_FILE_URI, item.fileUri);
            if (thumbData != null && thumbData.length > 0) {
                object.put(ITEM_THUMB_DATA, Base64.encodeToString(thumbData, Base64.NO_WRAP));
            }
            object.put(ITEM_FILE_SIZE, item.fileSize);
            object.put(ITEM_DURATION, item.duration);
            if (extraParams != null) {
                JSONObject extra = new JSONObject();
                for (Map.Entry<String, String> entry : extraParams.entrySet()) {
                    extra.put(entry.getKey(), entry.getValue());
                }
                object.put(ITEM_EXTRA_DATA, extra);
            }
            param.messageInfo = object.toString();
        } catch (JSONException e) {
            e.printStackTrace();
        }

        JCResult result = MtcEngine.getInstance().sendMessage(param);
        if (result.succ) {
            mMessages.put(result.cookie, item);
            item.state = ITEM_STATE_SENDING;
            notifySendUpdate(item);
        } else {
            JCLog.error(TAG, "Call failed to send");
            item.state = ITEM_STATE_SENDFAIL;
            notifySendUpdate(item);
        }
        return item;
    }

    @Override
    protected void addCallback(JCMessageChannelCallback callback) {
        mCallbacks.add(callback);
    }

    @Override
    protected void removeCallback(JCMessageChannelCallback callback) {
        mCallbacks.remove(callback);
    }

    @Override
    public void onLogin(boolean result, @JCClient.ClientReason int reason) {
        if (result) {
            JCLog.info(TAG, "Successful login, pull message");
            MtcEngine.getInstance().fetchMessage(new JCParam.MessageFetch());
        }
    }

    @Override
    public void onLogout(@JCClient.ClientReason int reason) {

    }

    @Override
    public void onClientStateChange(@JCClient.ClientState int state, @JCClient.ClientState int oldState) {

    }

    @Override
    public void onNotify(JCNotify notify) {
        if (notify.type == JCNotify.MESSAGE) {
            if (notify.messageNotify.type == JCNotify.MESSAGE_SEND_OK) {
                JCMessageChannelItem item = getMessageItem(notify.cookie);
                if (item != null) {
                    item.state = ITEM_STATE_SENDOK;
                    item.sentTime = System.currentTimeMillis();
                    notifySendUpdate(item);
                    removeMessageItem(notify.cookie);
                }
            } else if (notify.messageNotify.type == JCNotify.MESSAGE_SEND_FAIL) {
                JCMessageChannelItem item = getMessageItem(notify.cookie);
                if (item != null) {
                    item.state = ITEM_STATE_SENDFAIL;
                    notifySendUpdate(item);
                    removeMessageItem(notify.cookie);
                }
            } else if (notify.messageNotify.type == JCNotify.MESSAGE_RECV_MESSAGE) {
                JCNotify.Message.Recv recv = notify.messageNotify.recvMessage;
                if (TextUtils.equals(mClient.getUserId(), recv.userId)) {
                    return;
                }
                JCMessageChannelItem item = new JCMessageChannelItem();
                item.userId = recv.userId;
                item.displayName = recv.displayName;
                item.groupId = recv.groupId;
                item.messageType = recv.messageType;
                item.text = recv.messageContent;
                item.time = System.currentTimeMillis();
                item.sentTime = recv.time;
                // 自定义参数
                if (!TextUtils.isEmpty(recv.messageInfo)) {
                    try {
                        JSONObject object = new JSONObject(recv.messageInfo);
                        item.messageId = object.optString(ITEM_MESSAGE_ID);
                        item.fileUri = object.optString(ITEM_FILE_URI);
                        String thumb = object.optString(ITEM_THUMB_DATA);
                        if (!TextUtils.isEmpty(thumb)) {
                            byte[] thumbData = Base64.decode(thumb, Base64.NO_WRAP);
                            item.thumbPath = JCUtils.genFilePath(thumbDir, "thumb");
                            JCUtils.writeToFile(thumbData, item.thumbPath);
                        }
                        item.fileSize = object.optInt(ITEM_FILE_SIZE);
                        item.duration = object.optInt(ITEM_DURATION);
                        JSONObject extra = object.optJSONObject(ITEM_EXTRA_DATA);
                        if (extra != null) {
                            item.extraParams = new HashMap<>(16);
                            Iterator iterator = extra.keys();
                            while (iterator.hasNext()) {
                                String key = (String) iterator.next();
                                item.extraParams.put(key, extra.getString(key));
                            }
                        }
                    } catch (JSONException e) {
                        e.printStackTrace();
                    }
                }

                item.state = ITEM_STATE_RECEIVED;
                item.direction = DIRECTION_RECEIVE;
                item.type = translateFromMtcCategory(recv.category);

                notifyRecv(item);
            }
        }
    }

    private void notifySendUpdate(final JCMessageChannelItem item) {
        JCClientThreadImpl.getInstance().post(new Runnable() {
            @Override
            public void run() {
                JCLog.info(TAG, "Send status update");
                for (JCMessageChannelCallback callback : mCallbacks) {
                    callback.onMessageSendUpdate(item);
                }
            }
        });
    }

    private void notifyRecv(final JCMessageChannelItem item) {
        JCClientThreadImpl.getInstance().post(new Runnable() {
            @Override
            public void run() {
                JCLog.info(TAG, "Received the news");
                for (JCMessageChannelCallback callback : mCallbacks) {
                    callback.onMessageRecv(item);
                }
            }
        });
    }

    private
    @JCMessageChannel.Type
    int translateFromMtcCategory(int category) {
        switch (category) {
            case MtcImConstants.MTC_IM_ID_UID:
                return TYPE_1TO1;
            default:
                return TYPE_GROUP;
        }
    }

    private JCMessageChannelItem getMessageItem(int cookie) {
        JCMessageChannelItem item = mMessages.get(cookie);
        if (item == null) {
            JCLog.error(TAG, "The send message was not found");
        }
        return item;
    }

    private void removeMessageItem(int cookie) {
        mMessages.remove(cookie);
    }

}
