package com.juphoon.cloud;

import android.content.Context;
import android.media.AudioManager;
import android.os.Build;
import android.text.TextUtils;

import com.justalk.cloud.lemon.MtcCallDb;
import com.justalk.cloud.lemon.MtcCallDbConstants;
import com.justalk.cloud.lemon.MtcNumber;
import com.justalk.cloud.zmf.ZmfVideo;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.List;

class JCMediaDeviceImpl extends JCMediaDevice implements JCClientCallback, ZmfEngine.ZmfNotifyListener {

    static final String TAG = JCMediaDeviceImpl.class.getSimpleName();

    private List<JCMediaDeviceCallback> mCallbacks = new ArrayList<>();
    private JCClient mClient;
    private int mDefaultCameraIndex;
    private int mCameraIndex;
    private boolean mCameraOpen;
    private boolean mVideoFileOpen;
    private String mVideoFileId;
    private boolean mAudioStart;
    private boolean mSpeakerOn;
    private int mWidth;
    private int mHeight;
    private int mFramerate;
    private int mScreenCaptureWidth;
    private int mScreenCaptureHeight;
    private int mScreenCaptureFrameRate;
    private List<JCMediaDeviceVideoCanvas> mCameraCanvas = new ArrayList<>();
    private List<JCMediaDeviceVideoCanvas> mVideoFileCanvas = new ArrayList<>();
    private List<JCMediaDeviceVideoCanvas> mOtherCanvas = new ArrayList<>();
    private List<String> mCameras;
    private AudioManager mAudioManager;
    private boolean mSawtooth;
    private @AecMode int mAecMode;
    private @VideoMode int mVideoMode;
    private ByteBuffer mVideoFileFrameBuffer;
    private boolean mAgcOn;

    JCMediaDeviceImpl(JCClient client, JCMediaDeviceCallback callback) {
        JCParam.Init param = new JCParam.Init();
        param.context = client.getContext();
        ZmfEngine.getInstance().initialize(param);
        mCameraOpen = false;
        mVideoFileOpen = false;
        mAudioStart = false;
        mSpeakerOn = false;
        mSawtooth = false;
        mAecMode = AEC_OS_SDE;
        mVideoMode = VIDEO_MODE_S_S;
        mWidth = 352;
        mHeight = 282;
        mFramerate = 10;
        mScreenCaptureWidth = 640;
        mScreenCaptureHeight = 360;
        mScreenCaptureFrameRate = 10;
        mCameras = ZmfEngine.getInstance().getCameras();
        if (mCameras.size() > 0) {
            mDefaultCameraIndex = mCameraIndex = 0;
        } else {
            mDefaultCameraIndex = mCameraIndex = -1;
        }
        mVideoFileId = "6e946949562a5cee94987c91ae53162b";
        mClient = client;
        mCallbacks.add(callback);
        mClient.addCallback(this);
        ZmfEngine.getInstance().addZmfNotifyListener(this);
        mAudioManager = (AudioManager) mClient.getContext().getSystemService(Context.AUDIO_SERVICE);
        mAgcOn = false;
    }

    @Override
    protected void destroyObj() {
        mCameras.clear();
        mCallbacks.clear();
        mClient.removeCallback(this);
        mClient = null;
        ZmfEngine.getInstance().removeZmfNotifyListener(this);
        ZmfEngine.getInstance().uninitialize();
    }

    @Override
    public boolean isCameraOpen() {
        return mCameraOpen;
    }

    @Override
    public boolean isSpeakerOn() {
        return mSpeakerOn;
    }

    @Override
    public String getCamera() {
        if (mCameraIndex >= 0 && mCameraIndex < mCameras.size()) {
            return mCameras.get(mCameraIndex);
        }
        return null;
    }

    @Override
    public List<String> getCameras() {
        return mCameras;
    }

    @Override
    public JCMediaDeviceVideoCanvas startCameraVideo(@RenderType int renderType) {
        if (mCameras.size() == 0) {
            return null;
        }
        JCMediaDeviceVideoCanvas canvas = new JCMediaDeviceVideoCanvas(mClient.getContext(), mCameras.get(mCameraIndex), renderType);
        canvas.autoRotate = autoRotate;
        //canvas.rotateAngleToScreen = rotateAngleToScreen;
        canvas.resume();
        mCameraCanvas.add(canvas);
        startCamera();
        return canvas;
    }

    @Override
    public JCMediaDeviceVideoCanvas startVideo(String videoSource, @RenderType int renderType) {
        JCMediaDeviceVideoCanvas canvas = new JCMediaDeviceVideoCanvas(mClient.getContext(), videoSource, renderType);
        canvas.autoRotate = autoRotate;
        //canvas.rotateAngleToScreen = rotateAngleToScreen;
        canvas.resume();
        if (TextUtils.equals(videoSource, mVideoFileId)) {
            startVideoFile();
            mVideoFileCanvas.add(canvas);
        } else {
            mOtherCanvas.add(canvas);
        }
        return canvas;
    }

    @Override
    public void stopVideo(JCMediaDeviceVideoCanvas canvas) {
        canvas.pause();
        if (mCameraCanvas.contains(canvas)) {
            mCameraCanvas.remove(canvas);
            if (mCameraCanvas.size() == 0) {
                JCLog.info(TAG, "No local video object");
                stopCamera();
            }
        } else if (mVideoFileCanvas.contains(canvas)) {
            mVideoFileCanvas.remove(canvas);
            if (mVideoFileCanvas.size() == 0) {
                JCLog.info(TAG, "No file video object");
                stopVideoFile();
            }
        } else {
            mOtherCanvas.remove(canvas);
        }
    }

    @Override
    public boolean startAudio() {
        if (mAudioStart) {
            JCLog.info(TAG, "Audio is turned on");
        } else {
            JCParam.AudioDeal param = new JCParam.AudioDeal();
            param.start = true;
            param.watch = mVideoMode == VIDEO_MODE_S_S || mVideoMode == VIDEO_MODE_S_L;
            param.os = (mAecMode != AEC_SDE);
            param.agc = mAgcOn;
            if (ZmfEngine.getInstance().dealAudio(param).succ) {
                JCLog.info(TAG, "Turn on the audio");
            } else {
                JCLog.info(TAG, "Turn on the audio fail");
            }
            mAudioManager.setMode(getAudioMode());
            mAudioStart = true;
        }
        return mAudioStart;
    }

    @Override
    public boolean stopAudio() {
        if (mAudioStart) {
            JCParam.AudioDeal param = new JCParam.AudioDeal();
            param.start = false;
            if (ZmfEngine.getInstance().dealAudio(param).succ) {
                JCLog.info(TAG, "Turn off audio");
            } else {
                JCLog.info(TAG, "Turn off audio fail");
            }
            mAudioManager.setMode(AudioManager.MODE_NORMAL);
            mAudioStart = false;
        } else {
            JCLog.info(TAG, "Audio has been turned off");
        }
        return !mAudioStart;
    }

    @Override
    public boolean startCamera() {
        if (mCameraOpen) {
            JCLog.info(TAG, "Camera is on");
            return true;
        }
        stopVideoFile();
        mCameraIndex = mDefaultCameraIndex;
        JCParam.CameraDeal param = new JCParam.CameraDeal();
        param.type = JCParam.CameraDeal.START;
        param.camera = mCameras.get(mCameraIndex);
        param.width = mWidth;
        param.height = mHeight;
        param.framerate = mFramerate;
        if (ZmfEngine.getInstance().dealCamera(param).succ) {
            JCLog.info(TAG, "Camera open");
            mCameraOpen = true;
            notifyCameraUpdate();
            return true;
        } else {
            JCLog.error(TAG, "Camera failed to open");
        }
        return false;
    }

    @Override
    public boolean stopCamera() {
        if (mCameraOpen) {
            JCParam.CameraDeal param = new JCParam.CameraDeal();
            param.type = JCParam.CameraDeal.STOP;
            param.camera = mCameras.get(mCameraIndex);
            if (ZmfEngine.getInstance().dealCamera(param).succ) {
                JCLog.info(TAG, "Camera off");
            } else {
                JCLog.error(TAG, "Camera failed");
            }
            mCameraOpen = false;
            mCameraIndex = mDefaultCameraIndex;
            notifyCameraUpdate();
        } else {
            JCLog.info(TAG, "Camera is off");
        }
        return true;
    }

    @Override
    public boolean switchCamera() {
        if (mCameraOpen) {
            JCParam.CameraDeal param = new JCParam.CameraDeal();
            param.type = JCParam.CameraDeal.SWITCH;
            param.switchedCamera = mCameras.get(mCameraIndex);
            mCameraIndex = (mCameraIndex + 1) % mCameras.size();
            param.camera = mCameras.get(mCameraIndex);
            if (mCameras.size() == 0) {
                return false;
            }
            if (mCameras.size() == 1) {
                JCLog.info(TAG, "Only one camera");
                return true;
            }
            param.width = mWidth;
            param.height = mHeight;
            param.framerate = mFramerate;
            if (ZmfEngine.getInstance().dealCamera(param).succ) {
                for (JCMediaDeviceVideoCanvas canvas : mCameraCanvas) {
                    canvas.switchCamera(param.camera);
                }
                notifyCameraUpdate();
                return true;
            } else {
                JCLog.error(TAG, "Switching camera failed");
            }
        } else {
            JCLog.error(TAG, "Switching camera is not turned on");
        }
        return false;
    }

    @Override
    public void specifyCamera(int cameraIndex) {
        if (cameraIndex >= 0 && cameraIndex < mCameras.size()) {
            mDefaultCameraIndex = cameraIndex;
        }
    }

    @Override
    public void setCameraProperty(int width, int height, int frameRate) {
        mWidth = width;
        mHeight = height;
        mFramerate = frameRate;
    }

    @Override
    public void setScreenCaptureProperty(int width, int height, int frameRate) {
        mScreenCaptureWidth = width;
        mScreenCaptureHeight = height;
        mScreenCaptureFrameRate = frameRate;
    }

    @Override
    public @CameraType int getCameraType(int cameraIndex) {
        if (cameraIndex >= 0 && cameraIndex < mCameras.size()) {
            if (mCameras.get(cameraIndex).equals(ZmfVideo.CaptureFront())) {
                return CAMERA_FRONT;
            } else if (mCameras.get(cameraIndex).equals(ZmfVideo.CaptureBack())) {
                return CAMERA_BACK;
            } else {
                return CAMERA_UNKNOWN;
            }
        }
        return CAMERA_NONE;
    }

    @Override
    public void enableSpeaker(boolean enable) {
        if (mSpeakerOn != enable) {
            mSpeakerOn = enable;
            mAudioManager.setSpeakerphoneOn(mSpeakerOn);
            notifyAudioOutputTypeChange();
        }
    }

    @Override
    public void setVideoMode(int videoMode, VideoModeOtherParam otherParam) {
        JCLog.info(TAG, "setVideoMode %d", videoMode);
        mVideoMode = videoMode;
        if (mVideoMode == VIDEO_MODE_S_S) {
            mWidth = 352;
            mHeight = 288;
            mFramerate = 10;
        } else if (mVideoMode == VIDEO_MODE_S_L) {
            mWidth = 352;
            mHeight = 288;
            mFramerate = 10;
        } else {
            mWidth = 640;
            mHeight = 480;
            if (mVideoMode == VIDEO_MODE_L_L) {
                mFramerate = 30;
            } else {
                mFramerate = 10;
            }
        }
        configMeidaParam(otherParam);
    }

    @Override
    public int getVideoMode() {
        return mVideoMode;
    }

    @Override
    public void setAecMode(int aecMode) {
        mAecMode = aecMode;
    }

    @Override
    public int getAecMode() {
        return mAecMode;
    }

    @Override
    public void setSawtooth(boolean sawtooth) {
        mSawtooth = sawtooth;
    }

    @Override
    public boolean isVideoFileOpen() {
        return mVideoFileOpen;
    }

    @Override
    public String getVideoFileId() {
        return mVideoFileId;
    }

    @Override
    public boolean startVideoFile() {
        if (mVideoFileOpen) {
            JCLog.info(TAG, "Video file capture is turned on");
            return true;
        }
        stopCamera();
        mVideoFileOpen = true;
        notifyCameraUpdate();
        return true;
    }

    @Override
    public void setVideoFileFrame(byte[] data, int format, int width, int height, int angle, int mirror, boolean keyFrame) {
        if (!mVideoFileOpen) {
            JCLog.error(TAG, "Video file capture is not open");
            return;
        }
        int[] cfg = null;
        int fixAngle = width >= height ? 0 : 90;
        if (format == H264) {
            cfg = new int[11];
            cfg[0] = data.length;
            cfg[3] = 1;
            cfg[5] = keyFrame ? 1 : 0;
            if (mVideoFileFrameBuffer == null || mVideoFileFrameBuffer.array().length < data.length) {
                mVideoFileFrameBuffer = ByteBuffer.allocateDirect(data.length).order(ByteOrder.nativeOrder());
            }
            mVideoFileFrameBuffer.position(0);
            mVideoFileFrameBuffer.put(data);
        } else {
            ByteBuffer srcFrame = ByteBuffer.allocateDirect(data.length).order(ByteOrder.nativeOrder());
            srcFrame.position(0);
            srcFrame.put(data);
            int imgSize = width * height;
            int frameSize = imgSize * 4;
            if (mVideoFileFrameBuffer == null || mVideoFileFrameBuffer.array().length < frameSize) {
                mVideoFileFrameBuffer = ByteBuffer.allocateDirect(frameSize).order(ByteOrder.nativeOrder());
            }
            ZmfVideo.convertToI420(mVideoFileFrameBuffer, format, srcFrame, width, height, fixAngle, new int[]{width, height});
        }
        ZmfVideo.onVideoCapture(mVideoFileId, mirror, angle + 360 - fixAngle, angle + 360 - fixAngle,
                fixAngle == 0 ? new int[]{width, height} : new int[]{height, width}, mVideoFileFrameBuffer,  format == H264 ? "H264" : null, cfg);
    }

    @Override
    public boolean stopVideoFile() {
        if (!mVideoFileOpen) {
            JCLog.info(TAG, "Video file capture is off");
            return true;
        }
        JCParam.CameraDeal param = new JCParam.CameraDeal();
        param.type = JCParam.CameraDeal.STOP;
        param.camera = mVideoFileId;
        ZmfEngine.getInstance().dealCamera(param);
        mVideoFileOpen = false;
        mVideoFileFrameBuffer = null;
        notifyCameraUpdate();
        return true;
    }

    @Override
    public void readyForH264() {
        startVideoFile();
        byte[] buf = new byte[10];
        buf[0] = 0x5;
        setVideoFileFrame(buf, H264, 1, 1, 0, 0, false);
        stopVideoFile();
    }

    @Override
    protected void addCallback(JCMediaDeviceCallback callback) {
        mCallbacks.add(callback);
    }

    @Override
    protected void removeCallback(JCMediaDeviceCallback callback) {
        mCallbacks.remove(callback);
    }

    @Override
    protected void configMeidaParam(VideoModeOtherParam otherParam) {
        JCLog.info(TAG, "configMeidaParam videoMode=%d", mVideoMode);
        MtcCallDb.Mtc_CallDbSetAnUsePresetVideoParams(false);
        MtcCallDb.Mtc_CallDbSetAudioCodecEnable("PCMU",false);
        MtcCallDb.Mtc_CallDbSetAudioCodecEnable("G722",false);
        MtcCallDb.Mtc_CallDbSetAudioCodecEnable("AMR-WB",false);
        MtcCallDb.Mtc_CallDbSetAudioCodecEnable("PCMA",false);
        MtcCallDb.Mtc_CallDbSetAudioCodecEnable("iLBC",false);
        MtcCallDb.Mtc_CallDbSetAudioCodecEnable("opus",false);
        MtcCallDb.Mtc_CallDbSetAudioCodecEnable("AMR",true);
        MtcCallDb.Mtc_CallDbSetAudioCodecEnable("G729",false);
        MtcCallDb.Mtc_CallDbSetAudioCodecByPriority("AMR", (short) 0);
        MtcCallDb.Mtc_CallDbSetVideoCodecEnable("VP8", true);
        //audio的结构体 ST_MNP_VOICE里面没有码率控制的，无法设置 0 ~ 12k
        MtcCallDb.Mtc_CallDbSetArsEnable(true);
        MtcCallDb.Mtc_CallDbSetVoiceArsParam(12000, 2000);
        MtcCallDb.Mtc_CallDbSetVadEnable(true);
        MtcCallDb.Mtc_CallDbSetAudioRed(false);
        MtcCallDb.Mtc_CallDbSetRxAnrEnable(false);
        MtcNumber Aec = new MtcNumber();
        MtcNumber Anr = new MtcNumber();
        MtcNumber Agc = new MtcNumber();
        MtcNumber Vad = new MtcNumber();
        MtcCallDb.Mtc_CallDbGetAudioQos(Aec,Anr,Agc,Vad);
        boolean bAec = false;
        boolean bAgc = false;
        boolean bVad = false;
        if (Aec.getValue() != 0) {
            bAec = true;
        }
        if (Agc.getValue() != 0) {
            bAgc = true;
        }
        if (Vad.getValue() != 0) {
            bVad = true;
        }
        MtcCallDb.Mtc_CallDbSetAudioRtxEnable(false);
        MtcCallDb.Mtc_CallDbSetSrtpCryptoType(MtcCallDbConstants.EN_MTC_DB_SRTP_CRYPTO_OFF);
        MtcCallDb.Mtc_CallDbSetAecMode(mAecMode == AEC_OS ? (short)MtcCallDb.EN_MTC_EC_OS : (short)MtcCallDb.EN_MTC_EC_AEC_SDE);
        MtcCallDb.Mtc_CallDbSetResolutionControl(false);
        MtcCallDb.Mtc_CallDbSetVideoArs(true);
        MtcCallDb.Mtc_CallDbSetSmallNaluEnable(true);
        MtcCallDb.Mtc_CallDbSetVideoRedFec(false);
        MtcCallDb.Mtc_CallDbSetVideoFramerate(mFramerate);
        if (mVideoMode == VIDEO_MODE_S_S || mVideoMode == VIDEO_MODE_S_L) {
            if (mSawtooth) {
                MtcCallDb.Mtc_CallDbSetAnVideoRecvResolution(240, 240);
            } else {
                MtcCallDb.Mtc_CallDbSetAnVideoRecvResolution(176, 144);
            }
            if (mVideoMode == VIDEO_MODE_S_S) {
                MtcCallDb.Mtc_CallDbSetVideoArsParm(80000, 10000, 0, 0);
                MtcCallDb.Mtc_CallDbSetVideoBitrate(50);
            } else {
                MtcCallDb.Mtc_CallDbSetVideoArsParm(200000, 10000, 0, 0);
                MtcCallDb.Mtc_CallDbSetVideoBitrate(150);
            }
            if (mVideoMode == VIDEO_MODE_S_S) {
                mAgcOn = false;
                MtcCallDb.Mtc_CallDbSetAudioQos(bAec, false, false,true);
            } else {
                mAgcOn = true;
                MtcCallDb.Mtc_CallDbSetAudioQos(bAec, false, true,true);
            }
        } else {
            if (mVideoMode == VIDEO_MODE_L_S) {
                mAgcOn = false;
                MtcCallDb.Mtc_CallDbSetAudioQos(bAec, false, false,true);
                MtcCallDb.Mtc_CallDbSetAnVideoRecvResolution(352, 288);
                if (mFramerate >= 15 || mSawtooth) {
                    MtcCallDb.Mtc_CallDbSetVideoArsParm(120000, 10000, 0, 0);
                } else {
                    MtcCallDb.Mtc_CallDbSetVideoArsParm(80000, 10000, 0, 0);
                }
                MtcCallDb.Mtc_CallDbSetVideoBitrate(50);
            } else {
                MtcCallDb.Mtc_CallDbSetAnVideoRecvResolution(640, 360);
                MtcCallDb.Mtc_CallDbSetAudioQos(bAec, false, true,true);
                MtcCallDb.Mtc_CallDbSetAnUsePresetVideoParams(true);
                MtcCallDb.Mtc_CallDbSetVoiceArsParam(75000, 20000);
                MtcCallDb.Mtc_CallDbSetVideoArsParm(1500000, 50000, 0, 0);
                MtcCallDb.Mtc_CallDbSetVideoBitrate(800);
            }
        }
        if (otherParam != null && otherParam.agc != -1) {
            mAgcOn = true;
            MtcCallDb.Mtc_CallDbSetAudioQos(bAec, false, otherParam.agc > 0, true);
        }
    }

    @Override
    public void onLogin(boolean result, @JCClient.ClientReason int reason) {

    }

    @Override
    public void onLogout(@JCClient.ClientReason int reason) {

    }

    @Override
    public void onClientStateChange(@JCClient.ClientState int state, @JCClient.ClientState int oldState) {

    }

    private void notifyCameraUpdate() {
        JCClientThreadImpl.getInstance().post(new Runnable() {
            @Override
            public void run() {
                JCLog.info(TAG, "Camera change");
                for (JCMediaDeviceCallback callback : mCallbacks) {
                    callback.onCameraUpdate();
                }
            }
        });
    }

    private void notifyAudioOutputTypeChange() {
        JCClientThreadImpl.getInstance().post(new Runnable() {
            @Override
            public void run() {
                JCLog.info(TAG, "Speaker change");
                for (JCMediaDeviceCallback callback : mCallbacks) {
                    callback.onAudioOutputTypeChange(mSpeakerOn);
                }
            }
        });
    }

    private void notifyRenderReceived(final String renderId) {
        JCClientThreadImpl.getInstance().post(new Runnable() {
            @Override
            public void run() {
                JCMediaDeviceVideoCanvas canvas = getCanvasByRenderId(renderId);
                if (canvas == null) {
                    JCLog.error(TAG, "Render object not found");
                    return;
                }
                JCLog.info(TAG, "Received rendering data " + renderId);
                for (JCMediaDeviceCallback callback : mCallbacks) {
                    callback.onRenderReceived(canvas);
                }
            }
        });
    }

    private void notifyRenderStart(final String renderId) {
        JCClientThreadImpl.getInstance().post(new Runnable() {
            @Override
            public void run() {
                JCMediaDeviceVideoCanvas canvas = getCanvasByRenderId(renderId);
                if (canvas == null) {
                    JCLog.error(TAG, "Render object not found");
                    return;
                }
                JCLog.info(TAG, "Start of rendering");
                for (JCMediaDeviceCallback callback : mCallbacks) {
                    callback.onRenderStart(canvas);
                }
            }
        });
    }

    private int getAudioMode() {
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.HONEYCOMB) {
            return AudioManager.MODE_NORMAL;
        }
        return AudioManager.MODE_IN_COMMUNICATION;
    }

    @Override
    protected boolean enableScreenCapture(boolean enable) {
        JCParam.CameraDeal param = new JCParam.CameraDeal();
        if (enable) {
            param.type = JCParam.CameraDeal.START;
            param.camera = ZmfVideo.CaptureScreen;
            param.width = mScreenCaptureWidth;
            param.height = mScreenCaptureHeight;
            param.framerate = mScreenCaptureFrameRate;
            if (ZmfEngine.getInstance().dealCamera(param).succ) {
                JCLog.info(TAG, "Screen sharing Capture is successfully turned on");
                return true;
            } else {
                JCLog.error(TAG, "Screen sharing Capture failed to open");
                return false;
            }
        } else {
            param.type = JCParam.CameraDeal.STOP;
            param.camera = ZmfVideo.CaptureScreen;
            if (ZmfEngine.getInstance().dealCamera(param).succ) {
                JCLog.info(TAG, "Screen sharing Capture closed successfully");
                return true;
            } else {
                JCLog.error(TAG, "Screen sharing Capture close failed");
                return false;
            }
        }
    }

    private JCMediaDeviceVideoCanvas getCanvasByRenderId(String renderId) {
        for (JCMediaDeviceVideoCanvas c : mCameraCanvas) {
            if (TextUtils.equals(c.getVideoSource(), renderId)) {
                return c;
            }
        }
        for (JCMediaDeviceVideoCanvas c : mOtherCanvas) {
            if (TextUtils.equals(c.getVideoSource(), renderId)) {
                return c;
            }
        }
        return null;
    }

    @Override
    public void onZmfNotify(JCNotify notify) {
        if (notify.type == JCNotify.MEDIA) {
            JCNotify.Media mediaNotify = notify.mediaNotify;
            if (mediaNotify.type == JCNotify.MEDIA_TYPE_RENDER_RECEIVED) {
                notifyRenderReceived(mediaNotify.renderReceived.renderId);
            } else if (mediaNotify.type == JCNotify.MEDIA_TYPE_RENDER_START) {
                notifyRenderStart(mediaNotify.renderStart.renderId);
            }
        }
    }
}
