package com.juphoon.cloud;

/**
 * 涂鸦模块
 *
 * @author juphoon
 */
public abstract class JCDoodle {

    /**
     * 默认的 Doodle 数据类型。开发者可以用其作为 Doodle 数据类型参数。
     */
    public static final String DATA_TYPE_DOODLE = "DATA_TYPE_DOODLE";

    /**
     * 开始涂鸦
     */
    public static final int ACTION_START = 0x100;
    /**
     * 无效的的 Action
     */
    public static final int ACTION_INVALID = ACTION_START - 1;
    /**
     * 结束涂鸦
     */
    public static final int ACTION_STOP = ACTION_START + 1;
    /**
     * 撤销
     */
    public static final int ACTION_UNDO = ACTION_STOP + 1;
    /**
     * 清除涂鸦
     */
    public static final int ACTION_CLEAR = ACTION_UNDO + 1;
    /**
     * 画曲线
     */
    public static final int ACTION_DRAW = ACTION_CLEAR + 1;
    /**
     * 橡皮擦
     */
    public static final int ACTION_ERASER = ACTION_DRAW + 1;
    /**
     * 贴图
     */
    public static final int ACTION_STICKER = ACTION_ERASER + 1;
    /**
     * 翻页
     */
    public static final int ACTION_SELECT_PAGE = ACTION_STICKER + 1;
    /**
     * 获取数据
     */
    public static final int ACTION_FETCH = ACTION_SELECT_PAGE + 1;
    /**
     * 请求
     */
    public static final int ACTION_REQUEST = ACTION_FETCH + 1;
    /**
     * 回应
     */
    public static final int ACTION_RESPONSE = ACTION_REQUEST + 1;
    /**
     * 自定义类型
     */
    public static final int ACTION_EXTRA_BASE = 0x200;

    static final String KEY_STICKER_CONTENT = "sticker_content";
    static final String KEY_STICKER_USER_DEFINE = "sticker_user_define";
    static final String KEY_CUSTOM_STICKER_EMOJI_UNICODE = "emoji_unicode";
    static final String KEY_CUSTOM_STICKER_NAME = "name";
    static final String KEY_CUSTOM_STICKER_X = "x";
    static final String KEY_CUSTOM_STICKER_Y = "y";
    static final String KEY_CUSTOM_STICKER_WIDTH = "width";
    static final String KEY_CUSTOM_STICKER_HEIGHT = "height";
    static final String KEY_CUSTOM_STICKER_ROTATE = "rotate";

    private static JCDoodle sDoodle;

    /**
     * 创建 JCDoodle 对象。
     *
     * @param callback {@link JCDoodleCallback} 回调接口，用于接收未处理或自定义的
     *                 {@link JCDoodleAction}。
     * @return 返回 JCDoodle 对象。
     */
    public static JCDoodle create(JCDoodleCallback callback) {
        if (sDoodle == null) {
            sDoodle = new JCDoodleImpl(callback, DoodleEngine.getInstance());
        }
        return sDoodle;
    }

    /**
     * 销毁 JCDoodle 对象
     */
    public static void destroy() {
        if (sDoodle != null) {
            JCClientThreadImpl.getInstance().post(new Runnable() {
                @Override
                public void run() {
                    sDoodle.destroyObj();
                    sDoodle = null;
                }
            });
        }
    }

    /**
     * 销毁对象
     */
    protected abstract void destroyObj();

    /**
     * 绑定 Doodle UI 控件
     *
     * @param interactor UI 控件实例
     */
    public abstract void bindDoodleInteractor(JCDoodleInteractor interactor);

    /**
     * 将 UI 控件产生的 {@link JCDoodleAction} 注入 SDK。
     * <p>
     * SDK 会通过 {@link JCDoodleCallback#onDoodleActionGenerated(JCDoodleAction)} 回调此 Doodle 动作。
     * 本方法仅供 UI 控件调用。
     *
     * @param doodleAction Doodle 动作。
     */
    public abstract void generateDoodleAction(JCDoodleAction doodleAction);

    /**
     * 将 {@link JCDoodleAction} 转化为 {@link String}。转换后的字符串可以用于数据传输。
     *
     * @param doodleAction Doodle 动作。
     * @return Doodle 动作对应的 {@link String}。
     * 可通过 {@link JCDoodle#doodleActionFromString(String)} 转换回对应的 {@link JCDoodleAction}
     */
    public abstract String stringFromDoodleAction(JCDoodleAction doodleAction);

    /**
     * 解析涂鸦数据。收到 {@link String} 数据后调用此接口，若数据的类型符合，其将转化为 {@link JCDoodleAction}，
     * 并通过 {@link JCDoodleInteractor#onDoodleReceived(JCDoodleAction)} 回调至 UI 控件用于渲染。
     * 若数据类型不符合，则返回 null。
     *
     * @param doodleActionData Doodle 动作对应的 {@link String}
     * @return Doodle 动作。
     */
    public abstract JCDoodleAction doodleActionFromString(String doodleActionData);
}
