package com.juphoon.cloud;

import android.text.TextUtils;

import com.justalk.cloud.lemon.MtcConfConstants;
import com.justalk.cloud.zmf.ZmfVideo;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

class JCMediaChannelImpl extends JCMediaChannel implements MtcEngine.MtcNotifyListener,
        JCClientCallback, JCMediaDeviceCallback {

    private static final String TAG = JCMediaChannelImpl.class.getSimpleName();
    private static final int PART_UPDATE_MIN_INTERNAL = 2000;
    private static final int MIN_CAPACITY = 0;
    private static final int MAX_CAPACITY = 16;
    private final static int VOLUME_ZERO = 1;
    private final static int VOLUME_LOW = 30;
    private final static int VOLUME_MID = 60;
    private final static String MESSAGE_TYPE = "MESSAGE_TYPE";
    private final static String MESSAGE_SENDER = "MESSAGE_SENDER";
    private final static String DEFAULT_PASSWORD = "123456";

    private JCClient mClient;
    private JCMediaDevice mMediaDevice;
    private List<JCMediaChannelCallback> mCallbacks = new ArrayList<>();
    private int mConfId;
    private String mDeliveryUri;
    private List<JCMediaChannelParticipant> mParticipants = new ArrayList<>();
    private JCMediaChannelParticipant mSelfParticipant;
    private String mChannelId;
    private int mChannelNumber;
    private String mTitle;
    @MediaChannelState
    private int mState;
    private boolean mUploadLocalAudio;
    private boolean mUploadLocalVideo;
    private boolean mAudioOutput;
    private String mScreenRenderId;
    private String mScreenUserId;
    @RecordState
    private int mRecordState;
    @CdnState
    private int mCdnState;
    private String mCdnUri;
    private String mRecordParam;
    private String mCapacity = "6";
    private String mSipCallerNum = null;
    private String mSipCoreNetwork = null;
    private String mPassword = DEFAULT_PASSWORD;
    private boolean mNotifyVolumeChange = true;

    private Map<Integer, String> mMapQuery = new HashMap<>();
    private Map<String, String> mMapCustomProperty = new HashMap<>();

    JCMediaChannelImpl(JCClient client, JCMediaDevice mediaDevice, JCMediaChannelCallback eventHandler) {
        if (client == null) {
            throw new RuntimeException("JCMediaChannel client cannot be null!");
        }
        if (mediaDevice == null) {
            throw new RuntimeException("JCMediaChannel mediaDevice cannot be null!");
        }
        if (eventHandler == null) {
            throw new RuntimeException("JCMediaChannel eventHandler cannot be null!");
        }
        mCallbacks.add(eventHandler);
        mClient = client;
        mMediaDevice = mediaDevice;
        mClient.addCallback(this);
        mMediaDevice.addCallback(this);
        MtcEngine.getInstance().addMtcNotifyListener(this);

        mTitle = "";
        mUploadLocalVideo = true;
        mUploadLocalAudio = false;
        mAudioOutput = true;
    }

    @Override
    public void onCameraUpdate() {
        if (mState == STATE_JOINED) {
            if (mSelfParticipant != null && mSelfParticipant.isVideo()
                    && (mMediaDevice.isCameraOpen() || mMediaDevice.isVideoFileOpen())) {
                JCParam.ConfMedia param = new JCParam.ConfMedia();
                param.type = JCParam.ConfMedia.BIND_CAMERA;
                param.confId = mConfId;
                if (mMediaDevice.isVideoFileOpen()) {
                    param.camera = mMediaDevice.getVideoFileId();
                } else {
                    param.camera = mMediaDevice.getCamera();
                }
                MtcEngine.getInstance().confMedia(param);
            }
        }
    }

    @Override
    public void onAudioOutputTypeChange(boolean speaker) {

    }

    @Override
    public void onRenderReceived(JCMediaDeviceVideoCanvas canvas) {

    }

    @Override
    public void onRenderStart(JCMediaDeviceVideoCanvas canvas) {

    }

    @Override
    public void onLogin(boolean result, @JCClient.ClientReason int reason) {

    }

    @Override
    public void onLogout(@JCClient.ClientReason int reason) {
        /// 登出状态下如果有会议存在则退出
        if (mState != STATE_IDLE) {
            notifyLeave(JCMediaChannel.REASON_NOT_LOGIN);
            setState(JCMediaChannel.STATE_IDLE);
        }
    }

    @Override
    public void onClientStateChange(@JCClient.ClientState int state, @JCClient.ClientState int oldState) {

    }

    @Override
    protected void destroyObj() {
        mCallbacks.clear();
        mClient.removeCallback(this);
        mMediaDevice.removeCallback(this);
        MtcEngine.getInstance().removeMtcNotifyListener(this);
        mMediaDevice = null;
        mClient = null;
    }

    @Override
    public String getChannelId() {
        return mChannelId;
    }

    @Override
    public int getChannelNumber() {
        return mChannelNumber;
    }

    @Override
    public String getTitle() {
        return mTitle;
    }

    @Override
    public int getState() {
        return mState;
    }

    @Override
    public List<JCMediaChannelParticipant> getParticipants() {
        return mParticipants;
    }

    @Override
    public boolean getUploadLocalAudio() {
        return mUploadLocalAudio;
    }

    @Override
    public boolean getUploadLocalVideo() {
        return mUploadLocalVideo;
    }

    @Override
    public boolean getAudioOutput() {
        return mAudioOutput;
    }

    @Override
    public String getScreenRenderId() {
        return mScreenRenderId;
    }

    @Override
    public String getScreenUserId() {
        return mScreenUserId;
    }

    @Override
    public int getRecordState() {
        return mRecordState;
    }

    @Override
    public int getCdnState() {
        return mCdnState;
    }

    @Override
    public boolean setConfig(String key, String value) {
        boolean ret = true;
        if (TextUtils.equals(key, CONFIG_CAPACITY)) {
            if (!TextUtils.isEmpty(value) && Integer.valueOf(mCapacity) > MIN_CAPACITY && Integer.valueOf(mCapacity) < MAX_CAPACITY) {
                mCapacity = value;
            } else {
                ret = false;
            }
        } else if (TextUtils.equals(key, CONFIG_SIP_CALLER_NUMBER)) {
            mSipCallerNum = value;
        } else if (TextUtils.equals(key, CONFIG_SIP_CORE_NETWORK)) {
            mSipCoreNetwork = value;
        } else if (TextUtils.equals(key, CONFIG_NOTIFY_VOLUME_CHANGE)) {
            mNotifyVolumeChange = Integer.valueOf(value) > 0;
        } else {
            ret = false;
        }
        return ret;
    }

    @Override
    public String getConfig(String key) {
        if (TextUtils.equals(key, CONFIG_CAPACITY)) {
            return mCapacity;
        } else if (TextUtils.equals(key, CONFIG_SIP_CALLER_NUMBER)) {
            return mSipCallerNum;
        } else if (TextUtils.equals(key, CONFIG_SIP_CORE_NETWORK)) {
            return mSipCoreNetwork;
        } else if (TextUtils.equals(key, CONFIG_NOTIFY_VOLUME_CHANGE)) {
            return mNotifyVolumeChange ? "1" : "0";
        }
        return null;
    }

    @Override
    public int query(String channelId) {
        JCParam.ConfQuery param = new JCParam.ConfQuery();
        param.channelId = channelId;
        JCResult result = MtcEngine.getInstance().queryConf(param);
        if (result.succ) {
            JCLog.info(TAG, "查询处理 操作号:%d", result.cookie);
            mMapQuery.put(result.cookie, channelId);
        } else {
            JCLog.error(TAG, "查询调用失败");
            JCMediaChannelQueryInfo queryInfo = new JCMediaChannelQueryInfo(channelId, -1, -1, null);
            notifyQuery(result.cookie, false, REASON_CALL_FUNCTION_ERROR, queryInfo);
        }
        return result.cookie;
    }

    @Override
    public boolean join(String channelId, Map<String, String> params) {
        if (mClient.getState() != JCClient.STATE_LOGINED) {
            notifyJoin(false, REASON_NOT_LOGIN);
        } else if (mState == STATE_IDLE) {
            int region = REGION_CHINA;
            int maxResolution = MAX_RESOLUTION_360p;
            boolean smoothMode = true;
            if (params != null) {
                if (params.containsKey(JOIN_PARAM_CDN)) {
                    mCdnUri = params.get(JOIN_PARAM_CDN);
                    JCLog.info(TAG, "param cdn=%s", mCdnUri);
                }
                if (params.containsKey(JOIN_PARAM_RECORD)) {
                    mRecordParam = params.get(JOIN_PARAM_RECORD);
                    JCLog.info(TAG, "param record=%s", mRecordParam);
                }
                if (params.containsKey(JOIN_PARAM_REGION)) {
                    region = Integer.valueOf(params.get(JOIN_PARAM_REGION));
                    JCLog.info(TAG, "param region=%d", region);
                }
                if (params.containsKey(JOIN_PARAM_PASSWORD)) {
                    mPassword = params.get(JOIN_PARAM_PASSWORD);
                    JCLog.info(TAG, "param password set");
                } else {
                    mPassword = DEFAULT_PASSWORD;
                }
                if (params.containsKey(JOIN_PARAM_MAX_RESOLUTION)) {
                    maxResolution = Integer.valueOf(params.get(JOIN_PARAM_MAX_RESOLUTION));
                    JCLog.info(TAG, "param maxResolution=%d set", maxResolution);
                }
                if (params.containsKey(JOIN_SMOOTH_MODE)) {
                    smoothMode = Boolean.parseBoolean(params.get(JOIN_SMOOTH_MODE));
                    JCLog.info(TAG, "param smoothMode= " + smoothMode);
                }
            }
            JCParam.ConfJoin param = new JCParam.ConfJoin();
            param.channelId = channelId;
            param.displayName = TextUtils.isEmpty(mClient.displayName) ? mClient.getUserId() : mClient.displayName;
            param.video = true;
            param.localAudio = mUploadLocalAudio;
            param.localVideo = mUploadLocalVideo;
            param.regionId = translateToMtcRegion(region);
            param.capacity = Integer.valueOf(mCapacity);
            param.password = mPassword;
            param.maxResolution = maxResolution;
            param.smoothMode = smoothMode;
            if (!TextUtils.isEmpty(mCdnUri)) {
                param.webCastingUri = mCdnUri;
            } else if (!TextUtils.isEmpty(mRecordParam)) {
                param.webCastingUri = "http://record/";
            }

            JCResult result = MtcEngine.getInstance().joinConf(param);
            if (result.succ) {
                mConfId = (int) result.longValue;
                mChannelId = channelId;
                setState(STATE_JOINING);
                JCLog.info(TAG, "join");
                return true;
            } else {
                JCLog.error(TAG, "join 调用失败");
                notifyJoin(false, REASON_CALL_FUNCTION_ERROR);
            }
        } else {
            JCLog.error(TAG, "已有会议加入");
            notifyJoin(false, REASON_ALREADY_JOINED);
        }
        return false;
    }

    @Override
    public boolean leave() {
        if (mState == STATE_JOINED) {
            JCParam.ConfLeave param = new JCParam.ConfLeave();
            param.confId = mConfId;
            param.type = JCParam.ConfLeave.LEAVE;
            if (MtcEngine.getInstance().leaveConf(param).succ) {
                JCLog.info(TAG, "leave 离开会议");
            } else {
                JCLog.error(TAG, "leave 调用失败");
                notifyLeave(REASON_CALL_FUNCTION_ERROR);
            }
        } else {
            JCLog.error(TAG, "leave 未在会议中");
        }
        return true;
    }

    @Override
    public boolean stop() {
        if (mState != STATE_IDLE) {
            JCParam.ConfLeave param = new JCParam.ConfLeave();
            param.confId = mChannelNumber;
            param.type = JCParam.ConfLeave.TERMINATE;
            if (MtcEngine.getInstance().leaveConf(param).succ) {
                JCLog.info(TAG, "stop");
            } else {
                JCLog.error(TAG, "stop 调用失败");
                notifyLeave(REASON_CALL_FUNCTION_ERROR);
            }
        } else {
            JCLog.error(TAG, "stop 无会议");
        }
        return true;
    }

    @Override
    public boolean enableUploadAudioStream(boolean enable) {
        if (mState == STATE_JOINED) {
            if (mSelfParticipant != null) {
                JCParam.ConfMedia param = new JCParam.ConfMedia();
                param.type = JCParam.ConfMedia.LOCAL_AUDIO;
                param.confId = mConfId;
                param.on = enable;
                if (MtcEngine.getInstance().confMedia(param).succ) {
                    mUploadLocalAudio = enable;
                    mSelfParticipant.audio = enable;
                    notifyMediaChannelPropertyChange();
                    JCLog.info(TAG, "enableLocalAudioStream %b", enable);
                    return true;
                } else {
                    JCLog.error(TAG, "enableLocalAudioStream 调用失败");
                }
            } else {
                JCLog.error(TAG, "enableLocalAudioStream 无自身对象");
            }
        } else {
            JCLog.info(TAG, "enableLocalAudioStream 未在会议中");
            mUploadLocalAudio = enable;
            return true;
        }
        return false;
    }

    @Override
    public boolean enableUploadVideoStream(boolean enable) {
        if (mState == STATE_JOINED) {
            if (mSelfParticipant != null) {
                JCParam.ConfMedia param = new JCParam.ConfMedia();
                param.type = JCParam.ConfMedia.LOCAL_VIDEO;
                param.confId = mConfId;
                param.on = enable;
                if (MtcEngine.getInstance().confMedia(param).succ) {
                    mUploadLocalVideo = enable;
                    mSelfParticipant.video = enable;
                    notifyMediaChannelPropertyChange();
                    JCLog.info(TAG, "enableLocalVideoStream %b", enable);
                    return true;
                } else {
                    JCLog.error(TAG, "enableLocalVideoStream 调用失败");
                }
            } else {
                JCLog.error(TAG, "enableLocalVideoStream 无自身对象");
            }
        } else {
            JCLog.info(TAG, "enableLocalVideoStream 未在会议中");
            mUploadLocalVideo = enable;
            return true;
        }
        return false;
    }

    @Override
    public boolean enableAudioOutput(boolean enable) {
        if (mState == STATE_JOINED) {
            if (mSelfParticipant != null) {
                JCParam.ConfMedia param = new JCParam.ConfMedia();
                param.type = JCParam.ConfMedia.LOCAL_AUDIO_OUT;
                param.confId = mConfId;
                param.on = enable;
                if (MtcEngine.getInstance().confMedia(param).succ) {
                    mAudioOutput = enable;
                    notifyMediaChannelPropertyChange();
                    JCLog.info(TAG, "enableAudioOutput");
                    return true;
                } else {
                    JCLog.error(TAG, "enableAudioOutput 调用失败");
                }
            } else {
                JCLog.error(TAG, "enableAudioOutput 无自身对象");
            }
            return false;
        } else {
            JCLog.info(TAG, "enableAudioOutput 未在会议中");
            mAudioOutput = enable;
        }
        return true;
    }

    @Override
    public boolean requestVideo(JCMediaChannelParticipant participant, @PictureSize int pictureSize) {
        if (mState == STATE_JOINED) {
            if (participant.isVideo()) {
                JCParam.ConfMedia param = new JCParam.ConfMedia();
                param.type = JCParam.ConfMedia.REQUEST_VIDEO;
                param.confId = mConfId;
                param.uri = participant.getRenderId();
                param.pictureSize = translateToMtcPictureSize(pictureSize);
                param.frameRate = 30;
                if (MtcEngine.getInstance().confMedia(param).succ) {
                    participant.pictureSize = pictureSize;
                    notifyParticipantUpdate(participant);
                    JCLog.info(TAG, "requestVideo");
                    return true;
                } else {
                    JCLog.error(TAG, "requestVideo 调用失败");
                }
            } else {
                JCLog.error(TAG, "requestVideo 对方未开启视频");
            }
        } else {
            JCLog.error(TAG, "requestVideo 未在会议中");
        }
        return false;
    }

    @Override
    public boolean requestScreenVideo(String screenUri, @PictureSize int pictureSize) {
        if (mState == STATE_JOINED) {
            JCParam.ConfMedia param = new JCParam.ConfMedia();
            param.type = JCParam.ConfMedia.REQUEST_VIDEO;
            param.confId = mConfId;
            param.uri = screenUri;
            param.pictureSize = translateToMtcPictureSize(pictureSize);
            param.frameRate = 30;
            if (MtcEngine.getInstance().confMedia(param).succ) {
                JCLog.info(TAG, "requestScreenVideo");
                return true;
            } else {
                JCLog.error(TAG, "requestScreenVideo 调用失败");
            }
        } else {
            JCLog.error(TAG, "requestVideo 未在会议中");
        }
        return false;
    }

    @Override
    public boolean enableCdn(boolean enable) {
        if (mCdnState == CDN_STATE_NONE) {
            JCLog.error(TAG, "无法使用Cdn推流");
            return false;
        }
        if (enable && mCdnState == CDN_STATE_RUNNING) {
            JCLog.error(TAG, "已经开启Cdn推流");
            return false;
        } else if (!enable && mCdnState == CDN_STATE_READY) {
            JCLog.error(TAG, "已经关闭Cdn推流");
            return false;
        }
        JCParam.ConfMedia param = new JCParam.ConfMedia();
        param.type = JCParam.ConfMedia.CDN;
        param.confId = mConfId;
        param.on = enable;
        if (MtcEngine.getInstance().confMedia(param).succ) {
            JCLog.info(TAG, "enableCdn");
            if (mCdnState == CDN_STATE_READY) {
                mCdnState = CDN_STATE_RUNNING;
            } else {
                mCdnState = CDN_STATE_READY;
            }
            notifyMediaChannelPropertyChange();
            return true;
        } else {
            JCLog.error(TAG, "enableCdn 调用失败");
        }
        return false;
    }

    @Override
    public boolean enableRecord(boolean enable) {
        if (mRecordState == RECORD_STATE_NONE) {
            JCLog.error(TAG, "无法使用视频录制");
            return false;
        }
        if (enable && mRecordState == RECORD_STATE_RUNNING) {
            JCLog.error(TAG, "已经开启屏幕录制");
            return false;
        } else if (!enable && mRecordState == RECORD_STATE_READY) {
            JCLog.error(TAG, "已经关闭屏幕录制");
            return false;
        }
        JCParam.ConfMedia param = new JCParam.ConfMedia();
        param.type = JCParam.ConfMedia.RECORD;
        param.confId = mConfId;
        param.on = enable;
        param.recordParam = mRecordParam;
        if (MtcEngine.getInstance().confMedia(param).succ) {
            JCLog.info(TAG, "enableRecord");
            if (mRecordState == RECORD_STATE_READY) {
                mRecordState = RECORD_STATE_RUNNING;
            } else {
                mRecordState = RECORD_STATE_READY;
            }
            notifyMediaChannelPropertyChange();
            return true;
        } else {
            JCLog.error(TAG, "enableRecord 调用失败");
        }
        return false;
    }

    @Override
    public boolean enableScreenShare(boolean enable) {
        if (mState != STATE_JOINED) {
            JCLog.error(TAG, "当前未加入会议");
            return false;
        }
        if (enable) {
            if (!TextUtils.isEmpty(mScreenUserId)) {
                JCLog.error(TAG, "enableScreenShare 已有屏幕分享");
                return false;
            }
        } else {
            if (TextUtils.isEmpty(mScreenUserId)) {
                JCLog.error(TAG, "enableScreenShare 没有开启屏幕分享");
                return false;
            }
            if (!TextUtils.equals(mScreenUserId, mClient.getUserId())) {
                JCLog.error(TAG, "不能关闭非自己的屏幕共享");
                return false;
            }
        }
        JCParam.ConfMedia param = new JCParam.ConfMedia();
        param.type = JCParam.ConfMedia.SCREEN_SHARE;
        param.confId = mConfId;
        param.on = enable;
        param.camera = ZmfVideo.CaptureScreen;
        if (MtcEngine.getInstance().confMedia(param).succ) {
            mMediaDevice.enableScreenCapture(enable);
            JCLog.info(TAG, "enableScreenShare " + enable);
            return true;
        } else {
            JCLog.error(TAG, "enableScreenShare调用失败");
            return false;
        }
    }

    @Override
    public JCMediaChannelParticipant getParticipant(String userId) {
        for (JCMediaChannelParticipant participant : mParticipants) {
            if (TextUtils.equals(participant.getUserId(), userId)) {
                return participant;
            }
        }
        return null;
    }

    @Override
    public String getStatistics() {
        if (mState == STATE_JOINED) {
            JCParam.ConfStatistics param = new JCParam.ConfStatistics();
            param.confId = mConfId;
            for (JCMediaChannelParticipant participant : mParticipants) {
                param.parts.add(participant.getUserId());
            }
            return MtcEngine.getInstance().confStatistics(param).strValue;
        } else {
            JCLog.error(TAG, "getStatistics 未在会议中");
        }
        return "";
    }

    @Override
    public boolean setCustomProperty(String key, String value) {
        if (TextUtils.isEmpty(value)) {
            mMapCustomProperty.remove(key);
        } else {
            mMapCustomProperty.put(key, value);
        }
        JCParam.ConfCustomProperty param = new JCParam.ConfCustomProperty();
        param.confId = mConfId;
        param.mapCustomProperty = mMapCustomProperty;
        return MtcEngine.getInstance().confUpdateCustomProperty(param).succ;
    }

    @Override
    public String getCustomProperty(String key) {
        return mMapCustomProperty.get(key);
    }

    @Override
    public boolean sendMessage(String type, String content, String toUserId) {
        if (mState == STATE_JOINED) {
            if (TextUtils.isEmpty(content)) {
                JCLog.error(TAG, "发送信息失败，内容不能为空");
                return false;
            }
            JCParam.ConfMessage param = new JCParam.ConfMessage();
            param.confId = mConfId;
            try {
                JSONObject object = new JSONObject();
                object.put(MESSAGE_TYPE, type);
                object.put(MESSAGE_SENDER, mClient.getUserId());
                param.type = object.toString();
                param.toUserID = toUserId;
                param.content = content;
                if (MtcEngine.getInstance().sendConfMessage(param).succ) {
                    return true;
                } else {
                    JCLog.error(TAG, "发送信息失败");
                }
            } catch (JSONException e) {
                e.printStackTrace();
            }
        } else {
            JCLog.error(TAG, "sendMessage 未在会议中");
        }
        return false;
    }

    @Override
    public boolean sendCommand(String name, String param) {
        if (mState == STATE_JOINED) {
            JCParam.ConfCommand paramCommand = new JCParam.ConfCommand();
            paramCommand.confId = mConfId;
            paramCommand.name = name;
            paramCommand.param = param;
            if(MtcEngine.getInstance().confSendCmd(paramCommand).succ) {
                JCLog.info(TAG, "发送指令成功");
                return true;
            } else {
                JCLog.error(TAG, "发送指令失败");
                return false;
            }
        } else {
            JCLog.error(TAG, "未在会议中,无法发送指令");
            return false;
        }
    }

    @Override
    public int inviteSipUser(String userId) {
        if (mState == STATE_JOINED) {
            JCParam.ConfInviteSipUser param = new JCParam.ConfInviteSipUser();
            param.channelNumber = mChannelNumber;
            param.userId = userId;
            param.password = mPassword;
            param.callerNum = mSipCallerNum;
            param.coreNetId = mSipCoreNetwork;
            JCResult result = MtcEngine.getInstance().confInviteSipUser(param);
            if (result.succ) {
                return result.cookie;
            } else {
                JCLog.error(TAG, "inviteSipUser 失败");
            }
        } else {
            JCLog.error(TAG, "inviteSipUser 未在会议中");
        }
        return -1;
    }

    @Override
    public void onNotify(JCNotify notify) {
        if (notify.type == JCNotify.CONF) {
            if (notify.confNotify.type == JCNotify.CONF_JOIN_OK) {
                JCNotify.Conf.JoinOk joinOk = notify.confNotify.joinOk;
                mChannelNumber = joinOk.number;
                mScreenUserId = joinOk.screenUserId;
                mScreenRenderId = joinOk.screenRenderId;
                mDeliveryUri = joinOk.deliveryUri;
                mTitle = joinOk.title;
                for (JCNotify.Conf.Partp partp : joinOk.partps) {
                    if (TextUtils.equals(partp.uri, mDeliveryUri)) {
                        JCLog.info(TAG, "过滤delivery");
                        dealDeliveryJoin();
                        continue;
                    }
                    JCMediaChannelParticipant participant = new JCMediaChannelParticipant();
                    participant.userId = partp.userId;
                    participant.displayName = partp.displayName;
                    participant.renderId = partp.uri;
                    participant.audio = hasAudioFromMtcRoleState(partp.state);
                    participant.video = hasVideoFromMtcRoleState(partp.state);
                    participant.talking = hasSipTalking(partp.state);
                    participant.type = translateFromMtcRole(partp.role);
                    mParticipants.add(participant);
                    if (TextUtils.equals(participant.getUserId(), mClient.getUserId())) {
                        mSelfParticipant = participant;
                    }
                    mMapCustomProperty = joinOk.customProperty;
                }
                notifyJoin(true, REASON_NONE);
            } else if (notify.confNotify.type == JCNotify.CONF_JOIN_FAIL) {
                notifyJoin(false, translateFromMtcReason(notify.confNotify.joinFail.reason));
            } else if (notify.confNotify.type == JCNotify.CONF_LEAVE) {
                notifyLeave(translateFromMtcReason(notify.confNotify.leave.reason));
            } else if (notify.confNotify.type == JCNotify.CONF_CANCEL_RESERVATION_Ok) {
                notifyStop(true, JCMediaChannel.REASON_NONE);
            } else if (notify.confNotify.type == JCNotify.CONF_CANCEL_RESERVATION_FAIL) {
                notifyStop(false, translateFromMtcReason(notify.confNotify.stop.reason));
            } else if (notify.confNotify.type == JCNotify.CONF_QUERY_OK) {
                String channelId = getQueryChannelId(notify.cookie);
                if (!TextUtils.isEmpty(channelId)) {
                    JCMediaChannelQueryInfo queryInfo = new JCMediaChannelQueryInfo(channelId,
                            notify.confNotify.queryOk.number, notify.confNotify.queryOk.clientCount, notify.confNotify.queryOk.mMembers);
                    notifyQuery(notify.cookie, true, REASON_NONE, queryInfo);
                }
            } else if (notify.confNotify.type == JCNotify.CONF_QUERY_FAIL) {
                String channelId = getQueryChannelId(notify.cookie);
                if (!TextUtils.isEmpty(channelId)) {
                    JCMediaChannelQueryInfo queryInfo = new JCMediaChannelQueryInfo(channelId, -1, -1, null);
                    notifyQuery(notify.cookie, false, translateFromMtcReason(notify.confNotify.queryFail.reason), queryInfo);
                }
            } else if (notify.confNotify.type == JCNotify.CONF_ADD_PARTICIPANT) {
                if (notify.confNotify.addParticipant.number == mChannelNumber) {
                    JCNotify.Conf.Partp partp = notify.confNotify.addParticipant.partp;
                    if (TextUtils.equals(partp.uri, mDeliveryUri)) {
                        JCLog.info(TAG, "过滤delivery");
                        dealDeliveryJoin();
                        return;
                    }
                    JCMediaChannelParticipant participant = new JCMediaChannelParticipant();
                    participant.userId = partp.userId;
                    participant.displayName = partp.displayName;
                    participant.renderId = partp.uri;
                    participant.audio = hasAudioFromMtcRoleState(partp.state);
                    participant.video = hasVideoFromMtcRoleState(partp.state);
                    participant.talking = hasSipTalking(partp.state);
                    participant.type = translateFromMtcRole(partp.role);
                    mParticipants.add(participant);
                    notifyParticipantJoin(participant);
                } else {
                    JCLog.error(TAG, "会议号不对");
                }
            } else if (notify.confNotify.type == JCNotify.CONF_REMOVE_PARTICIPANT) {
                if (notify.confNotify.removeParticipant.number == mChannelNumber) {
                    JCMediaChannelParticipant participant = getParticipant(notify.confNotify.removeParticipant.partp.userId);
                    if (participant != null) {
                        mParticipants.remove(participant);
                        notifyParticipantLeft(participant);
                    } else {
                        JCLog.error(TAG, "无该成员");
                    }
                } else {
                    JCLog.error(TAG, "会议号不对");
                }
            } else if (notify.confNotify.type == JCNotify.CONF_PARTICIPANT_CNANGED) {
                if (notify.confNotify.participantChanged.number == mChannelNumber) {
                    JCNotify.Conf.ParticipantChanged partp = notify.confNotify.participantChanged;
                    JCMediaChannelParticipant participant = getParticipant(partp.userId);
                    if (participant != null) {
                        participant.displayName = partp.displayName;
                        participant.audio = hasAudioFromMtcRoleState(partp.state);
                        participant.video = hasVideoFromMtcRoleState(partp.state);
                        participant.talking = hasSipTalking(partp.state);
                        participant.type = translateFromMtcRole(partp.role);
                        notifyParticipantUpdate(participant);
                    } else {
                        JCLog.error(TAG, "无该成员");
                    }
                } else {
                    JCLog.error(TAG, "会议号不对");
                }
            } else if (notify.confNotify.type == JCNotify.CONF_VOLUME_CHANGED) {
                if (notify.confNotify.volumeChanged.number == mChannelNumber) {
                    JCNotify.Conf.VolumeChanged volumeChanged = notify.confNotify.volumeChanged;
                    for (String userId : volumeChanged.mapVolume.keySet()) {
                        JCMediaChannelParticipant participant = getParticipant(userId);
                        if (participant != null) {
                            int newStatus = translateFromMtcVolumeStatus(volumeChanged.mapVolume.get(userId));
                            if (participant.getVolumeStatus() != newStatus) {
                                participant.volumeStatus = newStatus;
                                if (mNotifyVolumeChange) {
                                    notifyParticipantUpdate(participant);
                                }
                            }
                        } else {
                            JCLog.error(TAG, "无该成员");
                        }
                    }
                } else {
                    JCLog.error(TAG, "会议号不对");
                }
            } else if (notify.confNotify.type == JCNotify.CONF_ERROR_EVENT) {

            } else if (notify.confNotify.type == JCNotify.CONF_PROP_CNANGED) {
                mScreenRenderId = notify.confNotify.propChange.screenRenderId;
                mScreenUserId = notify.confNotify.propChange.screenUserId;
                mTitle = notify.confNotify.propChange.title;
                mMapCustomProperty = notify.confNotify.propChange.customProperty;
                notifyMediaChannelPropertyChange();
            } else if (notify.confNotify.type == JCNotify.CONF_MESSAGE_RECEIVED) {
                try {
                    JSONObject object = new JSONObject(notify.confNotify.messageReceived.type);
                    String type = object.optString(MESSAGE_TYPE);
                    String sender = object.optString(MESSAGE_SENDER);
                    notifyMessageReceived(type, notify.confNotify.messageReceived.content, sender);
                } catch (JSONException e) {
                    e.printStackTrace();
                    if (!TextUtils.isEmpty(notify.confNotify.messageReceived.fromUserId)) {
                        // 兼容webrtc，使用text接口发送文本
                        notifyMessageReceived("text", notify.confNotify.messageReceived.content, notify.confNotify.messageReceived.fromUserId);
                    }
                }
            } else if (notify.confNotify.type == JCNotify.CONF_SIP_INVITE_OK) {
                notifyInviteSipUserResult(notify.cookie, true, REASON_NONE);
            } else if (notify.confNotify.type == JCNotify.CONF_SIP_INVITE_FAIL) {
                notifyInviteSipUserResult(notify.cookie, false, REASON_OTHER);
            }
        }
    }

    @Override
    protected void addCallback(JCMediaChannelCallback eventHandler) {
        mCallbacks.add(eventHandler);
    }

    @Override
    protected void removeCallback(JCMediaChannelCallback eventHandler) {
        mCallbacks.remove(eventHandler);
    }

    /*************** private ***************/

    private void notifyJoin(final boolean result, @MediaChannelReason final int reason) {
        JCClientThreadImpl.getInstance().post(new Runnable() {
            @Override
            public void run() {
                JCLog.info(TAG, "加入结果:%b 原因:%d", result, reason);
                for (JCMediaChannelCallback callback : mCallbacks) {
                    callback.onJoin(result, reason, mChannelId);
                }
            }
        });
        if (result) {
            setState(STATE_JOINED);
        } else {
            setState(STATE_IDLE);
        }
    }

    private void notifyStop(final boolean result, @MediaChannelReason final int reason) {
        JCClientThreadImpl.getInstance().post(new Runnable() {
            @Override
            public void run() {
                JCLog.info(TAG, "解散结果:%b 原因:%d", result, reason);
                for (JCMediaChannelCallback callback : mCallbacks) {
                    callback.onStop(result, reason);
                }
            }
        });
    }

    private void notifyLeave(@MediaChannelReason final int reason) {
        JCClientThreadImpl.getInstance().post(new Runnable() {
            @Override
            public void run() {
                JCLog.info(TAG, "离开原因:%d", reason);
                for (JCMediaChannelCallback callback : mCallbacks) {
                    callback.onLeave(reason, mChannelId);
                }
            }
        });
        setState(STATE_IDLE);
    }

    private void notifyQuery(final int operationId, final boolean result, @MediaChannelReason final int reason, final JCMediaChannelQueryInfo queryInfo) {
        JCClientThreadImpl.getInstance().post(new Runnable() {
            @Override
            public void run() {
                JCLog.info(TAG, "查询 操作号:%d 结果:%b 原因:%d", operationId, result, reason);
                for (JCMediaChannelCallback callback : mCallbacks) {
                    callback.onQuery(operationId, result, reason, queryInfo);
                }
            }
        });
    }

    private void notifyStateChange(@MediaChannelState final int state, @MediaChannelState final int oldState) {
        JCClientThreadImpl.getInstance().post(new Runnable() {
            @Override
            public void run() {
                JCLog.info(TAG, "状态变化 %d->%d", oldState, state);
                for (JCMediaChannelCallback callback : mCallbacks) {
                    callback.onMediaChannelStateChange(state, oldState);
                }
            }
        });
    }

    private void notifyMediaChannelPropertyChange() {
        JCClientThreadImpl.getInstance().post(new Runnable() {
            @Override
            public void run() {
                JCLog.info(TAG, "属性变化");
                for (JCMediaChannelCallback callback : mCallbacks) {
                    callback.onMediaChannelPropertyChange();
                }
            }
        });
    }

    private void notifyParticipantJoin(final JCMediaChannelParticipant participant) {
        JCClientThreadImpl.getInstance().post(new Runnable() {
            @Override
            public void run() {
                JCLog.info(TAG, "用户加入%s", participant.getUserId());
                for (JCMediaChannelCallback callback : mCallbacks) {
                    callback.onParticipantJoin(participant);
                }
            }
        });
    }

    private void notifyParticipantLeft(final JCMediaChannelParticipant participant) {
        JCClientThreadImpl.getInstance().post(new Runnable() {
            @Override
            public void run() {
                JCLog.info(TAG, "用户离开%s", participant.getUserId());
                for (JCMediaChannelCallback callback : mCallbacks) {
                    callback.onParticipantLeft(participant);
                }
            }
        });
    }

    private void notifyParticipantUpdate(final JCMediaChannelParticipant participant) {
        if (System.currentTimeMillis() - participant.lastUpdateTime > PART_UPDATE_MIN_INTERNAL) {
            participant.lastUpdateTime = System.currentTimeMillis();
            JCClientThreadImpl.getInstance().postDelayed(new Runnable() {
                @Override
                public void run() {
                    JCLog.info(TAG, "用户更新%s", participant.getUserId());
                    for (JCMediaChannelCallback callback : mCallbacks) {
                        callback.onParticipantUpdate(participant);
                    }
                }
            }, PART_UPDATE_MIN_INTERNAL);
        }
    }

    private void notifyMessageReceived(final String type, final String content, final String fromUserId) {
        JCClientThreadImpl.getInstance().post(new Runnable() {
            @Override
            public void run() {
                JCLog.info(TAG, "收到%s发送的信息 类型:%s内容:%s", fromUserId, type, content);
                for (JCMediaChannelCallback callback : mCallbacks) {
                    callback.onMessageReceive(type, content, fromUserId);
                }
            }
        });
    }

    private void notifyInviteSipUserResult(final int operationId, final boolean result, final int reason) {
        JCClientThreadImpl.getInstance().post(new Runnable() {
            @Override
            public void run() {
                JCLog.info(TAG, "sip邀请 操作号:%d 结果:%b 原因:%d", operationId, result, reason);
                for (JCMediaChannelCallback callback : mCallbacks) {
                    callback.onInviteSipUserResult(operationId, result, reason);
                }
            }
        });
    }

    private void setState(@MediaChannelState int state) {
        if (mState != state) {
            int oldState = mState;
            mState = state;
            if (mState == STATE_IDLE) {
                doWhenIdle();
            } else if (mState == STATE_JOINED) {
                doWhenJoin();
            }
            notifyStateChange(mState, oldState);
        }
    }

    private void doWhenJoin() {
        if (mSelfParticipant.isVideo() && mSelfParticipant.isAudio()) {
            JCParam.ConfMedia param = new JCParam.ConfMedia();
            param.type = JCParam.ConfMedia.LOCAL_ALL;
            param.confId = mConfId;
            param.on = true;
            MtcEngine.getInstance().confMedia(param);
        } else if (mSelfParticipant.isVideo()) {
            JCParam.ConfMedia param = new JCParam.ConfMedia();
            param.type = JCParam.ConfMedia.LOCAL_VIDEO;
            param.confId = mConfId;
            param.on = true;
            MtcEngine.getInstance().confMedia(param);
        } else if (mSelfParticipant.isAudio()) {
            JCParam.ConfMedia param = new JCParam.ConfMedia();
            param.type = JCParam.ConfMedia.LOCAL_AUDIO;
            param.confId = mConfId;
            param.on = true;
            MtcEngine.getInstance().confMedia(param);
        }
        enableAudioOutput(mAudioOutput);
        if (mMediaDevice != null) {
            onCameraUpdate();
            mMediaDevice.startAudio();
        } else {
            JCLog.error(TAG, "已销毁");
        }
        MtcEngine.getInstance().hasMediaChannel = true;
    }

    private void doWhenIdle() {
        mChannelId = "";
        mTitle = "";
        mScreenUserId = null;
        mScreenRenderId = null;
        mDeliveryUri = null;
        mRecordState = RECORD_STATE_NONE;
        mCdnState = CDN_STATE_NONE;
        mCdnUri = null;
        mRecordParam = null;
        mParticipants.clear();
        if (mMediaDevice != null) {
            mMediaDevice.stopAudio();
            mMediaDevice.enableSpeaker(false);
        } else {
            JCLog.error(TAG, "已销毁");
        }
        MtcEngine.getInstance().hasMediaChannel = false;
        mMapCustomProperty.clear();
        mMapQuery.clear();
    }

    private int translateToMtcPictureSize(@JCMediaChannel.PictureSize int pictureSize) {
        if (pictureSize == PICTURESIZE_MIN) {
            return MtcConfConstants.MTC_CONF_PS_MIN;
        } else if (pictureSize == PICTURESIZE_SMALL) {
            return MtcConfConstants.MTC_CONF_PS_SMALL;
        } else if (pictureSize == PICTURESIZE_LARGE) {
            return MtcConfConstants.MTC_CONF_PS_LARGE;
        } else if (pictureSize == PICTURESIZE_MAX) {
            return MtcConfConstants.MTC_CONF_PS_MAX;
        }
        return MtcConfConstants.MTC_CONF_PS_OFF;
    }

    private void dealDeliveryJoin() {
        if (!TextUtils.isEmpty(mRecordParam)) {
            mRecordState = RECORD_STATE_READY;
        }
        if (!TextUtils.isEmpty(mCdnUri)) {
            mCdnState = CDN_STATE_READY;
        }
        notifyMediaChannelPropertyChange();
    }

    private boolean hasAudioFromMtcRoleState(int state) {
        return (state & MtcConfConstants.MTC_CONF_STATE_AUDIO) > 0;
    }

    private boolean hasVideoFromMtcRoleState(int state) {
        return (state & MtcConfConstants.MTC_CONF_STATE_VIDEO) > 0;
    }

    private boolean hasSipTalking(int state) {
        return (state & 0x1000000) > 0;
    }

    private @ParticipantType
    int translateFromMtcRole(int role) {
        if ((role & MtcConfConstants.MTC_CONF_ROLE_DELIVERY_PSTN) > 0) {
            return PARTICIPANT_TYPE_PSTN;
        } else if ((role & MtcConfConstants.MTC_CONF_ROLE_DELIVERY_WEBRTC) > 0) {
            return PARTICIPANT_TYPE_WEBRTC;
        } else {
            return PARTICIPANT_TYPE_NORMAL;
        }
    }

    @MediaChannelReason
    private int translateFromMtcReason(int reason) {
        switch (reason) {
            case MtcConfConstants.EN_MTC_CONF_REASON_LEAVED:
                return REASON_QUIT;
            case MtcConfConstants.EN_MTC_CONF_REASON_KICKED:
                return REASON_KICKED;
            case MtcConfConstants.EN_MTC_CONF_REASON_OFFLINE:
                return REASON_OFFLINE;
            case MtcConfConstants.EN_MTC_CONF_REASON_OVER:
                return REASON_OVER;
            case MtcConfConstants.EN_MTC_CONF_REASON_TIMEOUT:
                return REASON_TIMEOUT;
            case MtcConfConstants.EN_MTC_CONF_REASON_MEMBER_FULL:
                return REASON_FULL;
            case MtcConfConstants.EN_MTC_CONF_REASON_INVALID_PASSWORD:
                return REASON_INVALID_PASSWORD;
            default:
                return REASON_OTHER;
        }
    }

    @VolumeStatus
    private int translateFromMtcVolumeStatus(int status) {
        if (status == 0) {
            return VOLUME_STATUS_NONE;
        } else if (status == VOLUME_ZERO) {
            return VOLUME_STATUS_ZERO;
        } else if (status > VOLUME_ZERO && status <= VOLUME_LOW) {
            return VOLUME_STATUS_LOW;
        } else if (status > VOLUME_LOW && status <= VOLUME_MID) {
            return VOLUME_STATUS_MID;
        } else {
            return VOLUME_STATUS_HIGH;
        }
    }

    private int translateToMtcRegion(@Region int region) {
        if (region == REGION_CHINA) {
            return 10;
        }
        return 30;
    }

    private String getQueryChannelId(int cookie) {
        if (mMapQuery.containsKey(cookie)) {
            return mMapQuery.get(cookie);
        }
        JCLog.error(TAG, "找不到会议查询id");
        return null;
    }
}
