package com.juphoon.cloud;

import android.content.Context;
import android.media.AudioManager;
import android.os.Build;

import com.justalk.cloud.lemon.MtcCallDb;
import com.justalk.cloud.lemon.MtcCallDbConstants;
import com.justalk.cloud.lemon.MtcNumber;
import com.justalk.cloud.zmf.ZmfVideo;

import java.util.ArrayList;
import java.util.List;

class JCMediaDeviceImpl extends JCMediaDevice implements JCClientCallback {

    static final String TAG = JCMediaDeviceImpl.class.getSimpleName();

    private List<JCMediaDeviceCallback> mCallbacks = new ArrayList<>();
    private JCClient mClient;
    private int mDefaultCameraIndex;
    private int mCameraIndex;
    private boolean mCameraOpen;
    private boolean mAudioStart;
    private boolean mSpeakerOn;
    private int mWidth;
    private int mHeight;
    private int mFramerate;
    private int mScreenCaptureWidth;
    private int mScreenCaptureHeight;
    private int mScreenCaptureFrameRate;
    private List<JCMediaDeviceVideoCanvas> mCameraCanvas = new ArrayList<>();
    private List<JCMediaDeviceVideoCanvas> mOtherCanvas = new ArrayList<>();
    private List<String> mCameras;
    private AudioManager mAudioManager;
    private boolean mSawtooth;
    private @AecMode int mAecMode;
    private @VideoMode int mVideoMode;

    JCMediaDeviceImpl(JCClient client, JCMediaDeviceCallback callback) {
        JCParam.Init param = new JCParam.Init();
        param.context = client.getContext();
        ZmfEngine.getInstance().initialize(param);
        mCameraOpen = false;
        mAudioStart = false;
        mSpeakerOn = false;
        mSawtooth = false;
        mAecMode = AEC_OS_SDE;
        mVideoMode = VIDEO_MODE_S_S;
        mWidth = 352;
        mHeight = 282;
        mFramerate = 10;
        mScreenCaptureWidth = 640;
        mScreenCaptureHeight = 360;
        mScreenCaptureFrameRate = 10;
        mCameras = ZmfEngine.getInstance().getCameras();
        if (mCameras.size() > 0) {
            mDefaultCameraIndex = mCameraIndex = 0;
        } else {
            mDefaultCameraIndex = mCameraIndex = -1;
        }

        mClient = client;
        mCallbacks.add(callback);
        mClient.addCallback(this);
        mAudioManager = (AudioManager) mClient.getContext().getSystemService(Context.AUDIO_SERVICE);
    }

    @Override
    protected void destroyObj() {
        mCameras.clear();
        mCallbacks.clear();
        mClient.removeCallback(this);
        mClient = null;
        ZmfEngine.getInstance().uninitialize();
    }

    @Override
    public boolean isCameraOpen() {
        return mCameraOpen;
    }

    @Override
    public boolean isSpeakerOn() {
        return mSpeakerOn;
    }

    @Override
    public String getCamera() {
        if (mCameraIndex >= 0 && mCameraIndex < mCameras.size()) {
            return mCameras.get(mCameraIndex);
        }
        return null;
    }

    @Override
    public List<String> getCameras() {
        return mCameras;
    }

    @Override
    public JCMediaDeviceVideoCanvas startCameraVideo(@RenderType int renderType) {
        if (mCameras.size() == 0) {
            return null;
        }
        JCMediaDeviceVideoCanvas canvas = new JCMediaDeviceVideoCanvas(mClient.getContext(), mCameras.get(mCameraIndex), renderType);
        canvas.autoRotate = autoRotate;
        //canvas.rotateAngleToScreen = rotateAngleToScreen;
        canvas.resume();
        mCameraCanvas.add(canvas);
        startCamera();
        return canvas;
    }

    @Override
    public JCMediaDeviceVideoCanvas startVideo(String videoSource, @RenderType int renderType) {
        JCMediaDeviceVideoCanvas canvas = new JCMediaDeviceVideoCanvas(mClient.getContext(), videoSource, renderType);
        canvas.autoRotate = autoRotate;
        //canvas.rotateAngleToScreen = rotateAngleToScreen;
        canvas.resume();
        mOtherCanvas.add(canvas);
        return canvas;
    }

    @Override
    public void stopVideo(JCMediaDeviceVideoCanvas canvas) {
        canvas.pause();
        if (mCameraCanvas.contains(canvas)) {
            mCameraCanvas.remove(canvas);
            if (mCameraCanvas.size() == 0) {
                JCLog.info(TAG, "无本地视频对象");
                stopCamera();
            }
        } else {
            mOtherCanvas.remove(canvas);
        }
    }

    @Override
    public boolean startAudio() {
        if (mAudioStart) {
            JCLog.info(TAG, "已经打开音频");
        } else {
            JCParam.AudioDeal param = new JCParam.AudioDeal();
            param.start = true;
            param.watch = mVideoMode == VIDEO_MODE_S_S || mVideoMode == VIDEO_MODE_S_L;
            param.os = (mAecMode != AEC_SDE);
            if (ZmfEngine.getInstance().dealAudio(param).succ) {
                JCLog.info(TAG, "打开音频");
            } else {
                JCLog.info(TAG, "打开音频失败");
            }
            mAudioManager.setMode(getAudioMode());
            mAudioStart = true;
        }
        return mAudioStart;
    }

    @Override
    public boolean stopAudio() {
        if (mAudioStart) {
            JCParam.AudioDeal param = new JCParam.AudioDeal();
            param.start = false;
            if (ZmfEngine.getInstance().dealAudio(param).succ) {
                JCLog.info(TAG, "关闭音频");
            } else {
                JCLog.info(TAG, "关闭音频失败");
            }
            mAudioManager.setMode(AudioManager.MODE_NORMAL);
            mAudioStart = false;
        } else {
            JCLog.info(TAG, "已经关闭音频");
        }
        return !mAudioStart;
    }

    @Override
    public boolean startCamera() {
        if (mCameraOpen) {
            JCLog.info(TAG, "摄像头已打开");
            return true;
        }
        mCameraIndex = mDefaultCameraIndex;
        JCParam.CameraDeal param = new JCParam.CameraDeal();
        param.type = JCParam.CameraDeal.START;
        param.camera = mCameras.get(mCameraIndex);
        param.width = mWidth;
        param.height = mHeight;
        param.framerate = mFramerate;
        if (ZmfEngine.getInstance().dealCamera(param).succ) {
            JCLog.info(TAG, "摄像头打开");
            mCameraOpen = true;
            notifyCameraUpdate();
            return true;
        } else {
            JCLog.error(TAG, "摄像头打开失败");
        }
        return false;
    }

    @Override
    public boolean stopCamera() {
        if (mCameraOpen) {
            JCParam.CameraDeal param = new JCParam.CameraDeal();
            param.type = JCParam.CameraDeal.STOP;
            param.camera = mCameras.get(mCameraIndex);
            if (ZmfEngine.getInstance().dealCamera(param).succ) {
                JCLog.info(TAG, "摄像头关闭");
            } else {
                JCLog.error(TAG, "摄像头失败");
            }
            mCameraOpen = false;
            mCameraIndex = mDefaultCameraIndex;
        } else {
            JCLog.info(TAG, "摄像头已关闭");
        }
        return true;
    }

    @Override
    public boolean switchCamera() {
        if (mCameraOpen) {
            JCParam.CameraDeal param = new JCParam.CameraDeal();
            param.type = JCParam.CameraDeal.SWITCH;
            param.switchedCamera = mCameras.get(mCameraIndex);
            mCameraIndex = (mCameraIndex + 1) % mCameras.size();
            param.camera = mCameras.get(mCameraIndex);
            if (mCameras.size() == 0) {
                return false;
            }
            if (mCameras.size() == 1) {
                JCLog.info(TAG, "只有一个摄像头");
                return true;
            }
            param.width = mWidth;
            param.height = mHeight;
            param.framerate = mFramerate;
            if (ZmfEngine.getInstance().dealCamera(param).succ) {
                for (JCMediaDeviceVideoCanvas canvas : mCameraCanvas) {
                    canvas.switchCamera(param.camera);
                }
                notifyCameraUpdate();
                return true;
            } else {
                JCLog.error(TAG, "切换摄像头失败");
            }
        } else {
            JCLog.error(TAG, "未打开切换摄像头");
        }
        return false;
    }

    @Override
    public void specifyCamera(int cameraIndex) {
        if (cameraIndex >= 0 && cameraIndex < mCameras.size()) {
            mDefaultCameraIndex = cameraIndex;
        }
    }

    @Override
    public void setCameraProperty(int width, int height, int frameRate) {
        mWidth = width;
        mHeight = height;
        mFramerate = frameRate;
    }

    @Override
    public void setScreenCaptureProperty(int width, int height, int frameRate) {
        mScreenCaptureWidth = width;
        mScreenCaptureHeight = height;
        mScreenCaptureFrameRate = frameRate;
    }

    @Override
    public @CameraType int getCameraType(int cameraIndex) {
        if (cameraIndex >= 0 && cameraIndex < mCameras.size()) {
            if (mCameras.get(cameraIndex).equals(ZmfVideo.CaptureFront())) {
                return CAMERA_FRONT;
            } else if (mCameras.get(cameraIndex).equals(ZmfVideo.CaptureBack())) {
                return CAMERA_BACK;
            } else {
                return CAMERA_UNKNOWN;
            }
        }
        return CAMERA_NONE;
    }

    @Override
    public void enableSpeaker(boolean enable) {
        if (mSpeakerOn != enable) {
            mSpeakerOn = enable;
            mAudioManager.setSpeakerphoneOn(mSpeakerOn);
            notifyAudioOutputTypeChange();
        }
    }

    @Override
    public void setVideoMode(int videoMode, VideoModeOtherParam otherParam) {
        JCLog.info(TAG, "setVideoMode %d", videoMode);
        mVideoMode = videoMode;
        if (mVideoMode == VIDEO_MODE_S_S) {
            mWidth = 352;
            mHeight = 282;
            mFramerate = 10;
        } else if (mVideoMode == VIDEO_MODE_S_L) {
            mWidth = 352;
            mHeight = 282;
            mFramerate = 10;
        } else {
            mWidth = 640;
            mHeight = 480;
            if (mVideoMode == VIDEO_MODE_L_L) {
                mFramerate = 30;
            } else {
                mFramerate = 10;
            }
        }
        configMeidaParam(otherParam);
    }

    @Override
    public int getVideoMode() {
        return mVideoMode;
    }

    @Override
    public void setAecMode(int aecMode) {
        mAecMode = aecMode;
    }

    @Override
    public int getAecMode() {
        return mAecMode;
    }

    @Override
    public void setSawtooth(boolean sawtooth) {
        mSawtooth = sawtooth;
    }

    @Override
    protected void addCallback(JCMediaDeviceCallback callback) {
        mCallbacks.add(callback);
    }

    @Override
    protected void removeCallback(JCMediaDeviceCallback callback) {
        mCallbacks.remove(callback);
    }

    @Override
    protected void configMeidaParam(VideoModeOtherParam otherParam) {
        JCLog.info(TAG, "configMeidaParam videoMode=%d", mVideoMode);
        MtcCallDb.Mtc_CallDbSetAnUsePresetVideoParams(false);
        MtcCallDb.Mtc_CallDbSetAudioCodecEnable("PCMU",false);
        MtcCallDb.Mtc_CallDbSetAudioCodecEnable("G722",false);
        MtcCallDb.Mtc_CallDbSetAudioCodecEnable("AMR-WB",false);
        MtcCallDb.Mtc_CallDbSetAudioCodecEnable("PCMA",false);
        MtcCallDb.Mtc_CallDbSetAudioCodecEnable("iLBC",false);
        MtcCallDb.Mtc_CallDbSetAudioCodecEnable("opus",false);
        MtcCallDb.Mtc_CallDbSetAudioCodecEnable("AMR",true);
        MtcCallDb.Mtc_CallDbSetAudioCodecEnable("G729",false);
        MtcCallDb.Mtc_CallDbSetAudioCodecByPriority("AMR", (short) 0);
        MtcCallDb.Mtc_CallDbSetVideoCodecEnable("VP8", true);
        //audio的结构体 ST_MNP_VOICE里面没有码率控制的，无法设置 0 ~ 12k
        MtcCallDb.Mtc_CallDbSetArsEnable(true);
        MtcCallDb.Mtc_CallDbSetVoiceArsParam(12000, 2000);
        MtcCallDb.Mtc_CallDbSetVadEnable(true);
        MtcCallDb.Mtc_CallDbSetAudioRed(false);
        MtcCallDb.Mtc_CallDbSetRxAnrEnable(false);
        MtcNumber Aec = new MtcNumber();
        MtcNumber Anr = new MtcNumber();
        MtcNumber Agc = new MtcNumber();
        MtcNumber Vad = new MtcNumber();
        MtcCallDb.Mtc_CallDbGetAudioQos(Aec,Anr,Agc,Vad);
        boolean bAec = false;
        boolean bAgc = false;
        boolean bVad = false;
        if (Aec.getValue() != 0) {
            bAec = true;
        }
        if (Agc.getValue() != 0) {
            bAgc = true;
        }
        if (Vad.getValue() != 0) {
            bVad = true;
        }
        MtcCallDb.Mtc_CallDbSetAudioRtxEnable(false);
        MtcCallDb.Mtc_CallDbSetSrtpCryptoType(MtcCallDbConstants.EN_MTC_DB_SRTP_CRYPTO_OFF);
        MtcCallDb.Mtc_CallDbSetAecMode(mAecMode == AEC_OS ? (short)MtcCallDb.EN_MTC_EC_OS : (short)MtcCallDb.EN_MTC_EC_AEC_SDE);
        MtcCallDb.Mtc_CallDbSetResolutionControl(false);
        MtcCallDb.Mtc_CallDbSetVideoArs(true);
        MtcCallDb.Mtc_CallDbSetSmallNaluEnable(true);
        MtcCallDb.Mtc_CallDbSetVideoRedFec(false);
        MtcCallDb.Mtc_CallDbSetVideoFramerate(mFramerate);
        if (mVideoMode == VIDEO_MODE_S_S || mVideoMode == VIDEO_MODE_S_L) {
            if (mSawtooth) {
                MtcCallDb.Mtc_CallDbSetAnVideoRecvResolution(240, 240);
            } else {
                MtcCallDb.Mtc_CallDbSetAnVideoRecvResolution(174, 144);
            }
            if (mVideoMode == VIDEO_MODE_S_S) {
                MtcCallDb.Mtc_CallDbSetVideoArsParm(80000, 10000, 0, 0);
                MtcCallDb.Mtc_CallDbSetVideoArsFixBitrate(50);
            } else {
                MtcCallDb.Mtc_CallDbSetVideoArsParm(200000, 10000, 0, 0);
                MtcCallDb.Mtc_CallDbSetVideoArsFixBitrate(150);
            }
            if (mVideoMode == VIDEO_MODE_S_S) {
                MtcCallDb.Mtc_CallDbSetAudioQos(bAec, false, false,true);
            } else {
                MtcCallDb.Mtc_CallDbSetAudioQos(bAec, false, true,true);
            }
        } else {
            if (mVideoMode == VIDEO_MODE_L_S) {
                MtcCallDb.Mtc_CallDbSetAudioQos(bAec, false, false,true);
                MtcCallDb.Mtc_CallDbSetAnVideoRecvResolution(352, 282);
                if (mFramerate >= 15 || mSawtooth) {
                    MtcCallDb.Mtc_CallDbSetVideoArsParm(120000, 10000, 0, 0);
                } else {
                    MtcCallDb.Mtc_CallDbSetVideoArsParm(80000, 10000, 0, 0);
                }
                MtcCallDb.Mtc_CallDbSetVideoArsFixBitrate(50);
            } else {
                MtcCallDb.Mtc_CallDbSetAnVideoRecvResolution(640, 360);
                MtcCallDb.Mtc_CallDbSetAudioQos(bAec, false, true,true);
                MtcCallDb.Mtc_CallDbSetAnUsePresetVideoParams(true);
                MtcCallDb.Mtc_CallDbSetVoiceArsParam(75000, 20000);
                MtcCallDb.Mtc_CallDbSetVideoArsParm(1500000, 50000, 0, 0);
                MtcCallDb.Mtc_CallDbSetVideoArsFixBitrate(800);
            }
        }
        if (otherParam != null && otherParam.agc != -1) {
            MtcCallDb.Mtc_CallDbSetAudioQos(bAec, false, otherParam.agc > 0, true);
        }
    }

    @Override
    public void onLogin(boolean result, @JCClient.ClientReason int reason) {

    }

    @Override
    public void onLogout(@JCClient.ClientReason int reason) {

    }

    @Override
    public void onClientStateChange(@JCClient.ClientState int state, @JCClient.ClientState int oldState) {

    }

    private void notifyCameraUpdate() {
        JCClientThreadImpl.getInstance().post(new Runnable() {
            @Override
            public void run() {
                JCLog.info(TAG, "摄像头变换");
                for (JCMediaDeviceCallback callback : mCallbacks) {
                    callback.onCameraUpdate();
                }
            }
        });
    }

    private void notifyAudioOutputTypeChange() {
        JCClientThreadImpl.getInstance().post(new Runnable() {
            @Override
            public void run() {
                JCLog.info(TAG, "扬声器改变");
                for (JCMediaDeviceCallback callback : mCallbacks) {
                    callback.onAudioOutputTypeChange(mSpeakerOn);
                }
            }
        });
    }

    private int getAudioMode() {
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.HONEYCOMB) {
            return AudioManager.MODE_NORMAL;
        }
        return AudioManager.MODE_IN_COMMUNICATION;
    }

    @Override
    protected boolean enableScreenCapture(boolean enable) {
        JCParam.CameraDeal param = new JCParam.CameraDeal();
        if (enable) {
            param.type = JCParam.CameraDeal.START;
            param.camera = ZmfVideo.CaptureScreen;
            param.width = mScreenCaptureWidth;
            param.height = mScreenCaptureHeight;
            param.framerate = mScreenCaptureFrameRate;
            if (ZmfEngine.getInstance().dealCamera(param).succ) {
                JCLog.info(TAG, "屏幕共享Capture开启成功");
                return true;
            } else {
                JCLog.error(TAG, "屏幕共享Capture开启失败");
                return false;
            }
        } else {
            param.type = JCParam.CameraDeal.STOP;
            param.camera = ZmfVideo.CaptureScreen;
            if (ZmfEngine.getInstance().dealCamera(param).succ) {
                JCLog.info(TAG, "屏幕共享Capture关闭成功");
                return true;
            } else {
                JCLog.error(TAG, "屏幕共享Capture关闭失败");
                return false;
            }
        }
    }
}
