package com.jimi.jmsmartftputils;

import android.app.Activity;
import android.util.Log;

import com.jimi.jmsmartftputils.bean.FtpProcessBean;
import com.jimi.jmsmartftputils.bean.FTPDownInfoBean;
import com.jimi.jmsmartftputils.bean.FileBean;
import com.jimi.jmsmartftputils.listener.JMFtpFindAllPathFilesCallBack;
import com.jimi.jmsmartftputils.listener.JMFtpProcessCallBack;
import com.jimi.jmsmartftputils.listener.JMFtpFindFilesCallBack;
import com.jimi.jmsmartftputils.listener.JMFtpListener;
import com.jimi.jmsmartutils.system.JMThread;

import org.apache.commons.net.ftp.FTPClient;
import org.apache.commons.net.ftp.FTPConnectionClosedException;
import org.apache.commons.net.ftp.FTPFile;
import org.apache.commons.net.ftp.FTPFileFilter;
import org.apache.commons.net.ftp.FTPReply;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import io.reactivex.Flowable;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.disposables.Disposable;
import io.reactivex.functions.Consumer;
import io.reactivex.schedulers.Schedulers;

import static com.jimi.jmsmartftputils.JMFtpConstant.FTP_MODE_ACTIVE;
import static com.jimi.jmsmartftputils.JMFtpConstant.FTP_MODE_PASSIVE;
import static com.jimi.jmsmartftputils.JMFtpErrorCode.FTP_CONNECTION_CLOSED_EXCEPTION;
import static com.jimi.jmsmartftputils.JMFtpErrorCode.FTP_ERROR_CONECT_BREAK;
import static com.jimi.jmsmartftputils.JMFtpErrorCode.FTP_ERROR_CONECT_BREAK_FAIL;
import static com.jimi.jmsmartftputils.JMFtpErrorCode.FTP_ERROR_CONNET;
import static com.jimi.jmsmartftputils.JMFtpErrorCode.FTP_ERROR_DELETE_FILE;
import static com.jimi.jmsmartftputils.JMFtpErrorCode.FTP_ERROR_DOWNLOAD;
import static com.jimi.jmsmartftputils.JMFtpErrorCode.FTP_ERROR_GET_FILE;
import static com.jimi.jmsmartftputils.JMFtpErrorCode.FTP_ERROR_MODE;
import static com.jimi.jmsmartftputils.JMFtpErrorCode.FTP_ERROR_MOVE_FILE;
import static com.jimi.jmsmartftputils.JMFtpErrorCode.FTP_ERROR_NOTAG;
import static com.jimi.jmsmartftputils.JMFtpErrorCode.FTP_ERROR_UNCONFIG;
import static com.jimi.jmsmartftputils.JMFtpErrorCode.FTP_ERROR_UPLOAD;
import static com.jimi.jmsmartftputils.JMFtpErrorCode.FTP_FILE_EMPTY;

public class JMFtp implements JMFtpImp {
    private String TAG = "JMFtp";
    private static final String temp = "tem";
    private FTPClient pClient;
    private String baseUrl;
    private String mode;
    private String account;
    private String password;
    private int port;
    private Map<String, FTPDownInfoBean> tagMap = new HashMap<>();
    private static JMFtp instance;

    private Disposable disposable;
    private volatile int speed;

    public static JMFtp getInstance() {
        synchronized (JMFtp.class) {
            if (instance == null) {
                instance = new JMFtp();
            }
            return instance;
        }
    }

    /**
     * 配置ftp连接信息
     *
     * @param baseUrl       ftp服务器地址
     * @param mode          模式 -被动和主动（见 JMFtoConstant类中)
     * @param port          端口
     * @param account       账号
     * @param password      密码
     * @param jmFtpListener 回调接口
     */
    @Override
    public void configFtpSyncFile(String baseUrl, String mode, int port, String account, String password, JMFtpListener jmFtpListener) {
        Log.e(TAG, "配置: baseUrl:" + baseUrl + "\n mode " + mode + "\n port " + port + "\n account " + account + "\n password " + password);

        if (baseUrl == null || mode == null || port < 0 || account == null || password == null) {
            if (jmFtpListener != null) jmFtpListener.onFail(FTP_ERROR_UNCONFIG);
            return;
        }

        if (!FTP_MODE_PASSIVE.equals(mode) && !FTP_MODE_ACTIVE.equals(mode)) {
            if (jmFtpListener != null) jmFtpListener.onFail(FTP_ERROR_MODE);
            return;
        }

        if (baseUrl.trim().startsWith("ftp://")) {
            String strUrl[] = baseUrl.split("//");
            if (strUrl.length == 2) {
                baseUrl = strUrl[1];
            } else {
                baseUrl = strUrl[0];
            }
        }
        this.baseUrl = baseUrl;
        this.mode = mode;
        this.port = port;
        this.account = account;
        this.password = password;
        if (jmFtpListener != null) jmFtpListener.onSuccess(null);
    }

    /**
     * 连接Ftp
     *
     * @param jmFtpListener 回调接口
     */
    @Override
    public void connectFTP(final JMFtpListener jmFtpListener) {

        JMThread.getCachedPool().execute(new Runnable() {
            @Override
            public void run() {
                try {
                    int reply;
                    if (pClient == null) {
                        pClient = new FTPClient();
                    }
                    if (mode.equals(FTP_MODE_PASSIVE)) {
                        pClient.enterLocalPassiveMode();
                    } else {
                        pClient.enterLocalActiveMode();
                    }
                    pClient.connect(baseUrl, port);
                    pClient.login(account, password);
                    pClient.setFileType(FTPClient.BINARY_FILE_TYPE);//很重要，不然下载的视频不完整

                    // 看返回的值是不是230，如果是，表示登陆成功
                    reply = pClient.getReplyCode();
                    if (!FTPReply.isPositiveCompletion(reply)) {
                        // 断开
                        pClient.disconnect();
                        Log.e(TAG, "FTP链接失败");
                        JMThread.runOnUiThread(new Runnable() {
                            @Override
                            public void run() {
                                jmFtpListener.onFail(FTP_ERROR_CONNET);
                            }
                        });

                    } else {
                        Log.e(TAG, "FTP链接成功");
                        JMThread.runOnUiThread(new Runnable() {
                            @Override
                            public void run() {
                                jmFtpListener.onSuccess(null);
                            }
                        });
                    }

                } catch (Exception e) {
                    e.printStackTrace();
                    Log.e(TAG, "FTP链接失败");
                    JMThread.runOnUiThread(new Runnable() {
                        @Override
                        public void run() {
                            jmFtpListener.onFail(FTP_ERROR_CONNET);
                        }
                    });
                }
            }
        });

    }

    public void resumeConnectFTP(final JMFtpListener jmFtpListener) {
        closeFTP(jmFtpListener);
        connectFTP(jmFtpListener);
    }

    /**
     * 查找文件
     *
     * @param fileBeans 文件路径数组
     * @param callBack  回调
     */
    public void findFTPFileList(final List<String> fileBeans, final JMFtpFindAllPathFilesCallBack callBack) {
        if (fileBeans == null || fileBeans.size() == 0) {
            JMThread.runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    callBack.onFail(FTP_ERROR_GET_FILE);
                }
            });
            return;
        }
        if (pClient == null || !pClient.isConnected()) {
            JMThread.runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    callBack.onFail(FTP_ERROR_CONECT_BREAK);
                }
            });
            Log.e(TAG, "FTP链接断开");
            return;
        }

        JMThread.getCachedPool().execute(new Runnable() {
            @Override
            public void run() {
                try {
                    final HashMap<String, List<FileBean>> map = new HashMap();
                    for (int i = 0; i < fileBeans.size(); i++) {
                        String filePath = fileBeans.get(i);
                        if (!changeDirectory(filePath)) {
                            continue;
                        }
                        FTPFile[] ftpFiles = pClient.listFiles();
                        if (ftpFiles == null || ftpFiles.length <= 0) {
                            continue;
                        }
                        List<FileBean> fileBeans = new ArrayList<>();
                        for (int index = 0; index < ftpFiles.length; index++) {
                            try {
                                FileBean fileBean = changeFileBean(filePath, ftpFiles[index]);
                                if (changeDirectory(fileBean.getFilePath())) {
                                    FTPFile[] childFiles = pClient.listFiles();
                                    if (childFiles != null && childFiles.length > 0) {
                                        List<FileBean> childFileBeans = new ArrayList<>();
                                        for (int j = 0; j < childFiles.length; j++) {
                                            String childFileName = childFiles[j].getName();
                                            int suffixIndex = childFileName.indexOf(".");
                                            if (childFiles[j].isFile() && suffixIndex != -1 &&
                                                    !childFileName.substring(suffixIndex).contains("jpg")
                                                    && !childFileName.substring(suffixIndex).contains("png")
                                                    && !childFileName.substring(suffixIndex).contains("temp")) {
                                                FileBean childFileBean = changeFileBean(fileBean.getFilePath(),
                                                        childFiles[j]);
                                                childFileBeans.add(childFileBean);
                                            }
                                        }
                                        fileBean.setChildFiles(childFileBeans);
                                    }
                                }
                                fileBeans.add(fileBean);
                                if (map.get(fileBean.getFileName()) == null && fileBean.getChildFiles() != null) {
                                    map.put(fileBean.getFileName(), fileBean.getChildFiles());
                                } else if (map.get(fileBean.getFileName()) != null && fileBean.getChildFiles() != null) {
                                    map.get(fileBean.getFileName()).addAll(fileBean.getChildFiles());
                                }
                            } catch (Exception e) {
                                e.printStackTrace();
                            }

                        }

                    }
                    JMThread.runOnUiThread(new Runnable() {
                        @Override
                        public void run() {
                            callBack.onSuccess(map);
                        }
                    });
                } catch (IOException e) {
                    e.printStackTrace();
                    Log.e(TAG, "获取文件失败");
                    JMThread.runOnUiThread(new Runnable() {
                        @Override
                        public void run() {
                            callBack.onFail(FTP_ERROR_GET_FILE);
                        }
                    });

                }
            }
        });
    }

    private FileBean changeFileBean(String filePath, FTPFile ftpFile) {
        FileBean fileBean = new FileBean();
        fileBean.setFileName(ftpFile.getName());
        fileBean.setFileSize(ftpFile.getSize());
        fileBean.setFilePath(filePath + File.separator + ftpFile.getName());
        fileBean.setFileType(ftpFile.getType());
        return fileBean;
    }

    /**
     * 查找文件
     *
     * @param subPath  文件/文件夹 地址 例如 /datafile
     * @param callBack 回调
     */
    @Override
    public void findFTPFlies(final String subPath, final JMFtpFindFilesCallBack callBack) {
        if (subPath == null) {
            JMThread.runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    callBack.onFail(FTP_ERROR_GET_FILE);
                }
            });

            return;
        }

        if (pClient == null || !pClient.isConnected()) {
            JMThread.runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    callBack.onFail(FTP_ERROR_CONECT_BREAK);
                }
            });
            Log.e(TAG, "FTP链接断开");
            return;
        }

        JMThread.getCachedPool().execute(new Runnable() {
            @Override
            public void run() {
                try {
                    if (!changeDirectory(subPath)) {
                        callBack.onFail(FTP_ERROR_GET_FILE);
                        return;
                    }
                    FTPFile[] ftpFiles = pClient.listFiles();
                    if (ftpFiles == null || ftpFiles.length <= 0) {
                        Log.e(TAG, "findFTPFlies: 为空了");
                        JMThread.runOnUiThread(new Runnable() {
                            @Override
                            public void run() {
                                callBack.onSuccess(new ArrayList<FileBean>());
                            }
                        });
                        return;
                    }
                    List<FileBean> fileBeans = new ArrayList<>();
                    for (int index = 0; index < ftpFiles.length; index++) {
                        try {
                            FileBean fileBean = new FileBean();
                            fileBean.setFileName(ftpFiles[index].getName());
                            fileBean.setFileSize(ftpFiles[index].getSize());
                            fileBean.setFilePath(subPath + "/" + ftpFiles[index].getName());
                            fileBean.setFileType(ftpFiles[index].getType());
                            fileBeans.add(fileBean);
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }
                    final List<FileBean> list = fileBeans;
                    Log.e(TAG, "findFTPFlies: 长度" + fileBeans.size());
                    JMThread.runOnUiThread(new Runnable() {
                        @Override
                        public void run() {
                            callBack.onSuccess(list);
                        }
                    });
                } catch (IOException e) {
                    e.printStackTrace();
                    Log.e(TAG, "获取文件失败");
                    JMThread.runOnUiThread(new Runnable() {
                        @Override
                        public void run() {
                            callBack.onFail(FTP_ERROR_GET_FILE);
                        }
                    });
                }
            }
        });
    }

    private boolean changeDirectory(String directoryName) {
        boolean isOk;
        try {
            isOk = pClient.changeWorkingDirectory(directoryName);
            Log.e("MainActivity", "isOk : " + isOk);
            return isOk;
            // FTPFile[] files=TujunApp.instance.mClient.list();
        } catch (IllegalStateException | IOException e) {
            e.printStackTrace();
            return false;
        }

    }

    /**
     * 下载文件
     *
     * @param url             要下的文件地址 /data/add_health_gaozhi.png
     * @param locaUrl         手机本地文件夹路径
     * @param fileName        手机将要保存的文件名称
     * @param tag             本次下的标记
     * @param downloaCallBack 回调
     */
    @Override
    public void downFTPFile(final String url, String locaUrl, final String fileName, final String tag, final JMFtpProcessCallBack downloaCallBack) {
        if (url == null || locaUrl == null || fileName == null) {
            JMThread.runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    downloaCallBack.onFail(FTP_ERROR_DOWNLOAD);
                }
            });
            return;
        }

        if (pClient == null || !pClient.isConnected()) {
            JMThread.runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    downloaCallBack.onFail(FTP_ERROR_CONECT_BREAK);
                }
            });
            Log.e(TAG, "FTP链接断开");
            return;
        }


        final StringBuffer stringBuffer = new StringBuffer();

        if (!locaUrl.endsWith("/")) {
            locaUrl = locaUrl + "/";
        }
        stringBuffer.append(locaUrl);

        JMThread.getCachedPool().execute(new Runnable() {
            @Override
            public void run() {
                try {
                    tagMap.put(tag, new FTPDownInfoBean(url, fileName, stringBuffer.toString(), tag, true, FTPDownInfoBean.DOWNLOD, downloaCallBack));
                    FTPFile[] files = pClient.listFiles(url);
                    if (files == null || files.length <= 0) {
                        JMThread.runOnUiThread(new Runnable() {
                            @Override
                            public void run() {
                                downloaCallBack.onFail(FTP_FILE_EMPTY);
                            }
                        });
                        Log.e(TAG, "files为空");
                        return;
                    }
                    File localFile = new File(stringBuffer.toString() + fileName);
                    long localBeginSize = 0;
                    if (localFile.exists()) {
                        localBeginSize = localFile.length();
                        if (localBeginSize == files[0].getSize()) {
                            Log.e(TAG, "文件已经存在");
                            final FtpProcessBean processBean = new FtpProcessBean();
                            processBean.setTag(tag);
                            processBean.setPath(localFile.getPath());
                            processBean.setProgress(1);
                            removeTag(tag);
                            JMThread.runOnUiThread(new Runnable() {
                                @Override
                                public void run() {
                                    downloaCallBack.onSuccess(processBean);
                                }
                            });
                            return;
                        } else {
//                            localFile.delete();
                            localBeginSize = localFile.length();
                        }
                    }
                    pClient.setFileType(FTPClient.BINARY_FILE_TYPE);//很重要，不然下载的视频不完整
                    if (downloadByUnit(url, fileName, stringBuffer.toString(), localBeginSize, files[0].getSize(), tag, downloaCallBack)) {
                        final FtpProcessBean processBean = new FtpProcessBean();
                        processBean.setTag(tag);
                        processBean.setPath(localFile.getPath());
                        processBean.setProgress(1);
                        JMThread.runOnUiThread(new Runnable() {
                            @Override
                            public void run() {
                                downloaCallBack.onSuccess(processBean);
                            }
                        });
                        removeTag(tag);
                    }

                } catch (Exception e) {
                    e.printStackTrace();
                    Log.e(TAG, "文件下载错误");
                    JMThread.runOnUiThread(new Runnable() {
                        @Override
                        public void run() {
                            downloaCallBack.onFail(FTP_ERROR_DOWNLOAD);
                        }
                    });
                }
            }
        });
    }

    private Boolean downloadByUnit(String url, String fileName, String locaUrl, long beginSize, long endSize, String tag, final JMFtpProcessCallBack callBack) throws Exception {

        if (pClient == null || !pClient.isConnected()) {
            JMThread.runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    callBack.onFail(FTP_ERROR_CONECT_BREAK);
                }
            });
            Log.e(TAG, "FTP链接断开");
            return false;
        }

        File dirFile = new File(locaUrl);

        if (!dirFile.exists()) {
            dirFile.mkdirs();
        }

        File localFile = new File(locaUrl + fileName);

        long waitSize = endSize - beginSize;
        Log.e(TAG, "waitSize" + waitSize + "endSize " + endSize + "beginSize" + beginSize);
        //切换服务器空间
        changeWorkingDirectory(url, callBack);

        //进行断点续传，并记录状态
        FileOutputStream out = new FileOutputStream(localFile, true);
        //把文件指针移动到 开始位置
        pClient.setRestartOffset(beginSize);
        pClient.setBufferSize(1024 * 1024);
        InputStream in = pClient.retrieveFileStream(new String(url.getBytes("GBK"), "iso-8859-1"));
        byte[] bytes = new byte[1024];
        int c;
        double finishSize = 0;
        double finishPercent = 0;
        long startTime = System.currentTimeMillis();
        stopDisposable();
        disposable = Flowable.interval(1, TimeUnit.SECONDS)
                .observeOn(AndroidSchedulers.mainThread())
                .onBackpressureBuffer()
                .doOnNext(new Consumer<Long>() {
                    @Override
                    public void accept(Long aLong) throws Exception {
                        callBack.onSpeed(speed);
                        speed = 0;
                    }
                }).subscribe();
        while ((c = in.read(bytes)) != -1 && tagMap.get(tag).isRun) {
            out.write(bytes, 0, c);
            speed += c;
            finishSize += c;
//            Log.e(TAG, "下载大小" + finishSize);
//            Log.e(TAG, "总大小" + waitSize);
//            Log.e(TAG, "百分比" + finishSize / waitSize);
            if (finishSize >= waitSize) {
                Log.e(TAG, "下载完成,下载大小MB：" + finishSize / 1024 / 1024);
                Log.e(TAG, "下载所用时间 秒：" + (System.currentTimeMillis() - startTime) / 1000);
                finishPercent = (finishSize + beginSize) / endSize;
                final FtpProcessBean processBean = new FtpProcessBean();
                processBean.setPath(locaUrl + fileName);
                processBean.setProgress(finishPercent);
                processBean.setTag(tag);
                JMThread.runOnUiThread(new Runnable() {
                    @Override
                    public void run() {
                        callBack.onProcess(processBean);
                    }
                });
            }
            if ((finishSize / waitSize) - finishPercent > 0.01) {
//                finishPercent = finishSize / waitSize;
                finishPercent = (finishSize + beginSize) / endSize;
//                Log.e(TAG, "文件下载进度" + finishPercent);
                final FtpProcessBean processBean = new FtpProcessBean();
                processBean.setPath(locaUrl + fileName);
                processBean.setProgress(finishPercent);
                processBean.setTag(tag);
                processBean.currentNum = speed;
                JMThread.runOnUiThread(new Runnable() {
                    @Override
                    public void run() {
                        callBack.onProcess(processBean);
                    }
                });
            }
        }
        in.close();
        out.close();
        stopDisposable();
        Log.e(TAG, "文件最后大小" + localFile.length());

        return pClient.completePendingCommand();
    }

    public void stopDisposable() {
        if (disposable != null && !disposable.isDisposed()) {
            disposable.dispose();
        }
        disposable = null;
    }

    /**
     * 上传文件
     *
     * @param remoteFolder   上传到服务器的文件夹路径 例如 /myproject/pic/
     * @param remoteFileName 上传到服务器的文件名称     mypic.png
     * @param localFilePath  本地文件
     * @param overWrite      如果有重名文件 是否覆盖
     * @param tag            本次上传的标记
     * @param callBack       回调
     */
    @Override
    public void uploadFTPFile(final String remoteFolder, final String remoteFileName, final String localFilePath, boolean overWrite, final String tag, final JMFtpProcessCallBack callBack) {
        if (remoteFileName == null || remoteFolder == null || localFilePath == null || tag == null) {
            JMThread.runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    callBack.onFail(FTP_ERROR_UPLOAD);
                }
            });
            return;
        }

        if (pClient == null || !pClient.isConnected()) {
            JMThread.runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    callBack.onFail(FTP_ERROR_CONECT_BREAK);
                }
            });
            Log.e(TAG, "FTP链接断开");
            return;
        }

        final FTPDownInfoBean ftpDownInfoBean = new FTPDownInfoBean(remoteFolder, remoteFileName, localFilePath, tag, true, FTPDownInfoBean.UPDATA, callBack);
        ftpDownInfoBean.overWrite = overWrite;
        ftpDownInfoBean.tempFileName = remoteFileName + temp;
        ftpDownInfoBean.fileName = remoteFileName;

        tagMap.put(tag, ftpDownInfoBean);

        JMThread.getCachedPool().execute(new Runnable() {
            @Override
            public void run() {
                upload(remoteFolder, localFilePath, ftpDownInfoBean.tempFileName, tag, callBack);
            }
        });
    }

    private void upload(String remoteFolder, String localFilePath, String remoteFileName, String tag, final JMFtpProcessCallBack callBack) {
        try {
            FTPDownInfoBean ftpDownInfoBean = tagMap.get(tag);
            FTPFile[] files = pClient.listFiles(remoteFolder + remoteFileName);
            File localFile = new File(localFilePath);
            long remoteSize = 0;
            long localSize = localFile.length();
            if (files.length >= 1) {
                //判断文件是否存在
                remoteSize = files[0].getSize();
                if (remoteSize == localSize) {
                    Log.e(TAG, "文件已经存在");
                    pClient.rename(remoteFolder + remoteFileName, remoteFolder + ftpDownInfoBean.fileName);
                    final FtpProcessBean processBean = new FtpProcessBean();
                    processBean.setPath(remoteFolder + remoteFileName);
                    processBean.setProgress(1);
                    processBean.setTag(tag);
                    JMThread.runOnUiThread(new Runnable() {
                        @Override
                        public void run() {
                            callBack.onSuccess(processBean);
                        }
                    });
                    removeTag(tag);
                    return;
                } else if (remoteSize > localSize) {
                    remoteSize = 0;
                    pClient.dele(remoteFolder + remoteFileName);
                    Log.e(TAG, "文件异常");
                }
            } else {
                pClient.makeDirectory(remoteFolder);
            }

            changeWorkingDirectory(remoteFolder, callBack);

            if (remoteSize < 0) {
                remoteSize = 0;
            }

            Log.e(TAG, "remoteSize" + remoteSize);

            //等待写入的文件大小
            long writeSize = localSize - remoteSize;
            if (writeSize <= 0) {
                return;
            }
            //获取百分单位是 1-100
            RandomAccessFile raf = new RandomAccessFile(localFile, "r");
            OutputStream out = pClient.appendFileStream(new String((remoteFileName).getBytes("GBK"), "iso-8859-1"));
//            OutputStream out = pClient.storeFileStream(new String((remoteFolder + remoteFileName).getBytes("GBK"), "iso-8859-1"));
            //把文件指针移动到 开始位置
            pClient.setRestartOffset(remoteSize);
            raf.seek(remoteSize);
            //定义最小移动单位是 1024字节 也就是1kb
            byte[] bytes = new byte[1024];
            int c;
            double finishSize = 0;
            double finishPercent = 0;
            boolean isOk = false;
            //存在一个bug 当分布移动的时候  可能会出现下载重复的问题 后期需要修改
            while ((c = raf.read(bytes)) != -1 && tagMap.get(tag).isRun) {
                out.write(bytes, 0, c);
                finishSize += c;

                if ((finishSize / writeSize) - finishPercent > 0.01 || finishSize == writeSize) {
                    finishPercent = finishSize / writeSize;

                    Log.e(TAG, "finshSize" + finishSize + "write" + writeSize);
                    Log.e(TAG, "文件上传进度" + finishPercent);
                    final FtpProcessBean processBean = new FtpProcessBean();
                    processBean.setPath(remoteFolder + ftpDownInfoBean.fileName);
                    processBean.setProgress(finishPercent);
                    processBean.setTag(tag);
                    JMThread.runOnUiThread(new Runnable() {
                        @Override
                        public void run() {
                            callBack.onProcess(processBean);
                        }
                    });
                    if (finishPercent >= 1) {
                        isOk = true;
                        Log.e(TAG, "上传完成");
                        out.flush();
                        raf.close();
                        out.close();
                        pClient.completePendingCommand();

                        changeWorkingDirectory(remoteFolder, null);
                        //如果覆盖文件
                        if (ftpDownInfoBean.overWrite) {
                            processBean.setPath(ftpDownInfoBean + ftpDownInfoBean.fileName);
                            //先删除服务器的文件,然后重命名
                            if (pClient.listFiles(remoteFolder + ftpDownInfoBean.fileName).length > 0) {
                                Log.e(TAG, "文件存在,先删除");
                                pClient.dele(remoteFolder + ftpDownInfoBean.fileName);
                            }
                            Log.e(TAG, "重命名");
                            //重命名文件
                            pClient.rename(remoteFolder + ftpDownInfoBean.tempFileName, remoteFolder + ftpDownInfoBean.fileName);

                        } else {
                            String createName = remoteFolder + ftpDownInfoBean.fileName;

                            if (pClient.listFiles(remoteFolder + ftpDownInfoBean.fileName).length > 0) {
                                Log.e(TAG, "文件存在,改名");
                                createName = remoteFolder + System.currentTimeMillis() + ftpDownInfoBean.fileName;
                            }
                            //重命名文件
                            pClient.rename(remoteFolder + ftpDownInfoBean.tempFileName, createName);

                            //不覆盖
                            processBean.setPath(remoteFolder + createName);
                        }
                        JMThread.runOnUiThread(new Runnable() {
                            @Override
                            public void run() {
                                callBack.onProcess(processBean);
                                callBack.onSuccess(processBean);
                            }
                        });
                        removeTag(tag);
                        Log.e(TAG, "上传完成1" + pClient.printWorkingDirectory());

                        break;
                    }
                }
            }

            if (!isOk) {
                out.flush();
                raf.close();
                out.close();
                pClient.completePendingCommand();
            }

            Log.e(TAG, "上传结束");
        } catch (Exception e) {
            e.printStackTrace();
            Log.e(TAG, "上传异常");
            try {
                pClient.completePendingCommand();
            } catch (Exception e1) {
                e1.printStackTrace();
            }
        }
    }

    /**
     * 删除文件
     *
     * @param remoteFile    将要删除的文件名称
     * @param jmFtpListener 回调
     */
    @Override
    public void deleteFTPFile(final String remoteFile, final JMFtpListener jmFtpListener) {
        if (!isConnectSuccess(jmFtpListener)) {
            JMThread.runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    jmFtpListener.onFail(FTP_ERROR_CONECT_BREAK);
                }
            });
            return;
        }

        JMThread.getCachedPool().execute(new Runnable() {
            @Override
            public void run() {
                try {
                    if (pClient.listFiles(remoteFile).length > 0) {
                        pClient.deleteFile(remoteFile);
                        JMThread.runOnUiThread(new Runnable() {
                            @Override
                            public void run() {
                                jmFtpListener.onSuccess(remoteFile);
                            }
                        });
                    }
                } catch (IOException e) {
                    e.printStackTrace();
                    JMThread.runOnUiThread(new Runnable() {
                        @Override
                        public void run() {
                            jmFtpListener.onFail(FTP_ERROR_DELETE_FILE);
                        }
                    });
                }
            }
        });

    }

    /**
     * 移动文件到别的位置
     *
     * @param from          需要移动位置的文件路径 例如 /myproject/op/up2.png
     * @param to            移动至该文件路径  /myproject/newOp/up2new.png
     * @param overWrite     如果有重名的文件是否覆盖
     * @param jmFtpListener 回调
     */
    @Override
    public void moveFTPFile(final String from, final String to, final boolean overWrite, final JMFtpListener jmFtpListener) {
        if (from == null || to == null) {
            JMThread.runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    jmFtpListener.onFail(FTP_ERROR_MOVE_FILE);
                }
            });
        }

        if (!isConnectSuccess(jmFtpListener)) {
            JMThread.runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    jmFtpListener.onFail(FTP_ERROR_CONECT_BREAK);
                }
            });
            return;
        }
        JMThread.getCachedPool().execute(new Runnable() {
            @Override
            public void run() {
                try {
                    String parentPath = to;
                    try {
                        parentPath = to.substring(0, to.lastIndexOf("/"));
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                    pClient.makeDirectory(parentPath);
                    pClient.changeWorkingDirectory(parentPath);

                    //如果是覆盖移动
                    if (overWrite) {
                        //先将目标文件夹的文件重命名,做一个备份
                        if (pClient.listFiles(to).length > 0) {
                            pClient.dele(to + temp);
                            if (pClient.rename(to, to + temp)) {
                                if (pClient.rename(from, to)) {
                                    Log.e(TAG, "moveFTPFile: 移动成功");
                                    pClient.dele(to + temp);
                                    moveFileSucees(to, jmFtpListener);
                                } else {
                                    Log.e(TAG, "moveFTPFile: 移动失败,还原备份");
                                    pClient.rename(to + temp, to);
                                    moveFileFail(jmFtpListener);
                                }
                            }
                        } else {
                            if (pClient.rename(from, to)) {
                                Log.e(TAG, "moveFTPFile: 移动成功");
                                moveFileSucees(to, jmFtpListener);
                            } else {
                                Log.e(TAG, "moveFTPFile: 移动失败");
                                moveFileFail(jmFtpListener);
                            }
                        }

                    } else {
                        //先判断本地有没有,如果有需要重命名一下.
                        if (pClient.listFiles(to).length > 0) {
                            StringBuilder sb = new StringBuilder(to);
                            sb.insert(to.lastIndexOf("/") + 1, System.currentTimeMillis());
                            String path = sb.toString();
                            if (pClient.rename(from, path)) {
                                Log.e(TAG, "moveFTPFile: 移动成功");
                                moveFileSucees(path, jmFtpListener);
                            } else {
                                Log.e(TAG, "moveFTPFile: 移动失败");
                                moveFileFail(jmFtpListener);
                            }

                        } else {
                            if (pClient.rename(from, to)) {
                                Log.e(TAG, "moveFTPFile: 移动成功");
                                moveFileSucees(to, jmFtpListener);
                            } else {
                                Log.e(TAG, "moveFTPFile: 移动失败");
                                moveFileFail(jmFtpListener);
                            }
                        }
                    }

                    Log.e(TAG, "moveFTPFile: " + parentPath);
                    Log.e(TAG, "moveFTPFile: " + pClient.printWorkingDirectory());
                    pClient.rename(from, to);
                } catch (Exception e) {
                    e.printStackTrace();
                    moveFileFail(jmFtpListener);
                }
            }
        });
    }

    /**
     * 上传、下载任务暂停
     *
     * @param tag           上传、下载的标记
     * @param jmFtpListener 回调
     */
    @Override
    public void ftpPause(String tag, JMFtpListener jmFtpListener) {
        if (!isConnect(jmFtpListener)) {
            jmFtpListener.onFail(FTP_ERROR_CONECT_BREAK);
            return;
        }
        try {
            FTPDownInfoBean ftpDownInfoBean = tagMap.get(tag);
            if (ftpDownInfoBean == null) {
                jmFtpListener.onFail(FTP_ERROR_NOTAG);
                return;
            }
            ftpDownInfoBean.isRun = false;
            tagMap.put(tag, ftpDownInfoBean);
            jmFtpListener.onSuccess(null);
            Log.e(TAG, "ftp暂停操作 : tag: " + tag);
        } catch (Exception e) {
            jmFtpListener.onFail(FTP_ERROR_NOTAG);
            e.printStackTrace();
        }
    }

    /**
     * 下载任务和继续任务
     */
    public void ftpDownResume(final String url, String locaUrl, final String fileName, final String tag,
                              final JMFtpProcessCallBack downloaCallBack, final JMFtpListener jmFtpListener) {
        FTPDownInfoBean ftpDownInfoBean = tagMap.get(tag);
        if (ftpDownInfoBean == null) {
            final StringBuffer stringBuffer = new StringBuffer();

            if (!locaUrl.endsWith("/")) {
                locaUrl = locaUrl + "/";
            }
            stringBuffer.append(locaUrl);
            ftpDownInfoBean = new FTPDownInfoBean(url, fileName, stringBuffer.toString(), tag, true, FTPDownInfoBean.DOWNLOD, downloaCallBack);
            tagMap.put(tag, ftpDownInfoBean);
        }
        ftpDownInfoBean.callBack = downloaCallBack;
        ftpResume(tag, jmFtpListener);
    }

    /**
     * 上传、下载任务继续
     *
     * @param tag           上传、下载的标记
     * @param jmFtpListener 回调
     */
    @Override
    public void ftpResume(final String tag, final JMFtpListener jmFtpListener) {
        if (!isConnect(jmFtpListener)) {
            JMThread.runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    if (jmFtpListener != null) jmFtpListener.onFail(FTP_ERROR_CONECT_BREAK);
                }
            });
            return;
        }
        try {
            JMThread.getCachedPool().execute(new Runnable() {
                @Override
                public void run() {
                    FTPDownInfoBean ftpDownInfoBean = tagMap.get(tag);
                    if (ftpDownInfoBean == null) {
                        JMThread.runOnUiThread(new Runnable() {
                            @Override
                            public void run() {
                                if (jmFtpListener != null) jmFtpListener.onFail(FTP_ERROR_NOTAG);
                            }
                        });
                        return;
                    }
                    ftpDownInfoBean.isRun = true;
                    tagMap.put(tag, ftpDownInfoBean);
                    //jmFtpListener.onSuccess(null);
                    boolean isResult = changeWorkingDirectory(ftpDownInfoBean.url, ftpDownInfoBean.callBack);
                    if (!isResult) {
                        JMThread.runOnUiThread(new Runnable() {
                            @Override
                            public void run() {
                                jmFtpListener.onFail(FTP_FILE_EMPTY);
                            }
                        });
                        return;
                    }
                    if (ftpDownInfoBean.type == FTPDownInfoBean.DOWNLOD) {
                        downFTPFile(ftpDownInfoBean.url, ftpDownInfoBean.locaUrl, ftpDownInfoBean.fileName, ftpDownInfoBean.tag, ftpDownInfoBean.callBack);
                    } else {
                        upload(ftpDownInfoBean.url, ftpDownInfoBean.locaUrl, ftpDownInfoBean.tempFileName, tag, ftpDownInfoBean.callBack);
                    }
                    Log.e(TAG, "ftp恢复操作 : tag: " + tag);
                }
            });

        } catch (Exception e) {
            JMThread.runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    if (jmFtpListener != null) jmFtpListener.onFail(FTP_ERROR_NOTAG);
                }
            });
            e.printStackTrace();
        }
    }

    /**
     * 关闭FTP连接
     *
     * @param jmFtpListener 回调
     */
    @Override
    public void closeFTP(final JMFtpListener jmFtpListener) {
        JMThread.getCachedPool().execute(new Runnable() {
            @Override
            public void run() {
                if (pClient != null) {
                    try {
                        if (pClient.logout()) {
                            pClient.disconnect();
                            pClient = null;
                            JMThread.runOnUiThread(new Runnable() {
                                @Override
                                public void run() {
                                    if (jmFtpListener != null) jmFtpListener.onSuccess(null);
                                }
                            });

                        } else {
                            JMThread.runOnUiThread(new Runnable() {
                                @Override
                                public void run() {
                                    if (jmFtpListener != null)
                                        jmFtpListener.onFail(FTP_ERROR_CONECT_BREAK_FAIL);
                                }
                            });

                        }
                    } catch (IOException e) {
                        e.printStackTrace();
                        JMThread.runOnUiThread(new Runnable() {
                            @Override
                            public void run() {
                                if (jmFtpListener != null)
                                    jmFtpListener.onFail(FTP_ERROR_CONECT_BREAK_FAIL);
                            }
                        });
                    }
                }
            }
        });
    }

    /**
     * 取消某个tag的上传或者下载任务
     *
     * @param tag           上传或者下载任务标记
     * @param jmFtpListener 回调
     */
    @Override
    public void ftpCancel(String tag, JMFtpListener jmFtpListener) {
        if (!isConnect(jmFtpListener)) {
            return;
        }
        removeTag(tag);
        jmFtpListener.onSuccess(tag);
    }

    private void moveFileSucees(final String toPath, final JMFtpListener jmFtpListener) {
        JMThread.runOnUiThread(new Runnable() {
            @Override
            public void run() {
                jmFtpListener.onSuccess(toPath);
            }
        });

    }

    private void moveFileFail(final JMFtpListener jmFtpListener) {
        JMThread.runOnUiThread(new Runnable() {
            @Override
            public void run() {
                jmFtpListener.onFail(FTP_ERROR_MOVE_FILE);
            }
        });
    }


    @Override
    public void destroy() {
        try {
            for (String key : tagMap.keySet()) {
                try {
                    FTPDownInfoBean ftpDownInfoBean = tagMap.get(key);
                    ftpDownInfoBean.isRun = false;
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
            tagMap.clear();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private boolean isConnectSuccess(final JMFtpListener jmFtpListener) {
        if (pClient == null || !pClient.isConnected()) {
            JMThread.runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    jmFtpListener.onFail(FTP_ERROR_CONECT_BREAK);
                }
            });
            Log.e(TAG, "FTP链接断开");
            return false;
        }
        return true;
    }

    private boolean isConnect(final JMFtpListener jmFtpListener) {
        if (pClient == null || !pClient.isConnected()) {
            if (jmFtpListener != null) jmFtpListener.onFail(FTP_ERROR_CONECT_BREAK);
            Log.e(TAG, "FTP链接断开");
            return false;
        }
        return true;
    }

    private boolean changeWorkingDirectory(String path, final JMFtpProcessCallBack callBack) {
        try {
            if (mode.equals(FTP_MODE_PASSIVE)) {
                pClient.enterLocalPassiveMode();
            } else {
                pClient.enterLocalActiveMode();
            }
            pClient.setFileType(FTPClient.BINARY_FILE_TYPE);//很重要，不然下载的视频不完整

            try {
                //切换服务器空间
                pClient.changeWorkingDirectory(path);
                Log.e(TAG, "空间路径：" + pClient.printWorkingDirectory());
                return true;
            } catch (FTPConnectionClosedException closeException) {
                if (callBack != null) callBack.onFail(FTP_CONNECTION_CLOSED_EXCEPTION);
                return false;
            } catch (Exception e) {
                e.printStackTrace();
                return false;
            }
        } catch (FTPConnectionClosedException closeException) {
            if (callBack != null) callBack.onFail(FTP_CONNECTION_CLOSED_EXCEPTION);
            return false;
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }

    }

    private long getRemoteFileSize(String remote, String fileName) {
        //获取远程文件大小,方面做断点续传
        try {
            FTPFile[] ftpFiles = pClient.listFiles(remote);
            if (ftpFiles.length > 0) {
                return ftpFiles[0].getSize();
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        return 0;
    }

    private void removeTag(String tag) {
        try {
            FTPDownInfoBean ftpDownInfoBean = tagMap.get(tag);
            ftpDownInfoBean.isRun = false;
            tagMap.remove(tag);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

}
