package com.jimi.jmmonitorview;

import android.content.Context;
import android.graphics.Bitmap;
import android.opengl.GLSurfaceView;
import android.util.AttributeSet;
import android.view.SurfaceHolder;

public class JMGLMonitor extends GLSurfaceView {

    private JMGLSurfaceViewRenderer mGLSurfaceViewRenderer;
    private boolean mIsBackground = false;

    public JMGLMonitor(Context context) {
        this(context, null);
    }

    public JMGLMonitor(Context context, AttributeSet attrs) {
        super(context, attrs);

        mGLSurfaceViewRenderer = new JMGLSurfaceViewRenderer(context);
        setEGLContextClientVersion(2);
        setRenderer(mGLSurfaceViewRenderer);
        setRenderMode(GLSurfaceView.RENDERMODE_WHEN_DIRTY);
    }

    @Override
    public void surfaceDestroyed(SurfaceHolder holder) {
        super.surfaceDestroyed(holder);

        mIsBackground = true;
        if (mGLSurfaceViewRenderer != null) {
            mGLSurfaceViewRenderer.releaseGL();
        }
    }

    /*
     * 显示YUV420P数据视图
     * */
    public void displayYUVData(int w, int h, byte[] y, byte[] u, byte[] v) {
        if (!mIsBackground && mGLSurfaceViewRenderer != null && w > 0 && h > 0 && y.length > 0 && u.length > 0 && v.length > 0) {
            synchronized (mGLSurfaceViewRenderer) {
                mGLSurfaceViewRenderer.displayYUVData(w, h, y, u, v);
                requestRender();
            }
        }
    }

    /*
     * 显示Bitmap位图
     * */
    public void displayBitmap(Bitmap bitmap) {
        if (bitmap != null) {
            int width = bitmap.getWidth();
            int height = bitmap.getHeight();
            int size = width * height;
            int pixels[] = new int[size];
            bitmap.getPixels(pixels, 0, width, 0, 0, width, height);

            rgb2YCbCr420(pixels, width, height);
        }
    }

    /*
     * 截图(图像为YUV数据视图，非OpenGL显示视图)
     * */
    public Bitmap snapshot() {
        if (mGLSurfaceViewRenderer != null) {
            return mGLSurfaceViewRenderer.snapshot();
        }

        return null;
    }

    /*rgb24转YUV420P*/
    private void rgb2YCbCr420(int[] pixels, int width, int height) {
        int len = width * height;
        byte[] dataY = new byte[len];
        byte[] dataU = new byte[len / 4];
        byte[] dataV = new byte[len / 4];

        int y, u, v;
        for (int i = 0; i < height; i++) {
            for (int j = 0; j < width; j++) {
                int rgb = pixels[i * width + j] & 0x00FFFFFF;   //屏蔽透明值
                int r = rgb & 0xFF;
                int g = (rgb >> 8) & 0xFF;
                int b = (rgb >> 16) & 0xFF;

                y = ((66 * r + 129 * g + 25 * b + 128) >> 8) + 16;
                u = ((112 * r - 94 * g - 18 * b + 128) >> 8) + 128;
                v = ((-38 * r - 74 * g + 112 * b + 128) >> 8) + 128;
                y = y < 16 ? 16 : (y > 255 ? 255 : y);
                u = u < 0 ? 0 : (u > 255 ? 255 : u);
                v = v < 0 ? 0 : (v > 255 ? 255 : v);

                dataY[i * width + j] = (byte) y;
                dataU[(i / 2) * (width / 2) + (j / 2)] = (byte) u;
                dataV[(i / 2) * (width / 2) + (j / 2)] = (byte) v;
            }
        }

        displayYUVData(width, height, dataY, dataU, dataV);     //显示图像
    }

    @Override
    public void onResume() {
        super.onResume();
    }

    public void onResumeGl(){
        mIsBackground = false;
        if (mGLSurfaceViewRenderer != null) {
            mGLSurfaceViewRenderer.initGL();
        }
    }

    public void onPauseGl(){
        mIsBackground = true;
    }

}