package com.jimi.jimivideoplayer.encoder;

import android.annotation.SuppressLint;
import android.media.AudioFormat;
import android.media.AudioRecord;
import android.media.MediaCodec;
import android.media.MediaCodec.BufferInfo;
import android.media.MediaCodecInfo;
import android.media.MediaFormat;
import android.media.MediaRecorder;

import com.jimi.jimivideoplayer.JMVideoStreamPlayerJni;
import com.jimi.jimivideoplayer.log.JMLogUtil;

import java.io.IOException;
import java.nio.ByteBuffer;

public class HWAACEncoder implements Runnable {
    public static final String MIME_TYPE = "audio/mp4a-latm";
    public static final int SAMPLE_RATE = 8000;
    public static final int BIT_RATE = 128000;
    public static final int CHANNEL_CONFIG = AudioFormat.CHANNEL_IN_MONO;
    public static final int AUDIO_FORMAT = AudioFormat.ENCODING_PCM_16BIT;
    public static MediaFormat audioOutputFormat = null;

    private static final int SAMPLES_PER_FRAME = 1024;

    private static final String TAG = "HWAACEncoder";
    private AudioRecord audioRecoder;
    private MediaCodec mAudioEncoder;
    private boolean isRun = false;
    private Thread thread;

    public boolean isRun() {
        return isRun;
    }

    public void startEncoder() {
        isRun = true;
        if (thread == null) {
            thread = new Thread(this);
            thread.start();
        }
    }

    @SuppressLint("NewApi")
    public void stopEncoder() {
        if (!isRun) {
            return;
        }
        isRun = false;
        if (thread != null) {
            try {
                thread.join();
                thread = null;
                JMLogUtil.i("HWAACEncoder has been released");
            } catch (InterruptedException e) {
                JMLogUtil.e(TAG, " stopEncoder " + e.getMessage());
                e.printStackTrace();
            }
        }
    }

    @SuppressLint("NewApi")
    private void initAudioCodec() {
        int minBufferSize = AudioRecord.getMinBufferSize(SAMPLE_RATE, CHANNEL_CONFIG, AUDIO_FORMAT);
        JMLogUtil.d(TAG, "audio init minimum buffer size : " + minBufferSize);
        int buffer_size = SAMPLES_PER_FRAME * 10;
        if (buffer_size < minBufferSize) {
            buffer_size = ((minBufferSize / SAMPLES_PER_FRAME) + 1) * SAMPLES_PER_FRAME * 2;
        }

        MediaFormat audioFormat = new MediaFormat();
        audioFormat.setString(MediaFormat.KEY_MIME, MIME_TYPE);
        audioFormat.setInteger(MediaFormat.KEY_AAC_PROFILE, MediaCodecInfo.CodecProfileLevel.AACObjectLC);
        audioFormat.setInteger(MediaFormat.KEY_SAMPLE_RATE, SAMPLE_RATE);
        audioFormat.setInteger(MediaFormat.KEY_CHANNEL_COUNT, 1);
        audioFormat.setInteger(MediaFormat.KEY_BIT_RATE, BIT_RATE);
        audioFormat.setInteger(MediaFormat.KEY_MAX_INPUT_SIZE, 16384);

        try {
            mAudioEncoder = MediaCodec.createEncoderByType(MIME_TYPE);

            mAudioEncoder.configure(audioFormat, null, null, MediaCodec.CONFIGURE_FLAG_ENCODE);
            mAudioEncoder.start();

            audioRecoder = new AudioRecord(MediaRecorder.AudioSource.MIC, SAMPLE_RATE, CHANNEL_CONFIG, AUDIO_FORMAT,
                    buffer_size);
            audioRecoder.startRecording();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    @Override
    public void run() {
        final long TimeUsPerFrame = 20000;// 160 *1000*1000/ 8000
        initAudioCodec();
        long lastEncodedAudioTimeStamp = 0;
        BufferInfo bufferInfo = new BufferInfo();
        while (isRun) {

            if (audioRecoder != null) {
                ByteBuffer[] inputBuffers = mAudioEncoder.getInputBuffers();
                int inputBufferIndex = mAudioEncoder.dequeueInputBuffer(20);
                if (inputBufferIndex >= 0) {
                    ByteBuffer inputBuffer = inputBuffers[inputBufferIndex];
                    inputBuffer.clear();
                    long presentationTimeNs = System.nanoTime();
                    int inputLength = audioRecoder.read(inputBuffer, 320);
                    if (inputLength == AudioRecord.ERROR_BAD_VALUE
                            || inputLength == AudioRecord.ERROR_INVALID_OPERATION) {
                        JMLogUtil.e(TAG, "Audio read error!!!");
                    }
                    // enqueue
                    mAudioEncoder.queueInputBuffer(inputBufferIndex, 0, inputLength, presentationTimeNs / 1000, 0);
                    // dequeue
                    ByteBuffer[] outputBuffers = mAudioEncoder.getOutputBuffers();
                    while (isRun) {
                        final int TIMEOUT_USEC = 20;
                        int encoderStatus = mAudioEncoder.dequeueOutputBuffer(bufferInfo, TIMEOUT_USEC);
                        if (encoderStatus == MediaCodec.INFO_TRY_AGAIN_LATER) {
                            break;
                        } else if (encoderStatus == MediaCodec.INFO_OUTPUT_BUFFERS_CHANGED) {
                            // not expected
                            outputBuffers = mAudioEncoder.getOutputBuffers();
                        } else if (encoderStatus == MediaCodec.INFO_OUTPUT_FORMAT_CHANGED) {
                            // should happen before receiving buffers, and should only happen once
                            audioOutputFormat = mAudioEncoder.getOutputFormat();

                        } else if (encoderStatus < 0) {
                            JMLogUtil.w(TAG, "unexpected result from encoder.dequeueOutputBuffer: " + encoderStatus);
                        } else {
                            ByteBuffer encodedData = outputBuffers[encoderStatus];
                            if (encodedData == null) {
                                throw new RuntimeException("encoderOutputBuffer " + encoderStatus + " was null");
                            }

                            if ((bufferInfo.flags & MediaCodec.BUFFER_FLAG_CODEC_CONFIG) != 0) {
                                bufferInfo.size = 0;
                            }

                            if (bufferInfo.size != 0) {
                                encodedData.position(bufferInfo.offset);
                                encodedData.limit(bufferInfo.offset + bufferInfo.size);
                                byte[] tempBytes = new byte[bufferInfo.size];
                                encodedData.get(tempBytes);

                                if (lastEncodedAudioTimeStamp == 0) {
                                    bufferInfo.presentationTimeUs = 0;
                                    lastEncodedAudioTimeStamp = TimeUsPerFrame;
                                } else {
                                    bufferInfo.presentationTimeUs = lastEncodedAudioTimeStamp + TimeUsPerFrame;
                                    lastEncodedAudioTimeStamp = bufferInfo.presentationTimeUs;
                                }
                                //ffmpeg推数据
                                JMVideoStreamPlayerJni.SendTalkData(tempBytes);
                            }
                            mAudioEncoder.releaseOutputBuffer(encoderStatus, false);
                        }
                    }
                }
            }
        }
        if (audioRecoder != null) {
            audioRecoder.stop();
            audioRecoder.release();
            audioRecoder = null;
        }

        if (mAudioEncoder != null) {
            mAudioEncoder.stop();
            mAudioEncoder.release();
            mAudioEncoder = null;
        }
    }
}
