package com.jimi.jimivideoplayer;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.SharedPreferences;
import android.graphics.Bitmap;
import android.media.AudioFormat;
import android.media.AudioManager;
import android.media.AudioTrack;
import android.os.PowerManager;
import android.telephony.TelephonyManager;
import android.text.TextUtils;

import com.jimi.jimivideoplayer.bean.FrameInfo;
import com.jimi.jimivideoplayer.encoder.HWAACEncoder;
import com.jimi.jimivideoplayer.log.JMLogUtil;
import com.jimi.jimivideoplayer.opengl.GLMonitor;
import com.jimi.jimivideoplayer.util.CommonUtils;
import com.jimi.jimivideoplayer.util.MD5Util;
import com.jimi.jimivideoplayer.util.MediaCodecUtil;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static com.jimi.jimivideoplayer.JMVideoStreamPlayerListener.STREAM_TALK_STATUS_ERR_AUTHORITY;
import static com.jimi.jimivideoplayer.JMVideoStreamPlayerListener.STREAM_TALK_STATUS_NONE;
import static com.jimi.jimivideoplayer.JMVideoStreamPlayerListener.STREAM_VIDEO_STATUS_NONE;
import static com.jimi.jimivideoplayer.JMVideoStreamPlayerListener.STREAM_VIDEO_STATUS_PREPARE;
import static com.jimi.jimivideoplayer.JMVideoStreamPlayerListener.STREAM_VIDEO_STATUS_START;

public class JMVideoStreamPlayer {
    static {
        try {
            System.loadLibrary("avformat");
            System.loadLibrary("avfilter");
            System.loadLibrary("postproc");
            System.loadLibrary("avcodec");
            System.loadLibrary("swscale");
            System.loadLibrary("swresample");
            System.loadLibrary("avutil");
            System.loadLibrary("avdevice");
        } catch (UnsatisfiedLinkError ule) {
            JMLogUtil.i("loadLibrary share ffmpeg failed, may use a static library");
            JMLogUtil.d(ule.getMessage());
        }

        try {
            System.loadLibrary("JimiVideoPlayer");
        } catch (UnsatisfiedLinkError ule) {
            JMLogUtil.d(ule.getMessage());
        }
    }

    /*
     *----------------------------------------------------------------------------------------------------
     */

    /**
     * 获取Jar库版本
     *
     * @return 获取jar包版本
     */
    public static String GetJarVersionString() {
        String value = "1.4.7";
        return value;
    }

    /**
     * 获取So库版本
     *
     * @return 获取so库版本
     */
    public static String GetSoVersionString() {
        return JMVideoStreamPlayerJni.GetVersionString();
    }

    /*
     * 初始化SDK
     * #对应释放接口：DeInitialize
     * */
    public static boolean Initialize() {
        return JMVideoStreamPlayerJni.Initialize(false);
    }

    /*
     * 释放SDK
     * #释放之前需要先释放所有的JMVideoStreamPlayer对象
     * #对应初始化接口：Initialize
     * */
    public static boolean DeInitialize() {
        return JMVideoStreamPlayerJni.DeInitialize(false);
    }

    /**
     * 配置开发者信息
     * #调用此接口，在不再使用SDK时需要调用DeInitialize释放SDK
     *
     * @param key    开发者Key
     * @param secret 开发者Secret
     * @param token  业务Token,若不知晓请填null
     * @return 是否配置成功
     */
    public static boolean Config(String key, String secret, String token) {
        if (TextUtils.isEmpty(token)) {
            token = "123456";
        }
        if (TextUtils.isEmpty(key) || TextUtils.isEmpty(secret)) {
            JMLogUtil.e("This has a null value, Key:" + key + " Secret:" + secret);
            return false;
        }
        return JMVideoStreamPlayerJni.ConfigKey(key, secret, token);
    }

    /**
     * 配置Web服务器地址、端口及http、https相关访问超时时间(内部已经默认配置)
     * #需要初始化SDK之后再调用此接口
     *
     * @param sHosts          Web服务器域名或IP，默认live.jimivideo.com
     * @param port            Web服务器端口，默认80，小于等于0则使用默认
     * @param timeoutInterval http、https访问的超时时间，默认45秒
     * @return 是否配置成功
     */
    public static boolean ConfigWebServer(String sHosts, int port, double timeoutInterval) {
        if (TextUtils.isEmpty(sHosts)) {
            return false;
        }
        return JMVideoStreamPlayerJni.ConfigWebServer(sHosts, port, timeoutInterval);
    }

    /**
     * 配置网关服务器地址、端口及与服务器连接的心跳时间
     * #需要初始化SDK之后再调用此接口
     *
     * @param sHosts        网关服务器域名或IP，默认live.jimivideo.com
     * @param port          网关服务器端口，默认22100，小于等于0则使用默认
     * @param heartbeatTime 与服务器保持连接的心跳时间，内部默认60秒，小于等于0则使用默认
     * //@See 监听: JMVideoStreamPlayerListener:onServerReceiveData;
     * //@Ass: -stopGatewayServer，-stopGatewayServer；
     * @return 是否配置成功
     */
    public static boolean ConfigGatewayServer(String sHosts, int port, double heartbeatTime) {
        if (TextUtils.isEmpty(sHosts)) {
            return false;
        }
        return JMVideoStreamPlayerJni.ConfigGatewayServer(sHosts, port, heartbeatTime);
    }

    /*
     *----------------------------------------------------------------------------------------------------
     */

    private Context mContext;
    private GLMonitor mGLMonitor = null;        //OpenGL显示视图
    private PowerManager.WakeLock mWakeLock = null;     //电池锁
    private boolean mStayAwake;
    private JMVideoStreamPlayerListener mExternalListener = null;   //外部监听
    private HWAACEncoder mHWAACEncoder = null;
    private int mAudioSampleRate = 44100;
    private int mAudioChannels = 1;
    private int mPlayStatus = STREAM_VIDEO_STATUS_NONE;     //播放状态
    private int mTalkStatus = STREAM_TALK_STATUS_NONE;      //对讲状态
    private boolean mIsSupportHWDecode = true;      //是否支持硬件解码

    private AudioTrack mAudioTrack = null;
    private String mIMEI = null;
    private long mCameraSwitchTask = 0;
    private JMSwitchCameraListener mSwitchCameraListener = null;
    private boolean mCameraSwitchAutoPlay = true;

    public static int mVideoWidth = 0;              //视频宽度
    public static int mVideoHeight = 0;             //视频高度

    /**
     * 初始化视频播放器，并配置设备IMEI
     * #一旦使用此接口前必须调用Initialize初始化SDK，并调用configWithKey:secret:token:配置开发者信息，不在使用此调用DeInitialize释放SDK；
     * #适用于单一Camera场景
     *
     * @param context 上下文
     * @param imei    设备IMEI
     * @param listener    回调监听
     */
    public JMVideoStreamPlayer(Context context, String imei, JMVideoStreamPlayerListener listener) {
        if (context == null) return;
        this.mContext = context;
        this.mExternalListener = listener;
        this.mIMEI = imei;

        JMLogUtil.i("SDK Version:" + JMVideoStreamPlayer.GetJarVersionString() +
                ", So Version:" + JMVideoStreamPlayer.GetSoVersionString());
        if (!JMVideoStreamPlayerJni.IsInitialized()) {
            JMLogUtil.w("JimiVideoPlayer(SDK) is not initialized");
        } else {
            if (!TextUtils.isEmpty(imei)) {
                JMVideoStreamPlayerJni.InitWithIMEI(imei, getPhoneIMEI(context), this, mSelfListener);

                mIsSupportHWDecode = MediaCodecUtil.isSupportMediaCodecHardDecoder("video/avc");
                if (!mIsSupportHWDecode) {  //不支持H264硬件解码
                    JMVideoStreamPlayerJni.SetHWDecodeState(false);
                }
            } else {
                JMLogUtil.e("This has a null value, IMEI:" + imei);
            }
        }
    }

    /**
     * 初始化视频播放器，并配置设备IMEI、开发者信息
     * #使用此接口会自动调用Initialize、DeInitialize初始化及释放SDK；
     * #适用于多个Camera场景
     *
     * @param context 上下文
     * @param key     appkey，需要在Jimi开放平台申请获得
     * @param secret  SECRET  需要在Jimi开放平台申请获得
     * @param imei    设备IMEI
     * @param token   Token,为空时默认为"123456"
     * @param listener   回调监听
     */
    public JMVideoStreamPlayer(Context context, String key, String secret, String imei, String token, JMVideoStreamPlayerListener listener) {
        if (context == null) return;
        this.mContext = context;
        this.mExternalListener = listener;
        this.mIMEI = imei;

        JMLogUtil.i("SDK Version:" + JMVideoStreamPlayer.GetJarVersionString() +
                ", So Version:" + JMVideoStreamPlayer.GetSoVersionString());
        if (!TextUtils.isEmpty(key) && !TextUtils.isEmpty(secret) && !TextUtils.isEmpty(imei)) {
            if (TextUtils.isEmpty(token)) {
                token = "123456";
            }
            JMVideoStreamPlayerJni.Init(key, secret, token, imei, getPhoneIMEI(context), this, mSelfListener);

            mIsSupportHWDecode = MediaCodecUtil.isSupportMediaCodecHardDecoder("video/avc");
            if (!mIsSupportHWDecode) {  //不支持H264硬件解码
                JMVideoStreamPlayerJni.SetHWDecodeState(false);
            }
        } else {
            JMLogUtil.e("This has a null value, Key:" + key + " Secret:" + secret + " IMEI:" + imei);
        }
    }

    /**
     * 释放SDK资源
     */
    public void release() {
        releaseAudioRecord();
        JMVideoStreamPlayerJni.Release();
        stopAudioPlayer();
        JMLogUtil.w("JMVideoStreamPlayer has been released");
    }

    public void addVideoStreamPlayerListener(JMVideoStreamPlayerListener listener) {
        this.mExternalListener = listener;
    }

    /**
     * 内部监听，作为底层的默认监听接口，外部如果代理监听，则再次反馈给外部
     */
    private JMVideoStreamPlayerListener mSelfListener = new JMVideoStreamPlayerListener() {

        /*播放状态*/
        public void onStreamPlayerPlayStatus(int status, String errStr) {
            mPlayStatus = status;
            if (mExternalListener != null) {
                mExternalListener.onStreamPlayerPlayStatus(status, errStr);
            }
        }

        /*对讲状态*/
        public void onStreamPlayerTalkStatus(int status, String errStr) {
            mTalkStatus = status;
            if (mExternalListener != null) {
                mExternalListener.onStreamPlayerTalkStatus(status, errStr);
            }
        }

        /*视频录制状态*/
        public void onStreamPlayerRecordStatus(int status, String filePath) {
            if (mExternalListener != null) {
                mExternalListener.onStreamPlayerRecordStatus(status, filePath);
            }
        }

        /*音视频帧信息*/
        @Override
        public void onStreamPlayerReceiveFrameInfo(FrameInfo frameInfo) {
            mVideoWidth = frameInfo.videoWidth;
            mVideoHeight = frameInfo.videoHeight;
            if (mExternalListener != null) {
                mExternalListener.onStreamPlayerReceiveFrameInfo(frameInfo);
            }
        }

        /*设备透传的数据*/
        public void onServerReceiveData(String data, int type) {
            if (mExternalListener != null) {
                mExternalListener.onServerReceiveData(data, type);
            }
        }
    };

    /**
     * 获取手机端AppID(即IMEI)
     */
    @SuppressLint("MissingPermission")
    private String getPhoneIMEI(Context context) {
        SharedPreferences sp = context.getSharedPreferences("JMVideoPlayer_AppIDInfo", Context.MODE_PRIVATE);
        String appID = sp.getString("JMVideoPlayer_AppID", "");
        if (TextUtils.isEmpty(appID)) {
            if (CommonUtils.isReadPhontState(context)) {
                TelephonyManager telephonyManager = (TelephonyManager) context.getSystemService(Context.TELEPHONY_SERVICE);
                appID = telephonyManager.getDeviceId();
            }

            if (TextUtils.isEmpty(appID)) {
                appID = CommonUtils.getRandomImei();
            }

            appID = MD5Util.md5(appID, 12);
            SharedPreferences.Editor editor = sp.edit();
            editor.putString("JMVideoPlayer_AppID", appID);
            editor.apply();
        }

        return appID;
    }

    /***********************内部API接口,通过反射的方式被C代码调用********************************/
    /**
     * 加载(显示)YUV420P数据(内部使用，外部请勿轻易使用)
     */
    private void loadYUV420pData(byte[] y, byte[] u, byte[] v, int width, int height) {
        if (mGLMonitor != null) {
            if (y.length > 0 && u.length > 0 && v.length > 0 && width > 0 && height > 0) {
                mGLMonitor.displayYUVData(width, height, y, u, v);
            }
        }
    }

    /**
     * 加载(播放)PCM音频数据(内部使用，外部请勿轻易使用)
     */
    private void loadAudioData(byte[] audioData, int size, int sampleRate, int channels) {
        mAudioSampleRate = sampleRate;
        mAudioChannels = channels;

        if (mPlayStatus == STREAM_VIDEO_STATUS_START) {     //音频播放器
            startAudioPlayer(audioData, size);
        } else if (mPlayStatus != STREAM_VIDEO_STATUS_PREPARE) {    //释放音频播放器
            stopAudioPlayer();
        }
    }

    /**
     * 开启音频(麦克风)采集线程
     */
    private void startAudioRecord() {
        if (mHWAACEncoder != null && mHWAACEncoder.isRun()) {
            return;
        } else {
            mHWAACEncoder = new HWAACEncoder();
            mHWAACEncoder.startEncoder();
        }
    }

    /**
     * 释放音频采集器
     */
    private void releaseAudioRecord() {
        if (mHWAACEncoder != null) {
            mHWAACEncoder.stopEncoder();
            mHWAACEncoder = null;
        }
    }

    /**
     * 开启PCM音频播放器
     */
    private void startAudioPlayer(byte[] audioData, int size) {
        if (mAudioTrack == null) {
            int channelConfig = AudioFormat.CHANNEL_OUT_MONO;
            int audioFormat = AudioFormat.ENCODING_PCM_16BIT;
            int minBufSize = AudioTrack.getMinBufferSize(mAudioSampleRate, channelConfig, audioFormat);
            mAudioTrack = new AudioTrack(AudioManager.STREAM_MUSIC, mAudioSampleRate, channelConfig, audioFormat,
                    minBufSize, AudioTrack.MODE_STREAM);
            mAudioTrack.play();
        }

        if (audioData == null || size <= 0) {
            return;
        }

        int temp = mAudioTrack.write(audioData, 0, size);
        if (temp < 0) {
            JMLogUtil.e("mAudioTreack write error");
        }
    }

    /**
     * 关闭PCM音频播放器
     */
    private void stopAudioPlayer() {
        if (mAudioTrack != null) {
            mAudioTrack.stop();
            mAudioTrack.release();
            mAudioTrack = null;
        }
    }

    /************************成员属性API接口*******************************/

    /**
     * 硬解码状态，默认开启
     *
     * @param isHWDecode 是否使用硬解码
     * @return true:设置成功
     */
    public boolean setHWDecodeState(boolean isHWDecode) {
        if (!mIsSupportHWDecode && isHWDecode) {
            JMLogUtil.e("Not Support MediaCodec to decode video data");
            return false;
        }
        JMVideoStreamPlayerJni.SetHWDecodeState(isHWDecode);

        return true;
    }

    /**
     * 获取硬件解码状态(是否开启)
     *
     * @return 是否已开启硬解码
     */
    public boolean getHWDecodeState() {
        return JMVideoStreamPlayerJni.GetHWDecodeState();
    }

    /**
     * 设置音频是否静音，默认关闭
     *
     * @param mute 是否设置静音
     */
    public void setMute(boolean mute) {
        JMVideoStreamPlayerJni.SetMute(mute);
    }

    /**
     * 获取音频静音状态
     *
     * @return 是否静音
     */
    public boolean getMute() {
        return JMVideoStreamPlayerJni.GetMute();
    }

    /**
     * 允许播放的缓冲(延迟)时间(默认10.0毫秒)
     *
     * @param delayMaxTime 设置最大可缓冲时间
     */
    public void setDelayMaxTime(double delayMaxTime) {
        JMVideoStreamPlayerJni.SetDelayMaxTime(delayMaxTime);
    }

    /**
     * 获取播放的最大缓存时间
     *
     * @return 缓冲时间
     */
    public double getDelayMaxTime() {
        return JMVideoStreamPlayerJni.GetDelayMaxTime();
    }

    /**
     * 设置http请求超时时间(默认45.0秒)
     *
     * @param timeoutInterval 超时时间
     * 主要涉及接口：startPlayLive、startPlayback、stopPlay、startTalk、stopTalk、sendRequestWithParameterDic等接口中附带的请求
     */
    public void setTimeoutInterval(double timeoutInterval) {
        JMVideoStreamPlayerJni.SetHttpTimeoutInterval(timeoutInterval);
    }

    /**
     * 获取http请求超时时间
     *
     * @return 超时时间
     */
    public double getTimeoutInterval() {
        return JMVideoStreamPlayerJni.GetHttpTimeoutInterval();
    }

    /***********************功能API接口********************************/

    /**
     * 关联显示GL视图
     *
     * @param monitor 显示视图
     */
    public void attachGlMonitor(GLMonitor monitor) {
        mGLMonitor = monitor;
//        stayAwake(true);      //添加电池锁
    }

    /**
     * 移除显示视图关联
     */
    public void deattachMonitor() {
        mGLMonitor = null;
//        stayAwake(false);     //移除电池锁
    }

    private void stayAwake(boolean awake) {
        if (mWakeLock != null) {
            if (awake && !mWakeLock.isHeld()) {
                mWakeLock.acquire();
            } else if (!awake && mWakeLock.isHeld()) {
                mWakeLock.release();
            }
        }
        mStayAwake = awake;
        updateSurfaceScreenOn();
    }

    private void updateSurfaceScreenOn() {
        if (mGLMonitor != null) {
            mGLMonitor.setKeepScreenOn(mStayAwake);
        }
    }

    /**
     * 开始直播
     *
     * //@See 监听: JMVideoStreamPlayerListener:onStreamPlayerPlayStatus;
     * //@Ass(相关接口): -stopPlay；
     */
    public void startPlayLive() {
        JMVideoStreamPlayerJni.StartPlayLive();
    }

    /**
     * 开始回放
     *
     * @param fileNameList 视频文件列表，比如列表："2018-08-29-09-21-27_30268.mp4", "2018-08-29-09-25-59_35532.mp4"
     * //@See 监听: JMVideoStreamPlayerListener:onStreamPlayerPlayStatus;
     * //@Ass: -stopPlay；
     */
    public void startPlayback(List<String> fileNameList) {
        JMVideoStreamPlayerJni.StartPlayback(fileNameList);
    }

    /**
     * 开始播放，用URL进行播放
     *
     * @param url RTMP视频流URL
     * //@See 监听: JMVideoStreamPlayerListener:onStreamPlayerPlayStatus;
     * //@Ass: -stopPlay；
     */
    public void startPlay(String url) {
        JMVideoStreamPlayerJni.StartPlay(url);
    }

    /**
     * 停止直播、回放、URL播放等
     *
     * //@See 监听: JMVideoStreamPlayerListener:onStreamPlayerPlayStatus;
     */
    public void stopPlay() {
        JMVideoStreamPlayerJni.StopPlay();
    }

    /**
     * 开始对讲(需要先申请权限:麦克风权限)
     *
     * //@See 监听: JMVideoStreamPlayerListener:onStreamPlayerTalkStatus;
     * //@Ass: -stopTalk；
     */
    public void startTalk() {
        if (CommonUtils.isRecordAudio(this.mContext)) {
            startAudioRecord();
            JMVideoStreamPlayerJni.StartTalk();
        } else if (mExternalListener != null) {
            mExternalListener.onStreamPlayerTalkStatus(STREAM_TALK_STATUS_ERR_AUTHORITY, "");
        }
    }

    /**
     * 停止对讲
     *
     * //@See 监听: JMVideoStreamPlayerListener:onStreamPlayerTalkStatus;
     */
    public void stopTalk() {
        JMVideoStreamPlayerJni.StopTalk();
        releaseAudioRecord();
    }

    /**
     * 停止SDK内部所有已启动的功能，包括视频播放、对讲、视频录制、Http请求和网关服务
     */
    public void stop() {
        stopRecording();
        stopPlay();
        stopTalk();

        JMVideoStreamPlayerJni.StopWebServer();
        stopGatewayServer();
    }

    /**
     * 重启网关及Web服务
     */
    public void reStart() {
        JMVideoStreamPlayerJni.StartWebServer();
        startGatewayServer();
    }

    /**
     * 切换摄像头
     *
     * @param isFront  是否切换至前置摄像头(设备端不支持，此参数暂时无效)
     * @param autoPlay 是否由内部自动进行播放视频，若不需要URL自己来播放视频请写true
     * @param listener 切换摄像头回调监听
     * //@See 监听: JMVideoStreamPlayerListener:onStreamPlayerTalkStatus;
     */
    public void switchCamera(boolean isFront, boolean autoPlay, JMSwitchCameraListener listener) {
        if (mCameraSwitchTask > 0) {
            cancelAsyncRequest(mCameraSwitchTask);
            mCameraSwitchTask = 0;
        }
        mSwitchCameraListener = listener;
        mCameraSwitchAutoPlay = autoPlay;
        Map<String, String> mapPara = new HashMap<>();
        mapPara.put("code", "001");
        mapPara.put("id", mIMEI);
        mapPara.put("cameraId", isFront ? "1" : "0");

        mCameraSwitchTask = sendAsyncRequest(mapPara, new JMHttpRequestTaskListener() {
            @Override
            public void onCustomMsgHandler(boolean success, long statusCode, String dataJsonStr) {
                if (!success && statusCode != 200) {
                    if (mSwitchCameraListener != null) {
                        mSwitchCameraListener.onSwitchCameraHandler(false, null, statusCode, "");
                    }
                    return;
                }

                long code = 0;
                String errMsg = null;
                try {
                    JSONObject jsonObject = new JSONObject(dataJsonStr);

                    if (jsonObject.has("code") && jsonObject.getInt("code") != 0) {
                        code = jsonObject.getLong("code");
                        errMsg = jsonObject.getString("msg");
                    } else {
                        if (jsonObject.has("content")) {
                            String content = jsonObject.getString("content");
                            JSONObject contentObj = new JSONObject(content);

                            int result = -1;
                            if (contentObj.has("ret")) {
                                result = contentObj.getInt("ret");
                            } else if (contentObj.has("code")) {
                                result = contentObj.getInt("code");
                            }

                            int subCode = 0;
                            if (contentObj.has("subCode")) {
                                subCode = contentObj.getInt("subCode");
                            }
                            if ((result == 0 && subCode != 0) || (result == 1 && subCode == 0)) {
                                code = -4;
                                errMsg = "Error:Device reject request";    //设备拒绝切换
                                JMLogUtil.e("Switch cameras: Device reject request");
                            } else {
                                String url = jsonObject.getString("pullURL") + jsonObject.getString("token");
                                if (mCameraSwitchAutoPlay && result == 2 && !JMVideoStreamPlayerJni.IsPlaying() && !TextUtils.isEmpty(url)) {
                                    JMVideoStreamPlayerJni.StartPlay(url);
                                } else if (mCameraSwitchAutoPlay) {
                                    if (!TextUtils.isEmpty(url)) {
                                        JMLogUtil.d("Switch cameras: No need to replay");
                                    } else {
                                        JMLogUtil.e("Switch cameras: Play url is NULL");
                                    }
                                }

                                if (mSwitchCameraListener != null) {
                                    mSwitchCameraListener.onSwitchCameraHandler(true, url, 0, "");
                                    return;
                                }
                            }
                        } else {
                            code = -3;
                            errMsg = "Error:Abnormal data";
                            JMLogUtil.e("Switch cameras: Abnormal data: " + dataJsonStr);
                        }
                    }
                } catch (JSONException e) {
                    e.printStackTrace();
                    code = -2;
                    errMsg = "Error:Not json data";
                    JMLogUtil.e("Switch cameras: Not json data: " + dataJsonStr);
                }

                if (mSwitchCameraListener != null) {
                    mSwitchCameraListener.onSwitchCameraHandler(false, null, code, errMsg);
                }
            }
        });
    }

    /*
     *----------------------------------------------------------------------------------------------------
     */

    /**
     * 视频截图(同步)
     *
     * @return Bitmap
     */
    public Bitmap snapshot() {
        if (mGLMonitor != null) {
            return mGLMonitor.snapshot();
        }

        return null;
    }

    /**
     * 开始录制(需要先申请权限：存储卡读取及写入权限)
     * #若正在录制视频，视频被主动或被动停止均会自动停止视频录制
     *
     * @param filePath 保存路径
     * //@See 监听: JMVideoStreamPlayerListener:onStreamPlayerRecordStatus;
     * //@Ass: -stopRecording，-isRecording，-GetRecordingRuration；
     */
    public void startRecording(String filePath) {
        if (CommonUtils.isWriteExternalStorage(mContext) && CommonUtils.isReadExternalStorage(mContext)) {
            JMVideoStreamPlayerJni.StartRecording(filePath);
        } else if (mExternalListener != null) {
            mExternalListener.onStreamPlayerRecordStatus(JMVideoStreamPlayerListener.STREAM_RECORD_STATUS_ERR_AUTHORITY, "");
        }
    }

    /**
     * 停止录制
     * #若正在录制视频，视频被主动或被动停止均会自动停止视频录制
     *
     * //@See 监听: JMVideoStreamPlayerListener:onStreamPlayerRecordStatus;
     * //@Ass：-stopPlay
     */
    public void stopRecording() {
        JMVideoStreamPlayerJni.StopRecording();
    }

    /**
     * 是否正在录制
     *
     * @return true：是
     */
    public boolean isRecording() {
        return JMVideoStreamPlayerJni.IsRecording();
    }

    /**
     * 获取视频录制时长
     *
     * @return 时长(毫秒)，0表示未在进行视频录制，其他为已录制的视频时长
     */
    public long getRecordingRuration() {
        return JMVideoStreamPlayerJni.GetRecordingRuration();
    }


    /**
     * 是否开启音频降噪功能及级别
     *
     * @param level 级别：0~3，0表示关闭，默认3
     */
    public void setDenoiseLevel(int level) {
        JMVideoStreamPlayerJni.SetDenoiseLevel(level);
    }

    /**
     * 获取在线人数
     *
     * @return  在线人数
     */
    public int getOnlineCount() {
        return JMVideoStreamPlayerJni.GetOnlineCount();
    }


    /***********************高级API接口********************************/

    /**
     * 发送自定义消息请求(异步请求)
     * #建议使用同步接口：buildSyncRequest、startSyncRequest、cancelSyncRequest
     *
     * @param mapPara  请求参数的Map集合
     * @param listener 监听回调
     * @return 消息句柄（地址）
     */
    public long sendAsyncRequest(Map<String, String> mapPara, JMHttpRequestTaskListener listener) {
        if (mapPara == null) {
            return 0;
        }

        String[] paraArray = new String[mapPara.size() * 2];
        int i = 0;
        for (String key : mapPara.keySet()) {
            paraArray[2 * i] = key;
            paraArray[2 * i + 1] = mapPara.get(key);
            i++;
        }
        return JMVideoStreamPlayerJni.SendAsyncRequest(paraArray, listener);
    }

    /**
     * 取消自定义消息请求
     * #若已经请求且回调，则取消无效
     *
     * @param task 消息请求句柄(异步)
     */
    public void cancelAsyncRequest(long task) {
        if (task == 0) return;
        JMVideoStreamPlayerJni.CancelAsyncRequest(task);
    }

    /**
     * 生成(组建)自定义消息句柄
     *
     * @param mapPara 请求参数的Map集合
     * @return 消息句柄（地址）
     */
    public long buildSyncRequest(Map<String, String> mapPara) {
        if (mapPara == null || mapPara.size() == 0) {
            return 0;
        }

        String[] paraArray = new String[mapPara.size() * 2];
        int i = 0;
        for (String key : mapPara.keySet()) {
            paraArray[2 * i] = key;
            paraArray[2 * i + 1] = mapPara.get(key);
            i++;
        }
        return JMVideoStreamPlayerJni.BuildSyncRequest(paraArray);
    }

    /**
     * 开始自定义消息请求(同步请求)
     *
     * @param task 消息句柄
     * @return 如果请求失败，返回则为null，其他一般情况：jason数据，解析成功则为data数据内容，否则返回整个接收到的数据
     */
    public String startSyncRequest(long task) {
        if (task == 0) return "";
        return JMVideoStreamPlayerJni.StartSyncRequest(task);
    }

    /**
     * 取消自定义消息请求
     * #若已经请求且回调，则取消无效
     *
     * @param task 消息请求句柄(同步)
     */
    public void cancelSyncRequest(long task) {
        if (task == 0) return;
        JMVideoStreamPlayerJni.CancelSyncRequest(task);
    }

    /**
     * 启动网关服务器服务(初始化时默认开启)
     *
     * //@See 监听: JMVideoStreamPlayerListener:onServerReceiveData;
     * //@Ass: -stopGatewayServer；
     */
    public void startGatewayServer() {
        JMVideoStreamPlayerJni.StartGatewayServer();
    }

    /**
     * 停止网关服务
     */
    public void stopGatewayServer() {
        JMVideoStreamPlayerJni.StopGatewayServer();
    }

}
