package com.jimi.jimitalk.tools.recorder;

import android.hardware.display.DisplayManager;
import android.hardware.display.VirtualDisplay;
import android.media.AudioFormat;
import android.media.AudioTrack;
import android.media.MediaCodec;
import android.media.MediaCodecInfo;
import android.media.MediaFormat;
import android.media.MediaMuxer;
import android.media.projection.MediaProjection;
import android.os.Build;
import android.support.annotation.RequiresApi;
import android.util.Log;
import android.view.Surface;

import org.webrtc.PeerConnectionFactory;

import java.io.IOException;
import java.lang.ref.WeakReference;
import java.nio.ByteBuffer;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * @author Administrator
 * @date 2020/3/31
 */
public class ScreenRecorder extends Thread {
    private static final String TAG = "ScreenRecorder";

    private int mWidth;
    private int mHeight;
    private int mBitRate;
    private int mDpi;
    private String mDstPath;
    private MediaProjection mMediaProjection;

    private MediaMuxer mMuxer;
    private boolean mMuxerStarted = false;
    private AtomicBoolean mQuit = new AtomicBoolean(false);

    private ScreenRecorderCallback callback;

    public interface ScreenRecorderCallback {
        public void onScreenRecordSuccess(String filePath);
    }

    public ScreenRecorder(int width, int height, int bitrate, int dpi, MediaProjection mp, String dstPath, ScreenRecorderCallback callback) {
        super(TAG);
        mWidth = width;
        mHeight = height;
        mBitRate = bitrate;
        mDpi = dpi;
        mMediaProjection = mp;
        mDstPath = dstPath;
        this.callback = callback;
    }

    public final void quit() {
        mQuit.set(true);
        setMuxerStarted(false);
        release();

        callback.onScreenRecordSuccess(mDstPath);
    }

    @Override
    public void run() {
        try {
            Log.e(TAG, "init muxer");
            mMuxer = new MediaMuxer(mDstPath, MediaMuxer.OutputFormat.MUXER_OUTPUT_MPEG_4);
        } catch (IOException e) {
            e.printStackTrace();
        }

        VideoRecorder videoRecorder = new VideoRecorder(mWidth, mHeight, mBitRate, mDpi, mMediaProjection, new WeakReference<ScreenRecorder>(this));
        videoRecorder.start();

        AudioRecorder audioRecorder = new AudioRecorder(new WeakReference<>(this));
        audioRecorder.start();

        while (!mMuxerStarted) {
            if (videoRecorder.isTrackAdded && audioRecorder.isTrackAdded) {
                startMuxer();
                setMuxerStarted(true);
            }
        }
    }

    private boolean ismMuxerStarted() {
        return mMuxerStarted;
    }

    private void setMuxerStarted(boolean flag) {
        mMuxerStarted = flag;
    }

    private void startMuxer() {
        mMuxer.start();
    }

    private class VideoRecorder extends Thread {
        private static final String MIME_TYPE = "video/avc"; // H.264 Advanced Video Coding
        private static final int FRAME_RATE = 30; // 30 fps
        private static final int IFRAME_INTERVAL = 10; // 10 seconds between I-frames
        private static final int TIMEOUT_US = 10000;

        private MediaCodec mEncoder;
        private Surface mSurface;
        private AtomicBoolean mIsStart = new AtomicBoolean(false);
        private MediaCodec.BufferInfo mBufferInfo = new MediaCodec.BufferInfo();
        private int mVideoTrackIndex = -1;
        private VirtualDisplay mVirtualDisplay;
        private WeakReference<ScreenRecorder> screenRecorderWeakReference;
        private int width;
        private int height;
        private int bitrate;
        private int dpi;
        private MediaProjection mp;

        private boolean isTrackAdded  = false;

        VideoRecorder(int width, int height, int bitrate, int dpi, MediaProjection mp, WeakReference<ScreenRecorder> screenRecorderWeakReference) {
            this.width = width;
            this.height = height;
            this.dpi = dpi;
            this.bitrate = bitrate > 0 ? bitrate : 2000000;
            this.mp = mp;
            this.screenRecorderWeakReference = screenRecorderWeakReference;

            try {
                initEncoder();
            } catch (Exception ex) {
                ex.printStackTrace();
            }
        }

        private void initEncoder() throws IOException {
            Log.e(TAG, "init video encoder");
            //MediaFormat这个类是用来定义视频格式相关信息的
            //video/avc,这里的avc是高级视频编码Advanced Video Coding
            //mWidth和mHeight是视频的尺寸，这个尺寸不能超过视频采集时采集到的尺寸，否则会直接crash
            MediaFormat format = MediaFormat.createVideoFormat(MIME_TYPE, width, height);
            //COLOR_FormatSurface这里表明数据将是一个graphicbuffer元数据
            format.setInteger(MediaFormat.KEY_COLOR_FORMAT,
                    MediaCodecInfo.CodecCapabilities.COLOR_FormatSurface);
            //设置码率，通常码率越高，视频越清晰，但是对应的视频也越大，这个值默认设置成了2000000，也就是通常所说的2M
            format.setInteger(MediaFormat.KEY_BIT_RATE, bitrate);
            //设置帧率，通常这个值越高，视频会显得越流畅，一般默认设置成30，最低可以设置成24，不要低于这个值，低于24会明显卡顿
            format.setInteger(MediaFormat.KEY_FRAME_RATE, FRAME_RATE);
            //IFRAME_INTERVAL是指的帧间隔，这是个很有意思的值，它指的是，关键帧的间隔时间。通常情况下，你设置成多少问题都不大。
            //比如设置成10，那就是10秒一个关键帧。但是，如果有需求要做视频的预览，那最好设置成1
            //因为如果设置成10，那会发现，10秒内的预览都是一个截图
            format.setInteger(MediaFormat.KEY_I_FRAME_INTERVAL, IFRAME_INTERVAL);

            Log.d(TAG, "created video format: " + format);
            //创建一个MediaCodec的实例
            mEncoder = MediaCodec.createEncoderByType(MIME_TYPE);
            //定义这个实例的格式，也就是上面我们定义的format，其他参数不用过于关注
            mEncoder.configure(format, null, null, MediaCodec.CONFIGURE_FLAG_ENCODE);
            //这一步非常关键，它设置的，是MediaCodec的编码源，也就是说，我要告诉mEncoder，你给我解码哪些流。
            //很出乎大家的意料，MediaCodec并没有要求我们传一个流文件进去，而是要求我们指定一个surface
            //而这个surface，其实就是MediaProjection中用来展示屏幕采集数据的surface
            mSurface = mEncoder.createInputSurface();
            Log.e(TAG, "created input surface: " + mSurface);
            mEncoder.start();
            mIsStart.set(true);
        }

        private void recordVirtualDisplay() {
            //在使用 MediaProjection时爆出 Token is nullor IllegalStateExceptionor Invalid Media Projection，
            //此时可以排查当前的 MediaProjection对象，是否在其他地方已经将其release掉了，
            //可以考虑做成全局的MediaProjection，让它的生命周期和Application生命周期同步，以防止token非法问题
            //TODO
            mVirtualDisplay = mp.createVirtualDisplay(TAG + "-display",
                    width, height, dpi, DisplayManager.VIRTUAL_DISPLAY_FLAG_PUBLIC,
                    mSurface, null, null);

            Log.e(TAG, "video encoder start?"+mIsStart.get());
            while (!screenRecorderWeakReference.get().mQuit.get()) {
                int index = mEncoder.dequeueOutputBuffer(mBufferInfo, TIMEOUT_US);
                Log.d(TAG, "dequeue output buffer index=" + index);
                if (index == MediaCodec.INFO_OUTPUT_FORMAT_CHANGED) {
                    resetOutputFormat();

                } else if (index == MediaCodec.INFO_TRY_AGAIN_LATER) {
                    Log.d(TAG, "retrieving buffers time out!");
                    try {
                        // wait 10ms
                        Thread.sleep(10);
                    } catch (InterruptedException e) {
                        e.printStackTrace();
                    }
                } else if (index >= 0) {
                    encodeToVideoTrack(index);

                    mEncoder.releaseOutputBuffer(index, false);
                }
            }
        }

        private void encodeToVideoTrack(int index) {
            ByteBuffer encodedData = mEncoder.getOutputBuffer(index);

            if ((mBufferInfo.flags & MediaCodec.BUFFER_FLAG_CODEC_CONFIG) != 0) {
                // The codec config data was pulled out and fed to the muxer when we got
                // the INFO_OUTPUT_FORMAT_CHANGED status.
                // Ignore it.
                Log.d(TAG, "ignoring BUFFER_FLAG_CODEC_CONFIG");
                mBufferInfo.size = 0;
            }
            if (mBufferInfo.size == 0) {
                Log.d(TAG, "info.size == 0, drop it.");
                encodedData = null;
            } else {
                Log.d(TAG, "got buffer, info: size=" + mBufferInfo.size
                        + ", presentationTimeUs=" + mBufferInfo.presentationTimeUs
                        + ", offset=" + mBufferInfo.offset);
            }
            if (encodedData != null) {
                encodedData.position(mBufferInfo.offset);
                encodedData.limit(mBufferInfo.offset + mBufferInfo.size);
                ScreenRecorder muxer = screenRecorderWeakReference.get();
                Log.e(TAG, "添加视频数据 " + mBufferInfo.size);
                if (muxer != null && muxer.mMuxerStarted) {
                    muxer.writeSampleData(mVideoTrackIndex, encodedData, mBufferInfo);
                }
                Log.i(TAG, "sent " + mBufferInfo.size + " video bytes to muxer...");
            }
        }

        private void resetOutputFormat() {
            // should happen before receiving buffers, and should only happen once
            ScreenRecorder muxer = screenRecorderWeakReference.get();
            if (muxer.ismMuxerStarted()) {
                throw new IllegalStateException("output format already changed!");
            }
            MediaFormat newFormat = mEncoder.getOutputFormat();

            Log.e(TAG, "output format changed.\n new format: " + newFormat.toString());

            mVideoTrackIndex = muxer.addTrack(newFormat);
            isTrackAdded = true;

            Log.e(TAG, "started media muxer, videoIndex=" + mVideoTrackIndex);
        }

        private void release() {
            if (mEncoder != null) {
                mEncoder.stop();
                mEncoder.release();
                mEncoder = null;
            }

            if (mVirtualDisplay != null) {
                mVirtualDisplay.release();
                mVirtualDisplay = null;
            }

            Log.e(TAG, "stop video 录制...");
        }

        @Override
        public void run() {
            super.run();
            recordVirtualDisplay();
            Log.e(TAG, "video 录制线程 退出...");
            release();
        }
    }

    public void writeSampleData(int trackIndex, ByteBuffer encodedData, MediaCodec.BufferInfo bufferInfo) {
        mMuxer.writeSampleData(trackIndex, encodedData, bufferInfo);
    }

    public int addTrack(MediaFormat mediaFormat) {
        return mMuxer.addTrack(mediaFormat);
    }

    private class AudioRecorder extends Thread {
        private final int TIMEOUT_USEC = 10000;// 10[msec]
        private final String MIME_TYPE = "audio/mp4a-latm";
        private int BIT_RATE = 320;
        private int SAMPLE_RATE = 48000;
        private int CHANNEL_CONFIGS = AudioFormat.CHANNEL_IN_MONO;
        private int AUDIO_FORMAT = AudioFormat.ENCODING_PCM_16BIT;
        private int MODE = AudioTrack.MODE_STREAM;

        private MediaCodec mEncoder;// API >= 16(Android4.1.2)
        private volatile boolean isExit = false;
        private WeakReference<ScreenRecorder> screenRecorderWeakReference;
        private MediaCodec.BufferInfo mAudioBufferInfo = new MediaCodec.BufferInfo();// API >= 16(Android4.1.2)
        private MediaCodecInfo audioCodecInfo;
        private volatile boolean isStart = false;
        private int mAudioTrackIndex = -1;
        /**
         * previous presentationTimeUs for writing
         */
        private long prevOutputPTSUs = 0;

        private boolean isTrackAdded = false;

        public AudioRecorder(WeakReference<ScreenRecorder> screenRecorderWeakReference) {
            this.screenRecorderWeakReference = screenRecorderWeakReference;
        }

        private void initEncoder() throws IOException{
            MediaFormat audioFormat = MediaFormat.createAudioFormat(MIME_TYPE, SAMPLE_RATE, 1);
            audioFormat.setInteger(MediaFormat.KEY_BIT_RATE, BIT_RATE);
            audioFormat.setInteger(MediaFormat.KEY_CHANNEL_COUNT, 1);
            audioFormat.setInteger(MediaFormat.KEY_SAMPLE_RATE, SAMPLE_RATE);
            Log.e(TAG, "format: " + audioFormat);

            mEncoder = MediaCodec.createEncoderByType(MIME_TYPE);
            mEncoder.configure(audioFormat, null, null, MediaCodec.CONFIGURE_FLAG_ENCODE);
            mEncoder.start();
            Log.e(TAG, "prepare finishing");

            isStart = true;
        }

        private void resetOutputFormat() {
            // should happen before receiving buffers, and should only happen once
            ScreenRecorder muxer = screenRecorderWeakReference.get();
            if (muxer.ismMuxerStarted()) {
                throw new IllegalStateException("output format already changed!");
            }
            MediaFormat newFormat = mEncoder.getOutputFormat();

            Log.e(TAG, "output format changed.\n new format: " + newFormat.toString());

            mAudioTrackIndex = muxer.addTrack(newFormat);
            isTrackAdded = true;

            Log.e(TAG, "started media muxer, audioTrackIndex=" + mAudioTrackIndex);
        }

        private void stopMediaCodec() {
            if (mEncoder != null) {
                mEncoder.stop();
                mEncoder.release();
                mEncoder = null;
            }
            isStart = false;
            Log.e(TAG, "stop audio 录制...");
        }

        private void encode(final byte[] buffer, final long presentationTimeUs) {
            if (isExit) return;
            final ByteBuffer[] inputBuffers = mEncoder.getInputBuffers();
            final int inputBufferIndex = mEncoder.dequeueInputBuffer(TIMEOUT_USEC);
            /*向编码器输入数据*/
            if (inputBufferIndex >= 0) {
                final ByteBuffer inputBuffer = inputBuffers[inputBufferIndex];
                inputBuffer.clear();
                if (buffer != null) {
                    inputBuffer.put(buffer);
                }
	            Log.v(TAG, "encode:queueInputBuffer");
                if (buffer != null) {
                    if (buffer.length <= 0) {
                        // send EOS
                        Log.i(TAG, "send BUFFER_FLAG_END_OF_STREAM");
                        mEncoder.queueInputBuffer(inputBufferIndex, 0, 0,
                                presentationTimeUs, MediaCodec.BUFFER_FLAG_END_OF_STREAM);
                    } else {
                        mEncoder.queueInputBuffer(inputBufferIndex, 0, buffer.length,
                                presentationTimeUs, 0);
                    }
                }
            } else if (inputBufferIndex == MediaCodec.INFO_TRY_AGAIN_LATER) {
                // wait for MediaCodec encoder is ready to encode
                // nothing to do here because MediaCodec#dequeueInputBuffer(TIMEOUT_USEC)
                // will wait for maximum TIMEOUT_USEC(10msec) on each call
                //TODO
            }

            /*获取解码后的数据*/
            final ScreenRecorder muxer = screenRecorderWeakReference.get();
            if (muxer == null) {
                Log.w(TAG, "MediaMuxerRunnable is unexpectedly null");
                return;
            }
            ByteBuffer[] encoderOutputBuffers = mEncoder.getOutputBuffers();
            int index;

            do {
                index = mEncoder.dequeueOutputBuffer(mAudioBufferInfo, TIMEOUT_USEC);
                if (index == MediaCodec.INFO_TRY_AGAIN_LATER) {
                    //TODO
                } else if (index == MediaCodec.INFO_OUTPUT_BUFFERS_CHANGED) {
                    encoderOutputBuffers = mEncoder.getOutputBuffers();
                } else if (index == MediaCodec.INFO_OUTPUT_FORMAT_CHANGED) {
                    resetOutputFormat();
                } else if (index < 0) {
                    //TODO
                } else {
                    final ByteBuffer encodedData = encoderOutputBuffers[index];
                    if ((mAudioBufferInfo.flags & MediaCodec.BUFFER_FLAG_CODEC_CONFIG) != 0) {
                        // You shoud set output format to muxer here when you target Android4.3 or less
                        // but MediaCodec#getOutputFormat can not call here(because INFO_OUTPUT_FORMAT_CHANGED don't come yet)
                        // therefor we should expand and prepare output format from buffer data.
                        // This sample is for API>=18(>=Android 4.3), just ignore this flag here
                        Log.d(TAG, "drain:BUFFER_FLAG_CODEC_CONFIG");
                        mAudioBufferInfo.size = 0;
                    }

                    if (mAudioBufferInfo.size != 0) {
                        mAudioBufferInfo.presentationTimeUs = getPTSUs();
                        Log.e(TAG, "添加音频数据 " + mAudioBufferInfo.size);
                        if (muxer.mMuxerStarted) {
                            muxer.writeSampleData(mAudioTrackIndex, encodedData, mAudioBufferInfo);
                            prevOutputPTSUs = mAudioBufferInfo.presentationTimeUs;
                            Log.i(TAG, "sent " + mAudioBufferInfo.size + " audio bytes to muxer...");
                        }
                    }
                    // return buffer to encoder
                    mEncoder.releaseOutputBuffer(index, false);
                }
            } while (index >= 0);
        }

        private long getPTSUs() {
            long result = System.nanoTime() / 1000L;
            // presentationTimeUs should be monotonic
            // otherwise muxer fail to write
            if (result < prevOutputPTSUs)
                result = (prevOutputPTSUs - result) + result;
            return result;
        }

        public void exit() {
            isExit = true;
        }

        private boolean isAudioConfigured = false;

        @RequiresApi(api = Build.VERSION_CODES.M)
        @Override
        public void run() {
            super.run();

            Queue<RTCAudioRecordThread.PCMAudioData> pcmAudioDataList = new ConcurrentLinkedDeque<>();
            /*
             * type=0 record both
             * type=1 only record mine
             * type=2 only record opposite
             */
            PeerConnectionFactory.VoiceRecordCallback voiceRecordCallback = new PeerConnectionFactory.VoiceRecordCallback() {
                @Override
                public void onRecordVoice(int type, byte[] pcm, int sampleRate, int channels) {
                    if (type == 0) {
                        Log.e(TAG, "type=" + type + " sampleRate=" + sampleRate + " channels=" + channels);
                        pcmAudioDataList.add(new RTCAudioRecordThread.PCMAudioData(type, pcm, sampleRate, channels));

                        if (!isAudioConfigured) {
                            SAMPLE_RATE = sampleRate;
                            CHANNEL_CONFIGS = channels;

                            isAudioConfigured = true;
                        }
                    }
                }
            };
            PeerConnectionFactory.SetVoiceRecordCallback(voiceRecordCallback);

            try {
                initEncoder();
            } catch (Exception ex) {
                ex.printStackTrace();
            }

            while (!screenRecorderWeakReference.get().mQuit.get()) {
                if (!pcmAudioDataList.isEmpty()) {
                    try {
                        RTCAudioRecordThread.PCMAudioData pcmAudioData = pcmAudioDataList.poll();
                        if (pcmAudioData != null) {
                            encode(pcmAudioData.getPcm(), getPTSUs());
                        } else {
                            //没及时产生数据，那么先休眠5毫秒
                            sleep(5);
                        }
                    } catch (InterruptedException e) {
                        e.printStackTrace();
                    }
                }
            }
            Log.e(TAG, "Audio 录制线程 退出... ");
            PeerConnectionFactory.SetVoiceRecordCallback(null);
            stopMediaCodec();
        }
    }

    private void release() {
        if (mMuxer != null) {
            try {
                mMuxer.stop();
                mMuxer.release();
            } catch (Exception e) {
                e.printStackTrace();
            }
            mMuxer = null;
        }
    }
}
