package com.jimi.jimitalk.tools;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;
import android.util.Log;

import com.jimi.jimitalk.bean.TalkMessage;

import java.util.ArrayList;
import java.util.List;

/**
 * @author Administrator
 * @date 2020/5/14
 */
public class DBUtil {
    public static class ChatDBHelper extends SQLiteOpenHelper {
        private static final String DATABASE_NAME = "record.db";
        public static int DATABASE_VERSION = 2;

        public ChatDBHelper(Context context) {
            super(context, DATABASE_NAME, null, DATABASE_VERSION);
        }

        @Override
        public void onCreate(SQLiteDatabase db) {
            createPttRecordTable(db);
        }

        @Override
        public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
            //数据库升级
            LogUtil.d("PttRecord upgrade database ...");
            dropPttRecordTable(db);

            onCreate(db);
        }

        private void createPttRecordTable(SQLiteDatabase db) {
            String sql = "create table IF NOT EXISTS ptt_record(" +
                    "_id INTEGER PRIMARY KEY AUTOINCREMENT," +
                    "send_id smallint," +
                    "send_name VARCHAR," +
                    "receive_id smallint," +
                    "msgid VARCHAR," +
                    "send_time VARCHAR," +
                    "mediafilepath VARCHAR," +
                    "record_timestamp long)";
            LogUtil.d("ptt record create database ...");
            db.execSQL(sql);//创建数据库，存储PTT消息
        }

        private void dropPttRecordTable(SQLiteDatabase db) {
            db.execSQL("drop table if exists ptt_record");
        }
    }

    public static class ChatDBManager {
        private static ChatDBManager instance;
        private static ChatDBHelper mChatDBHelper;

        public static synchronized void initialize(Context context) {
            if (instance == null) {
                instance = new ChatDBManager(context);
            }
        }

        public static synchronized ChatDBManager getInstance() {
            if (instance == null) {
                throw new IllegalMonitorStateException(ChatDBManager.class.getName() + " is not initialized, call initialize(..) method first.");
            }

            return instance;
        }

        public synchronized SQLiteDatabase getReadableDatabase() {
            return mChatDBHelper.getReadableDatabase();
        }

        public synchronized SQLiteDatabase getWritableDatabase() {
            return mChatDBHelper.getWritableDatabase();
        }

        public ChatDBManager(Context context) {
            mChatDBHelper = new ChatDBHelper(context);
        }

        /**
         * 增加历史对讲消息（PTT）到数据库
         *
         * @param info
         * @return
         */
        public synchronized long addPttRecord(TalkMessage info) {
            SQLiteDatabase db = getWritableDatabase();
            ContentValues cv = new ContentValues();
            cv.put("send_id", info.getSendId());
            cv.put("send_name", info.getSendName());
            cv.put("receive_id", info.getReceiverId());
            cv.put("msgid", info.getMsgId());
            cv.put("send_time", info.getSendTime());
            cv.put("mediafilepath", info.getUri());
            cv.put("record_timestamp", info.getTimestamp());
            long result = db.insert("ptt_record", null, cv);
            Log.d("chat", "pttrecord addRecord result = " + result);
            db.close();
            return result;
        }

        public void deleteAllPttRecord() {
            Log.d("chat", "ptt record truncate");
            SQLiteDatabase db = getWritableDatabase();
            db.execSQL("Delete from ptt_record");
            db.close();
        }

        /**
         * 查询历史消息
         *
         * @param groupId 群组ID
         * @return
         */
        public List<TalkMessage> getPttRecordByGroupId(int groupId) {
            SQLiteDatabase db = getReadableDatabase();
            ArrayList<TalkMessage> infos = new ArrayList<>();
            Cursor cursor = db.query("ptt_record", null, "receive_id=?", new String[]{groupId + ""}, null, null, "record_timestamp");
            while (cursor.moveToNext()) {
                TalkMessage info = new TalkMessage();
                info.setSendId(cursor.getInt(cursor.getColumnIndex("send_id")));
                info.setSendName(cursor.getString(cursor.getColumnIndex("send_name")));
                info.setReceiverId(cursor.getInt(cursor.getColumnIndex("receive_id")));
                info.setSendTime(cursor.getString(cursor.getColumnIndex("send_time")));
                info.setUri(cursor.getString(cursor.getColumnIndex("mediafilepath")));
                info.setTimestamp(cursor.getLong(cursor.getColumnIndex("record_timestamp")));
                infos.add(info);
            }
            cursor.close();
            db.close();
            return infos;
        }

        public static void closeDBManager() {
            if (mChatDBHelper != null) {
                mChatDBHelper.close();
                mChatDBHelper = null;
            }

            if (instance != null) {
                instance = null;
            }
        }
    }
}
