package com.jimi.jimitalk.tools;


import android.content.Context;
import android.os.Environment;
import android.util.Log;

import java.io.File;
import java.io.FileOutputStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.io.Writer;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * UncaughtException处理类,当程序发生Uncaught异常的时候,有该类来接管程序,并记录发送错误报告.
 *
 * @author user
 *
 */
public class CrashHandler implements Thread.UncaughtExceptionHandler {

    public static final String TAG = "crashHandler";

    //系统默认的UncaughtException处理类
    private Thread.UncaughtExceptionHandler mDefaultHandler;
    //CrashHandler实例
    private static CrashHandler INSTANCE = new CrashHandler();
    //程序的Context对象
    private Context mContext;
    //用来存储设备信息和异常信息
    private Map<String, String> infos = new HashMap<String, String>();

    //用于格式化日期,作为日志文件名的一部分
    private DateFormat formatter = new SimpleDateFormat("yyyy-MM-dd-HH-mm-ss");

    //日志路径
    public static final String LOG_PATH = "/sdcard/crash/";

    /** 保证只有一个CrashHandler实例 */
    private CrashHandler() {
    }

    /** 获取CrashHandler实例 ,单例模式 */
    public static CrashHandler getInstance() {
        return INSTANCE;
    }

    /**
     * 初始化
     *
     * @param context
     */
    public void init(Context context) {
        mContext = context;
        //获取系统默认的UncaughtException处理器
        mDefaultHandler = Thread.getDefaultUncaughtExceptionHandler();
        //设置该CrashHandler为程序的默认处理器
        Thread.setDefaultUncaughtExceptionHandler(this);
    }

    /**
     * 当UncaughtException发生时会转入该函数来处理
     */
    @Override
    public void uncaughtException(Thread thread, Throwable ex) {
        if (!handleException(ex) && mDefaultHandler != null) {
            //如果用户没有处理则让系统默认的异常处理器来处理
            mDefaultHandler.uncaughtException(thread, ex);
        } else {
            try {
                Thread.sleep(3000);
            } catch (InterruptedException e) {
                Log.e(TAG, "error : ", e);
            }
//            //退出程序
//            android.os.Process.killProcess(android.os.Process.myPid());
//            System.exit(1);
            reStartApp(mContext);
        }
    }

    public static void reStartApp(Context mContext) {
//        Log.e("long","UncaughtExceptionHandler restartApp");
//        Intent mStartActivity = new Intent(mContext, LaunchActivity.class);
//        int mPendingIntentId = 123456;
//        PendingIntent mPendingIntent = PendingIntent.getActivity(mContext, mPendingIntentId,mStartActivity, PendingIntent.FLAG_CANCEL_CURRENT);
//        AlarmManager mgr = (AlarmManager)mContext.getSystemService(Context.ALARM_SERVICE);
//        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
//            mgr.setExactAndAllowWhileIdle(AlarmManager.RTC, System.currentTimeMillis() + 2000,mPendingIntent);
//        } else if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
//            mgr.setExact(AlarmManager.RTC, System.currentTimeMillis() + 2000,mPendingIntent);
//        }
        System.exit(0);
    }

    /**
     * 自定义错误处理,收集错误信息 发送错误报告等操作均在此完成.
     *
     * @param ex
     * @return true:如果处理了该异常信息;否则返回false.
     */
    private boolean handleException(Throwable ex) {
        if (ex == null) {
            return false;
        }
        //保存日志文件
        saveCrashInfo2File(ex);
        return true;
    }


    /**
     * 保存错误信息到文件中
     *
     * @param ex
     * @return  返回文件名称,便于将文件传送到服务器
     */
    private String saveCrashInfo2File(Throwable ex) {

        StringBuffer sb = new StringBuffer();
        for (Map.Entry<String, String> entry : infos.entrySet()) {
            String key = entry.getKey();
            String value = entry.getValue();
            sb.append(key + "=" + value + "\n");
        }

        Writer writer = new StringWriter();
        PrintWriter printWriter = new PrintWriter(writer);
        ex.printStackTrace(printWriter);
        Throwable cause = ex.getCause();
        while (cause != null) {
            cause.printStackTrace(printWriter);
            cause = cause.getCause();
        }
        printWriter.close();
        String result = writer.toString();
        sb.append(result);
        Log.e(TAG,"error " + sb.toString());//自己先打印一波到log
        try {
            long timestamp = System.currentTimeMillis();
            String time = formatter.format(new Date());
            String fileName = "crash-" + time + "-" + timestamp + ".log";
            if (Environment.getExternalStorageState().equals(Environment.MEDIA_MOUNTED)) {
                String path = LOG_PATH;
                File dir = new File(path);
                if (!dir.exists()) {
                    dir.mkdirs();
                }
                FileOutputStream fos = new FileOutputStream(path + fileName);
                fos.write(sb.toString().getBytes());
                fos.close();
            }
            return fileName;
        } catch (Exception e) {
            Log.e(TAG, "an error occured while writing file...", e);
        }
        return null;
    }

    /*   *//**
     * 上传错误日志
     *//*
    public static boolean uploadCrashFile(int senderID, String senderName, int receiverID, String receiverName, int type) {
        File crashFolder = new File(LOG_PATH);
        List<File> crashFiles = new ArrayList<>();
        List<File> uploaded = new ArrayList<>();

        // Traversing /sdcard/crash/
        if(crashFolder.isDirectory()) {
            File[] files = crashFolder.listFiles();
            if (files != null) {
                for (File tmp : files) {
                    if (tmp.isFile()) {
                        crashFiles.add(tmp);
                    }
                }
            }
        }else {
            Log.e(TAG, LOG_PATH + " is not exist");
        }
        Log.d(TAG, "this is crash files size=" + crashFiles.size());

        // Uploading crash files
        OkHttpClient client = new OkHttpClient();
        for (File file : crashFiles) {
            String fileName = file.getName();
            Log.d(TAG, "this is found crash file=" + fileName);
            long sendTime = System.currentTimeMillis();
            try {
                RequestBody fileBody = RequestBody.create(MediaType.parse("multipart/form-data"), new File(file.getPath()));
                RequestBody body = new MultipartBody.Builder()
                        .setType(MultipartBody.FORM)
                        .addFormDataPart("file", fileName, fileBody)
                        .addFormDataPart("id", senderID + "")
                        .addFormDataPart("SenderName", senderName)
                        .addFormDataPart("ReceiverType", type + "") //TODO 1是个人，2是群
                        .addFormDataPart("ReceiverId", receiverID + "")
                        .addFormDataPart("ReceiverName", receiverName)
                        .addFormDataPart("SendTime", sendTime + "")
                        .build();

                Request request = new Request.Builder()
                        .header("Authorization", "Client-ID " + UUID.randomUUID())
                        .url(AppTools.UPLOAD_URL)
                        .post(body)
                        .build();

                Response response = client.newCall(request).execute();
                Log.d(TAG, "this is upload response = " + response.isSuccessful());
                Log.d(TAG, "this is upload body = " + response.body());
                Log.d(TAG, "this is upload response code=" + (response != null?response.code():null));


                if (response.isSuccessful()) {
                    uploaded.add(file);
                }
            } catch (IOException e) {
                e.printStackTrace();
                return false;
            }
        }
        Log.d(TAG, "this is uploaded files size=" + uploaded.size());

        // Delete local file after uploading successfully
        for (File file : uploaded) {
            String fileName = file.getName();
            if (file.delete()) {
                Log.d(TAG, "this is file=" + fileName + " deleted");
            }
        }

        // If all crash files uploaded successfully
        if (crashFiles.size() == uploaded.size() && crashFiles.size() != 0) {
            return false;
        }

        return true;
    }*/
}
