package com.jimi.jimitalk.ptt.janusclientapi;

import com.jimi.jimitalk.tools.LogUtil;

import org.json.JSONException;
import org.json.JSONObject;
import org.webrtc.PeerConnection;

import java.math.BigInteger;
import java.util.List;
import java.util.Random;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;

/**
 * Created by ben.trent on 5/7/2015.
 */
public class JanusServer extends Thread implements
        IJanusMessageObserver,
        IJanusSessionCreationCallbacks,
        IJanusAttachPluginCallbacks,
        IJanusUserCallCallbacks {

    private class RandomString {
        final String str = "0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";
        final Random rnd = new Random();

        public String randomString(Integer length) {
            StringBuilder sb = new StringBuilder(length);
            for (int i = 0; i < length; i++) {
                sb.append(str.charAt(rnd.nextInt(str.length())));
            }
            return sb.toString();
        }
    }

    private final RandomString stringGenerator = new RandomString();
    private ConcurrentHashMap<BigInteger, JanusPluginHandle> attachedPlugins = new ConcurrentHashMap<BigInteger, JanusPluginHandle>();
    private Object attachedPluginsLock = new Object();
    private ConcurrentHashMap<String, ITransactionCallbacks> transactions = new ConcurrentHashMap<String, ITransactionCallbacks>();
    private Object transactionsLock = new Object();
    public final String serverUri;
    public final IJanusGatewayCallbacks gatewayObserver;
    public final List<PeerConnection.IceServer> iceServers;
    public final Boolean ipv6Support;
    public final Integer maxPollEvents;
    private BigInteger sessionId;
    private Boolean connected;
    private final IJanusMessenger serverConnection;
    private volatile Thread keep_alive;
    private Boolean peerConnectionFactoryInitialized = false;
    private int uid;

    private int keepalive_count = 0;

    @Override
    public void run() {
        Thread thisThread = Thread.currentThread();
        while (keep_alive == thisThread) {
            try {
                TimeUnit.SECONDS.sleep(5);
            } catch (InterruptedException ex) {
            }

            if (!connected || serverConnection.getMessengerType() != JanusMessengerType.websocket)
                return;

            JSONObject obj = new JSONObject();
            try {
                obj.put("janus", JanusMessageType.keepalive.toString());
                if (serverConnection.getMessengerType() == JanusMessengerType.websocket)
                    obj.put("session_id", sessionId);
                obj.put("transaction", stringGenerator.randomString(6));
                serverConnection.sendMessage(obj.toString(), sessionId);
            } catch (JSONException ex) {
                gatewayObserver.onCallbackError("Keep alive failed is Janus online?" + ex.getMessage());
                connected = false;
                return;
            }
        }

        LogUtil.w("Server connection failed, keep alive thread ended is Janus online?");
    }

    public JanusServer(IJanusGatewayCallbacks gatewayCallbacks, int myuid) {
        gatewayObserver = gatewayCallbacks;
        System.setProperty("java.net.preferIPv6Addresses", "false");
        System.setProperty("java.net.preferIPv4Stack", "true");
        serverUri = gatewayObserver.getServerUri();
        iceServers = gatewayObserver.getIceServers();
        ipv6Support = gatewayObserver.getIpv6Support();
        maxPollEvents = gatewayObserver.getMaxPollEvents();
        connected = false;
        sessionId = new BigInteger("-1");
        serverConnection = JanusMessagerFactory.createMessager(serverUri, this);
        this.uid = myuid;
    }

    private String putNewTransaction(ITransactionCallbacks transactionCallbacks) {
        String transaction = stringGenerator.randomString(6);
        synchronized (transactionsLock) {
            while (transactions.containsKey(transaction))
                transaction = stringGenerator.randomString(6);
            transactions.put(transaction, transactionCallbacks);
        }
        return transaction;
    }

    private void createSession() {
        try {
            JSONObject obj = new JSONObject();
            obj.put("janus", JanusMessageType.create);
            ITransactionCallbacks cb = JanusTransactionCallbackFactory.createNewTransactionCallback(this, TransactionType.create);
            String transaction = putNewTransaction(cb);
            obj.put("transaction", transaction);

            JSONObject user_info = new JSONObject();
            user_info.put("user_id", uid);
            user_info.put("display", uid+"");
            obj.put("user_info", user_info);

            serverConnection.sendMessage(obj.toString());
        } catch (JSONException ex) {
            onCallbackError(ex.getMessage());
        }
    }

    public Boolean isConnected() {
        return connected;
    }

    public BigInteger getSessionId() {
        return sessionId;
    }

    public void Attach(IJanusPluginCallbacks callbacks) {
        try {
            JSONObject obj = new JSONObject();
            obj.put("janus", JanusMessageType.attach);
            obj.put("plugin", callbacks.getPlugin());
            if (serverConnection.getMessengerType() == JanusMessengerType.websocket)
                obj.put("session_id", sessionId);
            ITransactionCallbacks tcb = JanusTransactionCallbackFactory.createNewTransactionCallback(JanusServer.this, TransactionType.attach, callbacks.getPlugin(), callbacks);
            String transaction = putNewTransaction(tcb);
            obj.put("transaction", transaction);
            serverConnection.sendMessage(obj.toString(), sessionId);
        } catch (JSONException ex) {
            onCallbackError(ex.getMessage());
        }
    }

    public void stopKeepAliveThread(){
        if(keep_alive != null){
            keep_alive.interrupt();
            keep_alive = null;
        }
   }

    public void Destroy() {
        serverConnection.disconnect();
        stopKeepAliveThread();
        connected = false;
        gatewayObserver.onDestroy();
        for (ConcurrentHashMap.Entry<BigInteger, JanusPluginHandle> handle : attachedPlugins.entrySet()) {
            handle.getValue().detach();
        }
        synchronized (transactionsLock) {
            for (Object trans : transactions.entrySet())
                transactions.remove(trans);
        }
    }

    public void Connect() {
        serverConnection.connect();
    }

    public void newMessageForPlugin(String message, BigInteger plugin_id) {
        JanusPluginHandle handle = null;
        synchronized (attachedPluginsLock) {
            handle = attachedPlugins.get(plugin_id);
        }
        if (handle != null) {
            handle.onMessage(message);
        }
    }

    @Override
    public void onCallbackError(String msg) {
        gatewayObserver.onCallbackError(msg);
    }

    public void sendMessage(JSONObject msg, JanusMessageType type, BigInteger handle) {
        try {
            msg.put("janus", type.toString());
            if (serverConnection.getMessengerType() == JanusMessengerType.websocket) {
                msg.put("session_id", sessionId);
                msg.put("handle_id", handle);
            }
            msg.put("transaction", stringGenerator.randomString(6));
            if (connected)
                serverConnection.sendMessage(msg.toString(), sessionId, handle);
            if (type == JanusMessageType.detach) {
                synchronized (attachedPluginsLock) {
                    if (attachedPlugins.containsKey(handle))
                        attachedPlugins.remove(handle);
                }
            }
        } catch (JSONException ex) {
            gatewayObserver.onCallbackError(ex.getMessage());
        }
    }

    public void sendMessage(TransactionType type, BigInteger handle, IPluginHandleSendMessageCallbacks callbacks, JanusSupportedPluginPackages plugin) {
        JSONObject msg = callbacks.getMessage();
        if (msg != null) {
            try {
                JSONObject newMessage = new JSONObject();
                newMessage.put("janus", JanusMessageType.message.toString());

                if (serverConnection.getMessengerType() == JanusMessengerType.websocket) {
                    newMessage.put("session_id", sessionId);
                    newMessage.put("handle_id", handle);
                }
                ITransactionCallbacks cb = JanusTransactionCallbackFactory.createNewTransactionCallback(this, TransactionType.plugin_handle_message, plugin, callbacks);
                String transaction = putNewTransaction(cb);
                newMessage.put("transaction", transaction);
                if (msg.has("message"))
                    newMessage.put("body", msg.getJSONObject("message"));
                if (msg.has("jsep"))
                    newMessage.put("jsep", msg.getJSONObject("jsep"));
                serverConnection.sendMessage(newMessage.toString(), sessionId, handle);
            } catch (JSONException ex) {
                callbacks.onCallbackError(ex.getMessage());
            }
        }
    }

    public void sendMessage(TransactionType type, BigInteger handle, IPluginHandleWebRTCCallbacks callbacks, JanusSupportedPluginPackages plugin) {
        try {
            JSONObject msg = new JSONObject();
            msg.put("janus", JanusMessageType.message.toString());
            if (serverConnection.getMessengerType() == JanusMessengerType.websocket) {
                msg.put("session_id", sessionId);
                msg.put("handle_id", handle);
            }
            ITransactionCallbacks cb = JanusTransactionCallbackFactory.createNewTransactionCallback(this, TransactionType.plugin_handle_webrtc_message, plugin, callbacks);
            String transaction = putNewTransaction(cb);
            msg.put("transaction", transaction);
            if (callbacks.getJsep() != null) {
                msg.put("jsep", callbacks.getJsep());
            }
            serverConnection.sendMessage(msg.toString(), sessionId, handle);
        } catch (JSONException ex) {
            callbacks.onCallbackError(ex.getMessage());
        }
    }

    public void sendMessage(int myId, String myName, int remoteId, int type, boolean isVideo, int isAccept) {
        try {
            JSONObject user_call = new JSONObject();
            user_call.put("janus", "user_call");

            JSONObject user_info = new JSONObject();
            user_info.put("user_id", myId);
            user_info.put("display", myName);
            user_call.put("user_info", user_info);

            JSONObject janus_ext = new JSONObject();
            janus_ext.put("name", "videocall");
            janus_ext.put("uid", remoteId);//对方Id
            janus_ext.put("type", type);//主叫方:0第一次呼叫，2取消     被叫方：1
            janus_ext.put("isVideo", isVideo);
            janus_ext.put("isAccept", isAccept);//主叫方：0发起，3取消  。被叫方：1拒绝，2接受, 4通话中
            user_call.put("janus_ext", janus_ext);

            if (serverConnection.getMessengerType() == JanusMessengerType.websocket) {
                user_call.put("session_id", sessionId);
            }
            ITransactionCallbacks cb = JanusTransactionCallbackFactory.createNewTransactionCallback(this, TransactionType.user_call);
            String transaction = putNewTransaction(cb);
            user_call.put("transaction", transaction);

            LogUtil.d("User_call sent: " + user_call.toString());

            serverConnection.sendMessage(user_call.toString());
        } catch (JSONException ex) {

        }
    }

    public void sendMessage(int myId, String myName, int remoteId, String plugin, boolean connect, int reason_code, String reason) {
        try {
            JSONObject user_call = new JSONObject();
            user_call.put("janus", "user_call");

            JSONObject user_info = new JSONObject();
            user_info.put("user_id", myId);
            user_info.put("display", myName);
            user_call.put("user_info", user_info);

            JSONObject janus_ext = new JSONObject();
            janus_ext.put("name", plugin);
            janus_ext.put("uid", remoteId);
            janus_ext.put("connect", connect);
            janus_ext.put("reason_code", reason_code);
            janus_ext.put("reason", reason);
            user_call.put("janus_ext", janus_ext);

            if (serverConnection.getMessengerType() == JanusMessengerType.websocket) {
                user_call.put("session_id", sessionId);
            }
            ITransactionCallbacks cb = JanusTransactionCallbackFactory.createNewTransactionCallback(this, TransactionType.user_call);
            String transaction = putNewTransaction(cb);
            user_call.put("transaction", transaction);

            LogUtil.d("User_call sent: " + user_call.toString());

            serverConnection.sendMessage(user_call.toString());
        } catch (JSONException ex) {

        }
    }

    //region MessageObserver
    @Override
    public void receivedNewMessage(JSONObject obj) {
        LogUtil.d("Received: " + obj.toString());
        try {
            JanusMessageType type = JanusMessageType.fromString(obj.getString("janus"));
            String transaction = null;
            BigInteger sender = null;
            if (obj.has("transaction")) {
                transaction = obj.getString("transaction");
            }

            if (obj.has("sender")) {
                sender = new BigInteger(obj.getString("sender"));
            }
            JanusPluginHandle handle = null;
            if (sender != null) {
                synchronized (attachedPluginsLock) {
                    handle = attachedPlugins.get(sender);
                }
            }

            switch (type) {
                case keepalive:
                    break;
                case ack:
                    //主要是心跳消息
                    //todo 把心跳从这里剥离 放到上面的ACK类里面去   目前心跳计数并不完善 上面多了一个import static janusclientapi.JanusMessageType.ack;
                    LogUtil.d("无回调消息的返回:" + transaction + " " +keepalive_count);
                    keepalive_count++;

                    gatewayObserver.onCallbackKeep_alive(obj);
                case success:
                case error: {
                    if (transaction != null) {
                        ITransactionCallbacks cb = null;
                        synchronized (transactionsLock) {
                            cb = transactions.get(transaction);
                            if (cb != null)
                                transactions.remove(transaction);
                        }
                        if (cb != null) {
                            cb.reportSuccess(obj);
                            transactions.remove(transaction);
                        }
                    }
                    break;
                }
                case hangup: {
//                    if(handle != null) {
//                        handle.hangUp();
//                        JSONObject plugin_data = null;
//                        if (obj.has("plugindata"))
//                            plugin_data = obj.getJSONObject("plugindata");
//                        if (plugin_data != null) {
//                            JSONObject data = null;
//                            JSONObject jsep = null;
//                            if (plugin_data.has("data"))
//                                data = plugin_data.getJSONObject("data");
//                            if (obj.has("jsep"))
//                                jsep = obj.getJSONObject("jsep");
//                            handle.onMessage(data, jsep);
//                        }
//                    }
                    break;
                }
                case detached: {
                    if (gatewayObserver != null) {
                        gatewayObserver.onDetached();
                    }
                    break;
                }
                case user_called: {
                    if (transaction != null) {
                        ITransactionCallbacks cb = null;
                        synchronized (transactionsLock) {
                            cb = transactions.get(transaction);
                            if (cb != null)
                                transactions.remove(transaction);
                        }
                        if (cb != null) {
                            cb.reportSuccess(obj);
                            transactions.remove(transaction);
                        }
                    }
                    break;
                }
                case user_call: {
                    if (gatewayObserver != null) {
                        gatewayObserver.onUserCallSuccess(obj);
                    }
                    break;
                }
                case timeout: {
                    if (gatewayObserver != null) {
                        gatewayObserver.onTimeOut(obj);
                    }
                    break;
                }
                case slowlink: {
                    if (gatewayObserver != null) {
                        gatewayObserver.onSlowLink(obj);
                    }
                    break;
                }
                case event: {
                    if (handle != null) {
                        JSONObject plugin_data = null;
                        if (obj.has("plugindata")) {
                            plugin_data = obj.getJSONObject("plugindata");
                        }
                        if (plugin_data != null) {
                            JSONObject data = null;
                            JSONObject jsep = null;
                            if (plugin_data.has("data"))
                                data = plugin_data.getJSONObject("data");
                            if (obj.has("jsep"))
                                jsep = obj.getJSONObject("jsep");
                            handle.onMessage(data, jsep);
                        }
                    }
                    break;
                }
                case media: {
                    gatewayObserver.onUserCallSuccess(obj);
                    break;
                }
            }
        } catch (JSONException ex) {
            gatewayObserver.onCallbackError(ex.getMessage());
        }
    }

    @Override
    public void onOpen() {
        createSession();
    }

    @Override
    public void onClose() {
        connected = false;
        gatewayObserver.onCallbackError("Connection to janus server is closed");
    }

    @Override
    public void onError(Exception ex) {
        gatewayObserver.onCallbackError("Error connected to Janus gateway. Exception: " + ex.getMessage());
    }
    //endregion

    //region SessionCreationCallbacks
    @Override
    public void onSessionCreationSuccess(JSONObject obj) {
        try {
            sessionId = new BigInteger(obj.getJSONObject("data").getString("id"));

            keep_alive = new Thread(this, "KeepAlive");
            keep_alive.start();

            connected = true;
            //TODO do we want to keep track of multiple sessions and servers?
            gatewayObserver.onSuccess();
        } catch (JSONException ex) {
            gatewayObserver.onCallbackError(ex.getMessage());
        }
    }

    public String sendAliveOnce() {//返回本次心跳的transaction值
        JSONObject obj = new JSONObject();
        try {
            obj.put("janus", JanusMessageType.keepalive.toString());
            if (serverConnection.getMessengerType() == JanusMessengerType.websocket)
                obj.put("session_id", sessionId);
            String aliveMsgTransaction = stringGenerator.randomString(6);
            obj.put("transaction", aliveMsgTransaction);
            serverConnection.sendMessage(obj.toString(), sessionId);
            return aliveMsgTransaction;
        } catch (JSONException ex) {
            gatewayObserver.onCallbackError("Keep alive failed is Janus online?" + ex.getMessage());
            connected = false;
            return null;
        }
    }

    public int getKeepalive_count() {
        return keepalive_count;
    }

    //endregion

    //region AttachPluginCallbacks

    @Override
    public void attachPluginSuccess(JSONObject obj, JanusSupportedPluginPackages plugin, IJanusPluginCallbacks pluginCallbacks) {
        try {
            BigInteger handle = new BigInteger(obj.getJSONObject("data").getString("id"));
            JanusPluginHandle pluginHandle = new JanusPluginHandle(this, plugin, handle, pluginCallbacks);
            synchronized (attachedPluginsLock) {
                attachedPlugins.put(handle, pluginHandle);
            }
            pluginCallbacks.success(pluginHandle);
        } catch (JSONException ex) {
            //or do we want to use the pluginCallbacks.error(ex.getMessage());
            gatewayObserver.onCallbackError(ex.getMessage());
        }
    }

    // user_call
    @Override
    public void onUserCallSuccess(JSONObject obj) {
        LogUtil.d("User_call received: " + obj.toString());

        try{
            if (obj.has("err_code")) {
                if (obj.getInt("err_code") != 0) {
                    if (obj.has("err_msg")) {
                        gatewayObserver.onCallbackError(obj.getString("err_msg"));
                    }
                }
            }
        }catch (JSONException ex) {
            gatewayObserver.onCallbackError(ex.getMessage());
        }
    }

    //endregion
}
