package com.jimi.jimitalk.ptt.api;

import com.jimi.jimitalk.ptt.janusclientapi.IJanusGatewayCallbacks;
import com.jimi.jimitalk.ptt.janusclientapi.JanusServer;
import com.jimi.jimitalk.tools.AppTool;
import com.jimi.jimitalk.tools.JimipttConfig;
import com.jimi.jimitalk.tools.LogUtil;
import com.jimi.jimitalk.tools.NetworkUtils;

import org.json.JSONObject;
import org.webrtc.PeerConnection;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.TimeUnit;

public class JanusSession {
    private JanusServer janusServer;
    private String serverUri;

    private LinkedList<IJanusSessionMessageCallbacks> messageCallbacks = new LinkedList<>();
    private boolean isShutdown;
    private volatile boolean isConnecting;
    private volatile boolean isConnected;

    private List<IJanusPlugin> plugins = new LinkedList<>();

    public JanusSession(String serverUri, int userId) {
        this.serverUri = serverUri;//TODO, 'serverUri' assignment must before 'janusServer'
        this.janusServer = new JanusServer(new JanusGlobalCallbacks(), userId);
        this.isShutdown = false;
        this.isConnected = false;
        this.isConnecting = false;
    }

    public void register(IJanusPlugin plugin) {
        if (!plugins.contains(plugin)) {
            plugins.add(plugin);
        }
    }

    public void setShutdown(boolean shutdown) {
        isShutdown = shutdown;
    }

    public void userCall(int myId, String myName, int remoteId, String plugin, boolean connect, int reason_code, String reason) {
        janusServer.sendMessage(myId, myName, remoteId, plugin, connect, reason_code, reason);
    }

    public void registerMessageCallbacks(IJanusSessionMessageCallbacks callbacks) {
        if (!messageCallbacks.contains(callbacks)) {
            messageCallbacks.add(callbacks);
        }
    }

    public void unregisterMessageCallbacks(IJanusSessionMessageCallbacks callbacks) {
        messageCallbacks.remove(callbacks);
    }

    public void connect() {
        janusServer.Connect();
    }

    public void disconnect() {
        janusServer.Destroy();
    }

    public boolean isConnected() {
        return isConnected;
    }

    public void attach(IJanusPlugin plugin) {
        janusServer.Attach(plugin.callbacks());
    }

    public void detach(IJanusPlugin plugin) {
        plugin.destroy();
    }

    public class JanusGlobalCallbacks implements IJanusGatewayCallbacks {
        @Override
        public void onSuccess() {
            isConnected = true;
            AppTool.sendBroadcast("jimi", "pocroom", "connected");

            for (IJanusPlugin plugin : plugins) {
                janusServer.Attach(plugin.callbacks());
            }
        }

        @Override
        public void onUserCallSuccess(JSONObject object) {
            LogUtil.d(object.toString());

            if (!messageCallbacks.isEmpty()) {
                for (IJanusSessionMessageCallbacks callbacks : messageCallbacks) {
                    callbacks.onMessage(object);
                }
            }
        }

        @Override
        public void onSlowLink(JSONObject object) {
            LogUtil.d(object.toString());
        }

        @Override
        public void onTimeOut(JSONObject object) {
            LogUtil.d(object.toString());

            reconnect();
        }

        @Override
        public void onDetached() {
        }

        @Override
        public void onDestroy() {
        }

        @Override
        public void onCallbackKeep_alive(JSONObject object) {
            LogUtil.d(object.toString());
        }

        @Override
        public String getServerUri() {
            //Such as "ws://www.yunptt.com:9188" or "wss://www.yunptt.com:8989"
            return serverUri;
        }


        @Override
        public List<PeerConnection.IceServer> getIceServers() {
            return new ArrayList<>();
        }

        @Override
        public Boolean getIpv6Support() {
            return Boolean.FALSE;
        }

        @Override
        public Integer getMaxPollEvents() {
            return 0;
        }

        @Override
        public void onCallbackError(String error) {
            LogUtil.e(error);
            reconnect();
        }
    }

    private synchronized void sessionReconnect() {
        LogUtil.d("JANUS RECONNECT STEP TWO, CHECK SIGNALS");
        LogUtil.d("isConnecting=" + isConnecting + " isShutdown=" + isShutdown + " isConnected=" + isConnected);
        AppTool.sendBroadcast("jimi", "pocroom", "disconnected");
        if (!isConnecting && !isShutdown) {
            isConnecting = true;

            // Need to send a heartbeat to detect first
            sendHeartbeatDetect();

            try {
                int reconnectDelay = 0;
                while (!isConnected) {
                    LogUtil.d("JANUS RECONNECT STEP SIX, START RECONNECT");
                    if (!NetworkUtils.isNetWorkAvailable(JimipttConfig.getAppContext())) {
                        LogUtil.d("JANUS RECONNECT STOP SEVEN, NO NETWORK, WAIT");
                        TimeUnit.SECONDS.sleep(5);
                    } else {
                        LogUtil.d("JANUS RECONNECT STOP EIGHT, TRY SERVER DISCONNECT & CONNECT");
                        reconnectDelay++;
                        disconnect();
                        connect();
                        TimeUnit.SECONDS.sleep(20 + reconnectDelay);
                    }
                    LogUtil.d("JANUS CONNECT STATUS " + isConnected +
                            " IF STILL DISCONNECTED, WAIT FOR " + (20 + reconnectDelay) +" SECONDS, TYR AGAIN LATER");
                }
            } catch (Exception ex) {
                ex.printStackTrace();
            }
            isConnecting = false;
        }
    }

    private void sendHeartbeatDetect() {
        LogUtil.d("JANUS RECONNECT STEP THREE, HEARTBEAT DETECT");

        try {
            int tmp = janusServer.getKeepalive_count();
            janusServer.sendAliveOnce();
            TimeUnit.SECONDS.sleep(2);
            if (janusServer.getKeepalive_count() > tmp) {
                // Network delay, nothing to do
                LogUtil.d("JANUS RECONNECT STEP FOUR, HEARTBEAT RECEIVED, RECONNECT FINISHED");
                isConnected = true;
            }
        } catch (Exception ex) {
            ex.printStackTrace();
        }
    }

    public void reconnect() {
        LogUtil.d("JANUS RECONNECT STEP ONE, SEND SIGNAL & RUN THREAD");
        isConnected = false;

        new Thread(new Runnable() {
            @Override
            public void run() {
                sessionReconnect();
            }
        }).start();
    }
}
