package com.jimi.jimitalk.ptt.api;

import android.content.Context;

import com.jimi.jimitalk.ptt.janusclientapi.IJanusPluginCallbacks;
import com.jimi.jimitalk.ptt.janusclientapi.IPluginHandleSendMessageCallbacks;
import com.jimi.jimitalk.ptt.janusclientapi.IPluginHandleWebRTCCallbacks;
import com.jimi.jimitalk.ptt.janusclientapi.JanusMediaConstraints;
import com.jimi.jimitalk.ptt.janusclientapi.JanusPluginHandle;
import com.jimi.jimitalk.ptt.janusclientapi.JanusSupportedPluginPackages;
import com.jimi.jimitalk.ptt.janusclientapi.PluginHandleSendMessageCallbacks;
import com.jimi.jimitalk.tools.AppTool;
import com.jimi.jimitalk.tools.JimipttConfig;
import com.jimi.jimitalk.tools.LogUtil;

import org.json.JSONException;
import org.json.JSONObject;
import org.webrtc.MediaStream;

import java.util.LinkedList;

/**
 * JanusPocRoomPlugin is a wrapper of PoC(Push to talk over cell), as a plugin in Janus, need to
 * be attached first, media data exchanged by rtc tunnel created.
 *
 * Despite talk & untalk, other methods can refer to Janus audiobridge plugin
 */
public class JanusPocRoomPlugin implements IJanusPlugin {
    private final String MESSAGE = "message";
    private final String REQUEST = "request";

    private JanusPluginHandle handle = null;
    private JanusPocRoomPluginCallbacks pluginCallback;
    private LinkedList<IJanusPluginMessageCallbacks> messageCallbacks = new LinkedList<>();
    private String userName;
    private int userId;
    
    private boolean isInOneRoom;
    private boolean isMediaConnected;
    private boolean isPluginAttached;

    public JanusPocRoomPlugin(String userName, int userId) {
        this.userName = userName;
        this.userId = userId;
        this.pluginCallback = new JanusPocRoomPluginCallbacks();
        this.isMediaConnected = false;
        this.isPluginAttached = false;
        this.isInOneRoom = false;
    }

    /** Register synchronous message callback, all plugin messages return by this method*/
    public void registerMessageCallbacks(IJanusPluginMessageCallbacks callback) {
        if (!messageCallbacks.contains(callback)) {
            messageCallbacks.add(callback);
        }
    }

    public void unregisterMessageCallbacks(IJanusPluginMessageCallbacks callback) {
        messageCallbacks.remove(callback);
    }

    public void removeAudioTrack() {
        if (handle != null) {
            handle.removeAudioTrack();
        }
    }

    public void addAudioTrack() {
        if (handle != null) {
            handle.addAudioTrack();
        }
    }

    public void joinRoom(int roomId) {
        try {
            final JSONObject obj = new JSONObject();
            final JSONObject msg = new JSONObject();

            obj.put(REQUEST, "join");
            obj.put("room", roomId);
            obj.put("id", userId);
            obj.put("muted", true);
            obj.put("display", userName);
            msg.put(MESSAGE, obj);

            if (handle != null) {
                LogUtil.d("Sent:" + msg.toString());
                handle.sendMessage(new PluginHandleSendMessageCallbacks(msg));
            }
        } catch(Exception e) {
            e.printStackTrace();
        }
    }

    public void talk(int roomId) {
        addAudioTrack();

        try {
            final JSONObject msg = new JSONObject();
            JSONObject body = new JSONObject();
            body.put(REQUEST, "talk");
            body.put("room", roomId);
            body.put("id", userId);
            body.put("muted", false);
            msg.put(MESSAGE, body);

            if (handle != null) {
                LogUtil.d("Sent:" + msg.toString());
                handle.sendMessage(new PluginHandleSendMessageCallbacks(msg) {
                    @Override
                    public void onSuccessSynchronous(JSONObject obj) {
                        LogUtil.d(obj.toString());

                        if (!messageCallbacks.isEmpty()) {
                            for (IJanusPluginMessageCallbacks callback : messageCallbacks) {
                                callback.onMessage(obj, null);
                            }
                        }
                    }

                    @Override
                    public void onSuccessAsynchronous() {

                    }

                    @Override
                    public JSONObject getMessage() {
                        return msg;
                    }

                    @Override
                    public void onCallbackError(String error) {
                        LogUtil.e(error);
                    }
                });
            }
        }catch (JSONException e) {
            e.printStackTrace();
        }
    }

    public void untalk(int roomId) {
        try {
            final JSONObject msg = new JSONObject();
            JSONObject body = new JSONObject();
            body.put(REQUEST, "untalk");
            body.put("room", roomId);
            body.put("id", userId);
            body.put("muted", true);
            msg.put(MESSAGE, body);

            if (handle != null) {
                LogUtil.d("Sent:" + msg.toString());
                handle.sendMessage(new PluginHandleSendMessageCallbacks(msg) {
                    @Override
                    public void onSuccessSynchronous(JSONObject obj) {
                        LogUtil.d(obj.toString());

                        if (!messageCallbacks.isEmpty()) {
                            for (IJanusPluginMessageCallbacks callback : messageCallbacks) {
                                callback.onMessage(obj, null);
                            }
                        }
                    }

                    @Override
                    public void onSuccessAsynchronous() {

                    }

                    @Override
                    public JSONObject getMessage() {
                        return msg;
                    }

                    @Override
                    public void onCallbackError(String error) {
                        LogUtil.e(error);
                    }
                });
            }
        }catch (JSONException e) {
            e.printStackTrace();
        }

        removeAudioTrack();
    }

    public void createPeerConnection(Context context, boolean isAudio, boolean isVideo) {
        if (handle != null) {
            handle.createPeerConnectionFactory(context, isAudio, isVideo);

            createPeerConnection();
        }
    }

    /** Create a rtc tunnel*/
    private void createPeerConnection() {
        if (handle != null) {
            handle.createPeerConnection(new IPluginHandleWebRTCCallbacks() {
                @Override
                public void onSuccess(JSONObject obj) {
                    try {
                        final JSONObject msg = new JSONObject();
                        JSONObject body = new JSONObject();
                        body.put(REQUEST, "configure");
                        body.put("muted", false);
                        body.put("display", userName);
                        msg.put(MESSAGE, body);
                        msg.put("jsep", obj);

                        LogUtil.d("Sent:" + msg.toString());
                        handle.sendMessage(new PluginHandleSendMessageCallbacks(msg) {
                            @Override
                            public void onSuccessSynchronous(JSONObject obj) {
                                LogUtil.d(obj.toString());
                                if (!messageCallbacks.isEmpty()) {
                                    for (IJanusPluginMessageCallbacks callback : messageCallbacks) {
                                        callback.onMessage(obj, null);
                                    }
                                }
                            }

                            @Override
                            public void onSuccessAsynchronous() {
                            }

                            @Override
                            public JSONObject getMessage() {
                                return msg;
                            }

                            @Override
                            public void onCallbackError(String error) {
                                LogUtil.e(error);
                            }
                        });
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }

                @Override
                public JSONObject getJsep() {
                    return null;
                }

                @Override
                public JanusMediaConstraints getMedia() {
                    JanusMediaConstraints cons = new JanusMediaConstraints();
                    cons.setRecvAudio(true);
                    cons.setRecvVideo(false);
                    cons.setSendAudio(true);
                    cons.setSendVideo(false);
                    return cons;
                }

                @Override
                public Boolean getTrickle() {
                    return true;
                }

                @Override
                public void onCallbackError(String error) {
                    LogUtil.e(error);
                }
            });
        }
    }

    public void changeRoom(int changeRoomId){
        try {
            final JSONObject msg = new JSONObject();
            JSONObject body = new JSONObject();
            body.put(REQUEST, "changeroom");
            body.put("room", changeRoomId);
            body.put("id", userId);
            body.put("display", userName);
            body.put("muted", true);
            msg.put(MESSAGE, body);

            if (handle != null) {
                LogUtil.d("Sent:" + msg.toString());
                handle.sendMessage(new PluginHandleSendMessageCallbacks(msg) {
                    @Override
                    public void onSuccessSynchronous(JSONObject obj) {
                        LogUtil.d(obj.toString());
                        if (!messageCallbacks.isEmpty()) {
                            for (IJanusPluginMessageCallbacks callback : messageCallbacks) {
                                callback.onMessage(obj, null);
                            }
                        }
                    }

                    @Override
                    public void onSuccessAsynchronous() {

                    }

                    @Override
                    public JSONObject getMessage() {
                        return msg;
                    }

                    @Override
                    public void onCallbackError(String error) {
                        LogUtil.e(error);
                    }
                });
            }
        }catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void leave() {
        try {
            final JSONObject msg = new JSONObject();
            JSONObject body = new JSONObject();
            body.put(REQUEST, "leave");
            msg.put(MESSAGE, body);

            if (handle != null) {
                LogUtil.d("Sent:" + msg.toString());
                handle.sendMessage(new PluginHandleSendMessageCallbacks(msg) {
                    @Override
                    public void onSuccessSynchronous(JSONObject obj) {
                        LogUtil.d(obj.toString());
                    }

                    @Override
                    public void onSuccessAsynchronous() {

                    }

                    @Override
                    public JSONObject getMessage() {
                        return msg;
                    }

                    @Override
                    public void onCallbackError(String error) {
                        LogUtil.e(error);
                    }
                });
            }
        }catch (Exception e) {
            e.printStackTrace();
        }
    }

    /** Configure rtc attributes*/
    public void configure(boolean muted){
        try {
            final JSONObject msg = new JSONObject();
            JSONObject body = new JSONObject();
            body.put(REQUEST, "configure");
            body.put("muted", muted);
            body.put("display", userName);
            msg.put(MESSAGE, body);

            if (handle != null) {
                LogUtil.d("Sent:" + msg.toString());
                handle.sendMessage(new PluginHandleSendMessageCallbacks(msg) {
                    @Override
                    public void onSuccessSynchronous(JSONObject obj) {
                        LogUtil.d(obj.toString());
                    }

                    @Override
                    public void onSuccessAsynchronous() {

                    }

                    @Override
                    public JSONObject getMessage() {
                        return msg;
                    }

                    @Override
                    public void onCallbackError(String error) {
                        LogUtil.e(error);
                    }
                });
            }
        }catch (JSONException e) {
            e.printStackTrace();
        }
    }

    public boolean isMediaConnected() {
        return isMediaConnected;
    }

    @Override
    public IJanusPluginCallbacks callbacks() {
        return pluginCallback;
    }

    @Override
    public void destroy() {
        if (handle != null) {
            handle.detach();
        }
    }

    public boolean isPluginAttached() {
        return isPluginAttached;
    }

    public boolean isInOneRoom() {
        return isInOneRoom;
    }

    public void kick(int roomId, int uid){
        JSONObject obj = new JSONObject();
        JSONObject msg = new JSONObject();
        try {
            obj.put(REQUEST, "kick");
            obj.put("room", roomId);
            obj.put("id", uid);
            msg.put(MESSAGE, obj);

            if (handle != null) {
                LogUtil.d("Sent:" + msg.toString());
                handle.sendMessage(new IPluginHandleSendMessageCallbacks() {
                    @Override
                    public void onSuccessSynchronous(JSONObject obj) {
                        joinRoom(roomId);
                        changeRoom(roomId);
                    }

                    @Override
                    public void onSuccessAsynchronous() {
                        joinRoom(roomId);
                        changeRoom(roomId);
                    }

                    @Override
                    public JSONObject getMessage() {
                        return msg;
                    }

                    @Override
                    public void onCallbackError(String error) {

                    }
                });
            }
        } catch(Exception ex) {
            ex.printStackTrace();
        }
    }

    public class JanusPocRoomPluginCallbacks implements IJanusPluginCallbacks {
        @Override
        public void success(JanusPluginHandle pluginHandle) {
            LogUtil.i("PoC plugin attached!");

            handle = pluginHandle;
            isPluginAttached = true;

            try {
                JSONObject msg = new JSONObject();
                msg.put("pocroom", "attached");

                if (!messageCallbacks.isEmpty()) {
                    for (IJanusPluginMessageCallbacks callback : messageCallbacks) {
                        callback.onMessage(msg, null);
                    }
                }
            } catch (JSONException ex) {
                ex.printStackTrace();
            }
        }

        @Override
        public void onMessage(JSONObject msg, final JSONObject jsepLocal) {
            LogUtil.i("msg=" + msg);
            LogUtil.i("jsepLocal=" + jsepLocal);

            if (!messageCallbacks.isEmpty()) {
                for (IJanusPluginMessageCallbacks callback : messageCallbacks) {
                    callback.onMessage(msg, null);
                }
            }

            try {
                if(msg.getString("pocroom").equals("joined")) {
                    if (msg.has("id") && msg.getInt("id") == userId) {
                        createPeerConnection(JimipttConfig.getAppContext(), true, false);
                        isInOneRoom = true;
                    }
                }

                if (msg.getString("pocroom").equals("destroyed")) { //Deleted current group
                    isInOneRoom = false;
                    isMediaConnected = false;
                }

                //讲话超过30S服务器会发timeout
                if (msg.getString("pocroom").equals("talked_timeout")) {
                    AppTool.sendBroadcast("jimi", "talkfreed", userId + "");
                    removeAudioTrack();
                }

                //讲话过程中被优先级更高的用户抢麦
                if (msg.getString("pocroom").equals("talked_handover")) {
                    removeAudioTrack();
                }

                if (jsepLocal != null && jsepLocal.getString("type").equals("answer")) {
                    if (handle != null) {
                        handle.JanusSetRemoteDescription(new IPluginHandleWebRTCCallbacks() {
                            @Override
                            public void onSuccess(JSONObject obj) {
                                LogUtil.d(obj.toString());
                            }

                            @Override
                            public JSONObject getJsep() {
                                return jsepLocal;
                            }

                            @Override
                            public JanusMediaConstraints getMedia() {
                                JanusMediaConstraints cons = new JanusMediaConstraints();
                                cons.setRecvAudio(true);
                                cons.setRecvVideo(false);
                                cons.setSendAudio(true);
                                cons.setSendVideo(false);
                                return cons;
                            }

                            @Override
                            public Boolean getTrickle() {
                                return Boolean.TRUE;
                            }

                            @Override
                            public void onCallbackError(String error) {
                                LogUtil.e(error);
                            }
                        }, false);
                    }
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }

        @Override
        public void onLocalStream(MediaStream stream) { }

        @Override
        public void onRemoteStream(MediaStream stream) { }

        @Override
        public void onDataOpen(Object data) {
            LogUtil.d(data.toString());
        }

        @Override
        public void onData(Object data) {
            LogUtil.d(data.toString());
        }

        @Override
        public void onCleanup() {
        }

        @Override
        public JanusSupportedPluginPackages getPlugin() {
            return JanusSupportedPluginPackages.JANUS_POC_ROOM;
        }

        @Override
        public void onCallbackError(String error) {
            LogUtil.e(error);
        }

        @Override
        public void onDetached() {
            LogUtil.d("JanusPocRoomPluginCallbacks onDetached");
        }

        @Override
        public void onIceConnectionChange(String msg) {
            LogUtil.e(msg);

            try {
              JSONObject message = new JSONObject();
              message.put("ice", msg);

                if (!messageCallbacks.isEmpty()) {
                    for (IJanusPluginMessageCallbacks callback : messageCallbacks) {
                        callback.onMessage(message, null);
                    }
                }
            } catch (JSONException e) {
                e.printStackTrace();
            }

            if(msg.contains("IceFailed")){
                isMediaConnected = false;
            }

            if(msg.contains("IceDisconnected")){
                isMediaConnected = false;
            }

            if(msg.contains("IceConnected")){
                //TODO
                addAudioTrack();
                removeAudioTrack();
                isMediaConnected = true;
            }

            if (msg.contains("IceClosed")){
                isMediaConnected = false;
                isInOneRoom = false;
            }
        }
    }
}
