package com.jimi.jimitalk.grpc;

import com.jimi.jimitalk.tools.SPUtil;

import io.grpc.ConnectivityState;
import io.grpc.ManagedChannel;
import io.grpc.ManagedChannelBuilder;
import io.grpc.Metadata;
import io.grpc.stub.MetadataUtils;
import io.grpc.stub.StreamObserver;
import talk_cloud.TalkCloudApp;
import talk_cloud.TalkCloudGrpc;
import talk_cloud.TalkCloudLocationGrpc;

public class GrpcConnection {
    private ManagedChannel channel;
    private TalkCloudGrpc.TalkCloudBlockingStub blockingStub;
    private TalkCloudLocationGrpc.TalkCloudLocationBlockingStub locationBlockingStub;

    private StreamObserver<TalkCloudApp.StreamResponse> response;
    private StreamObserver<TalkCloudApp.StreamRequest> request;

    /** Dispatch rpc stream messages*/
    private IGrpcStreamObserverCallbacks callbacks;

    public GrpcConnection(String ip, int port) throws IllegalArgumentException {
        this.channel = ManagedChannelBuilder.forAddress(ip, port).usePlaintext().build();

        this.blockingStub = TalkCloudGrpc.newBlockingStub(channel);

        this.locationBlockingStub = TalkCloudLocationGrpc.newBlockingStub(channel);
    }

    public StreamObserver<TalkCloudApp.StreamRequest> getRequest() {
        return request;
    }

    public void registerMessageCallbacks(IGrpcStreamObserverCallbacks callbacks) {
        this.callbacks = callbacks;
    }

    public void startGrpcStreamReceiver() {
        response = new StreamObserver<TalkCloudApp.StreamResponse>() {
            @Override
            public void onNext(TalkCloudApp.StreamResponse value) {
                if (callbacks != null) {
                    callbacks.onNext(value);
                }
            }

            @Override
            public void onError(Throwable t) {
                if (callbacks != null) {
                    callbacks.onError(t);
                }
            }

            @Override
            public void onCompleted() {
                if (callbacks != null) {
                    callbacks.onCompleted();
                }
            }
        };

        if (isGrpcAvailable()) {
            TalkCloudGrpc.TalkCloudStub stub = TalkCloudGrpc.newStub(channel);

            // Check grpc session id
            Metadata header = new Metadata();
            Metadata.Key<String> key = Metadata.Key.of("session-id", Metadata.ASCII_STRING_MARSHALLER);
            header.put(key, SPUtil.getGrpcSessionId());
            stub = MetadataUtils.attachHeaders(stub, header);

            request = stub.dataPublish(response);
        }
    }

    public void stopGrpcStreamReceiver() {
        if (request != null) {
            request.onCompleted();
            request = null;
        }

        if (response != null) {
            response.onCompleted();
            response = null;
        }
    }

    public boolean isGrpcAvailable() {
        if (channel == null) {
            return false;
        }
        ConnectivityState state = channel.getState(false);
        if (ConnectivityState.CONNECTING.equals(state)
                || ConnectivityState.READY.equals(state)
                || ConnectivityState.TRANSIENT_FAILURE.equals(state)
                || ConnectivityState.IDLE.equals(state)) {
            return true;
        }
        return  false;
    }

    public TalkCloudGrpc.TalkCloudBlockingStub getBlockingStub() {
        return blockingStub;
    }

    public ManagedChannel getManagedChannel() {
        return channel;
    }

    public TalkCloudLocationGrpc.TalkCloudLocationBlockingStub getLocationBlockingStub() {
        return locationBlockingStub;
    }

    public void closeGrpcConnection() {
        stopGrpcStreamReceiver();

        if (channel != null) {
            channel.shutdown();
        }
        channel = null;
        blockingStub = null;
        locationBlockingStub = null;
    }

    public void startGrpcConnection(String ip, int port) {
        channel = ManagedChannelBuilder.forAddress(ip, port).usePlaintext().build();
        blockingStub = TalkCloudGrpc.newBlockingStub(channel);
        locationBlockingStub = TalkCloudLocationGrpc.newBlockingStub(channel);
    }
}
