package com.jibestream.navigationkit.instructionfactory;

import android.graphics.PointF;
import android.support.annotation.NonNull;

import com.jibestream.jmapandroidsdk.collections.MapCollection;
import com.jibestream.jmapandroidsdk.components.Map;
import com.jibestream.jmapandroidsdk.components.Waypoint;
import com.jibestream.jmapandroidsdk.jcontroller.JController;
import com.jibestream.navigationkit.NavigationKit;
import com.jibestream.navigationkit.surroundings.SurroundingDefinition;

import java.util.ArrayList;

/**
 * Created by Andrew Adams on 2017-05-15.
 */
public class InstructionFactory {

    private NavigationKit navKit;
    private JController controller;

    private int visualLimitLeft;
    private int visualLimitRight;
    private int visualRange;

    private ArrayList<String> layersOfInterest;
    private ArrayList<String> visualObstacles;

    public InstructionFactory(@NonNull NavigationKit navKit, @NonNull JController controller) {
        this.navKit = navKit;
        this.controller = controller;

        //set default values
        this.visualLimitLeft = 60;
        this.visualLimitRight = 60;
        this.visualRange = 250;
        this.layersOfInterest = new ArrayList<>();
        this.visualObstacles = new ArrayList<>();
    }

    public Instruction createFromDirection(@NonNull Direction direction) {

        Instruction instruction = new Instruction(direction);

        setSurroundingDefinitionWithInstruction(instruction);
        setSurroundingElementsWithInstruction(instruction);

        return instruction;
    }

    public Instruction createEmpty() {
        return new Instruction();
    }

    public void setSurroundingElementsWithInstruction(@NonNull Instruction instruction) {
        instruction.setSurroundingElements(navKit.getSurroundingElements(instruction.getSurroundingDefinition()));
    }

    public void setSurroundingDefinitionWithInstruction(@NonNull Instruction instruction) {
        MapCollection mapCollection = controller.getActiveVenue().getMaps();
        Waypoint waypoint = instruction.getOriginPoint();
        Waypoint nextWaypoint = instruction.getDecisionPoint();

        if (nextWaypoint == null) {
            if (instruction.getCompletionPoint() != null) {
                nextWaypoint = instruction.getCompletionPoint();
            }
        }

        Map map = mapCollection.getById(waypoint.getMapId());

        if (map == null) {
            return;
        }

        PointF newPoint = new PointF(waypoint.getCoordinates()[0].floatValue(), waypoint.getCoordinates()[1].floatValue());

        SurroundingDefinition surroundingDefinition = new SurroundingDefinition(map, newPoint);
        surroundingDefinition.setGazeDirection(Direction.getAngle(waypoint, nextWaypoint));
        surroundingDefinition.setVisualLimitLeft(visualLimitLeft);
        surroundingDefinition.setVisualLimitRight(visualLimitRight);
        surroundingDefinition.setVisualRange(visualRange);
        surroundingDefinition.addAllLayersOfInterest(layersOfInterest);
        surroundingDefinition.addAllLayersOfInterest(visualObstacles);

        instruction.setSurroundingDefinition(surroundingDefinition);
    }

    public SurroundingDefinition getSurroundingDefinitionForInstruction(@NonNull Instruction instruction) {
        return instruction.getSurroundingDefinition();
    }

    public int getVisualLimitLeft() {
        return visualLimitLeft;
    }

    public void setVisualLimitLeft(int visualLimitLeft) {
        this.visualLimitLeft = visualLimitLeft;
    }

    public int getVisualLimitRight() {
        return visualLimitRight;
    }

    public void setVisualLimitRight(int visualLimitRight) {
        this.visualLimitRight = visualLimitRight;
    }

    public int getVisualRange() {
        return visualRange;
    }

    public void setVisualRange(int visualRange) {
        this.visualRange = visualRange;
    }

    public void addLayerOfInterest(@NonNull String layerName) {
        this.layersOfInterest.add(layerName);
    }

    public void removeLayerOfInterest(@NonNull String layerName) {
        this.layersOfInterest.remove(layerName);
    }

    public void addVisualObstacles(@NonNull String visualObstacleName) {
        this.visualObstacles.add(visualObstacleName);
    }

    public void removeVisualObstacles(@NonNull String visualObstacleName) {
        this.visualObstacles.remove(visualObstacleName);
    }
}
