package com.jibestream.mapuikit;

import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Matrix;
import android.graphics.Paint;
import android.graphics.Path;
import android.graphics.Rect;
import android.graphics.RectF;
import android.graphics.PointF;
import android.support.annotation.NonNull;
import android.util.Log;

import com.jibestream.jmapandroidsdk.rendering_engine.MapView;
import com.jibestream.jmapandroidsdk.rendering_engine.drawables.JDrawable;
import com.jibestream.jmapandroidsdk.styles.JStyle;
/**
 * Created by Ken Pangilinan on 2017-05-11.
 */

public class PopupDrawable extends JDrawable {
    private final boolean showActionButton;
    private String titleText;
    private JStyle titleTextStyle;
    private Rect titleTextBounds;

    private String subText;
    private JStyle subTextStyle;
    private Rect subTitleTextBounds;

    private String actionButtonText;
    private JStyle actionButtonTextStyle;
    private Rect actionButtonTextBounds;
    private JStyle actionButtonStyle;

    private String closeButtonText;
    private Rect closeButtonBounds;
    private JStyle closeButtonStyle;
    private RectF closeButtonBox;

    private JStyle popupStyle;

    private float pointerSize;
    private RectF popupBox;
    private Path popupPointer;
    private float padding;

    private int spacing;
    private RectF actionBox;

    private PopupDrawableOptions options;


    private float rotation;
    private float scale;

    private float opacity;

    public PopupDrawable(PopupDrawableOptions popupOptions) {
        this.titleText = popupOptions.titleText;
        this.subText = popupOptions.subText;
        this.actionButtonText = popupOptions.actionButtonText;
        this.actionButtonStyle = popupOptions.actionButtonStyle;
        this.actionButtonTextStyle = popupOptions.actionButtonTextStyle;
        this.showActionButton = popupOptions.showActionButton;

        //initialize paint objects
        this.titleTextStyle = popupOptions.titleTextStyle;
        this.subTextStyle = popupOptions.subTextStyle;

        this.popupStyle = popupOptions.popupStyle;
//        popupStyle.setShadowLayer(1, 2, 2, Color.DKGRAY);

        //measure size of titleText
        titleTextBounds = new Rect();
        titleTextStyle.getTextBounds(titleText, 0, titleText.length(), titleTextBounds);


        if (subText != null && !subText.isEmpty()) {
            subTitleTextBounds = new Rect();
            subTextStyle.getTextBounds(subText, 0, subText.length(), subTitleTextBounds);
        }

        if (actionButtonText != null && !actionButtonText.isEmpty()) {
            actionButtonTextBounds = new Rect();
            actionButtonTextStyle.getTextBounds(actionButtonText, 0, actionButtonText.length(), actionButtonTextBounds);
        }

        JStyle closeBtnStyle = new JStyle(popupOptions.titleTextStyle.getColor());
        this.closeButtonStyle = closeBtnStyle;
        this.closeButtonText = "X";
        closeButtonBounds = new Rect();
        this.closeButtonStyle.getTextBounds(closeButtonText, 0, closeButtonText.length(), closeButtonBounds);

        //default to heads up
        this.setHeadsUp(true);

        pointerSize = 10;
        padding = 15;
        spacing = 5;


        setVisible(true);
    }

    @Override
    public void draw(Canvas canvas) {

        int maxWidth = Math.max(titleTextBounds.width(), subTitleTextBounds != null ? subTitleTextBounds.width() : 0);
        popupBox = new RectF(
                getX() - (maxWidth / 2) - padding,
                getY() - pointerSize - titleTextBounds.height() - (padding * 2) -
                        (subTitleTextBounds != null ? subTitleTextBounds.height() + spacing : 0) -
                        ((actionButtonTextBounds != null && !actionButtonText.isEmpty() && showActionButton) ?
                                actionButtonTextBounds.height() + (padding * 3) : 0),
                getX() + (maxWidth / 2) + padding,
                getY() - pointerSize);

        popupPointer = new Path();
        popupPointer.moveTo(getX(), getY());
        popupPointer.rLineTo(-(pointerSize / 2), -pointerSize);
        popupPointer.rLineTo(pointerSize, 0);
        popupPointer.close();

        //popupStyle.setAlpha(50);
        canvas.drawRoundRect(popupBox, 5, 5, popupStyle);
        canvas.drawPath(popupPointer, popupStyle);

        float top = popupBox.top + padding + titleTextBounds.height();
        canvas.drawText(titleText, popupBox.left + padding, top, titleTextStyle);
        if (subTitleTextBounds != null) {
            top += subTitleTextBounds.height() + spacing;
            canvas.drawText(subText, popupBox.left + padding, top, subTextStyle);
        }

        if (actionButtonTextBounds != null && !actionButtonText.isEmpty() && showActionButton) {
            top += padding;
            actionBox = new RectF((int) (popupBox.left + padding), top, (int) (popupBox.right - padding), top + actionButtonTextBounds.height() + (padding *2));
            canvas.drawRoundRect(actionBox, 5, 5, actionButtonStyle);
            canvas.drawText(actionButtonText, actionBox.centerX() - actionButtonTextBounds.width()/2, actionBox.centerY() + actionButtonTextBounds.height()/2, actionButtonTextStyle);
        }

        // Set close button bounds
        float leftClose = popupBox.right - closeButtonBounds.width() - 5;
        float topClose = popupBox.top;
        float rightClose = popupBox.right;
        float bottomClose = popupBox.top + closeButtonBounds.height() + 10;
        closeButtonBox = new RectF(leftClose, topClose, rightClose, bottomClose);

        // Draw the close button on canvas
        canvas.drawText(closeButtonText, popupBox.right - closeButtonBounds.width() - 5, popupBox.top + closeButtonBounds.height() + 5, closeButtonStyle);
    }

    @Override
    public RectF getBounds() {
        // Check for heads up to adjust bounds of the popupBox
        if (this.isHeadsUp()) {
            RectF rotatedPopupBox = popupBox;
            Matrix m = new Matrix();
            m.setRotate(this.rotation, getX(), getY());
            m.mapRect(rotatedPopupBox);
            return rotatedPopupBox;
        }

        return popupBox;
    }

    /**
     * @return Options for popup drawable.
     */
    public PopupDrawableOptions getOptions() {
        return options;
    }

    /**
     * @param options Set options for popup drawable.
     */
    public void setOptions(@NonNull PopupDrawableOptions options) {
        this.options = options;
    }

    public void popupTapped(PointF point, @NonNull final OnPopupTapCallback onPopupTapCallback) {
        if (onPopupTapCallback == null) {
            return;
        }
        // Check for heads up to apply rotation to the bounds
        if (this.isHeadsUp()) {
            RectF rotatedActionBox = actionBox;
            Matrix m = new Matrix();
            m.setRotate(this.rotation, getX(), getY());
            m.mapRect(rotatedActionBox);
            this.actionBox = rotatedActionBox;

            m.setRotate(this.rotation, getX(), getY());
            RectF rotatedCloseBox = closeButtonBox;
            m.mapRect(rotatedCloseBox);
            this.closeButtonBox = rotatedCloseBox;
        }

        if (this.isCounterScaling()){
            RectF scaledActionBox = actionBox;
            RectF scaledCloseBox = closeButtonBox;

            Matrix m = new Matrix();
            m.setScale(2f/this.scale, 2f/this.scale, getX(), getY());
            m.mapRect(scaledActionBox);
            this.actionBox = scaledActionBox;

            m.setScale(2f/this.scale, 2f/this.scale, getX(), getY());
            m.mapRect(scaledCloseBox);
            this.closeButtonBox = scaledCloseBox;
        }
        if (this.actionBox.contains((int)point.x, (int)point.y)) {
            onPopupTapCallback.onActionButtonTapped();
            return;
        }
        //This means that for a x,y to be contained: left <= x < right and top <= y < bottom.
        if (this.closeButtonBox.contains((int)point.x, (int)point.y)) {
            onPopupTapCallback.onCloseButtonTapped();;
            return;
        }
    }

    public interface OnPopupTapCallback {
        void onActionButtonTapped();

        void onCloseButtonTapped();
    }

    public void setRotation(float rotation) {
        this.rotation = rotation;
    }
    public void setScale(float scale) {
        this.scale = scale;
    }

}