package com.jibestream.mapuikit;

import android.app.SearchManager;
import android.content.Context;
import android.database.Cursor;
import android.database.MatrixCursor;
import android.graphics.drawable.ColorDrawable;
import android.graphics.drawable.Drawable;
import android.graphics.drawable.GradientDrawable;
import android.graphics.drawable.ShapeDrawable;
import android.os.AsyncTask;
import android.os.Build;
import android.provider.BaseColumns;
import android.support.annotation.NonNull;
import android.support.v4.widget.SimpleCursorAdapter;
import android.support.v7.widget.SearchView;
import android.view.View;
import android.view.ViewGroup;
import android.widget.FrameLayout;
import android.widget.HorizontalScrollView;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.ScrollView;

import com.daimajia.androidanimations.library.Techniques;
import com.daimajia.androidanimations.library.YoYo;
import com.jibestream.jmapandroidsdk.components.Amenity;
import com.jibestream.jmapandroidsdk.components.BaseModel;
import com.jibestream.jmapandroidsdk.components.Destination;
import com.jibestream.jmapandroidsdk.components.Floor;
import com.jibestream.jmapandroidsdk.components.PathType;
import com.jibestream.jmapandroidsdk.jcontroller.JController;
import com.jibestream.jmapandroidsdk.jcore.JCore;
import com.jibestream.jmapandroidsdk.rendering_engine.MapLayer;
import com.jibestream.jmapandroidsdk.rendering_engine.MapView;
import com.jibestream.jmapandroidsdk.rendering_engine.Stage;
import com.jibestream.jmapandroidsdk.rendering_engine.Transform;
import com.jibestream.jmapandroidsdk.rendering_engine.drawables.MapDrawable;
import com.jibestream.jmapandroidsdk.styles.JStyle;

import java.util.ArrayList;
import java.util.Arrays;

/**
 * Created by Ken Pangilinan on 2018-03-28.
 */

public class MapUiKit {
    private static final int SEARCH_QUERY_THRESHOLD = 3;
    private static final int TYPE_DESTINATION = 0;
    private static final int TYPE_AMENITY = 1;
    private static final int TYPE_PATH_TYPE = 2;

    private final JCore core;
    private final JController controller;
    private final Stage stage;
    private final Context context;
    private RelativeLayout uiContainer;

    private FloorSelector floorSelector;
    private Compass compass;
    private Search search;

    final private int DEFAULT_MARGIN = 20;
    private Techniques animationIn, animationOut;

    private boolean allowConcurrentPopups;

    /**
     * Kit that aids in adding common UI components to Jibestream's SDK map.
     *
     * @param core       {@link JCore} reference from JMap Android SDK.
     * @param controller {@link JController} reference from JMap Android SDK.
     */
    public MapUiKit(@NonNull final JCore core, @NonNull final JController controller) {
        this.core = core;
        this.controller = controller;
        this.stage = controller.getStage();
        this.context = controller.getContext();

        //setup the initial layout that holds all the components
        setupMapUIContainer();
    }

    /**
     * Renders a {@link Compass} onto the map.
     *
     * @param compassOptions Pass in options to set properties such as compass position.
     * @return Compass view, used to customize natively. Returns {@code null} if compassOptions is {@code null}.
     */
    public Compass renderCompass(@NonNull final CompassOptions compassOptions) {
        if (compassOptions == null) {
            return null;
        }

        boolean addView = false;
        if (compass == null) {
            compass = new Compass(context);
            compass.setId(R.id.map_ui_compass);

            addView = true;
        }

        //set the options
        compass.setOptions(compassOptions);

        //if only one icon is set, set icon/rotateIcon respectively
        if (compassOptions.icon == R.drawable.ic_compass && compassOptions.rotatedIcon != R.drawable.ic_rotated_compass) {
            compassOptions.icon = compassOptions.rotatedIcon;
        } else if (compassOptions.icon != R.drawable.ic_compass && compassOptions.rotatedIcon == R.drawable.ic_rotated_compass) {
            compassOptions.rotatedIcon = compassOptions.icon;
        }

        //set the image resource
        compass.setImageResource(compassOptions.icon);

        //set the layout of the view
        RelativeLayout.LayoutParams params = new RelativeLayout.LayoutParams(
                ViewGroup.LayoutParams.WRAP_CONTENT, ViewGroup.LayoutParams.WRAP_CONTENT);
        params.setMargins(DEFAULT_MARGIN, DEFAULT_MARGIN, DEFAULT_MARGIN, DEFAULT_MARGIN);
        params.addRule(compassOptions.position[0]);
        params.addRule(compassOptions.position[1]);
        compass.setLayoutParams(params); //causes layout update

        //set controller's rotation listener so compass can rotate along w/ map
        controller.setOnRotateListener(new MapView.OnRotateListener() {
            @Override
            public void onRotateChanged(final float v) {
                final float delta = v - compass.getRotation();

                compass.post(new Runnable() {
                    @Override
                    public void run() {
                        if (compass.isCompassNorth) {
                            compass.setImageResource(compassOptions.rotatedIcon);
                            compass.isCompassNorth = false;
                        }
                        compass.setRotation(compass.getRotation() + delta);
                    }
                });
            }
        });

        //if compass tapped on, reset to point map/compass north
        if (compassOptions.resetRotationOnTap) {
            compass.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    Transform transform = new Transform();
                    transform.setRotate(0);
                    controller.setMapTransform(transform, 500, new MapView.OnAnimationCallback() {
                        @Override
                        public void onAnimationComplete() {
                            compass.postDelayed(new Runnable() {
                                @Override
                                public void run() {
                                    compass.setImageResource(compassOptions.icon);
                                    compass.isCompassNorth = true;
                                }
                            }, 300);
                        }

                        @Override
                        public void onAnimationInterrupted() {

                        }
                    });
                }
            });
        }

        if (addView) {
            //add compass to ui container
            uiContainer.post(new Runnable() {
                @Override
                public void run() {
                    uiContainer.addView(compass);
                }
            });
        }

        return compass;
    }

    /**
     * Renders a {@link Compass} onto the map.
     *
     * @return Compass view, used to customize natively. Returns {@code null} if compassOptions is {@code null}.
     */
    public Compass renderCompass() {
        return renderCompass(new CompassOptions());
    }

    /**
     * Renders a {@link FloorSelector} onto the map.
     *
     * @param floorSelectorOptions Pass in options to set properties such as floorSelector position.
     * @return FloorSelector view, used to customize natively. Returns {@code null} if floorSelectorOptions is {@code null}.
     */
    public FloorSelector renderFloorSelector(@NonNull final FloorSelectorOptions floorSelectorOptions) {
        if (floorSelectorOptions == null) {
            return null;
        }

        boolean addView = false;
        if (floorSelector == null) {
            floorSelector = new FloorSelector(context);
            floorSelector.setId(R.id.map_ui_floor_selector);

            addView = true;
        }

        //set the options
        floorSelector.setOptions(floorSelectorOptions);

        //set the layout of the view
        RelativeLayout.LayoutParams params = new RelativeLayout.LayoutParams(
                RelativeLayout.LayoutParams.WRAP_CONTENT, RelativeLayout.LayoutParams.WRAP_CONTENT);
        params.addRule(floorSelectorOptions.position[0]);
        params.addRule(floorSelectorOptions.position[1]);

        floorSelector.setLayoutParams(params); //causes layout update

        final LinearLayout floorButtonContainer = new LinearLayout(context);
        floorButtonContainer.setId(R.id.floor_container);

        final FrameLayout scrollView;

        boolean isVerticalBottom = false;
        boolean isHorizontalRight = false;

        //setup floor selector based on options configured
        if (floorSelectorOptions.vertical) {
            scrollView = new ScrollView(context);

            animationIn = Techniques.FadeInDown;
            animationOut = Techniques.FadeOutUp;

            floorSelector.setOrientation(LinearLayout.VERTICAL);
            floorButtonContainer.setOrientation(LinearLayout.VERTICAL);

            isVerticalBottom = floorSelectorOptions.position[1] == RelativeLayout.ALIGN_PARENT_BOTTOM;
        } else {
            scrollView = new HorizontalScrollView(context);

            animationIn = Techniques.FadeInLeft;
            animationOut = Techniques.FadeOutLeft;

            floorSelector.setOrientation(LinearLayout.HORIZONTAL);
            floorButtonContainer.setOrientation(LinearLayout.HORIZONTAL);

            isHorizontalRight = floorSelectorOptions.position[0] == RelativeLayout.ALIGN_PARENT_RIGHT;
        }

        //scroll view by default is invisible
        scrollView.setId(R.id.scroll_view);
        scrollView.setVisibility(View.INVISIBLE);

        //set the layout of the scroll view
        LinearLayout.LayoutParams scrollViewParams = new LinearLayout.LayoutParams(
                LinearLayout.LayoutParams.MATCH_PARENT, LinearLayout.LayoutParams.WRAP_CONTENT);
        scrollViewParams.weight = 1.0f;
        scrollView.setLayoutParams(scrollViewParams);
        scrollView.setPadding(0, 0, 0, 0);
        if (addView) {
            scrollView.post(new Runnable() {
                @Override
                public void run() {
                    scrollView.addView(floorButtonContainer);
                }
            });
        }

        //adjust how to animate button based on positioning on map
        if (isVerticalBottom || isHorizontalRight) {
            if (addView) {
                floorSelector.addView(scrollView, 0);
            }

            if (isVerticalBottom) {
                animationIn = Techniques.FadeInUp;
                animationOut = Techniques.FadeOutDown;
            } else {
                animationIn = Techniques.FadeInRight;
                animationOut = Techniques.FadeOutRight;
            }
        } else {
            if (addView) {
                floorSelector.addView(scrollView);
            }
        }

        //setup main floor button
        setupSelectedFloorButton(floorSelectorOptions);

        if (addView) {
            //add and setup floor buttons
            setupFloorButtons(floorButtonContainer, floorSelectorOptions);
        }

        if (addView) {
            //add floor selector to ui container
            uiContainer.post(new Runnable() {
                @Override
                public void run() {
                    uiContainer.addView(floorSelector);
                }
            });
        }

        return floorSelector;
    }

    /**
     * Renders a {@link FloorSelector} onto the map.
     *
     * @return FloorSelector view, used to customize natively. Returns {@code null} if floorSelectorOptions is {@code null}.
     */
    public FloorSelector renderFloorSelector() {
        return renderFloorSelector(new FloorSelectorOptions());
    }

    /**
     * Renders a {@link Search} onto the map.
     *
     * @param searchOptions Pass in options to set properties such as search position.
     * @return Search view, used to customize natively. Returns {@code null} if searchOptions is {@code null}.
     */
    public Search renderSearch(@NonNull final SearchOptions searchOptions) {
        if (searchOptions == null) {
            return null;
        }

        boolean addView = false;
        if (search == null) {
            search = new Search(context);
            search.setId(R.id.map_ui_search);

            addView = true;
        }

        //set the options
        search.setOptions(searchOptions);

        search.setQueryHint(searchOptions.hint);

        // populate with destinations, amenities and path types if searchArray is null
        if (searchOptions.searchArray == null) {
            Destination[] destinations = controller.getActiveVenue().getDestinations().getAll();
            Amenity[] amenities = controller.getActiveVenue().getAmenities().getAll();
            PathType[] pathTypes = controller.getActiveVenue().getPathTypes().getAll();
            ArrayList<BaseModel> baseModelArrayList = new ArrayList<>();

            baseModelArrayList.addAll(Arrays.asList(destinations));
            baseModelArrayList.addAll(Arrays.asList(amenities));
            baseModelArrayList.addAll(Arrays.asList(pathTypes));

            searchOptions.searchArray = new BaseModel[baseModelArrayList.size()];
            searchOptions.searchArray = baseModelArrayList.toArray(searchOptions.searchArray);
        }

        //add rounded corners to search, only if greater than or equal to API level 21
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            search.setBackground(context.getDrawable(R.drawable.rounded_corner));
        }

        //set the layout of the view
        RelativeLayout.LayoutParams params = new RelativeLayout.LayoutParams(
                RelativeLayout.LayoutParams.MATCH_PARENT, RelativeLayout.LayoutParams.WRAP_CONTENT);
        params.setMargins(DEFAULT_MARGIN, DEFAULT_MARGIN, DEFAULT_MARGIN, DEFAULT_MARGIN);
        params.addRule(searchOptions.position);
        search.setLayoutParams(params); //causes layout update

        search.post(new Runnable() {
            @Override
            public void run() {
                search.setSuggestionsAdapter(new SimpleCursorAdapter(
                        context, android.R.layout.simple_list_item_1, null,
                        new String[]{SearchManager.SUGGEST_COLUMN_TEXT_1},
                        new int[]{android.R.id.text1}));
                search.setOnSuggestionListener(new SearchView.OnSuggestionListener() {
                    @Override
                    public boolean onSuggestionSelect(int position) {

                        return false;
                    }

                    @Override
                    public boolean onSuggestionClick(int position) {
                        Cursor cursor = search.getSuggestionsAdapter().getCursor();
                        cursor.moveToPosition(position);

                        int id = cursor.getInt(0);
                        int type = cursor.getInt(2);

                        BaseModel baseModel = null;
                        if (type == TYPE_DESTINATION) {
                            baseModel = controller.getActiveVenue().getDestinations().getById(id);
                        } else if (type == TYPE_AMENITY) {
                            baseModel = controller.getActiveVenue().getAmenities().getById(id);
                        } else if (type == TYPE_PATH_TYPE) {
                            baseModel = controller.getActiveVenue().getPathTypes().getById(id);
                        }

                        searchOptions.onItemClickListener.onItemClick(baseModel);

                        return false;
                    }
                });
            }
        });

        search.setOnQueryTextListener(new SearchView.OnQueryTextListener() {

            @Override
            public boolean onQueryTextSubmit(String query) {
                // if user presses enter, do default search, ex:
                if (query.length() >= SEARCH_QUERY_THRESHOLD) {

                    return true;
                }

                return false;
            }

            @Override
            public boolean onQueryTextChange(String newText) {
                if (newText.length() >= SEARCH_QUERY_THRESHOLD) {
                    new FetchAutoCompleteSuggestionsTask().execute(newText);
                } else {
//                    search.getSuggestionsAdapter().changeCursor(null);
                }

                return true;
            }
        });

        if (addView) {
            //add search to ui container
            uiContainer.post(new Runnable() {
                @Override
                public void run() {
                    uiContainer.addView(search);
                }
            });
        }

        return search;
    }

    /**
     * Renders a {@link PopupDrawable} onto the map.
     *
     * @param popupOptions Pass in options to set properties such as currentPopupDrawable coordinates.
     * @return Popup drawable, used to customize natively. Returns {@code null} if popupOptions is {@code null}.
     */
    public PopupDrawable renderPopup(@NonNull final PopupDrawableOptions popupOptions) {
        if (popupOptions == null) {
            return null;
        }

        if (!allowConcurrentPopups) {
            //remove previously added popups from map
            for (MapDrawable mapDrawable : controller.getMapDrawables()) {
                MapLayer mapLayer = mapDrawable.getMapLayer("Popups");
                if (mapLayer != null) {
                    mapLayer.clearCustomDrawables();
                }
            }
        }

        final PopupDrawable popupDrawable = new PopupDrawable(popupOptions);
        //set the options
        popupDrawable.setOptions(popupOptions);

        //add popup drawable to map
        controller.addComponent(popupDrawable, popupOptions.map != null ?
                popupOptions.map : controller.getCurrentMap(), popupOptions.coordinates, "Popups");

        popupDrawable.setRotation(-controller.getMapView().getCurrentRotation());

        controller.setOnRotateListener(new MapView.OnRotateListener() {
            @Override
            public void onRotateChanged(final float v) {
                final float delta = -v;

                popupDrawable.setRotation(delta);
            }
        });

        return popupDrawable;
    }

    public void setAllowConcurrentPopups(final boolean allowConcurrentPopups) {
        this.allowConcurrentPopups = allowConcurrentPopups;
    }

    /**
     * @return layout that holds all the map UI components.
     */
    public RelativeLayout getUiContainer() {
        return uiContainer;
    }

    private void setupSelectedFloorButton(FloorSelectorOptions floorSelectorOptions) {
        Floor currentFloor = controller.getCurrentFloor();
        final FloorButton selectedFloorButton = floorSelector.findViewById(R.id.main_floor_button);

        selectedFloorButton.setMain(true);
        styleButton(selectedFloorButton, currentFloor, floorSelectorOptions.activeStyle, floorSelectorOptions.activeFontStyle);

        selectedFloorButton.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (!selectedFloorButton.isSelected()) {
                    //expand menu
                    expandFloorSelector();
                } else {
                    //collapse menu
                    collapseFloorSelector();
                }
            }
        });
    }

    private void collapseFloorSelector() {
        final FloorButton mainButton = floorSelector.findViewById(R.id.main_floor_button);
        mainButton.setSelected(false);

        FrameLayout scrollView = floorSelector.findViewById(R.id.scroll_view);

        YoYo.with(animationOut)
                .duration(700)
                .playOn(scrollView);
    }

    private void expandFloorSelector() {
        final FloorButton mainButton = floorSelector.findViewById(R.id.main_floor_button);
        mainButton.setSelected(true);

        FrameLayout scrollView = floorSelector.findViewById(R.id.scroll_view);

        scrollView.setVisibility(View.VISIBLE);
        YoYo.with(animationIn)
                .duration(700)
                .playOn(scrollView);
    }

    private void setupFloorButtons(final LinearLayout floorButtonContainer, final FloorSelectorOptions floorSelectorOptions) {
        final Floor[] floors = controller.getCurrentBuilding().getFloors().getAll();
        for (final Floor floor : floors) {
            final FloorButton floorButton = new FloorButton(context);
            styleButton(floorButton, floor, floorSelectorOptions.inactiveStyle, floorSelectorOptions.inactiveFontStyle);

            floorButtonContainer.post(new Runnable() {
                @Override
                public void run() {
                    if (floorSelectorOptions.position[1] == RelativeLayout.ALIGN_PARENT_BOTTOM ||
                            floorSelectorOptions.position[0] == RelativeLayout.ALIGN_PARENT_RIGHT) {
                        floorButtonContainer.addView(floorButton, 0);
                    } else {
                        floorButtonContainer.addView(floorButton);
                    }
                }
            });

            floorButton.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    final FloorButton mainButton = floorSelector.findViewById(R.id.main_floor_button);
                    mainButton.setText(floorButton.getText());

                    collapseFloorSelector();

                    controller.showMap(floor.getMap(), new JController.OnMapShownCallback() {
                        @Override
                        public void onBeforeMapShown(MapDrawable mapDrawable) {
                        }

                        @Override
                        public void onMapShown() {

                        }

                        @Override
                        public void onError(String s) {

                        }
                    });
                }
            });
        }
    }

    private void styleButton(FloorButton floorButton, Floor floor, JStyle style, JStyle fontStyle) {
        floorButton.setMapId(floor.getMap().getId());
        Drawable background = floorButton.getBackground();

        //button is styled this way to preserve the rounded corners
        if (background instanceof ShapeDrawable) {
            ShapeDrawable shapeDrawable = ((ShapeDrawable) background);
            shapeDrawable.getPaint().setColor(style.getColor());
        } else if (background instanceof GradientDrawable) {
            GradientDrawable gradientDrawable = ((GradientDrawable) background);
            gradientDrawable.setColor(style.getColor());
            gradientDrawable.setStroke(style.getStrokeColor(), (int) style.getStrokeWidth());
        } else if (background instanceof ColorDrawable) {
            ColorDrawable colorDrawable = ((ColorDrawable) background);
            colorDrawable.setColor(style.getColor());
        }
        floorButton.setText(floor.getShortName() != null ? floor.getShortName() : "");
        floorButton.setTextColor(fontStyle.getColor());
        floorButton.setTextSize(fontStyle.getTextSize());
    }

    private void setupMapUIContainer() {
        uiContainer = new RelativeLayout(context);
        uiContainer.setId(R.id.map_ui_container);

        uiContainer.setLayoutParams(new RelativeLayout.LayoutParams(
                RelativeLayout.LayoutParams.MATCH_PARENT, RelativeLayout.LayoutParams.MATCH_PARENT));

        //add ui container to stage view
        stage.post(new Runnable() {
            @Override
            public void run() {
                stage.addView(uiContainer);
            }
        });
    }

    private class FetchAutoCompleteSuggestionsTask extends AsyncTask<String, Void, Cursor> {

        private final String[] sAutocompleteColNames = new String[]{
                BaseColumns._ID,                        // necessary for adapter
                SearchManager.SUGGEST_COLUMN_TEXT_1,    // the full search term
                "type"                                  // base model type
        };

        @Override
        protected Cursor doInBackground(String... params) {
            Object[] baseModels = JController.getObjectsInArrayByString(search.getOptions().searchArray, params[0].toLowerCase(), null, search.getOptions().maxResults);

            MatrixCursor cursor = new MatrixCursor(sAutocompleteColNames);

            for (Object object : baseModels) {
                BaseModel baseModel = (BaseModel) object;
                String name = null;
                Integer type = null;
                if (baseModel instanceof Destination) {
                    name = ((Destination) baseModel).getName();
                    type = TYPE_DESTINATION;
                } else if (baseModel instanceof Amenity) {
                    name = ((Amenity) baseModel).getName();
                    type = TYPE_AMENITY;
                } else if (baseModel instanceof PathType) {
                    name = ((PathType) baseModel).getName();
                    type = TYPE_PATH_TYPE;
                }

                Object[] row = new Object[]{baseModel.getId(), name, type};
                cursor.addRow(row);
            }

            return cursor;
        }

        @Override
        protected void onPostExecute(Cursor result) {
            search.getSuggestionsAdapter().changeCursor(result);
        }

    }
}
