package com.jibestream.geofencekit;

import android.graphics.Color;
import android.graphics.Path;
import android.graphics.PointF;
import android.graphics.RectF;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.jibestream.jmapandroidsdk.components.Amenity;
import com.jibestream.jmapandroidsdk.components.Destination;
import com.jibestream.jmapandroidsdk.components.Map;
import com.jibestream.jmapandroidsdk.components.PathType;
import com.jibestream.jmapandroidsdk.components.Waypoint;
import com.jibestream.jmapandroidsdk.http.HttpClient;
import com.jibestream.jmapandroidsdk.jcontroller.JController;
import com.jibestream.jmapandroidsdk.jcore.JCore;
import com.jibestream.jmapandroidsdk.main.ErrorMessage;
import com.jibestream.jmapandroidsdk.main.Polygon;
import com.jibestream.jmapandroidsdk.rendering_engine.MapLayer;
import com.jibestream.jmapandroidsdk.rendering_engine.drawables.JShapeDrawable;
import com.jibestream.jmapandroidsdk.rendering_engine.drawables.MapDrawable;
import com.jibestream.jmapandroidsdk.rendering_engine.moving_objects.MovingObject;
import com.jibestream.jmapandroidsdk.styles.JStyle;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Created by Ken Pangilinan on 2018-02-26.
 */
public class GeofenceKit {
    private static final String LAYER_GEOFENCES = "Geofences";
    public GeofenceCollection geofenceCollection;

    private JCore core;
    private JController controller;

    private ArrayList<GeofenceInstance> geofenceInstanceArrayList;
    private HashMap<GeofenceInstance, Polygon> geofenceInstancePolygons;

    //a moving object can land on an overlaying geofence, hence an array list of geofences
    private HashMap<MovingObject, ArrayList<GeofenceInstance>> watchedMovingObjects;

    /**
     * Kit that aids in working with Jibestream's Geofences.
     *
     * @param core       {@link JCore} reference from JMap Android SDK.
     * @param controller {@link JController} reference from JMap Android SDK.
     */
    public GeofenceKit(@NonNull final JCore core, @NonNull final JController controller) {
        this.core = core;
        this.controller = controller;

        watchedMovingObjects = new HashMap<>();
        geofenceInstancePolygons = new HashMap<>();
    }

    /**
     * Draws all {@link Polygon} shapes of a {@link Geofence} with a defined {@link JStyle} for an optional {@link Map}.
     *
     * @param geofence Polygon shapes to be extracted from geofence.
     * @param map      Used to filter which map to draw polygons on. If {@code null} draw polygons on all maps.
     * @param style    Style for polygon shapes. If {@code null}, default is geofence's fill color. If no geofence fill color, default is black.
     */
    public synchronized void drawPolygonsOfGeofence(@NonNull final Geofence geofence, @Nullable final Map map, @Nullable JStyle style) {
        if (geofence == null) {
            return;
        }

        //if style is not provided use the geofence's default fill color
        if (style == null) {
            style = new JStyle();
            try {
                String BLACK_HEX_COLOR = "#000000";
                style.setColor(Color.parseColor(geofence.getColor() != null ? geofence.getColor() : BLACK_HEX_COLOR));
            } catch (IllegalArgumentException iae) {
                //set geofence color to black if invalid
                style.setColor(Color.BLACK);
            }
        }

        //loop through all geofence instances
        for (GeofenceInstance geofenceInstance : geofence.getGeofenceInstances()) {
            //ensure geofence instance is on specified map
            if (map == null || geofenceInstance.getFloor().getMap().getId() == map.getId()) {
                drawPolygonOfGeofenceInstance(geofenceInstance, style);
            }
        }
    }

    /**
     * Draws {@link Polygon} shape of a {@link GeofenceInstance} with a defined {@link JStyle}.
     *
     * @param geofenceInstance Polygon shape to be extracted from geofence instance.
     * @param style            Style for polygon shapes. If {@code null}, default fill is black.
     */
    public synchronized void drawPolygonOfGeofenceInstance(@NonNull final GeofenceInstance geofenceInstance, @Nullable JStyle style) {
        if (geofenceInstance == null) {
            return;
        }

        //if style is not provided use the geofence's default fill color
        if (style == null) {
            style = new JStyle();
            try {
                String BLACK_HEX_COLOR = "#000000";
                style.setColor(Color.parseColor(geofenceInstance.parent.getColor() != null ? geofenceInstance.parent.getColor() : BLACK_HEX_COLOR));
            } catch (IllegalArgumentException iae) {
                //set geofence color to black if invalid
                style.setColor(Color.BLACK);
            }
        }

        String geofenceInstanceId = geofenceInstance.floor.getMap().id + "0" + geofenceInstance.id;

        //if drawn already, update the style
        final JShapeDrawable[] geofenceShapes = controller.getShapesInLayer(LAYER_GEOFENCES, geofenceInstance.floor.getMap());
        for (JShapeDrawable geofenceShape : geofenceShapes) {
            if (geofenceShape.getId() == Integer.valueOf(geofenceInstanceId)) {
                geofenceShape.setStyle(style);

                return;
            }
        }

        //extract the polygon from the geofence instance
        Polygon geofencePolygon = getPolygonOfGeofenceInstance(geofenceInstance);

        //skip if polygon cannot be created, otherwise create a drawable using the polygon of the geofence and add it to the map
        if (geofencePolygon != null) {
            Path polygonPath = geofencePolygon.getPolygonPath();
            controller.drawShapeOnMap(polygonPath, geofenceInstance.floor.getMap(), style,
                    Integer.valueOf(geofenceInstanceId), geofenceInstance.name, LAYER_GEOFENCES);
        }
    }

    /**
     * Removes all {@link Polygon} shapes of a {@link Geofence} for an optional {@link Map}.
     *
     * @param geofence Polygon shapes to be extracted from geofence then removed from map.
     * @param map      Used to filter which map to remove polygons from. If {@code null} remove polygons from all maps.
     */
    public synchronized void removePolygonsOfGeofence(@NonNull final Geofence geofence, @Nullable final Map map) {
        if (geofence == null) {
            return;
        }

        //loop through all geofence instances
        for (GeofenceInstance geofenceInstance : geofence.getGeofenceInstances()) {
            //ensure geofence instance is on specified map
            if (map == null || geofenceInstance.getFloor().getMap().getId() == map.getId()) {
                removePolygonOfGeofenceInstance(geofenceInstance);
            }
        }
    }

    /**
     * Removes {@link Polygon} shape of a {@link GeofenceInstance} from the {@link Map}.
     *
     * @param geofenceInstance Polygon shape to be removed from map.
     */
    public synchronized void removePolygonOfGeofenceInstance(@NonNull final GeofenceInstance geofenceInstance) {
        if (geofenceInstance == null) {
            return;
        }

        String geofenceInstanceId = geofenceInstance.floor.getMap().id + "0" + geofenceInstance.id;

        //loop through all map drawables
        for (MapDrawable mapDrawable : controller.getMapDrawables()) {
            //get geofence map layer
            MapLayer geofenceMapLayer = mapDrawable.getMapLayer(LAYER_GEOFENCES);
            //check if geofence map layer exists in map drawable
            if (geofenceMapLayer != null) {
                //loop through all shape drawables
                for (JShapeDrawable shapeDrawable : geofenceMapLayer.getShapeDrawables()) {
                    //remove if shape drawable id matches geofence instance id
                    if (shapeDrawable.getId() == Integer.valueOf(geofenceInstanceId)) {
                        geofenceMapLayer.removeComponent(shapeDrawable);

                        return;
                    }
                }
            }
        }
    }

    /**
     * Get the bounds of a {@link Geofence} for a specific {@link Map}.
     *
     * @param geofence Bounds are extracted from geofence.
     * @param map      Used to filter which map to get bounds from.
     * @return Accumulated bounds of all geofence instances. If no instances exist, returns empty bounds.
     */
    public synchronized RectF getBoundsOfGeofenceOnMap(@NonNull final Geofence geofence, @NonNull final Map map) {
        RectF bounds = new RectF();

        if (geofence != null && map != null) {
            //loop through all geofence instances
            for (GeofenceInstance geofenceInstance : geofence.getGeofenceInstances()) {
                //ensure geofence instance is on specified map
                if (geofenceInstance.getFloor().getMap().getId() == map.getId()) {
                    //if bounds is empty compute initial bounds, otherwise add new bounds to existing
                    if (bounds.isEmpty()) {
                        bounds = geofenceInstance.getBounds();
                    } else {
                        bounds.union(geofenceInstance.getBounds());
                    }
                }
            }
        }

        return bounds;
    }

    /**
     * Get the {@link Polygon} shapes of a {@link Geofence} for a specific {@link Map}.
     *
     * @param geofence Polygon shapes to be extracted from geofence.
     * @param map      Used to filter which map to get polygons from.
     * @return Accumulated polygon shapes of all geofence instances. If no instances exist, returns an empty array.
     */
    public synchronized Polygon[] getPolygonsOfGeofenceOnMap(@NonNull final Geofence geofence, @NonNull final Map map) {
        ArrayList<Polygon> polygons = new ArrayList<>();

        if (geofence != null && map != null) {
            //loop through all geofence instances
            for (GeofenceInstance geofenceInstance : geofence.getGeofenceInstances()) {
                //ensure geofence instance is on specified map
                if (geofenceInstance.getFloor().getMap().getId() == map.getId()) {
                    //extract the polygon from the geofence instance
                    Polygon polygon = getPolygonOfGeofenceInstance(geofenceInstance);

                    //ensure polygon is valid
                    if (polygon != null) {
                        polygons.add(polygon);
                    }
                }
            }
        }

        return polygons.toArray(new Polygon[polygons.size()]);
    }

    /**
     * Tag a {@link MovingObject} to monitor whether it has entered or exited a {@link Geofence}.
     *
     * @param movingObject           Moving object to monitor.
     * @param onMovingObjectListener Listener to that fires events 'onEnteredGeofence' or 'onExitedGeofence'.
     */
    public synchronized void watchMovingObject(@NonNull final MovingObject movingObject, @NonNull final OnMovingObjectListener onMovingObjectListener) {
        if (movingObject == null || onMovingObjectListener == null) {
            return;
        }

        //add moving object to watched list (w/ empty array list) if it does not exist yet
        if (!watchedMovingObjects.containsKey(movingObject)) {
            watchedMovingObjects.put(movingObject, new ArrayList<GeofenceInstance>());
        }

        geofenceInstanceArrayList = watchedMovingObjects.get(movingObject);
        movingObject.setWatchMovingObjectCallback(new MovingObject.MovingObjectCallback() {
            @Override
            public void onLocationChanged(PointF location) {

            }

            @Override
            public void onAnimationComplete(PointF location) {
                //once moving object is done animating check if it has entered and/or exited a geofence or multiple geofences
                GeofenceInstance[] geofencesInstancesEntered = getGeofenceInstancesEntered(movingObject, location);
                if (geofencesInstancesEntered.length > 0) {
                    onMovingObjectListener.onEnteredGeofence(movingObject, geofencesInstancesEntered);
                }

                GeofenceInstance[] geofenceInstancesExited = getGeofenceInstancesExited(movingObject, location);
                if (geofenceInstancesExited.length > 0) {
                    onMovingObjectListener.onExitedGeofence(movingObject, geofenceInstancesExited);
                }
            }

            private GeofenceInstance[] getGeofenceInstancesEntered(MovingObject movingObject, PointF location) {
                ArrayList<GeofenceInstance> geofenceInstances = new ArrayList<>();

                //loop through all geofences
                for (Geofence geofence : geofenceCollection.getAll()) {
                    //loop through all geofence instances
                    for (GeofenceInstance geofenceInstance : geofence.getGeofenceInstances()) {
                        //skip if not on same map as moving object or already inside geofence instance
                        if (geofenceInstanceArrayList.contains(geofenceInstance) ||
                                geofenceInstance.getFloor().getMap().getId() != movingObject.getMapId()) {
                            continue;
                        }

                        Polygon polygon = geofenceInstance.getPolygon();

                        //ensure polygon is valid
                        if (polygon != null) {
                            //check if moving object is inside polygon
                            if (polygon.contains(location.x, location.y)) {
                                geofenceInstances.add(geofenceInstance);

                                break;
                            }
                        }
                    }
                }

                geofenceInstanceArrayList.addAll(geofenceInstances);

                return geofenceInstances.toArray(new GeofenceInstance[geofenceInstances.size()]);
            }

            private GeofenceInstance[] getGeofenceInstancesExited(MovingObject movingObject, PointF location) {
                ArrayList<GeofenceInstance> geofenceInstances = new ArrayList<>();

                //loop through all geofences
                for (Geofence geofence : geofenceCollection.getAll()) {
                    //loop through all geofence instances
                    for (GeofenceInstance geofenceInstance : geofence.getGeofenceInstances()) {
                        //check if inside geofence instance and if on same map as moving object
                        if (geofenceInstanceArrayList.contains(geofenceInstance) &&
                                geofenceInstance.getFloor().getMap().getId() == movingObject.getMapId()) {
                            Polygon polygon = geofenceInstance.getPolygon();

                            //ensure polygon is valid
                            if (polygon != null) {
                                //check if moving object is outside polygon
                                if (!polygon.contains(location.x, location.y)) {
                                    geofenceInstances.add(geofenceInstance);
                                }
                            }
                        }
                    }
                }

                geofenceInstanceArrayList.removeAll(geofenceInstances);

                return geofenceInstances.toArray(new GeofenceInstance[geofenceInstances.size()]);
            }
        });
    }

    /**
     * Untag all {@link MovingObject}s from being monitored.
     */
    public synchronized void unwatchMovingObjects() {
        //loop through all map drawables
        for (MapDrawable mapDrawable : controller.getMapDrawables()) {
            //loop through all map layers
            for (MapLayer mapLayer : mapDrawable.getAllMapLayers()) {
                // loop through all moving objects and unregister listener
                for (final MovingObject movingObject : mapLayer.getMovingObjects()) {
                    watchedMovingObjects.remove(movingObject);
                    movingObject.setWatchMovingObjectCallback(null);
                }
            }
        }
    }

    /**
     * Get a list of {@link GeofenceInstance}s for a given {@link PointF}.
     *
     * @param point Point used to filter if it is inside a geofence instance.
     * @return List of geofence instances.
     */
    public GeofenceInstance[] getGeofenceInstancesByPoint(@NonNull PointF point) {
        ArrayList<GeofenceInstance> geofenceInstanceArrayList = new ArrayList<>();

        for (Geofence geofence : geofenceCollection.getAll()) {
            for (GeofenceInstance geofenceInstance : geofence.getGeofenceInstances()) {
                if (geofenceInstance.getPolygon().contains(point.x, point.y) &&
                        geofenceInstance.floor.getMap().getId() == controller.getCurrentMap().getId()) {
                    geofenceInstanceArrayList.add(geofenceInstance);
                }
            }
        }

        return geofenceInstanceArrayList.toArray(new GeofenceInstance[geofenceInstanceArrayList.size()]);
    }

    /**
     * Get a list of {@link GeofenceInstance}s for a given {@link Waypoint}.
     *
     * @param waypoint Waypoint used to filter if it is inside a geofence instance.
     * @return List of geofence instances.
     */
    public GeofenceInstance[] getGeofenceInstancesByWaypoint(@NonNull Waypoint waypoint) {
        Float[] coordinates = waypoint.getCoordinates();

        return getGeofenceInstancesByPoint(new PointF(coordinates[0], coordinates[1]));

    }

    /**
     * Get a list of {@link Waypoint}s for a given {@link GeofenceInstance}.
     *
     * @param geofenceInstance Geofence instance used to filter if waypoint is inside it.
     * @return List of waypoints.
     */
    public Waypoint[] getWaypointsInGeofenceInstance(@NonNull GeofenceInstance geofenceInstance) {
        ArrayList<Waypoint> waypointArrayList = new ArrayList<>();

        Map map = controller.getActiveVenue().getMaps().getById(geofenceInstance.getFloor().getMap().getId());
        for (Waypoint waypoint : map.getWaypoints().getAll()) {
            if (!waypointArrayList.contains(waypoint) && geofenceInstance.getPolygon().contains(waypoint.getCoordinates()[0], waypoint.getCoordinates()[1])) {
                waypointArrayList.add(waypoint);
            }
        }

        return waypointArrayList.toArray(new Waypoint[waypointArrayList.size()]);
    }

    /**
     * Get a list of {@link Waypoint}s for a given {@link Geofence} and {@link Map}.
     *
     * @param geofence Geofence used to filter if waypoint is inside it.
     * @param map      Map used to filter if geofence is inside it. If {@code null} check all maps.
     * @return List of waypoints.
     */
    public Waypoint[] getWaypointsInGeofence(@NonNull Geofence geofence, @Nullable Map map) {
        ArrayList<Waypoint> waypointArrayList = new ArrayList<>();

        for (GeofenceInstance geofenceInstance : geofence.getGeofenceInstances()) {
            if (map == null || map.getId() == geofenceInstance.getFloor().getMap().getId()) {
                for (Waypoint waypoint : getWaypointsInGeofenceInstance(geofenceInstance)) {
                    if (!waypointArrayList.contains(waypoint) && geofenceInstance.getPolygon().contains(waypoint.getCoordinates()[0], waypoint.getCoordinates()[1])) {
                        waypointArrayList.add(waypoint);
                    }
                }
            }
        }

        return waypointArrayList.toArray(new Waypoint[waypointArrayList.size()]);
    }

    /**
     * Get a list of {@link Destination}s for a given {@link GeofenceInstance}.
     *
     * @param geofenceInstance Geofence instance used to filter if destination is inside it.
     * @return List of destinations.
     */
    public Destination[] getDestinationsInGeofenceInstance(@NonNull GeofenceInstance geofenceInstance) {
        ArrayList<Destination> destinationArrayList = new ArrayList<>();

        Destination[] destinations = controller.getActiveVenue().getDestinations().getByMap(geofenceInstance.getFloor().getMap());

        for (Destination destination : destinations) {
            for (Waypoint waypoint : destination.getWaypoints()) {
                if (!destinationArrayList.contains(destination) &&
                        geofenceInstance.getPolygon().contains(waypoint.getCoordinates()[0], waypoint.getCoordinates()[1])) {
                    destinationArrayList.add(destination);

                    break;
                }
            }
        }

        return destinationArrayList.toArray(new Destination[destinationArrayList.size()]);
    }

    /**
     * Get a list of {@link Destination}s for a given {@link Geofence} and {@link Map}.
     *
     * @param geofence Geofence used to filter if destination is inside it.
     * @param map      Map used to filter if geofence is inside it. If {@code null} check all maps.
     * @return List of destinations.
     */
    public Destination[] getDestinationsInGeofence(@NonNull Geofence geofence, @Nullable Map map) {
        ArrayList<Destination> destinationArrayList = new ArrayList<>();

        for (GeofenceInstance geofenceInstance : geofence.getGeofenceInstances()) {
            if (map == null || map.getId() == geofenceInstance.getFloor().getMap().getId()) {
                Destination[] destinations = controller.getActiveVenue().getDestinations().getByMap(geofenceInstance.getFloor().getMap());

                for (Destination destination : destinations) {
                    for (Waypoint waypoint : destination.getWaypoints()) {
                        if (!destinationArrayList.contains(destination) &&
                                geofenceInstance.getPolygon().contains(waypoint.getCoordinates()[0], waypoint.getCoordinates()[1])) {
                            destinationArrayList.add(destination);

                            break;
                        }
                    }
                }
            }
        }

        return destinationArrayList.toArray(new Destination[destinationArrayList.size()]);
    }

    /**
     * Get a list of {@link Amenity}s for a given {@link GeofenceInstance}.
     *
     * @param geofenceInstance Geofence instance used to filter if amenity is inside it.
     * @return List of amenities.
     */
    public Amenity[] getAmenitiesInGeofenceInstance(@NonNull GeofenceInstance geofenceInstance) {
        ArrayList<Amenity> amenityArrayList = new ArrayList<>();

        Amenity[] amenities = controller.getActiveVenue().getAmenities().getByMap(geofenceInstance.getFloor().getMap());

        for (Amenity amenity : amenities) {
            for (Waypoint waypoint : amenity.getWaypoints()) {
                if (!amenityArrayList.contains(amenity) &&
                        geofenceInstance.getPolygon().contains(waypoint.getCoordinates()[0], waypoint.getCoordinates()[1])) {
                    amenityArrayList.add(amenity);

                    break;
                }
            }
        }

        return amenityArrayList.toArray(new Amenity[amenityArrayList.size()]);
    }

    /**
     * Get a list of {@link Amenity}s for a given {@link Geofence} and {@link Map}.
     *
     * @param geofence Geofence used to filter if amenity is inside it.
     * @param map      Map used to filter if geofence is inside it. If {@code null} check all maps.
     * @return List of amenities.
     */
    public Amenity[] getAmenitiesInGeofence(@NonNull Geofence geofence, @Nullable Map map) {
        ArrayList<Amenity> amenityArrayList = new ArrayList<>();

        for (GeofenceInstance geofenceInstance : geofence.getGeofenceInstances()) {
            if (map == null || map.getId() == geofenceInstance.getFloor().getMap().getId()) {
                Amenity[] amenities = controller.getActiveVenue().getAmenities().getByMap(geofenceInstance.getFloor().getMap());

                for (Amenity amenity : amenities) {
                    for (Waypoint waypoint : amenity.getWaypoints()) {
                        if (!amenityArrayList.contains(amenity) &&
                                geofenceInstance.getPolygon().contains(waypoint.getCoordinates()[0], waypoint.getCoordinates()[1])) {
                            amenityArrayList.add(amenity);

                            break;
                        }
                    }
                }
            }
        }

        return amenityArrayList.toArray(new Amenity[amenityArrayList.size()]);
    }

    /**
     * Get a list of {@link PathType}s for a given {@link GeofenceInstance}.
     *
     * @param geofenceInstance Geofence instance used to filter if path type is inside it.
     * @return List of path types.
     */
    public PathType[] getPathTypesInGeofenceInstance(@NonNull GeofenceInstance geofenceInstance) {
        ArrayList<PathType> pathTypeArrayList = new ArrayList<>();

        PathType[] pathTypes = controller.getActiveVenue().getPathTypes().getByMap(geofenceInstance.getFloor().getMap());

        for (PathType pathType : pathTypes) {
            for (Waypoint waypoint : pathType.getWaypoints()) {
                if (!pathTypeArrayList.contains(pathType)) {
                    if (geofenceInstance.getPolygon().contains(waypoint.getCoordinates()[0], waypoint.getCoordinates()[1])) {
                        pathTypeArrayList.add(pathType);
                        break;
                    }
                }
            }
        }

        return pathTypeArrayList.toArray(new PathType[pathTypeArrayList.size()]);
    }

    /**
     * Get a list of {@link PathType}s for a given {@link Geofence} and {@link Map}.
     *
     * @param geofence Geofence used to filter if path type is inside it.
     * @param map      Map used to filter if geofence is inside it. If {@code null} check all maps.
     * @return List of path types.
     */
    public PathType[] getPathTypesInGeofence(@NonNull Geofence geofence, @Nullable Map map) {
        ArrayList<PathType> pathTypeArrayList = new ArrayList<>();

        for (GeofenceInstance geofenceInstance : geofence.getGeofenceInstances()) {
            if (map == null || map.getId() == geofenceInstance.getFloor().getMap().getId()) {
                PathType[] pathTypes = controller.getActiveVenue().getPathTypes().getByMap(geofenceInstance.getFloor().getMap());

                for (PathType pathType : pathTypes) {
                    for (Waypoint waypoint : pathType.getWaypoints()) {
                        if (!pathTypeArrayList.contains(pathType)) {
                            if (geofenceInstance.getPolygon().contains(waypoint.getCoordinates()[0], waypoint.getCoordinates()[1])) {
                                pathTypeArrayList.add(pathType);
                                break;
                            }
                        }
                    }
                }
            }
        }

        return pathTypeArrayList.toArray(new PathType[pathTypeArrayList.size()]);
    }

    /**
     * Get a list of {@link MovingObject}s for a given {@link GeofenceInstance}.
     *
     * @param geofenceInstance Geofence instance used to filter if moving object is inside it.
     * @return List of moving objects.
     */
    public MovingObject[] getMovingObjectsInGeofenceInstance(@NonNull GeofenceInstance geofenceInstance) {
        ArrayList<MovingObject> allMovingObjects = new ArrayList<>();
        for (MapDrawable mapDrawable : controller.getMapDrawables()) {
            if (mapDrawable.getId() == geofenceInstance.getFloor().getMap().getId()) {
                for (MapLayer mapLayer : mapDrawable.getAllMapLayers()) {
                    allMovingObjects.addAll(mapLayer.getMovingObjects());
                }
            }
        }

        ArrayList<MovingObject> movingObjectArrayList = new ArrayList<>();
        for (MovingObject movingObject : allMovingObjects) {
            if (geofenceInstance.getPolygon().contains(movingObject.getX(), movingObject.getY())) {
                movingObjectArrayList.add(movingObject);
            }
        }

        return movingObjectArrayList.toArray(new MovingObject[movingObjectArrayList.size()]);
    }

    /**
     * Get a list of {@link MovingObject}s for a given {@link Geofence} and {@link Map}.
     *
     * @param geofence Geofence used to filter if moving object is inside it.
     * @param map      Map used to filter if geofence is inside it. If {@code null} check all maps.
     * @return List of moving objects.
     */
    public MovingObject[] getMovingObjectsInGeofence(@NonNull Geofence geofence, @Nullable Map map) {
        ArrayList<MovingObject> movingObjectArrayList = new ArrayList<>();

        for (GeofenceInstance geofenceInstance : geofence.getGeofenceInstances()) {
            if (map == null || map.getId() == geofenceInstance.getFloor().getMap().getId()) {
                movingObjectArrayList.addAll(Arrays.asList(getMovingObjectsInGeofenceInstance(geofenceInstance)));
            }
        }

        return movingObjectArrayList.toArray(new MovingObject[movingObjectArrayList.size()]);
    }

    /**
     * Get closest {@link Waypoint} inside a {@link GeofenceInstance} from a given {@link PointF}.
     *
     * @param point            Point of reference to potential closet waypoint.
     * @param geofenceInstance Geofence instance that contains waypoints.
     * @return Waypoint that is closest to point inside the geofence instance.
     */
    public Waypoint getClosestWaypointInGeofenceInstance(@NonNull PointF point, @NonNull GeofenceInstance geofenceInstance) {
        Waypoint[] waypoints = getWaypointsInGeofenceInstance(geofenceInstance);

        return getClosestWaypointFromPoint(point, waypoints);
    }

    /**
     * Get closest {@link Waypoint} inside a {@link Geofence} and {@link Map} from a given {@link PointF}.
     *
     * @param point    Point of reference to potential closet waypoint.
     * @param geofence Geofence that contains waypoints.
     * @param map      Map used to filter if geofence is inside it. If {@code null} check all maps.
     * @return Waypoint that is closest to point inside the geofence.
     */
    public Waypoint getClosestWaypointInGeofence(@NonNull PointF point, @NonNull Geofence geofence, @Nullable Map map) {
        Waypoint[] waypoints = getWaypointsInGeofence(geofence, map);

        return getClosestWaypointFromPoint(point, waypoints);
    }

    /**
     * Asynchronous network call to get {@link Geofence}s. Use {@link OnGeofencesReadyCallback} to get the geofences status of return.
     *
     * @param onGeofencesReadyCallback {@link OnGeofencesReadyCallback#onSuccess(GeofenceCollection)} or {@link OnGeofencesReadyCallback#onError(String)})}.
     */
    public synchronized void getGeofences(@NonNull final OnGeofencesReadyCallback onGeofencesReadyCallback) {
        if (onGeofencesReadyCallback == null) {
            //log to console if callback is invalid
            Log.e(ErrorMessage.ERROR_TAG, ErrorMessage.ERROR_INVALID_CALLBACK);
            return;
        }

        String url = getGeofenceParentsCallUrl(core.getHostUrl(), core.getCustomerId(), controller.getActiveVenue().getId());

        //request to get Geofence
        core.getRequest(url, new HttpClient.DownloadCallbacks() {
            @Override
            public void onSuccess(String response) {
                Gson gson = new GsonBuilder().serializeNulls().create();

                final String regex = "\\{\"items\":.*]\\}";

                final Pattern pattern = Pattern.compile(regex);
                final Matcher matcher = pattern.matcher(response);

                matcher.find();
                String geofenceParentsResponse = matcher.group(0);

                geofenceCollection = gson.fromJson(geofenceParentsResponse, GeofenceCollection.class);

                String url = getGeofenceInstancesCallUrl(core.getHostUrl(), core.getCustomerId(), controller.getActiveVenue().getId());

                //request to get Geofence Instances of all Geofences
                core.getRequest(url, new HttpClient.DownloadCallbacks() {
                    @Override
                    public void onSuccess(String response) {
                        Gson gson = new GsonBuilder().serializeNulls().create();

                        GeofenceInstancesResponse geofenceInstancesResponse = gson.fromJson(response, GeofenceInstancesResponse.class);
                        for (GeofenceInstancesResponse.Feature feature : geofenceInstancesResponse.features) {
                            Geofence geofence = geofenceCollection.getById(feature.properties.parent.id);
                            if (geofence != null) {
                                //append geofence instance to respective geofence
                                geofence.addInstance(feature, controller.getActiveVenue());
                            }
                        }

                        onGeofencesReadyCallback.onSuccess(geofenceCollection);
                    }

                    @Override
                    public void onError(String message) {
                        onGeofencesReadyCallback.onError(message);
                    }
                });
            }

            @Override
            public void onError(String message) {
                onGeofencesReadyCallback.onError(message);
            }
        });
    }

    /**
     * Gets a specific geofence instance by id and map id.
     * @param geofenceInstanceId Id for geofence instance.
     * @param mapId Id for geofence instance's map
     * @return Geofence instance found. If cannot be found, returns {@code null}.
     */
    public synchronized GeofenceInstance getGeofenceInstanceByIdWithMapId(int geofenceInstanceId, int mapId) {
        Map map = controller.getActiveVenue().getMaps().getById(mapId);

        //before looping through all geofences, check if mapId even exists
        if (map == null) {
            return null;
        }

        //loop through all geofences
        for (Geofence geofence : geofenceCollection.getAll()) {
            //loop through all geofence instances
            for (GeofenceInstance geofenceInstance : geofence.getGeofenceInstances()) {
                //check if geofence instance/map id match params passed
                if (geofenceInstance.getId() == geofenceInstanceId &&
                        geofenceInstance.floor.getMap().getId() == mapId) {
                    return geofenceInstance;
                }
            }
        }

        return null;
    }

    private String getGeofenceParentsCallUrl(String url, int customerId, int venueId) {
        return url + "/JACS/api/customer/" + customerId + "/venue/" + venueId + "/geofence/full";
    }

    private String getGeofenceInstancesCallUrl(String url, int customerId, int venueId) {
        return url + "/JACS/api/customer/" + customerId + "/venue/" + venueId + "/geofence-instance/full";
    }

    private Polygon getPolygonOfGeofenceInstance(GeofenceInstance geofenceInstance) {
        return geofenceInstancePolygons.containsKey(geofenceInstance) ?
                geofenceInstancePolygons.get(geofenceInstance) : geofenceInstance.getPolygon();
    }

    private Waypoint getClosestWaypointFromPoint(PointF point, Waypoint[] waypoints) {
        Waypoint waypoint = controller.getActiveVenue().getClosestWaypointToCoordinate(point);

        return controller.getActiveVenue().getClosestWaypointInArrayToWaypoint(waypoint, waypoints);
    }

    /**
     * Interface for asynchronous callbacks.
     *
     * @see #getGeofences(OnGeofencesReadyCallback)
     */
    public interface OnGeofencesReadyCallback {
        void onSuccess(GeofenceCollection geofenceCollection);

        void onError(String message);
    }

    /**
     * Interface for asynchronous callbacks.
     *
     * @see #watchMovingObject(MovingObject, OnMovingObjectListener)
     */
    public interface OnMovingObjectListener {
        void onEnteredGeofence(MovingObject movingObject, GeofenceInstance[] geofenceInstances);

        void onExitedGeofence(MovingObject movingObject, GeofenceInstance[] geofenceInstances);
    }
}