package com.jibestream.geofencekit;

import android.graphics.PointF;
import android.graphics.RectF;

import com.jibestream.jmapandroidsdk.components.Floor;
import com.jibestream.jmapandroidsdk.main.PointD;
import com.jibestream.jmapandroidsdk.main.Polygon;
import com.jibestream.jmapandroidsdk.main.Utilities;

import java.util.ArrayList;

/**
 * Created by Ken Pangilinan on 2018-02-26.
 */

public class GeofenceInstance {
    int id;
    double[][][] coordinates;
    double[][][] localCoordinates;
    String name;
    Floor floor;
    Geofence parent;

    /**
     * @return {@link Geofence} / parent of geofence instance.
     */
    public Geofence getParent() {
        return parent;
    }

    /**
     * @return Name of geofence instance. ie. "Food Court A" or "Food Court B"
     */
    public String getName() {
        return name;
    }

    /**
     * @return Id of geofence instance.
     */
    public int getId() {
        return id;
    }

    /**
     * @return {@link Floor} object of geofence instance.
     */
    public Floor getFloor() {
        return floor;
    }

    /**
     * @return Centroid {@link PointF} of geofence instance.
     */
    public PointD getCentroid() {
        Polygon polygon = getPolygon();

        return Utilities.getCentroidOfPolygon(polygon);
    }

    /**
     * @return Bounds of geofence instance's polygon shape.
     */
    public RectF getBounds() {
        RectF bounds = new RectF();

        Polygon polygon = getPolygon();
        polygon.getPolygonPath().computeBounds(bounds, true);

        return bounds;
    }

    /**
     * @return {@link Polygon} shape of geofence instance. Return {@code null} if not enough points to create polygon.
     */
    public Polygon getPolygon() {
        return Utilities.getPolygonFromPoints(getLocalCoordinatePoints());
    }

    /**
     * @return Array of lat/lng doubles of the geofence instance.
     */
    public double[][] getCoordinates() {
        double[][] c = new double[0][];

        int i = 0;
        for (double[][] c1 : coordinates) {
            c = new double[c1.length][2];
            for (double[] c2 : c1) {
                c[i][0] = c2[0];
                c[i][1] = c2[1];
                i++;
            }
        }

        return c;
    }

    /**
     * @return Array of local x/y doubles of the geofence instance.
     */
    public double[][] getLocalCoordinates() {
        double[][] lc = new double[0][];

        int i = 0;
        for (double[][] c1 : localCoordinates) {
            lc = new double[c1.length][2];
            for (double[] c2 : c1) {
                lc[i][0] = c2[0];
                lc[i][1] = c2[1];
                i++;
            }
        }

        return lc;
    }

    /**
     * @return Array of local x/y {@link PointF}s of the geofence instance.
     */
    PointD[] getLocalCoordinatePoints() {
        ArrayList<PointD> lcp = new ArrayList<>();

        for (double[][] c1 : localCoordinates) {
            for (double[] c2 : c1) {
                lcp.add(new PointD(c2[0], c2[1]));
            }
        }

        return lcp.toArray(new PointD[lcp.size()]);
    }
}
