package com.jibestream.analyticskit;

import com.jibestream.jmapandroidsdk.analytics_kit.AnalyticsEvent;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Created by Ken Pangilinan on 2017-03-29.
 * <p>
 * The Analytics Kit is intended to help generate a structure event log to send to a specified server or
 * to manually handle and listen to events coming from the JMap Android SDK.
 */
public class AnalyticsKit {
    public static final String EVENT_ENDPOINT = "end_point";
    public static final String EVENT_REQUEST_START_TIME = "start_time";
    public static final String EVENT_REQUEST_END_TIME = "end_time";
    public static final String EVENT_TIME = "event_time";
    public static final String EVENT_DURATION = "duration";
    public static final String EVENT_ID = "id";
    public static final String EVENT_ERROR = "error";
    public static final String EVENT_COORDINATES = "coordinates";
    public static final String EVENT_ICON_TYPE = "type";
    public static final String EVENT_SHAPE_TYPE = "type";
    public static final String EVENT_WAYPOINT_ID = "waypoint_id";
    public static final String EVENT_WAYFIND_START_WAYPOINT_ID = "wayfind_start";
    public static final String EVENT_WAYFIND_END_WAYPOINT_ID = "wayfind_end";

    public static final String EVENT_TYPE = "event_type";
    public static final String EVENT_TYPE_API_CALL = "api_call";
    public static final String EVENT_TYPE_PARSE_MAP = "parse_map";
    public static final String EVENT_TYPE_SHOW_MAP = "show_map";
    public static final String EVENT_TYPE_GENERATE_WAYFIND = "generate_wayfind";
    public static final String EVENT_TYPE_TAP_MAP = "tap_map";
    public static final String EVENT_TYPE_TAP_ICON = "tap_icon";
    public static final String EVENT_TYPE_TAP_SHAPE = "tap_shape";


    private static AnalyticsKit ourInstance = new AnalyticsKit();

    //start of server request
    private Date serverRequestStart;
    //start of map parse
    private Date parseMapStart;
    private OnJMapEventListener onJMapEventListener;

    public static AnalyticsKit getInstance() {
        return ourInstance;
    }

    private AnalyticsKit() {
    }

    /**
     * Event listener that is fired when a AnalyticsEvent is posted from the JMap Android SDK.
     *
     * @param analyticsEvent
     * @see AnalyticsEvent
     */
    public void onEvent(AnalyticsEvent analyticsEvent) {
        //add to events log
        switch (analyticsEvent.getKey()) {
            case AnalyticsEvent.JMAP_CORE_SERVER_REQUEST_START: {
                //endpoint

                //timestamp start of network call
                serverRequestStart = getCurrentDateAndTime();

                break;
            }
            case AnalyticsEvent.JMAP_CORE_SERVER_REQUEST_SUCCESS:
            case AnalyticsEvent.JMAP_CORE_SERVER_REQUEST_FAILED: {
                //endpoint, [error]

                String endpoint = analyticsEvent.getData("endpoint");
                if (endpoint != null) {
                    String regex = "";
                    if (endpoint.contains("building")) {
                        regex = ".*building\\/(\\d*)";
                    } else if (endpoint.contains("venue")) {
                        regex = ".*venue\\/(\\d*)";
                    } else if (endpoint.contains("customer")) {
                        regex = ".*customer\\/(\\d*)";
                    }

                    final Pattern pattern = Pattern.compile(regex);
                    final Matcher matcher = pattern.matcher(endpoint);


                    String id = null;
                    while (matcher.find()) {
                        id = matcher.group(1);
                    }

                    if (id != null) {
                        //timestamp end of successful network call
                        Date serverRequestEnd = getCurrentDateAndTime();
                        String requestStartTime = getCurrentDateAndTimeFormatted(serverRequestStart);
                        String requestEndTime = getCurrentDateAndTimeFormatted(serverRequestEnd);
                        String duration = getDuration(serverRequestStart, serverRequestEnd);
                        String error = analyticsEvent.getData("error");

                        HashMap<String, String> event = new HashMap<>();
                        event.put(EVENT_TYPE, EVENT_TYPE_API_CALL);
                        event.put(EVENT_ID, id);
                        event.put(EVENT_ENDPOINT, endpoint);
                        event.put(EVENT_REQUEST_START_TIME, requestStartTime);
                        event.put(EVENT_REQUEST_END_TIME, requestEndTime);
                        event.put(EVENT_DURATION, duration);

                        //add error data only if it exists
                        if (error != null) {
                            event.put(EVENT_ERROR, error);
                        }

                        onJMapEventListener.onEvent(event);
                    }
                }

                break;
            }
            case AnalyticsEvent.JMAP_ENGINE_PARSING_START: {
                //mapId

                //timestamp start of parsing
                parseMapStart = getCurrentDateAndTime();

                break;
            }
            case AnalyticsEvent.JMAP_ENGINE_PARSING_SUCCESS:
            case AnalyticsEvent.JMAP_ENGINE_PARSING_FAILED: {
                //mapId, [error]

                String mapId = analyticsEvent.getData("mapId");
                if (mapId != null) {
                    //timestamp end of successful network call
                    Date serverRequestEnd = getCurrentDateAndTime();
                    String requestStartTime = getCurrentDateAndTimeFormatted(parseMapStart);
                    String requestEndTime = getCurrentDateAndTimeFormatted(serverRequestEnd);
                    String duration = getDuration(parseMapStart, serverRequestEnd);
                    String error = analyticsEvent.getData("error");

                    HashMap<String, String> event = new HashMap<>();
                    event.put(EVENT_TYPE, EVENT_TYPE_PARSE_MAP);
                    event.put(EVENT_ID, mapId);
                    event.put(EVENT_REQUEST_START_TIME, requestStartTime);
                    event.put(EVENT_REQUEST_END_TIME, requestEndTime);
                    event.put(EVENT_DURATION, duration);

                    //add error data only if it exists
                    if (error != null) {
                        event.put(EVENT_ERROR, error);
                    }

                    onJMapEventListener.onEvent(event);
                }

                break;
            }
            case AnalyticsEvent.JMAP_CONTROLLER_MAP_SHOW: {
                //mapId

                String mapId = analyticsEvent.getData("mapId");
                if (mapId != null) {
                    //timestamp when map is shown
                    String mapShownTime = getCurrentDateAndTimeFormatted();

                    HashMap<String, String> event = new HashMap<>();
                    event.put(EVENT_TYPE, EVENT_TYPE_SHOW_MAP);
                    event.put(EVENT_ID, mapId);
                    event.put(EVENT_TIME, mapShownTime);

                    onJMapEventListener.onEvent(event);
                }

                break;
            }
            case AnalyticsEvent.JMAP_CONTROLLER_GENERATE_WAYPATH: {
                //accessibility, waypointStartId, waypointEndId, [error]

                //timestamp when map is shown
                String waypathGeneratedAt = getCurrentDateAndTimeFormatted();
                String waypointStartId = analyticsEvent.getData("waypointStartId");
                String waypointEndId = analyticsEvent.getData("waypointEndId");
                String error = analyticsEvent.getData("error");

                HashMap<String, String> event = new HashMap<>();
                event.put(EVENT_TYPE, EVENT_TYPE_GENERATE_WAYFIND);
                event.put(EVENT_WAYFIND_START_WAYPOINT_ID, waypointStartId);
                event.put(EVENT_WAYFIND_END_WAYPOINT_ID, waypointEndId);
                event.put(EVENT_TIME, waypathGeneratedAt);

                //add error data only if it exists
                if (error != null) {
                    event.put(EVENT_ERROR, error);
                }

                onJMapEventListener.onEvent(event);

                break;
            }
            case AnalyticsEvent.JMAP_VIEW_TAP: {
                //mapId, coordinates

                String mapId = analyticsEvent.getData("mapId");
                if (mapId != null) {
                    String coordinates = analyticsEvent.getData("coordinates");
                    String timeTapped = getCurrentDateAndTimeFormatted();

                    HashMap<String, String> event = new HashMap<>();
                    event.put(EVENT_TYPE, EVENT_TYPE_TAP_MAP);
                    event.put(EVENT_ID, mapId);
                    event.put(EVENT_TIME, timeTapped);
                    event.put(EVENT_COORDINATES, coordinates);

                    onJMapEventListener.onEvent(event);
                }

                break;
            }
            case AnalyticsEvent.JMAP_VIEW_TAP_ICON: {
                //mapId, iconId, iconType, [waypointIds], [destinationIds]

//                String mapId = analyticsEvent.getData("mapId"));
                String iconId = analyticsEvent.getData("iconId");
                String iconType = analyticsEvent.getData("iconType");
                String waypointId = analyticsEvent.getData("waypointId");
                String timeTapped = getCurrentDateAndTimeFormatted();

                HashMap<String, String> event = new HashMap<>();
                event.put(EVENT_TYPE, EVENT_TYPE_TAP_ICON);
                event.put(EVENT_ID, iconId);
                event.put(EVENT_ICON_TYPE, iconType);
                event.put(EVENT_WAYPOINT_ID, waypointId);
                event.put(EVENT_TIME, timeTapped);

                onJMapEventListener.onEvent(event);

                break;
            }
            case AnalyticsEvent.JMAP_VIEW_TAP_SHAPE: {
                //mapId, shapeType, [waypointIds], [destinationIds]

//                String mapId = analyticsEvent.getData("mapId"));
                String shapeId = analyticsEvent.getData("shapeId");
                String shapeType = analyticsEvent.getData("shapeType");
                String waypointIds = analyticsEvent.getData("waypointIds");
                String timeTapped = getCurrentDateAndTimeFormatted();

                HashMap<String, String> event = new HashMap<>();
                event.put(EVENT_TYPE, EVENT_TYPE_TAP_SHAPE);
                event.put(EVENT_ID, shapeId);
                event.put(EVENT_SHAPE_TYPE, shapeType);
                event.put(EVENT_WAYPOINT_ID, waypointIds);
                event.put(EVENT_TIME, timeTapped);

                onJMapEventListener.onEvent(event);

                break;
            }
        }
    }

    private Date getCurrentDateAndTime() {
        return new Date();
    }

    private String getCurrentDateAndTimeFormatted() {
        return getCurrentDateAndTimeFormatted(new Date());
    }

    private String getCurrentDateAndTimeFormatted(Date date) {
        //ie. 2017-03-08T21:43:03.962Z
        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        return dateFormat.format(date);
    }

    private String getDuration(Date startDate, Date endDate) {
        long duration = endDate.getTime() - startDate.getTime();

        return String.valueOf(duration);
    }

    public void setOnJMapEventListener(OnJMapEventListener onJMapEventListener) {
        this.onJMapEventListener = onJMapEventListener;
    }

    public interface OnJMapEventListener {
        void onEvent(HashMap<String, String> event);
    }
}