/*
 * Decompiled with CFR 0.152.
 */
package com.jetdrone.vertx.mods.stomp;

import com.jetdrone.vertx.mods.stomp.Frame;
import com.jetdrone.vertx.mods.stomp.Protocol;
import com.jetdrone.vertx.mods.stomp.StompDecoder;
import com.jetdrone.vertx.mods.stomp.StompSubscriptions;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import java.io.IOException;
import java.util.LinkedList;
import java.util.Map;
import java.util.Queue;
import org.vertx.java.core.AsyncResult;
import org.vertx.java.core.AsyncResultHandler;
import org.vertx.java.core.Handler;
import org.vertx.java.core.Vertx;
import org.vertx.java.core.buffer.Buffer;
import org.vertx.java.core.logging.Logger;
import org.vertx.java.core.net.NetClient;
import org.vertx.java.core.net.NetSocket;

public class StompClient {
    private static final Frame ASYNCFRAME = new Frame("ASYNC");
    private final Heartbeat heartbeat = new Heartbeat();
    private final Queue<Handler<Frame>> replies = new LinkedList<Handler<Frame>>();
    private final StompSubscriptions subscriptions;
    private long pinger;
    private long ponger;
    private long serverActivity;
    private final Vertx vertx;
    private final Logger logger;
    private final String host;
    private final int port;
    private final String login;
    private final String passcode;
    private NetSocket netSocket;
    private State state = State.DISCONNECTED;

    private static String getSupportedVersions() {
        return Protocol.V1_2.version + "," + Protocol.V1_1.version + "," + Protocol.V1_0.version;
    }

    public StompClient(Vertx vertx, Logger logger, String host, int port, String login, String passcode, StompSubscriptions subscriptions) {
        this.vertx = vertx;
        this.logger = logger;
        this.host = host;
        this.port = port;
        this.login = login;
        this.passcode = passcode;
        this.subscriptions = subscriptions;
    }

    public void connect(final AsyncResultHandler<Void> resultHandler) {
        if (this.state == State.DISCONNECTED) {
            this.state = State.CONNECTING;
            NetClient client = this.vertx.createNetClient();
            client.connect(this.port, this.host, (Handler)new AsyncResultHandler<NetSocket>(){

                public void handle(final AsyncResult<NetSocket> asyncResult) {
                    if (asyncResult.failed()) {
                        StompClient.this.logger.error((Object)"Net client error", asyncResult.cause());
                        if (resultHandler != null) {
                            resultHandler.handle((Object)new AsyncResult<Void>(){

                                public Void result() {
                                    return null;
                                }

                                public Throwable cause() {
                                    return asyncResult.cause();
                                }

                                public boolean succeeded() {
                                    return asyncResult.succeeded();
                                }

                                public boolean failed() {
                                    return asyncResult.failed();
                                }
                            });
                        }
                        StompClient.this.disconnect();
                    } else {
                        StompClient.this.state = State.CONNECTED;
                        StompClient.this.netSocket = (NetSocket)asyncResult.result();
                        StompClient.this.init(StompClient.this.netSocket);
                        StompClient.this.netSocket.exceptionHandler((Handler)new Handler<Throwable>(){

                            public void handle(Throwable e) {
                                StompClient.this.logger.error((Object)"Socket client error", e);
                                StompClient.this.disconnect();
                            }
                        });
                        StompClient.this.netSocket.closeHandler((Handler)new Handler<Void>(){

                            public void handle(Void arg0) {
                                StompClient.this.logger.info((Object)"Socket closed");
                                StompClient.this.disconnect();
                            }
                        });
                        if (resultHandler != null) {
                            resultHandler.handle((Object)new AsyncResult<Void>(){

                                public Void result() {
                                    return null;
                                }

                                public Throwable cause() {
                                    return null;
                                }

                                public boolean succeeded() {
                                    return true;
                                }

                                public boolean failed() {
                                    return false;
                                }
                            });
                        }
                    }
                }
            });
        }
    }

    private void disconnect() {
        this.state = State.DISCONNECTED;
        if (this.pinger != 0L) {
            this.vertx.cancelTimer(this.pinger);
        }
        if (this.ponger != 0L) {
            this.vertx.cancelTimer(this.ponger);
        }
        if (this.netSocket != null) {
            this.netSocket.close();
        }
    }

    void send(final Frame frame, final boolean async, final Handler<Frame> replyHandler) {
        switch (this.state) {
            case CONNECTED: {
                this.netSocket.write(frame.command);
                this.netSocket.write("\n");
                for (Map.Entry<String, String> entry : frame.headers.entrySet()) {
                    String value = entry.getValue();
                    if (value == null) continue;
                    this.netSocket.write(entry.getKey());
                    this.netSocket.write(":");
                    this.netSocket.write(Frame.escape(entry.getValue()));
                    this.netSocket.write("\n");
                }
                if (frame.body != null) {
                    this.netSocket.write("content-length:");
                    this.netSocket.write(Integer.toString(frame.body.length()));
                    this.netSocket.write("\n");
                }
                this.netSocket.write("\n");
                if (frame.body != null) {
                    this.netSocket.write(frame.body);
                }
                this.netSocket.write("\u0000");
                if (async) {
                    replyHandler.handle((Object)ASYNCFRAME);
                    break;
                }
                this.replies.offer(replyHandler);
                break;
            }
            case DISCONNECTED: {
                this.logger.info((Object)"Got request when disconnected. Trying to connect.");
                this.connect(new AsyncResultHandler<Void>(){

                    public void handle(AsyncResult<Void> connection) {
                        if (connection.succeeded()) {
                            StompClient.this.send(frame, async, (Handler<Frame>)replyHandler);
                        } else {
                            Frame error = new Frame("ERROR");
                            error.body = "Unable to connect";
                            replyHandler.handle((Object)error);
                        }
                    }
                });
                break;
            }
            case CONNECTING: {
                this.logger.debug((Object)"Got send request while connecting. Will try again in a while.");
                this.vertx.setTimer(100L, (Handler)new Handler<Long>(){

                    public void handle(Long event) {
                        StompClient.this.send(frame, async, (Handler<Frame>)replyHandler);
                    }
                });
            }
        }
    }

    private void init(NetSocket netSocket) {
        this.netSocket = netSocket;
        final StompDecoder stompDecoder = new StompDecoder();
        this.heartbeat.sx = 10000;
        this.heartbeat.sy = 10000;
        netSocket.dataHandler((Handler)new Handler<Buffer>(){
            private ByteBuf read = null;

            public void handle(Buffer buffer) {
                StompClient.this.serverActivity = System.currentTimeMillis();
                ByteBuf byteBuf = buffer.getByteBuf();
                if (this.read != null) {
                    byteBuf = Unpooled.copiedBuffer((ByteBuf[])new ByteBuf[]{this.read, byteBuf});
                    this.read = null;
                }
                try {
                    Frame receive = stompDecoder.receive(byteBuf);
                    StompClient.this.handleReply(receive);
                    if (byteBuf.isReadable()) {
                        this.handle(new Buffer(Unpooled.copiedBuffer((ByteBuf)byteBuf)));
                    }
                }
                catch (IOException e) {
                    StompClient.this.logger.error((Object)"Error receiving data", (Throwable)e);
                    StompClient.this.disconnect();
                }
                catch (IndexOutOfBoundsException th) {
                    byteBuf.resetReaderIndex();
                    this.read = Unpooled.copiedBuffer((ByteBuf)byteBuf);
                }
            }
        });
        this.logger.debug((Object)"Socket Opened...");
        Frame connect = new Frame("CONNECT");
        connect.putHeader("accept-version", StompClient.getSupportedVersions());
        connect.putHeader("heart-beat", this.heartbeat.toString());
        connect.putHeader("vhost", this.host);
        if (this.login != null) {
            connect.putHeader("login", this.login);
        }
        if (this.passcode != null) {
            connect.putHeader("passcode", this.passcode);
        }
        this.send(connect, false, new Handler<Frame>(){

            public void handle(Frame frame) {
                StompClient.this.logger.debug((Object)("connected to server " + frame.headers.get("server")));
                StompClient.this.setupHeartbeat(frame.headers);
            }
        });
    }

    private void setupHeartbeat(Map<String, String> headers) {
        int ttl;
        if (headers.get("version").equals(Protocol.V1_0.version)) {
            return;
        }
        Heartbeat heartbeat = Heartbeat.parse(headers.get("heart-beat"));
        if (this.heartbeat.sx != 0 && heartbeat.sy != 0) {
            ttl = Math.max(this.heartbeat.sx, heartbeat.sy);
            this.logger.debug((Object)("send PING every " + ttl + "ms"));
            this.pinger = this.vertx.setPeriodic((long)ttl, (Handler)new Handler<Long>(){

                public void handle(Long event) {
                    if (StompClient.this.state == State.CONNECTED) {
                        StompClient.this.logger.debug((Object)"PING");
                        StompClient.this.netSocket.write("\n");
                    }
                }
            });
        }
        if (this.heartbeat.sy != 0 && heartbeat.sx != 0) {
            ttl = Math.max(this.heartbeat.sy, heartbeat.sx);
            this.logger.debug((Object)("check PONG every " + ttl + "ms"));
            this.ponger = this.vertx.setPeriodic((long)ttl, (Handler)new Handler<Long>(){

                public void handle(Long event) {
                    long delta = System.currentTimeMillis() - StompClient.this.serverActivity;
                    if (delta > (long)(ttl * 2)) {
                        StompClient.this.logger.debug((Object)("did not receive server activity for the last " + delta + "ms"));
                        StompClient.this.disconnect();
                    }
                }
            });
        }
    }

    void handleReply(Frame reply) throws IOException {
        if ("ERROR".equals(reply.command)) {
            this.logger.error((Object)reply.body);
            this.disconnect();
            return;
        }
        Handler<Frame> handler = this.replies.poll();
        if (handler != null) {
            handler.handle((Object)reply);
            return;
        }
        if ("MESSAGE".equals(reply.command) && (handler = this.subscriptions.getHandler(reply.headers.get("subscription"))) != null) {
            handler.handle((Object)reply);
            return;
        }
        System.out.println(reply.toJSON());
        throw new IOException("Received a non MESSAGE while in SUBSCRIBE mode");
    }

    private static enum State {
        DISCONNECTED,
        CONNECTING,
        CONNECTED;

    }

    private static class Heartbeat {
        int sx;
        int sy;

        private Heartbeat() {
        }

        static Heartbeat parse(String header) {
            String[] token = header.split(",");
            Heartbeat beat = new Heartbeat();
            beat.sx = Integer.parseInt(token[0]);
            beat.sy = Integer.parseInt(token[1]);
            return beat;
        }

        public String toString() {
            return this.sx + "," + this.sy;
        }
    }
}

