package com.ironsource.adapters.ironsource;

import android.app.Activity;
import android.text.TextUtils;
import android.view.Gravity;
import android.widget.FrameLayout;

import com.ironsource.adapters.supersonicads.SupersonicConfig;
import com.ironsource.mediationsdk.AbstractAdapter;
import com.ironsource.mediationsdk.AdapterUtils;
import com.ironsource.mediationsdk.AuctionDataUtils;
import com.ironsource.mediationsdk.BuildConfig;
import com.ironsource.mediationsdk.ISBannerSize;
import com.ironsource.mediationsdk.IronSourceBannerLayout;
import com.ironsource.mediationsdk.IronSourceObject;
import com.ironsource.mediationsdk.LoadWhileShowSupportState;
import com.ironsource.mediationsdk.logger.IronLog;
import com.ironsource.mediationsdk.logger.IronSourceError;
import com.ironsource.mediationsdk.logger.IronSourceLogger;
import com.ironsource.mediationsdk.metadata.MetaDataConstants;
import com.ironsource.mediationsdk.metadata.MetaDataUtils;
import com.ironsource.mediationsdk.sdk.BannerSmashListener;
import com.ironsource.mediationsdk.sdk.InternalOfferwallListener;
import com.ironsource.mediationsdk.sdk.InterstitialSmashListener;
import com.ironsource.mediationsdk.sdk.OfferwallAdapterApi;
import com.ironsource.mediationsdk.sdk.RewardedVideoSmashListener;
import com.ironsource.mediationsdk.utils.ContextProvider;
import com.ironsource.mediationsdk.utils.ErrorBuilder;
import com.ironsource.mediationsdk.utils.IronSourceConstants;
import com.ironsource.mediationsdk.utils.IronSourceUtils;
import com.ironsource.mediationsdk.utils.SessionDepthManager;
import com.ironsource.sdk.ISAdSize;
import com.ironsource.sdk.ISNAdView.ISNAdView;
import com.ironsource.sdk.IronSourceAdInstance;
import com.ironsource.sdk.IronSourceAdInstanceBuilder;
import com.ironsource.sdk.IronSourceNetwork;
import com.ironsource.sdk.constants.Constants;
import com.ironsource.sdk.data.ISNEnums;
import com.ironsource.sdk.data.ISNError;
import com.ironsource.sdk.listeners.OnBannerListener;
import com.ironsource.sdk.listeners.OnInterstitialListener;
import com.ironsource.sdk.listeners.OnNetworkSDKInitListener;
import com.ironsource.sdk.listeners.OnOfferWallListener;
import com.ironsource.sdk.utils.SDKUtils;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;

/*
This adapter is responsible for addressing several flows
    - mediation (reward video is loaded automatically)
            -  non bidder - reward video and interstitial
            -  bidder - reward video and interstitial
    - demand only  (reward video is loaded manually)
            - non bidder - reward video and interstitial
            - demand only - bidder - reward video and interstitial
*/
public class IronSourceAdapter extends AbstractAdapter implements OfferwallAdapterApi, OnOfferWallListener, ContextProvider.ContextLifeCycleListener {
    private final static String VERSION = BuildConfig.VERSION_NAME;
    private final static int IS_LOAD_EXCEPTION = 1000;
    private final static int IS_SHOW_EXCEPTION = 1001;
    private final static int RV_LOAD_EXCEPTION = 1002;
    private final static int RV_SHOW_EXCEPTION = 1003;

    private final String DYNAMIC_CONTROLLER_URL = "controllerUrl";
    private final String DYNAMIC_CONTROLLER_DEBUG_MODE = "debugMode";
    private final String DYNAMIC_CONTROLLER_CONFIG = "controllerConfig";

    private final String SESSION_ID = "sessionid";
    private final String SDK_PLUGIN_TYPE = "SDKPluginType";
    private final String CUSTOM_SEGMENT = "custom_Segment";
    private final String ADM_KEY = "adm";
    private final String IN_APP_BIDDING_KEY = "inAppBidding";
    private final String IN_APP_BIDDING_VALUE = "true";


    private final String DEMAND_SOURCE_NAME = "demandSourceName";

    private final String OW_PLACEMENT_ID = "placementId";
    private final String OW_CLIENT_SIDE_CALLBACKS = "useClientSideCallbacks";
    private InternalOfferwallListener mOfferwallListener;

    private final String LWS_SUPPORT_STATE = "isSupportedLWS";

    // Rewarded video
    private ConcurrentHashMap<String, IronSourceAdInstance> mDemandSourceToISAd;
    private ConcurrentHashMap<String, RewardedVideoSmashListener> mDemandSourceToRvSmash;

    // Interstitial
    private ConcurrentHashMap<String, IronSourceAdInstance> mDemandSourceToRvAd;
    private ConcurrentHashMap<String, InterstitialSmashListener> mDemandSourceToISSmash;

    // Banner
    private ConcurrentHashMap<String, BannerSmashListener> mDemandSourceToBNSmash;
    private static ISNAdView mIsnAdView;
    private boolean mIsAlreadyShowing;

    private static AtomicBoolean mDidInitSdk = new AtomicBoolean(false);
    private static String mediationSegment;

    public static IronSourceAdapter startAdapter(String providerName) {
        return new IronSourceAdapter(providerName);
    }

    private IronSourceAdapter(String providerName) {
        super(providerName);
        IronLog.INTERNAL.verbose(providerName + ": new instance");

        // rewarded video
        mDemandSourceToRvAd = new ConcurrentHashMap<>();
        mDemandSourceToRvSmash = new ConcurrentHashMap<>();

        // interstitial
        mDemandSourceToISAd = new ConcurrentHashMap<>();
        mDemandSourceToISSmash = new ConcurrentHashMap<>();

        // banner
        mDemandSourceToBNSmash = new ConcurrentHashMap<>();

        mediationSegment = null;

        // listen to life cycle events
        ContextProvider.getInstance().registerLifeCycleListener(this);
    }

    @Override
    public String getVersion() {
        return VERSION;
    }

    @Override
    public String getCoreSDKVersion() {
        return SDKUtils.getSDKVersion();
    }

    // ********** Base **********

    @Override
    public void onPause(Activity activity) {
        IronLog.ADAPTER_API.verbose("IronSourceNetwork.onPause");
        IronSourceNetwork.onPause(activity);
    }

    @Override
    public void onResume(Activity activity) {
        IronLog.ADAPTER_API.verbose("IronSourceNetwork.onResume");
        IronSourceNetwork.onResume(activity);
    }

    protected void setConsent(boolean consent) {
        IronLog.ADAPTER_API.verbose("(" + (consent ? "true" : "false") + ")");

        JSONObject consentParams = new JSONObject();
        try {
            consentParams.put(Constants.RequestParameters.CONSENT, consent ? "1" : "0");
        } catch (JSONException e) {
            IronLog.ADAPTER_API.error("exception " + e.getMessage());
            return;
        }

        IronSourceNetwork.updateMetadata(consentParams);
    }

    @Override
    public void setMediationSegment(String segment) {
        mediationSegment = segment;
    }

    private HashMap<String, String> getInitParams() {
        HashMap<String, String> params = new HashMap<>();

        String pluginType = getPluginType();
        if (!TextUtils.isEmpty(pluginType)) {
            params.put(SDK_PLUGIN_TYPE, pluginType);
        }

        if (!TextUtils.isEmpty(mediationSegment)) {
            params.put(CUSTOM_SEGMENT, mediationSegment);
        }

        if (!TextUtils.isEmpty(IronSourceObject.getInstance().getSessionId())) {
            params.put(SESSION_ID, IronSourceObject.getInstance().getSessionId());
        }

        return params;
    }

    public JSONObject getPlayerBiddingData() {
        IronLog.ADAPTER_API.verbose("");
        JSONObject biddingData = new JSONObject();
        JSONObject token = null;

        try {
            token = IronSourceNetwork.getRawToken(ContextProvider.getInstance().getApplicationContext());
        }
        // In case there is an exception while trying to fetch the token we wouldn't want to return a partial token
        // so instead we will return null. Since this is Demand Only related code we will add it on the IronSource adapter layer
        catch (Exception e) {
            IronLog.ADAPTER_API.error("getRawToken exception: " + e.getLocalizedMessage());
        }

        if (token != null) {
            biddingData = token;
        } else {
            IronLog.ADAPTER_API.error("Player's bidding token is null");
        }

        return biddingData;
    }

    // ********** Init SDK **************

    @Override
    public void earlyInit(final String appKey, final String userId, final JSONObject config) {
        if (appKey != null) {
            IronSourceUtils.sendAutomationLog(getDemandSourceName(config) + ": " + "earlyInit");
            initSDK(appKey, config);
        } else {
            IronLog.ADAPTER_API.error("Appkey is null for early init");
        }
    }

    private void initSDK(String appKey, JSONObject config) {

        if (mDidInitSdk.compareAndSet(false, true)) {
            //using the mediation user id and not the user id provided by the adapter that does not include the default value if not set by the publisher
            String userId = IronSourceUtils.getMediationUserId();
            int logLevel = config.optInt(DYNAMIC_CONTROLLER_DEBUG_MODE, IronSourceLogger.IronSourceLogLevel.VERBOSE);
            if (isAdaptersDebugEnabled()) {
                logLevel = IronSourceLogger.IronSourceLogLevel.ERROR;
            }
            IronLog.ADAPTER_API.verbose("setting debug mode to " + logLevel);
            SDKUtils.setDebugMode(logLevel);

            SDKUtils.setControllerUrl(config.optString(DYNAMIC_CONTROLLER_URL));
            IronLog.ADAPTER_API.verbose("IronSourceNetwork setting controller url to  " + config.optString(DYNAMIC_CONTROLLER_URL));

            SDKUtils.setControllerConfig(config.optString(DYNAMIC_CONTROLLER_CONFIG));
            IronLog.ADAPTER_API.verbose("IronSourceNetwork setting controller config to  " + config.optString(DYNAMIC_CONTROLLER_CONFIG));

            HashMap<String, String> initParams = getInitParams();
            IronLog.ADAPTER_API.verbose("with appKey=" + appKey + " userId=" + userId + " parameters " + initParams);

            IronSourceNetwork.setInitListener(new OnNetworkSDKInitListener() {
                @Override
                public void onSuccess() {
                    IronLog.ADAPTER_API.verbose(
                            "OnNetworkSDKInitListener success"
                    );
                }

                @Override
                public void onFail(ISNError error) {
                    IronLog.ADAPTER_API.verbose(
                            "OnNetworkSDKInitListener fail - code:" + error.getCode() + " message:" + error.getMessage()
                    );
                }
            });

            IronSourceNetwork.initSDK(
                    ContextProvider.getInstance().getApplicationContext(),
                    appKey,
                    userId,
                    initParams);
        }
    }

    // ********** RewardedVideo **********

    @Override
    public Map<String, Object> getRewardedVideoBiddingData(JSONObject config) {
        return getBiddingData();
    }

    // used for bidders in the mediation
    @Override
    public void initRewardedVideoForBidding(String appKey, String userId, JSONObject config, RewardedVideoSmashListener listener) {
        String demandSourceName = getDemandSourceName(config);
        IronLog.INTERNAL.verbose(demandSourceName + ": demandSourceName=" + demandSourceName);

        initRewardedVideoInternal(appKey, config, listener, demandSourceName);
        listener.onRewardedVideoInitSuccess();
    }

    // used for bidders and non bidders in demand only
    @Override
    public void initRewardedVideoForDemandOnly(String appKey, String userId, JSONObject config, RewardedVideoSmashListener listener) {
        String demandSourceName = getDemandSourceName(config);
        IronLog.INTERNAL.verbose(demandSourceName + ": demandSourceName=" + demandSourceName);

        initRewardedVideoInternal(appKey, config, listener, demandSourceName);
    }

    // used for non bidders in the mediation
    @Override
    public void initRewardedVideo(String appKey, String userId, JSONObject config, RewardedVideoSmashListener listener) {
        String demandSourceName = getDemandSourceName(config);
        IronLog.INTERNAL.verbose(demandSourceName + ": demandSourceName=" + demandSourceName);
        initRewardedVideoInternal(appKey, config, listener, demandSourceName);
        fetchRewardedVideoForAutomaticLoad(config, listener);
    }

    private void initRewardedVideoInternal(String appKey, JSONObject config, RewardedVideoSmashListener listener, String demandSourceName) {
        initSDK(appKey, config);
        mDemandSourceToRvSmash.put(demandSourceName, listener);
    }

    // TODO unify API calls

    // used for non bidders in the mediation
    @Override
    public void fetchRewardedVideoForAutomaticLoad(JSONObject config, RewardedVideoSmashListener listener) {
        String demandSourceName = getDemandSourceName(config);
        IronLog.ADAPTER_API.verbose(demandSourceName);

        try {
            loadAdInternal(demandSourceName, null, false, false, true);
        } catch (Exception e) {
            IronLog.ADAPTER_API.error("exception " + e.getMessage());

            listener = mDemandSourceToRvSmash.get(demandSourceName);
            if (listener != null) {
                IronLog.ADAPTER_API.error("exception " + e.getMessage());
                listener.onRewardedVideoLoadFailed(new IronSourceError(RV_LOAD_EXCEPTION, e.getMessage()));
                listener.onRewardedVideoAvailabilityChanged(false);
            }
        }
    }

    // used for non bidders in demand only
    @Override
    public void loadRewardedVideoForDemandOnly(JSONObject config, RewardedVideoSmashListener listener) {
        IronLog.ADAPTER_API.verbose(getDemandSourceName(config));

        try {
            loadAdInternal(getDemandSourceName(config), null, true, false, true);
        } catch (Exception e) {
            IronLog.ADAPTER_API.error("exception " + e.getMessage());
            listener.onRewardedVideoLoadFailed(new IronSourceError(RV_LOAD_EXCEPTION, e.getMessage()));
        }
    }

    // used for bidders in demand only
    @Override
    public void loadRewardedVideoForDemandOnlyForBidding(JSONObject config, RewardedVideoSmashListener listener, String serverData) {
        IronLog.ADAPTER_API.verbose(getDemandSourceName(config));

        try {
            loadAdInternal(getDemandSourceName(config), serverData, true, true, true);
        } catch (Exception e) {
            IronLog.ADAPTER_API.error("exception " + e.getMessage());
            listener.onRewardedVideoLoadFailed(new IronSourceError(RV_LOAD_EXCEPTION, e.getMessage()));
        }
    }

    // used  bidders in the mediation
    @Override
    public void loadRewardedVideoForBidding(JSONObject config, final RewardedVideoSmashListener listener, final String serverData) {
        IronLog.ADAPTER_API.verbose(getDemandSourceName(config));

        try {
            loadAdInternal(getDemandSourceName(config), serverData, false, true, true);
        } catch (Exception e) {
            IronLog.ADAPTER_API.error("exception " + e.getMessage());
            listener.onRewardedVideoLoadFailed(new IronSourceError(RV_LOAD_EXCEPTION, e.getMessage()));
            listener.onRewardedVideoAvailabilityChanged(false);
        }
    }

    @Override
    public void showRewardedVideo(JSONObject config, RewardedVideoSmashListener listener) {
        try {
            IronSourceAdInstance adInstance = mDemandSourceToRvAd.get(getDemandSourceName(config));
            showAdInternal(adInstance, SessionDepthManager.REWARDEDVIDEO);
        } catch (Exception e) {
            IronLog.ADAPTER_API.error("exception " + e.getMessage());
            listener.onRewardedVideoAdShowFailed(new IronSourceError(RV_SHOW_EXCEPTION, e.getMessage()));
        }

    }

    @Override
    public boolean isRewardedVideoAvailable(JSONObject config) {
        IronSourceAdInstance adInstance = mDemandSourceToRvAd.get(getDemandSourceName(config));
        return adInstance != null && IronSourceNetwork.isAdAvailableForInstance(adInstance);
    }

    Map<String, Object> getBiddingData(){
        IronLog.ADAPTER_API.verbose("");
        Map<String, Object> biddingData = new HashMap<>();
        String token = IronSourceNetwork.getToken(ContextProvider.getInstance().getApplicationContext());

        if (token != null) {
            biddingData.put(IronSourceConstants.IRONSOURCE_BIDDING_TOKEN_KEY, token);
        }
        else {
            IronLog.ADAPTER_API.error("bidding token is null");
            biddingData.put(IronSourceConstants.IRONSOURCE_BIDDING_TOKEN_KEY, "");
        }
        return biddingData;
    }


    // ********** Interstitial **********

    @Override
    public Map<String, Object> getInterstitialBiddingData(JSONObject config) {
        return getBiddingData();
    }

    // used for bidders in the mediation
    @Override
    public void initInterstitialForBidding(String appKey, String userId, JSONObject config, InterstitialSmashListener listener) {
        String demandSourceName = getDemandSourceName(config);
        IronLog.INTERNAL.verbose(demandSourceName + ": " + "demandSourceName=" + demandSourceName);
        initInterstitialInternal(appKey, config, listener, demandSourceName);
    }

    // used for non bidders in the mediation and bidders in demand only
    @Override
    public void initInterstitial(String appKey, String userId, JSONObject config, InterstitialSmashListener listener) {
        String demandSourceName = getDemandSourceName(config);
        IronLog.INTERNAL.verbose(demandSourceName + ": " + "demandSourceName=" + demandSourceName);
        initInterstitialInternal(appKey, config, listener, demandSourceName);
    }


    private void initInterstitialInternal(String appKey, JSONObject config, InterstitialSmashListener listener, String demandSourceName) {
        initSDK(appKey, config);
        mDemandSourceToISSmash.put(demandSourceName, listener);
        listener.onInterstitialInitSuccess();
    }

    // used for bidders in the mediation and demand only
    @Override
    public void loadInterstitialForBidding(final JSONObject config, final InterstitialSmashListener listener, final String serverData) {
        IronLog.ADAPTER_API.verbose(getDemandSourceName(config));
        try {
            loadAdInternal(getDemandSourceName(config), serverData, false, true, false);
        } catch (Exception e) {
            IronLog.ADAPTER_API.error("for bidding exception " + e.getMessage());
            listener.onInterstitialAdLoadFailed(new IronSourceError(IS_LOAD_EXCEPTION, e.getMessage()));
        }
    }

    // used for non bidders in the mediation and demand only
    @Override
    public void loadInterstitial(JSONObject config, InterstitialSmashListener listener) {
        IronLog.ADAPTER_API.verbose(getDemandSourceName(config));
        try {
            loadAdInternal(getDemandSourceName(config), null, false, false, false);
        } catch (Exception e) {
            IronLog.ADAPTER_API.error("exception " + e.getMessage());
            listener.onInterstitialAdLoadFailed(new IronSourceError(IS_LOAD_EXCEPTION, e.getMessage()));
        }
    }

    @Override
    public void showInterstitial(JSONObject config, InterstitialSmashListener listener) {
        IronLog.ADAPTER_API.verbose(getDemandSourceName(config));
        try {
            IronSourceAdInstance adInstance = mDemandSourceToISAd.get(getDemandSourceName(config));
            showAdInternal(adInstance, SessionDepthManager.INTERSTITIAL);
        } catch (Exception e) {
            IronLog.ADAPTER_API.error("exception " + e.getMessage());
            listener.onInterstitialAdShowFailed(new IronSourceError(IS_SHOW_EXCEPTION, e.getMessage()));
        }
    }

    @Override
    public boolean isInterstitialReady(JSONObject config) {
        IronSourceAdInstance adInstance = mDemandSourceToISAd.get(getDemandSourceName(config));
        return adInstance != null && IronSourceNetwork.isAdAvailableForInstance(adInstance);
    }

    // ********** OfferWall **********

    @Override
    public void setInternalOfferwallListener(InternalOfferwallListener listener) {
        mOfferwallListener = listener;
    }

    @Override
    public void showOfferwall(String placementId, JSONObject config) {
        IronLog.ADAPTER_API.verbose(getProviderName() + " showOfferWall");

        try {
            Map<String, String> extraParams = getOfferwallExtraParams();
            extraParams.put(OW_PLACEMENT_ID, placementId);
            IronSourceNetwork.showOfferWall(ContextProvider.getInstance().getCurrentActiveActivity(), extraParams);
        } catch (Exception e) {
            IronLog.ADAPTER_API.error("exception " + e.getMessage());
        }

    }

    private HashMap<String, String> getOfferwallExtraParams() {
        HashMap<String, String> owExtraParams = getInitParams();

        //Check client side callbacks configuration
        boolean clientSideCallbacks = SupersonicConfig.getConfigObj().getClientSideCallbacks();
        owExtraParams.put(OW_CLIENT_SIDE_CALLBACKS, String.valueOf(clientSideCallbacks));

        //Add Offer Wall customParams if set
        Map<String, String> customParams = SupersonicConfig.getConfigObj().getOfferwallCustomParams();
        if (customParams != null && !customParams.isEmpty()) {
            owExtraParams.putAll(customParams);
        }

        return owExtraParams;
    }

    @Override
    public boolean isOfferwallAvailable() {
        return true;
    }

    @Override
    public void getOfferwallCredits() {
        IronLog.ADAPTER_API.verbose(getProviderName() + " getOfferwallCredits");
        try {
            IronSourceNetwork.getOfferWallCredits(this);
        } catch (Exception e) {
            IronLog.ADAPTER_API.error("exception " + e.getMessage());
        }
    }

    @Override
    public void initOfferwall(
            String appKey,
            String userId,
            JSONObject config) {

        initSDK(appKey, config);
        IronLog.ADAPTER_API.verbose(":initOfferwall");

        postOnUIThread(new Runnable() {
            @Override
            public void run() {
                try {
                    Map<String, String> offerwallExtraParams = getOfferwallExtraParams();
                    IronSourceNetwork.initOfferWall(offerwallExtraParams, IronSourceAdapter.this);
                } catch (Exception e) {
                    IronLog.ADAPTER_API.error(getProviderName() + ":initOfferwall " + e);
                    mOfferwallListener.onOfferwallAvailable(
                            false,
                            ErrorBuilder.buildInitFailedError(
                                    "Adapter initialization failure - " + getProviderName() + " - " + e.getMessage(),
                                    IronSourceConstants.OFFERWALL_AD_UNIT
                            )
                    );
                }
            }
        });
    }


    // ********** Banner **********

    @Override
    public Map<String, Object> getBannerBiddingData(JSONObject config) {
        return getBiddingData();
    }

    @Override
    public void initBanners(String appKey, String userId, JSONObject config, BannerSmashListener listener) {
        String demandSourceName = getDemandSourceName(config);
        IronLog.ADAPTER_API.verbose(demandSourceName + ": " + "demandSourceName=" + demandSourceName);

        // init sdk
        initSDK(appKey, config);
        mDemandSourceToBNSmash.put(demandSourceName, listener);

        try {
            IronSourceNetwork.initBanner(demandSourceName, getInitParams(), new IronSourceBannerListener(listener, demandSourceName));
        } catch (Exception e) {
            listener.onBannerInitFailed(ErrorBuilder.buildInitFailedError(e.getMessage(), IronSourceConstants.BANNER_AD_UNIT));
        }
    }

    @Override
    public void loadBanner(IronSourceBannerLayout banner, JSONObject config, BannerSmashListener listener) {
        String demandSourceName = getDemandSourceName(config);
        IronLog.ADAPTER_API.verbose(demandSourceName + ": " + "demandSourceName=" + demandSourceName);

        //Remove ISNAdView instance from memory in order to prevent multiple instances of it
        //in scenarios when there's a failure in SDK6 that we're not aware of
        if (!mIsAlreadyShowing) {
            destroyBannerInternal();
            mIsnAdView = createBanner(banner.getActivity(), banner.getSize(), listener);
        }

        try {
            loadBannerInternal(mIsnAdView, listener, null);
        } catch (Exception e) {
            String errorString = IronSourceConstants.BANNER_AD_UNIT + " Load Fail, " + getProviderName() + " - " + e.getMessage();
            IronSourceError error = ErrorBuilder.buildLoadFailedError(errorString);
            listener.onBannerAdLoadFailed(error);
        }
    }

    @Override
    public void initBannerForBidding(String appKey, String userId, JSONObject config, BannerSmashListener listener) {
        String demandSourceName = getDemandSourceName(config);
        IronLog.ADAPTER_API.verbose(demandSourceName + ": " + "demandSourceName=" + demandSourceName);



        // init sdk
        initSDK(appKey, config);
        mDemandSourceToBNSmash.put(demandSourceName,listener);
        listener.onBannerInitSuccess();
    }

    @Override
    public void loadBannerForBidding(IronSourceBannerLayout banner, JSONObject config, BannerSmashListener listener, String serverData) {
        String demandSourceName = getDemandSourceName(config);
        IronLog.ADAPTER_API.verbose(demandSourceName + ": " + "demandSourceName=" + demandSourceName);

        try {

            //Remove ISNAdView instance from memory in order to prevent multiple instances of it
            //in scenarios when there's a failure in SDK6 that we're not aware of
            if (!mIsAlreadyShowing) {
                destroyBannerInternal();
                mIsnAdView = createBanner(banner.getActivity(), banner.getSize(), listener);
            }

            HashMap<String, String> initParams = getInitParams();
            initParams.put(IN_APP_BIDDING_KEY, IN_APP_BIDDING_VALUE);
            IronSourceNetwork.initBanner(demandSourceName, initParams, new IronSourceBannerListener(listener, demandSourceName));
            loadBannerInternal(mIsnAdView, listener, serverData);

        } catch (Exception e) {
            String errorString = IronSourceConstants.BANNER_AD_UNIT + " Load Fail, " + getProviderName() + " - " + e.getMessage();
            IronSourceError error = ErrorBuilder.buildLoadFailedError(errorString);
            listener.onBannerAdLoadFailed(error);
        }
    }

    @Override
    public void reloadBanner(IronSourceBannerLayout banner, JSONObject config, BannerSmashListener listener) {
        String demandSourceName = getDemandSourceName(config);
        IronLog.ADAPTER_API.verbose(demandSourceName + ": " + "demandSourceName=" + demandSourceName);
        loadBannerInternal(mIsnAdView, listener, null);
    }

    @Override
    public void destroyBanner(JSONObject config) {
        String demandSourceName = getDemandSourceName(config);
        IronLog.ADAPTER_API.verbose(demandSourceName + ": " + "demandSourceName=" + demandSourceName);
        mIsAlreadyShowing = false;

        destroyBannerInternal();
    }

    private void destroyBannerInternal() {
        if (mIsnAdView != null) {
            IronLog.ADAPTER_API.verbose(getProviderName() + " bannerView.performCleanup");
            mIsnAdView.performCleanup();
            mIsnAdView = null;
        }
    }

    @Override
    public boolean shouldBindBannerViewOnReload() {
        return true;
    }

    private void loadBannerInternal(ISNAdView bannerView, BannerSmashListener listener, String serverData) {
        try {
            final JSONObject loadParams = new JSONObject();
            loadParams.put(Constants.RequestParameters.DEMAND_SOURCE_NAME, getProviderName());
            loadParams.put(Constants.ParametersKeys.PRODUCT_TYPE, ISNEnums.ProductType.Banner);

            if (serverData != null) {
                loadParams.put(ADM_KEY, AuctionDataUtils.getInstance().getAdmFromServerData(serverData));
                loadParams.put(IN_APP_BIDDING_KEY, IN_APP_BIDDING_VALUE);
                Map<String, String> auctionResponseServerDataParams = AuctionDataUtils.getInstance().getAuctionResponseServerDataParams(serverData);
                JSONObject serverResJson = new JSONObject(auctionResponseServerDataParams);

                JSONArray keys = serverResJson.names ();
                for (int i = 0; i < keys.length (); i++) {
                    loadParams.put(keys.getString(i), serverResJson.get(keys.getString(i)));
                }
            }

            try {
                if (bannerView != null) {
                    IronLog.ADAPTER_API.verbose("bannerView.loadAd");
                    bannerView.loadAd(loadParams);
                }
            } catch (Exception e) {
                String errorString = IronSourceConstants.BANNER_AD_UNIT + " Load Fail, " + getProviderName() + " - " + e.getMessage();
                IronSourceError error = ErrorBuilder.buildLoadFailedError(errorString);
                listener.onBannerAdLoadFailed(error);
            }

        } catch (Exception e) {
            e.printStackTrace();
        }
    }
    // ********** Callbacks **********

    private class IronSourceRewardedVideoListener implements OnInterstitialListener {
        private String mDemandSourceName;
        RewardedVideoSmashListener mListener;
        boolean mIsRvDemandOnly;

        IronSourceRewardedVideoListener(RewardedVideoSmashListener listener, String demandSourceName) {
            mDemandSourceName = demandSourceName;
            mListener = listener;
            mIsRvDemandOnly = false;
        }

        IronSourceRewardedVideoListener(RewardedVideoSmashListener listener, String demandSourceName, boolean isDemandOnly) {
            mDemandSourceName = demandSourceName;
            mListener = listener;
            mIsRvDemandOnly = isDemandOnly;
        }

        @Override
        public void onInterstitialInitSuccess() {
            //Deprecated - Old SuperSonicAds API callback
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " rewardedVideoListener");
        }

        @Override
        public void onInterstitialInitFailed(String description) {
            //Deprecated - Old SuperSonicAds API callback
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " rewardedVideoListener");
        }

        @Override
        public void onInterstitialLoadSuccess() {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " rewardedVideoListener");
            if (mIsRvDemandOnly) {
                mListener.onRewardedVideoLoadSuccess();
            } else {
                mListener.onRewardedVideoAvailabilityChanged(true);
            }
        }

        @Override
        public void onInterstitialLoadFailed(String description) {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " rewardedVideoListener " + description);
            mListener.onRewardedVideoLoadFailed(ErrorBuilder.buildLoadFailedError(description));
            if (!mIsRvDemandOnly) {
                mListener.onRewardedVideoAvailabilityChanged(false);
            }
        }

        @Override
        public void onInterstitialOpen() {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " rewardedVideoListener");
            mListener.onRewardedVideoAdOpened();
        }

        @Override
        public void onInterstitialAdRewarded(String demandSourceId, int amount) {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " rewardedVideoListener demandSourceId=" + demandSourceId + " amount=" + amount);
            mListener.onRewardedVideoAdRewarded();
        }

        @Override
        public void onInterstitialClose() {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " rewardedVideoListener");
            mListener.onRewardedVideoAdClosed();
        }

        @Override
        public void onInterstitialShowSuccess() {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " rewardedVideoListener");
            if (!mIsRvDemandOnly) {
                mListener.onRewardedVideoAvailabilityChanged(false);
            }
        }

        @Override
        public void onInterstitialShowFailed(String description) {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + "rewardedVideoListener " + description);
            mListener.onRewardedVideoAdShowFailed(ErrorBuilder.buildShowFailedError(IronSourceConstants.REWARDED_VIDEO_AD_UNIT, description));
        }

        @Override
        public void onInterstitialClick() {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " rewardedVideoListener");
            mListener.onRewardedVideoAdClicked();
        }

        @Override
        public void onInterstitialEventNotificationReceived(String eventName, JSONObject extData) {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " rewardedVideoListener eventName=" + eventName);
            mListener.onRewardedVideoAdVisible();
        }
    }

    private class IronSourceInterstitialListener implements OnInterstitialListener {
        private String mDemandSourceName;
        private InterstitialSmashListener mListener;

        IronSourceInterstitialListener(InterstitialSmashListener listener, String demandSourceName) {
            mDemandSourceName = demandSourceName;
            mListener = listener;
        }

        @Override
        public void onInterstitialInitSuccess() {
            //Deprecated - Old SuperSonicAds API callback
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " interstitialListener");

        }

        @Override
        public void onInterstitialInitFailed(String description) {
            //Deprecated - Old SuperSonicAds API callback
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " interstitialListener");

        }

        @Override
        public void onInterstitialLoadSuccess() {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " interstitialListener");
            mListener.onInterstitialAdReady();
        }

        @Override
        public void onInterstitialLoadFailed(String description) {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " interstitialListener " + description);
            mListener.onInterstitialAdLoadFailed(ErrorBuilder.buildLoadFailedError(description));
        }

        @Override
        public void onInterstitialOpen() {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " interstitialListener");
            mListener.onInterstitialAdOpened();
        }

        @Override
        public void onInterstitialAdRewarded(String demandSourceId, int amount) {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " interstitialListener demandSourceId=" + demandSourceId + " amount=" + amount);
        }

        @Override
        public void onInterstitialClose() {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " interstitialListener");
            mListener.onInterstitialAdClosed();
        }

        @Override
        public void onInterstitialShowSuccess() {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " interstitialListener");
            mListener.onInterstitialAdShowSucceeded();
        }

        @Override
        public void onInterstitialShowFailed(String description) {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " interstitialListener " + description);
            mListener.onInterstitialAdShowFailed(ErrorBuilder.buildShowFailedError(IronSourceConstants.INTERSTITIAL_AD_UNIT, description));
        }

        @Override
        public void onInterstitialClick() {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " interstitialListener");
            mListener.onInterstitialAdClicked();
        }

        @Override
        public void onInterstitialEventNotificationReceived(String eventName, JSONObject extData) {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " interstitialListener eventName=" + eventName);
            mListener.onInterstitialAdVisible();
        }
    }

    private class IronSourceBannerListener implements OnBannerListener {
        private String mDemandSourceName;
        private BannerSmashListener mListener;

        IronSourceBannerListener(BannerSmashListener listener, String demandSourceName) {
            mDemandSourceName = demandSourceName;
            mListener = listener;
        }

        @Override
        public void onBannerInitSuccess() {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " bannerListener");
            mListener.onBannerInitSuccess();

        }

        @Override
        public void onBannerInitFailed(String description) {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " bannerListener");
            mListener.onBannerInitFailed(new IronSourceError(IronSourceError.ERROR_BN_INSTANCE_INIT_ERROR, description));
        }

        @Override
        public void onBannerClick() {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " bannerListener");
            mListener.onBannerAdClicked();
        }

        @Override
        public void onBannerLoadSuccess() {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " bannerListener");
            mIsAlreadyShowing = true;

            if (mIsnAdView != null && mIsnAdView.getAdViewSize() != null) {
                FrameLayout.LayoutParams layoutParams = new FrameLayout.LayoutParams(mIsnAdView.getAdViewSize().getWidth(), mIsnAdView.getAdViewSize().getHeight());
                layoutParams.gravity = Gravity.CENTER;
                mListener.onBannerAdLoaded(mIsnAdView, layoutParams);
                mListener.onBannerAdShown();
            }

        }


        @Override
        public void onBannerLoadFail(String description) {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " bannerListener");
            mListener.onBannerAdLoadFailed(ErrorBuilder.buildLoadFailedError(getProviderName() + " load failed - error = " + description));
        }
    }

    // ********** Helpers **********

    private void loadAdInternal(String demandSourceName, String serverData, boolean isDemandOnlyForRv, boolean isBidder, boolean isRewarded) throws Exception {
        HashMap<String, String> loadConfig = new HashMap<>();
        if (serverData != null) {
            loadConfig.put(ADM_KEY, AuctionDataUtils.getInstance().getAdmFromServerData(serverData));
            Map<String, String> auctionResponseServerDataParams = AuctionDataUtils.getInstance().getAuctionResponseServerDataParams(serverData);
            loadConfig.putAll(auctionResponseServerDataParams);
        }
        IronSourceAdInstance adInstance = getAdInstance(demandSourceName, isDemandOnlyForRv, isBidder, isRewarded);
        printInstanceExtraParams(loadConfig);
        IronLog.ADAPTER_API.verbose("demandSourceName=" + adInstance.getName());
        IronSourceNetwork.loadAd(ContextProvider.getInstance().getCurrentActiveActivity(), adInstance, loadConfig);
    }

    private void showAdInternal(IronSourceAdInstance adInstance, int adUnit) throws Exception {
        int sessionDepth = SessionDepthManager.getInstance().getSessionDepth(adUnit);
        Map<String, String> showParams = new HashMap<>();
        showParams.put(Constants.RequestParameters.SESSION_DEPTH, String.valueOf(sessionDepth));
        IronLog.ADAPTER_API.verbose("demandSourceName=" + adInstance.getName() + " showParams=" + showParams);
        IronSourceNetwork.showAd(adInstance, showParams);

    }

    // get ad instance for iab, traditional, demand only, for the first time will create the ad
    private IronSourceAdInstance getAdInstance(String demandSourceName, boolean isDemandOnlyForRv, boolean isBidder, boolean isRewarded) {
        IronSourceAdInstance adInstance = isRewarded ?
                mDemandSourceToRvAd.get(demandSourceName) :
                mDemandSourceToISAd.get(demandSourceName);

        if (adInstance == null) {
            IronLog.ADAPTER_API.verbose("creating ad instance for " + demandSourceName + " isDemandOnlyForRv=" + isDemandOnlyForRv + " isBidder=" + isBidder + " isRewarded=" + isRewarded);
            IronSourceAdInstanceBuilder builder;
            if (isRewarded) {
                builder = new IronSourceAdInstanceBuilder(demandSourceName, new IronSourceRewardedVideoListener(mDemandSourceToRvSmash.get(demandSourceName), demandSourceName, isDemandOnlyForRv))
                        .setExtraParams(getInitParams());
                builder.setRewarded();
            } else {
                builder = new IronSourceAdInstanceBuilder(demandSourceName, new IronSourceInterstitialListener(mDemandSourceToISSmash.get(demandSourceName), demandSourceName))
                        .setExtraParams(getInitParams());
            }

            if (isBidder) {
                builder.setInAppBidding();
            }
            adInstance = builder.build();
            if (isRewarded) {
                mDemandSourceToRvAd.put(demandSourceName, adInstance);
            } else {
                mDemandSourceToISAd.put(demandSourceName, adInstance);
            }
        }
        return adInstance;
    }

    private ISNAdView createBanner(final Activity activity, final ISBannerSize bannerSize, BannerSmashListener listener) {
        ISNAdView adView = null;

        final String BANNER = "BANNER",
                LARGE = "LARGE",
                RECTANGLE = "RECTANGLE",
                SMART = "SMART",
                CUSTOM = "CUSTOM";

        final int STANDARD_SCREEN_BANNER_WIDTH = 320,
                STANDARD_BANNER_SIZE_HEIGHT = 50,
                LARGE_SCREEN_BANNER_WIDTH = 728,
                LARGE_BANNER_SIZE_HEIGHT = 90;

        String sizeLabel = bannerSize.getDescription();

        int widthDp = STANDARD_SCREEN_BANNER_WIDTH;
        int heightDp = STANDARD_BANNER_SIZE_HEIGHT;

        switch (sizeLabel) {
            case BANNER: {
                widthDp = STANDARD_SCREEN_BANNER_WIDTH;
                heightDp = STANDARD_BANNER_SIZE_HEIGHT;
                break;
            }
            case LARGE: {
                widthDp = STANDARD_SCREEN_BANNER_WIDTH;
                heightDp = LARGE_BANNER_SIZE_HEIGHT;
                break;
            }
            case SMART: {
                boolean isLargeScreen = AdapterUtils.isLargeScreen(activity);
                widthDp = isLargeScreen ? LARGE_SCREEN_BANNER_WIDTH : STANDARD_SCREEN_BANNER_WIDTH;
                heightDp = isLargeScreen ? LARGE_BANNER_SIZE_HEIGHT : STANDARD_BANNER_SIZE_HEIGHT;
                break;
            }
            case CUSTOM: {
                widthDp = STANDARD_SCREEN_BANNER_WIDTH;
                heightDp = bannerSize.getHeight();
                if (heightDp != STANDARD_BANNER_SIZE_HEIGHT && heightDp != LARGE_BANNER_SIZE_HEIGHT) { //Limitation for CUSTOM banner size (JIRA UC-460)
                    if (listener != null) {
                        listener.onBannerAdLoadFailed(ErrorBuilder.unsupportedBannerSize(getProviderName()));
                    }
                    return null;
                }
                break;
            }
            case RECTANGLE:
            default: {
                if (listener != null) {
                    listener.onBannerAdLoadFailed(ErrorBuilder.unsupportedBannerSize(getProviderName()));
                }
                return null;
            }
        }
        int widthPixels = AdapterUtils.dpToPixels(activity, widthDp);
        int heightPixels = AdapterUtils.dpToPixels(activity, heightDp);
        ISAdSize bannerAdSize = new ISAdSize(widthPixels, heightPixels, sizeLabel);

        try {
            adView = IronSourceNetwork.createBanner(activity, bannerAdSize);
        } catch (Exception e) {
            String errorString = IronSourceConstants.BANNER_AD_UNIT + " Load Fail, " + getProviderName() + " - " + e.getMessage();
            IronSourceError error = ErrorBuilder.buildLoadFailedError(errorString);
            listener.onBannerAdLoadFailed(error);
        }

        return adView;
    }

    @Override
    public void onOWShowSuccess(String placementId) {
        IronLog.ADAPTER_CALLBACK.verbose(getProviderName() + ":onOWShowSuccess(placementId:" + placementId + ")");

        if (mOfferwallListener != null) {
            mOfferwallListener.onOfferwallOpened();
        }
    }

    @Override
    public void onOWShowFail(String description) {
        IronLog.ADAPTER_CALLBACK.verbose(getProviderName());

        if (mOfferwallListener != null) {
            IronSourceError ise = ErrorBuilder.buildGenericError(description);
            mOfferwallListener.onOfferwallShowFailed(ise);
        }
    }

    @Override
    public boolean onOWAdCredited(
            int credits,
            int totalCredits,
            boolean totalCreditsFlag) {

        IronLog.ADAPTER_CALLBACK.verbose(getProviderName());

        return mOfferwallListener != null && mOfferwallListener.onOfferwallAdCredited(
                credits,
                totalCredits,
                totalCreditsFlag
        );
    }

    @Override
    public void onGetOWCreditsFailed(String description) {
        IronLog.ADAPTER_CALLBACK.verbose(getProviderName());

        if (mOfferwallListener != null) {
            IronSourceError ise = ErrorBuilder.buildGenericError(description);
            mOfferwallListener.onGetOfferwallCreditsFailed(ise);
        }
    }

    @Override
    public void onOWAdClosed() {
        IronLog.ADAPTER_CALLBACK.verbose(getProviderName());

        if (mOfferwallListener != null) {
            mOfferwallListener.onOfferwallClosed();
        }
    }

    @Override
    public void onOfferwallInitSuccess() {
        IronLog.ADAPTER_CALLBACK.verbose(getProviderName());

        if (mOfferwallListener != null) {
            mOfferwallListener.onOfferwallAvailable(true);
        }
    }

    @Override
    public void onOfferwallInitFail(String description) {
        IronLog.ADAPTER_CALLBACK.verbose(getProviderName());

        if (mOfferwallListener != null) {
            IronSourceError ise = ErrorBuilder.buildGenericError(description);
            mOfferwallListener.onOfferwallAvailable(
                    false,
                    ise
            );
        }
    }

    @Override
    public void onOfferwallEventNotificationReceived(String eventName, JSONObject extData) {
        if (extData != null) {
            IronLog.ADAPTER_CALLBACK.verbose(getProviderName());
        }
    }

    private String getDemandSourceName(JSONObject config) {

        if (!TextUtils.isEmpty(config.optString(DEMAND_SOURCE_NAME))) {
            return config.optString(DEMAND_SOURCE_NAME);
        } else {
            return getProviderName();
        }
    }


    // ********** Helpers **********

    private boolean isValidMetaData(String key, String value) {
        // In case the formatted CCPA Value is empty it means we didn't get a valid boolean string value and therefore we don't want to pass it to the network
        if (key.equals(MetaDataConstants.META_DATA_CCPA_KEY)) {
            return MetaDataUtils.isValidCCPAMetaData(key, value);
        }

        return true;
    }

    @Override
    protected void setMetaData(String key, String value) {
        if (!mDidInitSdk.get()) {
            IronLog.ADAPTER_API.verbose("key=" + key + ", value=" + value);

            if (!isValidMetaData(key, value)) {
                IronLog.ADAPTER_API.verbose("MetaData not valid");

                return;
            }

            JSONObject metaDataJson = new JSONObject();
            try {
                metaDataJson.put(key, value);
                IronSourceNetwork.updateMetadata(metaDataJson);
            } catch (JSONException e) {
                IronLog.ADAPTER_API.error("error - " + e);

                e.printStackTrace();
            }

        }
    }

    private void printInstanceExtraParams(Map<String, String> params) {
        if (params != null && params.size() > 0) {
            IronLog.ADAPTER_API.verbose("instance extra params:");
            for (String key : params.keySet()) {
                IronLog.ADAPTER_API.verbose(key + "=" + params.get(key));

            }
        }
    }

    // ability to override the adapter flag with a platform configuration from the instance in order to support load while show for bidder instances
    @Override
    public LoadWhileShowSupportState getLoadWhileShowSupportState(JSONObject mAdUnitSettings) {
        LoadWhileShowSupportState loadWhileShowSupportState = mLWSSupportState;
        if (mAdUnitSettings != null) {
            boolean isSupportedLWS = mAdUnitSettings.optBoolean(LWS_SUPPORT_STATE);
            if (isSupportedLWS) {
                loadWhileShowSupportState = LoadWhileShowSupportState.LOAD_WHILE_SHOW_BY_INSTANCE;
            }
        }
        return loadWhileShowSupportState;
    }
}