package com.ironsource.adapters.ironsource;

import android.app.Activity;
import android.text.TextUtils;

import com.ironsource.mediationsdk.AbstractAdapter;
import com.ironsource.mediationsdk.AuctionDataUtils;
import com.ironsource.mediationsdk.BuildConfig;
import com.ironsource.mediationsdk.IronSourceObject;
import com.ironsource.mediationsdk.logger.IronLog;
import com.ironsource.mediationsdk.logger.IronSourceError;
import com.ironsource.mediationsdk.logger.IronSourceLogger;
import com.ironsource.mediationsdk.metadata.MetaDataConstants;
import com.ironsource.mediationsdk.metadata.MetaDataUtils;
import com.ironsource.mediationsdk.sdk.InterstitialSmashListener;
import com.ironsource.mediationsdk.sdk.RewardedVideoSmashListener;
import com.ironsource.mediationsdk.utils.ContextProvider;
import com.ironsource.mediationsdk.utils.ErrorBuilder;
import com.ironsource.mediationsdk.utils.IronSourceConstants;
import com.ironsource.mediationsdk.utils.IronSourceUtils;
import com.ironsource.mediationsdk.utils.SessionDepthManager;
import com.ironsource.sdk.IronSourceAdInstance;
import com.ironsource.sdk.IronSourceAdInstanceBuilder;
import com.ironsource.sdk.IronSourceNetwork;
import com.ironsource.sdk.constants.Constants;
import com.ironsource.sdk.listeners.OnInterstitialListener;
import com.ironsource.sdk.utils.SDKUtils;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;

/*
This adapter is responsible for addressing several flows
    - mediation (reward video is loaded automatically)
            -  non bidder - reward video and interstitial
            -  bidder - reward video and interstitial
    - demand only  (reward video is loaded manually)
            - non bidder - reward video and interstitial
            - demand only - bidder - reward video and interstitial
*/
public class IronSourceAdapter extends AbstractAdapter implements ContextProvider.ContextLifeCycleListener {
    private final static String VERSION = BuildConfig.VERSION_NAME;
    private final static int IS_LOAD_EXCEPTION = 1000;
    private final static int IS_SHOW_EXCEPTION = 1001;
    private final static int RV_LOAD_EXCEPTION = 1002;
    private final static int RV_SHOW_EXCEPTION = 1003;

    private final String DYNAMIC_CONTROLLER_URL = "controllerUrl";
    private final String DYNAMIC_CONTROLLER_DEBUG_MODE = "debugMode";
    private final String DYNAMIC_CONTROLLER_CONFIG = "controllerConfig";

    private final String APPLICATION_USER_GENDER = "applicationUserGender";
    private final String APPLICATION_USER_AGE_GROUP = "applicationUserAgeGroup";
    private final String SESSION_ID = "sessionid";
    private final String SDK_PLUGIN_TYPE = "SDKPluginType";
    private final String CUSTOM_SEGMENT = "custom_Segment";
    private final String ADM_KEY = "adm";

    private final String DEMAND_SOURCE_NAME = "demandSourceName";

    private ConcurrentHashMap<String, IronSourceAdInstance> mDemandSourceToISAd;
    private ConcurrentHashMap<String, IronSourceAdInstance> mDemandSourceToRvAd;

    private ConcurrentHashMap<String, RewardedVideoSmashListener> mDemandSourceToRvSmash;
    private ConcurrentHashMap<String, InterstitialSmashListener> mDemandSourceToISSmash;


    private static AtomicBoolean mDidInitSdk = new AtomicBoolean(false);
    private static String userAgeGroup;
    private static String userGender;
    private static String mediationSegment;


    public static IronSourceAdapter startAdapter(String providerName) {
        return new IronSourceAdapter(providerName);
    }

    private IronSourceAdapter(String providerName) {
        super(providerName);
        IronLog.INTERNAL.verbose(providerName + ": new instance");
        mDemandSourceToISAd = new ConcurrentHashMap<>();
        mDemandSourceToRvAd = new ConcurrentHashMap<>();
        mDemandSourceToRvSmash = new ConcurrentHashMap<>();
        mDemandSourceToISSmash = new ConcurrentHashMap<>();
        userAgeGroup = null;
        userGender = null;
        mediationSegment = null;

        // listen to life cycle events
        ContextProvider.getInstance().registerLifeCycleListener(this);
    }

    @Override
    public String getVersion() {
        return VERSION;
    }

    @Override
    public String getCoreSDKVersion() {
        return SDKUtils.getSDKVersion();
    }

    // ********** Base **********

    @Override
    public void onPause(Activity activity) {
        IronLog.ADAPTER_API.verbose("IronSourceNetwork.onPause");
        IronSourceNetwork.onPause(activity);
    }

    @Override
    public void onResume(Activity activity) {
        IronLog.ADAPTER_API.verbose("IronSourceNetwork.onResume");
        IronSourceNetwork.onResume(activity);
    }

    protected void setConsent(boolean consent) {
        IronLog.ADAPTER_API.verbose("(" + (consent ? "true" : "false") + ")" );

        JSONObject consentParams = new JSONObject();
        try {
            consentParams.put(Constants.RequestParameters.CONSENT, consent ? "1" : "0");
        } catch (JSONException e) {
            IronLog.ADAPTER_API.error("exception " + e.getMessage());

            return;
        }

        IronSourceNetwork.updateMetadata(consentParams);
    }

    @Override
    public void setAge(int age) {
        IronLog.INTERNAL.verbose(": " + age);

        if (age >= 13 && age <= 17) {
            userAgeGroup = "1";
        } else if (age >= 18 && age <= 20) {
            userAgeGroup = "2";
        } else if (age >= 21 && age <= 24) {
            userAgeGroup = "3";
        } else if (age >= 25 && age <= 34) {
            userAgeGroup = "4";
        } else if (age >= 35 && age <= 44) {
            userAgeGroup = "5";
        } else if (age >= 45 && age <= 54) {
            userAgeGroup = "6";
        } else if (age >= 55 && age <= 64) {
            userAgeGroup = "7";
        } else if (age > 65 && age <= 120) {
            userAgeGroup = "8";
        } else {
            userAgeGroup = "0";
        }
    }

    @Override
    public void setGender(String gender) {
        IronLog.INTERNAL.verbose(gender);

        userGender = gender;
    }

    @Override
    public void setMediationSegment(String segment) {
        mediationSegment = segment;
    }

    private HashMap<String, String> getInitParams() {
        HashMap<String, String> params = new HashMap<>();

        if (!TextUtils.isEmpty(userAgeGroup)) {
            params.put(APPLICATION_USER_AGE_GROUP, userAgeGroup);
        }

        if (!TextUtils.isEmpty(userGender)) {
            params.put(APPLICATION_USER_GENDER, userGender);
        }

        String pluginType = getPluginType();
        if (!TextUtils.isEmpty(pluginType)) {
            params.put(SDK_PLUGIN_TYPE, pluginType);
        }

        if (!TextUtils.isEmpty(mediationSegment)) {
            params.put(CUSTOM_SEGMENT, mediationSegment);
        }

        if(!TextUtils.isEmpty(IronSourceObject.getInstance().getSessionId())) {
            params.put(SESSION_ID, IronSourceObject.getInstance().getSessionId());
        }

        return params;
    }

    public JSONObject getPlayerBiddingData() {
        IronLog.ADAPTER_API.verbose("");
        JSONObject biddingData = new JSONObject();
        JSONObject token = null;

        try {
            token = IronSourceNetwork.getRawToken(ContextProvider.getInstance().getApplicationContext());
        }
        // In case there is an exception while trying to fetch the token we wouldn't want to return a partial token
        // so instead we will return null. Since this is Demand Only related code we will add it on the IronSource adapter layer
        catch (Exception e) {
            IronLog.ADAPTER_API.error("getRawToken exception: " + e.getLocalizedMessage());
        }

        if (token != null) {
            biddingData = token;
        }
        else {
            IronLog.ADAPTER_API.error("Player's bidding token is null");
        }

        return biddingData;
    }

    // ********** Init SDK **************

    @Override
    public void earlyInit (final String appKey, final String userId, final JSONObject config) {
        if(appKey != null){
            IronSourceUtils.sendAutomationLog(getDemandSourceName(config) + ": " + "earlyInit");
            initSDK(appKey, config);
        }
        else {
            IronLog.ADAPTER_API.error("Appkey is null for early init");
        }
    }

    // ********** Interstitial **********

    @Override
    public Map<String, Object> getInterstitialBiddingData(JSONObject config) {
        IronLog.ADAPTER_API.verbose("");

        Map<String, Object> ret = new HashMap<>();
        String token = IronSourceNetwork.getToken(ContextProvider.getInstance().getApplicationContext());

        if (token != null) {
            ret.put(IronSourceConstants.IRONSOURCE_BIDDING_TOKEN_KEY, token);
        }
        else {
            IronLog.ADAPTER_API.error("IS bidding token is null");
            ret.put(IronSourceConstants.IRONSOURCE_BIDDING_TOKEN_KEY, "");
        }

        return ret;
    }

    // used for bidders in the mediation
    @Override
    public void initInterstitialForBidding(String appKey, String userId, JSONObject config, InterstitialSmashListener listener) {
        String demandSourceName = getDemandSourceName(config);
        IronLog.INTERNAL.verbose(demandSourceName + ": " + "demandSourceName=" + demandSourceName);
        initInterstitialInternal(appKey, config, listener, demandSourceName);
    }

    // used for non bidders in the mediation and bidders in demand only
    @Override
    public void initInterstitial(String appKey, String userId, JSONObject config, InterstitialSmashListener listener) {
        String demandSourceName = getDemandSourceName(config);
        IronLog.INTERNAL.verbose(demandSourceName + ": " + "demandSourceName=" + demandSourceName);
        initInterstitialInternal(appKey, config, listener, demandSourceName);
    }


    private void initInterstitialInternal(String appKey, JSONObject config, InterstitialSmashListener listener, String demandSourceName) {
        initSDK(appKey, config);
        mDemandSourceToISSmash.put(demandSourceName,listener);
        listener.onInterstitialInitSuccess();
    }

    // used for bidders in the mediation and demand only
    @Override
    public void loadInterstitialForBidding(final JSONObject config, final InterstitialSmashListener listener, final String serverData) {
        IronLog.ADAPTER_API.verbose(getDemandSourceName(config));
        try {
            loadAdInternal(getDemandSourceName(config), serverData, false, true, false);
        } catch (Exception e) {
            IronLog.ADAPTER_API.error("for bidding exception " + e.getMessage());
            listener.onInterstitialAdLoadFailed(new IronSourceError(IS_LOAD_EXCEPTION, e.getMessage()));
        }

    }

    // used for non bidders in the mediation and demand only
    @Override
    public void loadInterstitial(JSONObject config, InterstitialSmashListener listener) {
        IronLog.ADAPTER_API.verbose(getDemandSourceName(config));
        try {
            loadAdInternal(getDemandSourceName(config), null, false, false, false);
        } catch (Exception e) {
            IronLog.ADAPTER_API.error("exception " + e.getMessage());
            listener.onInterstitialAdLoadFailed(new IronSourceError(IS_LOAD_EXCEPTION, e.getMessage()));
        }
    }

    @Override
    public void showInterstitial(JSONObject config, InterstitialSmashListener listener) {
        IronLog.ADAPTER_API.verbose(getDemandSourceName(config));
        try {
            IronSourceAdInstance adInstance = mDemandSourceToISAd.get(getDemandSourceName(config));
            showAdInternal(adInstance, SessionDepthManager.INTERSTITIAL);
        } catch (Exception e) {
            IronLog.ADAPTER_API.error("exception " + e.getMessage());
            listener.onInterstitialAdShowFailed(new IronSourceError(IS_SHOW_EXCEPTION, e.getMessage()));
        }
    }

    @Override
    public boolean isInterstitialReady(JSONObject config) {
        IronSourceAdInstance adInstance = mDemandSourceToISAd.get(getDemandSourceName(config));
        return adInstance != null && IronSourceNetwork.isAdAvailableForInstance(adInstance);
    }

    // ********** RewardedVideo **********

    @Override
    public Map<String, Object> getRewardedVideoBiddingData(JSONObject config) {
        IronLog.ADAPTER_API.verbose("getRewardedVideoBiddingData");
        Map<String, Object> biddingData = new HashMap<>();
        String token = IronSourceNetwork.getToken(ContextProvider.getInstance().getApplicationContext());

        if (token != null) {
            biddingData.put(IronSourceConstants.IRONSOURCE_BIDDING_TOKEN_KEY, token);
        }
        else {
            IronLog.ADAPTER_API.error("RV bidding token is null");
            biddingData.put(IronSourceConstants.IRONSOURCE_BIDDING_TOKEN_KEY, "");
        }
        return biddingData;
    }

    // used for bidders in the mediation
    @Override
    public void initRewardedVideoForBidding(String appKey, String userId, JSONObject config, RewardedVideoSmashListener listener) {
        String demandSourceName = getDemandSourceName(config);
        IronLog.INTERNAL.verbose(demandSourceName + ": demandSourceName=" + demandSourceName);

        initRewardedVideoInternal(appKey, config, listener, demandSourceName);
        listener.onRewardedVideoInitSuccess();
    }

    // used for bidders and non bidders in demand only
    @Override
    public void initRewardedVideoForDemandOnly(String appKey, String userId, JSONObject config, RewardedVideoSmashListener listener) {
        String demandSourceName = getDemandSourceName(config);
        IronLog.INTERNAL.verbose(demandSourceName + ": demandSourceName=" + demandSourceName);

        initRewardedVideoInternal(appKey, config, listener, demandSourceName);
    }

    // used for non bidders in the mediation
    @Override
    public void initRewardedVideo(String appKey, String userId, JSONObject config, RewardedVideoSmashListener listener) {
        String demandSourceName = getDemandSourceName(config);
        IronLog.INTERNAL.verbose(demandSourceName + ": demandSourceName=" + demandSourceName);
        initRewardedVideoInternal(appKey, config, listener, demandSourceName);
        fetchRewardedVideoForAutomaticLoad(config, listener);
    }

    private void initRewardedVideoInternal(String appKey, JSONObject config, RewardedVideoSmashListener listener, String demandSourceName) {
        initSDK(appKey, config);
        mDemandSourceToRvSmash.put(demandSourceName,listener);
    }


    // get ad instance for iab, traditional, demand only, for the first time will create the ad
    private IronSourceAdInstance getAdInstance(String demandSourceName, boolean isDemandOnlyForRv, boolean isBidder, boolean isRewarded) {
        IronSourceAdInstance adInstance = isRewarded ?
                mDemandSourceToRvAd.get(demandSourceName):
                mDemandSourceToISAd.get(demandSourceName);

        if (adInstance == null) {
            IronLog.ADAPTER_API.verbose("creating ad instance for " + demandSourceName + " isDemandOnlyForRv=" + isDemandOnlyForRv + " isBidder=" + isBidder + " isRewarded=" + isRewarded);
            IronSourceAdInstanceBuilder builder;
            if (isRewarded) {
                builder = new IronSourceAdInstanceBuilder(demandSourceName, new IronSourceRewardedVideoListener(mDemandSourceToRvSmash.get(demandSourceName), demandSourceName, isDemandOnlyForRv))
                        .setExtraParams(getInitParams());
                builder.setRewarded();
            }
            else {
                builder = new IronSourceAdInstanceBuilder(demandSourceName,new IronSourceInterstitialListener(mDemandSourceToISSmash.get(demandSourceName), demandSourceName))
                        .setExtraParams(getInitParams());
            }

            if (isBidder) {
                builder.setInAppBidding();
            }
            adInstance = builder.build();
            if (isRewarded) {
                mDemandSourceToRvAd.put(demandSourceName,adInstance);
            }
            else {
                mDemandSourceToISAd.put(demandSourceName,adInstance);
            }
        }
        return adInstance;
    }
    // TODO unify API calls

    // used for non bidders in the mediation
    @Override
    public void fetchRewardedVideoForAutomaticLoad(JSONObject config, RewardedVideoSmashListener listener) {
        String demandSourceName = getDemandSourceName(config);
        IronLog.ADAPTER_API.verbose(demandSourceName);

        try {
            loadAdInternal(demandSourceName,null, false, false, true);
        } catch (Exception e) {
            IronLog.ADAPTER_API.error("exception " + e.getMessage());

            listener = mDemandSourceToRvSmash.get(demandSourceName);
            if (listener != null) {
                IronLog.ADAPTER_API.error("exception " + e.getMessage());
                listener.onRewardedVideoLoadFailed(new IronSourceError(RV_LOAD_EXCEPTION, e.getMessage()));
                listener.onRewardedVideoAvailabilityChanged(false);
            }
        }
    }

    // used for non bidders in demand only
    @Override
    public void loadRewardedVideoForDemandOnly(JSONObject config, RewardedVideoSmashListener listener) {
        IronLog.ADAPTER_API.verbose(getDemandSourceName(config));

        try {
            loadAdInternal(getDemandSourceName(config), null, true, false, true);
        } catch (Exception e) {
            IronLog.ADAPTER_API.error("exception " + e.getMessage());
            listener.onRewardedVideoLoadFailed(new IronSourceError(RV_LOAD_EXCEPTION, e.getMessage()));
        }
    }

    // used for bidders in demand only
    @Override
    public void loadRewardedVideoForDemandOnlyForBidding(JSONObject config, RewardedVideoSmashListener listener, String serverData) {
        IronLog.ADAPTER_API.verbose(getDemandSourceName(config));

        try {
            loadAdInternal(getDemandSourceName(config), serverData, true, true, true);
        } catch (Exception e) {
            IronLog.ADAPTER_API.error("exception " + e.getMessage());
            listener.onRewardedVideoLoadFailed(new IronSourceError(RV_LOAD_EXCEPTION, e.getMessage()));
        }
    }

    // used  bidders in the mediation
    @Override
    public void loadRewardedVideoForBidding(JSONObject config, final RewardedVideoSmashListener listener, final String serverData) {
        IronLog.ADAPTER_API.verbose(getDemandSourceName(config));

        try {
            loadAdInternal(getDemandSourceName(config), serverData, false, true, true);
        } catch (Exception e) {
            IronLog.ADAPTER_API.error("exception " + e.getMessage());
            listener.onRewardedVideoLoadFailed(new IronSourceError(RV_LOAD_EXCEPTION, e.getMessage()));
            listener.onRewardedVideoAvailabilityChanged(false);
        }
    }


    private void loadAdInternal(String demandSourceName, String serverData, boolean isDemandOnlyForRv, boolean isBidder, boolean isRewarded) throws Exception {
        HashMap<String, String> loadConfig = new HashMap<>();
        if (serverData != null) {
            loadConfig.put(ADM_KEY, AuctionDataUtils.getInstance().getAdmFromServerData(serverData));
            Map<String, String> auctionResponseServerDataParams = AuctionDataUtils.getInstance().getAuctionResponseServerDataParams(serverData);
            loadConfig.putAll(auctionResponseServerDataParams);
        }
        IronSourceAdInstance adInstance = getAdInstance(demandSourceName, isDemandOnlyForRv, isBidder, isRewarded);
        printInstanceExtraParams(loadConfig);
        IronLog.ADAPTER_API.verbose("demandSourceName=" + adInstance.getName());
        IronSourceNetwork.loadAd(adInstance, loadConfig);
    }

    @Override
    public void showRewardedVideo(JSONObject config, RewardedVideoSmashListener listener) {
        try {
            IronSourceAdInstance adInstance = mDemandSourceToRvAd.get(getDemandSourceName(config));
            showAdInternal(adInstance, SessionDepthManager.REWARDEDVIDEO);
        } catch (Exception e) {
            IronLog.ADAPTER_API.error("exception " + e.getMessage());
            listener.onRewardedVideoAdShowFailed(new IronSourceError(RV_SHOW_EXCEPTION, e.getMessage()));
        }

    }


    private void showAdInternal(IronSourceAdInstance adInstance, int adUnit) throws Exception {
        int sessionDepth = SessionDepthManager.getInstance().getSessionDepth(adUnit);
        Map<String, String> showParams = new HashMap<>();
        showParams.put(Constants.RequestParameters.SESSION_DEPTH, String.valueOf(sessionDepth));
        IronLog.ADAPTER_API.verbose("demandSourceName=" + adInstance.getName() + " showParams=" + showParams);
        IronSourceNetwork.showAd(adInstance, showParams);

    }
    @Override
    public boolean isRewardedVideoAvailable(JSONObject config) {
        IronSourceAdInstance adInstance = mDemandSourceToRvAd.get(getDemandSourceName(config));
        return adInstance != null && IronSourceNetwork.isAdAvailableForInstance(adInstance);
    }

    private void initSDK(String appKey, JSONObject config) {

        if (mDidInitSdk.compareAndSet(false,true)) {
            // using the mediation user id and not the user id provided by the adapter that does not include the default value if not set by the publisher
            String userId = IronSourceUtils.getMediationUserId();
            int logLevel = config.optInt(DYNAMIC_CONTROLLER_DEBUG_MODE, IronSourceLogger.IronSourceLogLevel.VERBOSE);
            if (isAdaptersDebugEnabled()) {
                logLevel = IronSourceLogger.IronSourceLogLevel.ERROR;
            }
            IronLog.ADAPTER_API.verbose("etting debug mode to " + logLevel);
            SDKUtils.setDebugMode(logLevel);

            SDKUtils.setControllerUrl(config.optString(DYNAMIC_CONTROLLER_URL));
            IronLog.ADAPTER_API.verbose("IronSourceNetwork setting controller url to  "+ config.optString(DYNAMIC_CONTROLLER_URL));

            SDKUtils.setControllerConfig(config.optString(DYNAMIC_CONTROLLER_CONFIG));
            IronLog.ADAPTER_API.verbose("IronSourceNetwork setting controller config to  "+ config.optString(DYNAMIC_CONTROLLER_CONFIG));

            HashMap<String, String> initParams = getInitParams();
            IronLog.ADAPTER_API.verbose("with appKey=" + appKey+ " userId=" + userId+ " parameters "+ initParams);
            IronSourceNetwork.initSDK(ContextProvider.getInstance().getCurrentActiveActivity(), appKey, userId, initParams);
        }
    }

    // ********** Callbacks **********

    private class IronSourceInterstitialListener implements OnInterstitialListener {
        private String mDemandSourceName;
        private InterstitialSmashListener mListener;

        IronSourceInterstitialListener (InterstitialSmashListener listener, String demandSourceName) {
            mDemandSourceName = demandSourceName;
            mListener = listener;
        }

        @Override
        public void onInterstitialInitSuccess() {
            //Deprecated - Old SuperSonicAds API callback
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " interstitialListener");

        }

        @Override
        public void onInterstitialInitFailed(String description) {
            //Deprecated - Old SuperSonicAds API callback
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " interstitialListener");

        }

        @Override
        public void onInterstitialLoadSuccess() {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " interstitialListener");
            mListener.onInterstitialAdReady();
        }

        @Override
        public void onInterstitialLoadFailed(String description) {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " interstitialListener " + description);
            mListener.onInterstitialAdLoadFailed(ErrorBuilder.buildLoadFailedError(description));
        }

        @Override
        public void onInterstitialOpen() {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " interstitialListener");
            mListener.onInterstitialAdOpened();
        }

        @Override
        public void onInterstitialAdRewarded(String demandSourceId, int amount) {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " interstitialListener demandSourceId=" + demandSourceId + " amount=" + amount);
        }

        @Override
        public void onInterstitialClose() {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " interstitialListener");
            mListener.onInterstitialAdClosed();
        }

        @Override
        public void onInterstitialShowSuccess() {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " interstitialListener");
            mListener.onInterstitialAdShowSucceeded();
        }

        @Override
        public void onInterstitialShowFailed(String description) {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " interstitialListener " + description);
            mListener.onInterstitialAdShowFailed(ErrorBuilder.buildShowFailedError(IronSourceConstants.INTERSTITIAL_AD_UNIT, description));
        }

        @Override
        public void onInterstitialClick() {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " interstitialListener");
            mListener.onInterstitialAdClicked();
        }

        @Override
        public void onInterstitialEventNotificationReceived(String eventName, JSONObject extData) {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " interstitialListener eventName=" + eventName);
            mListener.onInterstitialAdVisible();
        }
    }


    private class IronSourceRewardedVideoListener implements OnInterstitialListener {
        private String mDemandSourceName;
        RewardedVideoSmashListener mListener;
        boolean mIsRvDemandOnly;

        IronSourceRewardedVideoListener (RewardedVideoSmashListener listener, String demandSourceName) {
            mDemandSourceName = demandSourceName;
            mListener = listener;
            mIsRvDemandOnly = false;
        }

        IronSourceRewardedVideoListener (RewardedVideoSmashListener listener, String demandSourceName, boolean isDemandOnly) {
            mDemandSourceName = demandSourceName;
            mListener = listener;
            mIsRvDemandOnly = isDemandOnly;
        }

        @Override
        public void onInterstitialInitSuccess() {
            //Deprecated - Old SuperSonicAds API callback
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " rewardedVideoListener");
        }

        @Override
        public void onInterstitialInitFailed(String description) {
            //Deprecated - Old SuperSonicAds API callback
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " rewardedVideoListener");
        }

        @Override
        public void onInterstitialLoadSuccess() {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " rewardedVideoListener");
            if (mIsRvDemandOnly) {
                mListener.onRewardedVideoLoadSuccess();
            }
            else {
                mListener.onRewardedVideoAvailabilityChanged(true);
            }
        }

        @Override
        public void onInterstitialLoadFailed(String description) {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " rewardedVideoListener " + description);
            mListener.onRewardedVideoLoadFailed(ErrorBuilder.buildLoadFailedError(description));
            if (!mIsRvDemandOnly) {
                mListener.onRewardedVideoAvailabilityChanged(false);
            }
        }

        @Override
        public void onInterstitialOpen() {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " rewardedVideoListener");
            mListener.onRewardedVideoAdOpened();
        }

        @Override
        public void onInterstitialAdRewarded(String demandSourceId, int amount) {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " rewardedVideoListener demandSourceId=" + demandSourceId + " amount=" + amount);
            mListener.onRewardedVideoAdRewarded();
        }

        @Override
        public void onInterstitialClose() {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " rewardedVideoListener");
            mListener.onRewardedVideoAdClosed();
        }

        @Override
        public void onInterstitialShowSuccess() {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " rewardedVideoListener");
            if (!mIsRvDemandOnly) {
                mListener.onRewardedVideoAvailabilityChanged(false);
            }
        }

        @Override
        public void onInterstitialShowFailed(String description) {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + "rewardedVideoListener " + description);
            mListener.onRewardedVideoAdShowFailed(ErrorBuilder.buildShowFailedError(IronSourceConstants.REWARDED_VIDEO_AD_UNIT, description));
        }

        @Override
        public void onInterstitialClick() {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " rewardedVideoListener");
            mListener.onRewardedVideoAdClicked();
        }

        @Override
        public void onInterstitialEventNotificationReceived(String eventName, JSONObject extData) {
            IronLog.ADAPTER_CALLBACK.verbose(mDemandSourceName + " rewardedVideoListener eventName=" + eventName);
            mListener.onRewardedVideoAdVisible();
        }
    }


    private String getDemandSourceName(JSONObject config) {

        if (!TextUtils.isEmpty(config.optString(DEMAND_SOURCE_NAME))) {
            return config.optString(DEMAND_SOURCE_NAME);
        }
        else {
            return getProviderName();
        }
    }


    // ********** Helpers **********

    private boolean isValidMetaData(String key, String value) {
        // In case the formatted CCPA Value is empty it means we didn't get a valid boolean string value and therefore we don't want to pass it to the network
        if (key.equals(MetaDataConstants.META_DATA_CCPA_KEY)) {
            return MetaDataUtils.isValidCCPAMetaData(key, value);
        }

        return true;
    }

    @Override
    protected void setMetaData(String key, String value) {
        if (!mDidInitSdk.get()) {
            IronLog.ADAPTER_API.verbose("key=" + key +", value=" + value);

            if (!isValidMetaData(key, value)) {
                IronLog.ADAPTER_API.verbose("MetaData not valid");

                return;
            }

            JSONObject metaDataJson = new JSONObject();
            try {
                metaDataJson.put(key, value);
                IronSourceNetwork.updateMetadata(metaDataJson);
            } catch (JSONException e) {
                IronLog.ADAPTER_API.error("error - " + e);

                e.printStackTrace();
            }

        }
    }

    private void printInstanceExtraParams(Map<String, String> params) {
        if (params != null && params.size() > 0) {
            IronLog.ADAPTER_API.verbose("instance extra params:" );
            for (String key : params.keySet()) {
                IronLog.ADAPTER_API.verbose(key+"=" + params.get(key));

            }
        }
    }
}