package com.ironsource.adapters.ironsource;

import android.app.Activity;

import com.ironsource.mediationsdk.AbstractAdapter;
import com.ironsource.mediationsdk.BuildConfig;
import com.ironsource.mediationsdk.logger.IronSourceError;
import com.ironsource.mediationsdk.logger.IronSourceLogger;
import com.ironsource.mediationsdk.logger.IronSourceLoggerManager;
import com.ironsource.mediationsdk.sdk.InterstitialSmashListener;
import com.ironsource.mediationsdk.sdk.RewardedVideoSmashListener;
import com.ironsource.mediationsdk.utils.ErrorBuilder;
import com.ironsource.mediationsdk.utils.IronSourceConstants;
import com.ironsource.mediationsdk.utils.SessionDepthManager;
import com.ironsource.sdk.IronSourceAdInstance;
import com.ironsource.sdk.IronSourceAdInstanceBuilder;
import com.ironsource.sdk.IronSourceNetwork;
import com.ironsource.sdk.constants.Constants;
import com.ironsource.sdk.listeners.OnInterstitialListener;
import com.ironsource.sdk.utils.SDKUtils;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.HashMap;
import java.util.Map;


public class IronSourceAdapter extends AbstractAdapter {
    private final static String VERSION = BuildConfig.VERSION_NAME;
    private final static int IS_LOAD_EXCEPTION = 1000;
    private final static int IS_SHOW_EXCEPTION = 1001;
    private final static int RV_LOAD_EXCEPTION = 1002;
    private final static int RV_SHOW_EXCEPTION = 1003;

    private final String DYNAMIC_CONTROLLER_URL = "controllerUrl";
    private final String DYNAMIC_CONTROLLER_DEBUG_MODE = "debugMode";
    private final String DYNAMIC_CONTROLLER_CONFIG = "controllerConfig";

    private IronSourceAdInstance mInterstitialAdInstance;
    private InterstitialSmashListener mIsSmashListener;

    private IronSourceAdInstance mRewardedVideoAdInstance;
    private RewardedVideoSmashListener mRvSmashListener;

    public static IronSourceAdapter startAdapter(String providerName) {
        return new IronSourceAdapter(providerName);
    }

    private IronSourceAdapter(String providerName) {
        super(providerName);
        mRewardedVideoAdInstance = null;
        mRvSmashListener = null;
        mInterstitialAdInstance = null;
        mIsSmashListener = null;
    }

    @Override
    public String getVersion() {
        return VERSION;
    }

    @Override
    public String getCoreSDKVersion() {
        return SDKUtils.getSDKVersion();
    }


    // ********** Base **********

    @Override
    public void onPause(Activity activity) {
        IronSourceNetwork.onPause(activity);
    }

    @Override
    public void onResume(Activity activity) {
        IronSourceNetwork.onResume(activity);
    }

    protected void setConsent(boolean consent) {
        JSONObject consentParams = new JSONObject();
        try {
            consentParams.put(Constants.RequestParameters.GDPR_CONSENT_STATUS, String.valueOf(consent));
            consentParams.put(Constants.RequestParameters.DEMAND_SOURCE_NAME, getProviderName());
        } catch (JSONException e) {
            logError("setConsent exception " + e.getMessage());
            return;
        }

        IronSourceNetwork.updateConsentInfo(consentParams);
    }

    // ********** Interstitial **********

    @Override
    public void initInterstitial(Activity activity, String appKey, String userId, JSONObject config, InterstitialSmashListener listener) {
        mIsSmashListener = listener;
        initSDK(activity, appKey, userId, config);
        mInterstitialAdInstance = new IronSourceAdInstanceBuilder(getProviderName(), interstitialListener).build();
        mIsSmashListener.onInterstitialInitSuccess();
    }

    @Override
    public void loadInterstitial(JSONObject config, InterstitialSmashListener listener) {
        try {
            IronSourceNetwork.loadAd(mInterstitialAdInstance);
        } catch (Exception e) {
            logError("loadInterstitial exception " + e.getMessage());
            mIsSmashListener.onInterstitialAdLoadFailed(new IronSourceError(IS_LOAD_EXCEPTION, e.getMessage()));
        }
    }

    @Override
    public void showInterstitial(JSONObject config, InterstitialSmashListener listener) {
        try {
            int sessionDepth = SessionDepthManager.getInstance().getSessionDepth(SessionDepthManager.INTERSTITIAL);
            Map<String, String> showParams = new HashMap<>();
            showParams.put(Constants.RequestParameters.SESSION_DEPTH, String.valueOf(sessionDepth));
            IronSourceNetwork.showAd(mInterstitialAdInstance, showParams);
        } catch (Exception e) {
            logError("showInterstitial exception " + e.getMessage());
            mIsSmashListener.onInterstitialAdShowFailed(new IronSourceError(IS_SHOW_EXCEPTION, e.getMessage()));
        }
    }

    @Override
    public boolean isInterstitialReady(JSONObject config) {
        return mInterstitialAdInstance != null && IronSourceNetwork.isAdAvailableForInstance(mInterstitialAdInstance);
    }

    // ********** RewardedVideo **********

    @Override
    public void initRewardedVideo(Activity activity, String appKey, String userId, JSONObject config, RewardedVideoSmashListener listener) {
        mRvSmashListener = listener;
        initSDK(activity, appKey, userId, config);
        mRewardedVideoAdInstance = new IronSourceAdInstanceBuilder(getProviderName(), rewardedVideoListener).setRewarded().build();
    }

    @Override
    public void fetchRewardedVideo(JSONObject config) {
        try {
            IronSourceNetwork.loadAd(mRewardedVideoAdInstance);
        } catch (Exception e) {
            logError("fetchRewardedVideo exception " + e.getMessage());
            mRvSmashListener.onRewardedVideoLoadFailed(new IronSourceError(RV_LOAD_EXCEPTION, e.getMessage()));
        }
    }

    @Override
    public void loadVideo(JSONObject config, RewardedVideoSmashListener listener) {
        try {
            IronSourceNetwork.loadAd(mRewardedVideoAdInstance);
        } catch (Exception e) {
            logError("loadVideo exception " + e.getMessage());
            mRvSmashListener.onRewardedVideoLoadFailed(new IronSourceError(RV_LOAD_EXCEPTION, e.getMessage()));
        }
    }

    @Override
    public void showRewardedVideo(JSONObject config, RewardedVideoSmashListener listener) {
        try {
            int sessionDepth = SessionDepthManager.getInstance().getSessionDepth(SessionDepthManager.REWARDEDVIDEO);
            Map<String, String> showParams = new HashMap<>();
            showParams.put(Constants.RequestParameters.SESSION_DEPTH, String.valueOf(sessionDepth));
            IronSourceNetwork.showAd(mRewardedVideoAdInstance, showParams);
        } catch (Exception e) {
            logError("showRewardedVideo exception " + e.getMessage());
            mRvSmashListener.onRewardedVideoAdShowFailed(new IronSourceError(RV_SHOW_EXCEPTION, e.getMessage()));
        }
    }

    @Override
    public boolean isRewardedVideoAvailable(JSONObject config) {
        return mRewardedVideoAdInstance != null && IronSourceNetwork.isAdAvailableForInstance(mRewardedVideoAdInstance);
    }

    private synchronized void initSDK(Activity activity, String appKey, String userId, JSONObject config) {
        SDKUtils.setControllerUrl(config.optString(DYNAMIC_CONTROLLER_URL));
        SDKUtils.setDebugMode(config.optInt(DYNAMIC_CONTROLLER_DEBUG_MODE, 0));
        SDKUtils.setControllerConfig(config.optString(DYNAMIC_CONTROLLER_CONFIG, ""));
        IronSourceNetwork.initSDK(activity, appKey, userId, null);
    }

    // ********** Callbacks **********

    private OnInterstitialListener interstitialListener = new OnInterstitialListener() {
        @Override
        public void onInterstitialInitSuccess() {
            //will never be called
        }

        @Override
        public void onInterstitialInitFailed(String description) {
            //will never be called
        }

        @Override
        public void onInterstitialLoadSuccess() {
            log("interstitialListener onInterstitialLoadSuccess");
            if (mIsSmashListener != null) {
                mIsSmashListener.onInterstitialAdReady();
            }
        }

        @Override
        public void onInterstitialLoadFailed(String description) {
            log("interstitialListener onInterstitialLoadFailed " + description);
            if (mIsSmashListener != null) {
                mIsSmashListener.onInterstitialAdLoadFailed(ErrorBuilder.buildLoadFailedError(description));
            }
        }

        @Override
        public void onInterstitialOpen() {
            log("interstitialListener onInterstitialOpen");
            if (mIsSmashListener != null) {
                mIsSmashListener.onInterstitialAdOpened();
            }
        }

        @Override
        public void onInterstitialAdRewarded(String demandSourceId, int amount) {
            log("interstitialListener onInterstitialAdRewarded demandSourceId=" + demandSourceId + " amount=" + amount);
        }

        @Override
        public void onInterstitialClose() {
            log("interstitialListener onInterstitialClose");
            if (mIsSmashListener != null) {
                mIsSmashListener.onInterstitialAdClosed();
            }
        }

        @Override
        public void onInterstitialShowSuccess() {

        }

        @Override
        public void onInterstitialShowFailed(String description) {
            log("interstitialListener onInterstitialShowFailed " + description);
            if (mIsSmashListener != null) {
                mIsSmashListener.onInterstitialAdShowFailed(ErrorBuilder.buildShowFailedError(IronSourceConstants.INTERSTITIAL_AD_UNIT, description));
            }
        }

        @Override
        public void onInterstitialClick() {
            log("interstitialListener onInterstitialClick");
            if (mIsSmashListener != null) {
                mIsSmashListener.onInterstitialAdClicked();
            }
        }

        @Override
        public void onInterstitialEventNotificationReceived(String eventName, JSONObject extData) {
            log("interstitialListener onInterstitialEventNotificationReceived eventName=" + eventName);
            if (mIsSmashListener != null) {
                mIsSmashListener.onInterstitialAdVisible();
            }
        }
    };

    private OnInterstitialListener rewardedVideoListener = new OnInterstitialListener() {
        @Override
        public void onInterstitialInitSuccess() {
            //will never be called
        }

        @Override
        public void onInterstitialInitFailed(String description) {
            //will never be called
        }

        @Override
        public void onInterstitialLoadSuccess() {
            log("rewardedVideoListener onInterstitialLoadSuccess");
            if (mRvSmashListener != null) {
                mRvSmashListener.onRewardedVideoLoadSuccess();
            }
        }

        @Override
        public void onInterstitialLoadFailed(String description) {
            log("rewardedVideoListener onInterstitialLoadFailed " + description);
            if (mRvSmashListener != null) {
                mRvSmashListener.onRewardedVideoLoadFailed(ErrorBuilder.buildLoadFailedError(description));
            }
        }

        @Override
        public void onInterstitialOpen() {
            log("rewardedVideoListener onInterstitialOpen");
            if (mRvSmashListener != null) {
                mRvSmashListener.onRewardedVideoAdOpened();
            }
        }

        @Override
        public void onInterstitialAdRewarded(String demandSourceId, int amount) {
            log("rewardedVideoListener onInterstitialAdRewarded demandSourceId=" + demandSourceId + " amount=" + amount);
            if (mRvSmashListener != null) {
                mRvSmashListener.onRewardedVideoAdRewarded();
            }
        }

        @Override
        public void onInterstitialClose() {
            log("rewardedVideoListener onInterstitialClose");
            if (mRvSmashListener != null) {
                mRvSmashListener.onRewardedVideoAdClosed();
            }
        }

        @Override
        public void onInterstitialShowSuccess() {
            log("rewardedVideoListener onInterstitialShowSuccess");
        }

        @Override
        public void onInterstitialShowFailed(String description) {
            log("rewardedVideoListener onInterstitialShowSuccess " + description);
            if (mRvSmashListener != null) {
                mRvSmashListener.onRewardedVideoAdShowFailed(ErrorBuilder.buildShowFailedError(IronSourceConstants.REWARDED_VIDEO_AD_UNIT, description));
            }
        }

        @Override
        public void onInterstitialClick() {
            log("rewardedVideoListener onInterstitialClick");
            if (mRvSmashListener != null) {
                mRvSmashListener.onRewardedVideoAdClicked();
            }
        }

        @Override
        public void onInterstitialEventNotificationReceived(String eventName, JSONObject extData) {
            log("rewardedVideoListener onInterstitialEventNotificationReceived eventName=" + eventName);
            if (mRvSmashListener != null) {
                mRvSmashListener.onRewardedVideoAdVisible();
            }
        }
    };

    // ********** Helpers **********


    private void logError(String text) {
        IronSourceLoggerManager.getLogger().log(IronSourceLogger.IronSourceTag.INTERNAL, "IronSourceAdapter " + text, IronSourceLogger.IronSourceLogLevel.ERROR);
    }

    private void log(String text) {
        IronSourceLoggerManager.getLogger().log(IronSourceLogger.IronSourceTag.INTERNAL, "IronSourceAdapter " + text, IronSourceLogger.IronSourceLogLevel.VERBOSE);
    }

}
