package com.ironsource.adapters.supersonicads;

import android.app.Activity;
import android.text.TextUtils;

import com.ironsource.mediationsdk.AbstractAdapter;
import com.ironsource.mediationsdk.AbstractSmash;
import com.ironsource.mediationsdk.IntegrationData;
import com.ironsource.mediationsdk.IronSourceObject;
import com.ironsource.mediationsdk.logger.IronSourceError;
import com.ironsource.mediationsdk.logger.IronSourceLogger.IronSourceLogLevel;
import com.ironsource.mediationsdk.logger.IronSourceLogger.IronSourceTag;
import com.ironsource.mediationsdk.logger.IronSourceLoggerManager;
import com.ironsource.mediationsdk.model.Placement;
import com.ironsource.mediationsdk.sdk.InternalOfferwallListener;
import com.ironsource.mediationsdk.sdk.InterstitialSmashListener;
import com.ironsource.mediationsdk.sdk.OfferwallAdapterApi;
import com.ironsource.mediationsdk.sdk.RewardedVideoSmashListener;
import com.ironsource.mediationsdk.utils.ErrorBuilder;
import com.ironsource.mediationsdk.utils.IronSourceConstants;
import com.ironsource.mediationsdk.utils.IronSourceUtils;
import com.ironsource.mediationsdk.utils.SessionDepthManager;
import com.ironsource.sdk.SSAFactory;
import com.ironsource.sdk.SSAPublisher;
import com.ironsource.sdk.data.AdUnitsReady;
import com.ironsource.sdk.listeners.OnInterstitialListener;
import com.ironsource.sdk.listeners.OnOfferWallListener;
import com.ironsource.sdk.listeners.OnRewardedVideoListener;
import com.ironsource.sdk.constants.Constants;
import com.ironsource.sdk.utils.SDKUtils;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.HashMap;
import java.util.Map;


class SupersonicAdsAdapter extends AbstractAdapter implements OfferwallAdapterApi, OnOfferWallListener, OnInterstitialListener, OnRewardedVideoListener {
    private final static String VERSION = "6.7.6";

    private final String TIMESTAMP = "timestamp";
    private final String ITEM_SIGNATURE = "itemSignature";
    private final String SDK_PLUGIN_TYPE = "SDKPluginType";
    private final String OW_PLACEMENT_ID = "placementId";

    private SSAPublisher mSSAPublisher;

    private InternalOfferwallListener mOfferwallListener;

    private boolean mIsRVAvailable = false;
    private boolean mDidSetInitParams = false;

    private String mUserAgeGroup;
    private String mUserGender;
    private String mMediationSegment;

    private final String DYNAMIC_CONTROLLER_URL = "controllerUrl";
    private final String DYNAMIC_CONTROLLER_DEBUG_MODE = "debugMode";
    private final String DYNAMIC_CONTROLLER_CONFIG = "controllerConfig";

    private final String APPLICATION_USER_GENDER = "applicationUserGender";
    private final String APPLICATION_USER_AGE_GROUP = "applicationUserAgeGroup";

    private final String LANGUAGE = "language";
    private final String MAX_VIDEO_LENGTH = "maxVideoLength";
    private final String CAMPAIGN_ID = "campaignId";
    private final String CUSTOM_PARAM_PREFIX = "custom_";
    private final String CUSTOM_SEGMENT = CUSTOM_PARAM_PREFIX + "Segment";

    private final String ITEM_NAME = "itemName";
    private final String ITEM_COUNT = "itemCount";
    private final String APPLICATION_PRIVATE_KEY = "privateKey";

    private final String CLIENT_SIDE_CALLBACKS = "useClientSideCallbacks";
    private final String AD_VISIBLE_EVENT_NAME = "impressions";

    public static SupersonicAdsAdapter startAdapter(String providerName) {
        return new SupersonicAdsAdapter(providerName);
    }

    private SupersonicAdsAdapter(String providerName) {
        super(providerName);
    }

    public static IntegrationData getIntegrationData(Activity activity) {
        IntegrationData ret = new IntegrationData("SupersonicAds", VERSION);
        ret.activities = new String[]{"com.ironsource.sdk.controller.ControllerActivity",
                "com.ironsource.sdk.controller.InterstitialActivity", "com.ironsource.sdk.controller.OpenUrlActivity"};
        return ret;
    }

    @Override
    public String getVersion() {
        return VERSION;
    }

    @Override
    public String getCoreSDKVersion() {
        return SDKUtils.getSDKVersion();
    }

    // ********** Base **********

    @Override
    public void onPause(Activity activity) {
        if (mSSAPublisher != null)
            mSSAPublisher.onPause(activity);
    }

    @Override
    public void onResume(Activity activity) {
        if (mSSAPublisher != null)
            mSSAPublisher.onResume(activity);
    }

    @Override
    public void setAge(int age) {
        if (age >= 13 && age <= 17) {
            mUserAgeGroup = "1";
        } else if (age >= 18 && age <= 20) {
            mUserAgeGroup = "2";
        } else if (age >= 21 && age <= 24) {
            mUserAgeGroup = "3";
        } else if (age >= 25 && age <= 34) {
            mUserAgeGroup = "4";
        } else if (age >= 35 && age <= 44) {
            mUserAgeGroup = "5";
        } else if (age >= 45 && age <= 54) {
            mUserAgeGroup = "6";
        } else if (age >= 55 && age <= 64) {
            mUserAgeGroup = "7";
        } else if (age > 65 && age <= 120) {
            mUserAgeGroup = "8";
        } else {
            mUserAgeGroup = "0";
        }
    }

    @Override
    public void setGender(String gender) {
        mUserGender = gender;
    }

    @Override
    public void setMediationSegment(String segment) {
        mMediationSegment = segment;
    }

    // ********** RewardedVideo **********

    @Override
    public void initRewardedVideo(final Activity activity, final String appKey, final String userId, final JSONObject config, RewardedVideoSmashListener listener) {
        if (!mDidSetInitParams)
            setParamsBeforeInit(config);

        activity.runOnUiThread(new Runnable() {

            @Override
            public void run() {
                try {
                    mSSAPublisher = SSAFactory.getPublisherInstance(activity);
                    HashMap<String, String> rewardedVideoExtraParams = getRewardedVideoExtraParams(config);

                    mSSAPublisher.initRewardedVideo(appKey, userId, getProviderName(), rewardedVideoExtraParams, SupersonicAdsAdapter.this);
                } catch (Exception e) {
                    e.printStackTrace();
                    onRVInitFail("initRewardedVideo");
                }
            }

        });
    }

    @Override
    public void fetchRewardedVideo(JSONObject config) {

    }

    @Override
    public void showRewardedVideo(JSONObject config, RewardedVideoSmashListener listener) {
        mActiveRewardedVideoSmash = listener;

        if (mSSAPublisher != null) {
            int sessionDepth = SessionDepthManager.getInstance().getSessionDepth(SessionDepthManager.REWARDEDVIDEO);

            JSONObject showParams = new JSONObject();
            try {
                showParams.put(Constants.RequestParameters.DEMAND_SOURCE_NAME, getProviderName());
                showParams.put(Constants.RequestParameters.SESSION_DEPTH, sessionDepth);
            } catch (JSONException e) {
                e.printStackTrace();
            }

            mSSAPublisher.showRewardedVideo(showParams);
        } else {
            mIsRVAvailable = false;

            if (mActiveRewardedVideoSmash != null)
                mActiveRewardedVideoSmash.onRewardedVideoAdShowFailed(ErrorBuilder.buildNoAdsToShowError(IronSourceConstants.REWARDED_VIDEO_AD_UNIT));

            for (RewardedVideoSmashListener smash : mAllRewardedVideoSmashes) {
                if (smash != null)
                    smash.onRewardedVideoAvailabilityChanged(false);
            }
        }
    }

    @Override
    public synchronized boolean isRewardedVideoAvailable(JSONObject config) {
        return mIsRVAvailable;
    }

    // ********** Interstitial **********

    @Override
    public void initInterstitial(final Activity activity, final String appKey, final String userId, final JSONObject config, InterstitialSmashListener listener) {
        if (!mDidSetInitParams)
            setParamsBeforeInit(config);

        activity.runOnUiThread(new Runnable() {
            @Override
            public void run() {
                try {
                    mSSAPublisher = SSAFactory.getPublisherInstance(activity);

                    final HashMap<String, String> interstitialExtraParams = getInterstitialExtraParams();

                    mSSAPublisher.initInterstitial(appKey, userId, getProviderName(), interstitialExtraParams, SupersonicAdsAdapter.this);
                } catch (Exception e) {
                    e.printStackTrace();
                    onInterstitialInitFailed(e.getMessage());
                }

            }
        });
    }

    @Override
    public void loadInterstitial(JSONObject config, InterstitialSmashListener listener) {
        if (mSSAPublisher != null) {
            JSONObject loadParams = new JSONObject();
            try {
                loadParams.put(Constants.RequestParameters.DEMAND_SOURCE_NAME, getProviderName());
            } catch (JSONException e) {
                e.printStackTrace();
            }
            mSSAPublisher.loadInterstitial(loadParams);
        } else {
            log(IronSourceTag.NATIVE, "Please call initInterstitial before calling loadInterstitial", IronSourceLogLevel.WARNING);

            for (InterstitialSmashListener smash : mAllInterstitialSmashes) {
                if (smash != null)
                    smash.onInterstitialAdLoadFailed(ErrorBuilder.buildLoadFailedError("Load was called before Init"));
            }
        }
    }

    @Override
    public void showInterstitial(JSONObject config, InterstitialSmashListener listener) {
        mActiveInterstitialSmash = listener;

        if (mSSAPublisher != null) {
            int sessionDepth = SessionDepthManager.getInstance().getSessionDepth(SessionDepthManager.INTERSTITIAL);

            JSONObject showParams = new JSONObject();
            try {
                showParams.put(Constants.RequestParameters.DEMAND_SOURCE_NAME, getProviderName());
                showParams.put(Constants.RequestParameters.SESSION_DEPTH, sessionDepth);
            } catch (JSONException e) {
                e.printStackTrace();
            }

            mSSAPublisher.showInterstitial(showParams);
        } else {
            log(IronSourceTag.NATIVE, "Please call loadInterstitial before calling showInterstitial", IronSourceLogLevel.WARNING);

            if (mActiveInterstitialSmash != null)
                mActiveInterstitialSmash.onInterstitialAdShowFailed(ErrorBuilder.buildNoAdsToShowError(IronSourceConstants.INTERSTITIAL_AD_UNIT));
        }
    }

    @Override
    public boolean isInterstitialReady(JSONObject config) {
        return mSSAPublisher != null && mSSAPublisher.isInterstitialAdAvailable(getProviderName());
    }

    // ********** Offerwall **********

    @Override
    public void setInternalOfferwallListener(InternalOfferwallListener listener) {
        mOfferwallListener = listener;
    }

    @Override
    public void initOfferwall(final Activity activity, final String appKey, final String userId, final JSONObject config) {
        if (!mDidSetInitParams)
            setParamsBeforeInit(config);

        activity.runOnUiThread(new Runnable() {
            @Override
            public void run() {
                try {
                    Map<String, String> offerwallExtraParams = getOfferwallExtraParams(config);
                    mSSAPublisher = SSAFactory.getPublisherInstance(activity);
                    mSSAPublisher.initOfferWall(appKey, userId, offerwallExtraParams, SupersonicAdsAdapter.this);
                } catch (Exception e) {
                    e.printStackTrace();
                    IronSourceLoggerManager.getLogger().logException(IronSourceTag.ADAPTER_API, getProviderName() + ":initOfferwall(userId:" + userId + ")", e);
                    mOfferwallListener.onOfferwallAvailable(false,
                            ErrorBuilder.buildInitFailedError("Adapter initialization failure - " + getProviderName() + " - " + e.getMessage(), IronSourceConstants.OFFERWALL_AD_UNIT));
                }
            }
        });
    }

    @Override
    public void getOfferwallCredits() {

        if (mSSAPublisher != null) {
            String appKey = IronSourceObject.getInstance().getIronSourceAppKey();
            String userId = IronSourceObject.getInstance().getIronSourceUserId();

            mSSAPublisher.getOfferWallCredits(appKey, userId, this);
        } else
            log(IronSourceTag.NATIVE, "Please call init before calling getOfferwallCredits", IronSourceLogLevel.WARNING);
    }

    @Override
    public void showOfferwall(String placementId, JSONObject config) {
        Map<String, String> offerwallExtraParams = getOfferwallExtraParams(config);
        if (offerwallExtraParams != null) {
            offerwallExtraParams.put(OW_PLACEMENT_ID, placementId);
        }

        if (mSSAPublisher != null) {
            mSSAPublisher.showOfferWall(offerwallExtraParams);
        } else {
            log(IronSourceTag.NATIVE, "Please call init before calling showOfferwall", IronSourceLogLevel.WARNING);
        }
    }

    @Override
    public boolean isOfferwallAvailable() {
        return true;
    }

    // ********** Helpers **********

    private synchronized void setParamsBeforeInit(JSONObject config) {
        mDidSetInitParams = true;

        SDKUtils.setControllerUrl(config.optString(DYNAMIC_CONTROLLER_URL));

        if (isAdaptersDebugEnabled()) {
            SDKUtils.setDebugMode(3);
        } else {
            SDKUtils.setDebugMode(config.optInt(DYNAMIC_CONTROLLER_DEBUG_MODE, 0));
        }

        SDKUtils.setControllerConfig(config.optString(DYNAMIC_CONTROLLER_CONFIG, ""));
    }

    private HashMap<String, String> getGenenralExtraParams() {
        HashMap<String, String> params = new HashMap<>();

        //Check and add user age group configuration
        if (!TextUtils.isEmpty(mUserAgeGroup))
            params.put(APPLICATION_USER_AGE_GROUP, mUserAgeGroup);

        //Check and add user creation gender configuration
        if (!TextUtils.isEmpty(mUserGender))
            params.put(APPLICATION_USER_GENDER, mUserGender);

        String pluginType = getPluginType();
        if (!TextUtils.isEmpty(pluginType))
            params.put(SDK_PLUGIN_TYPE, pluginType);

        return params;
    }

    private HashMap<String, String> getRewardedVideoExtraParams(JSONObject config) {
        HashMap<String, String> rvExtraParams = getGenenralExtraParams();

        //Check and add language configuration
        String language = config.optString(LANGUAGE);
        if (!TextUtils.isEmpty(language)) {
            rvExtraParams.put(LANGUAGE, language);
        }

        //Check and add maxVideoLength configuration
        String maxVideoLength = config.optString(MAX_VIDEO_LENGTH);
        if (!TextUtils.isEmpty(maxVideoLength)) {
            rvExtraParams.put(MAX_VIDEO_LENGTH, maxVideoLength);
        }

        //Check and add campaignId configuration
        String campaignId = config.optString(CAMPAIGN_ID);
        if (!TextUtils.isEmpty(campaignId))
            rvExtraParams.put(CAMPAIGN_ID, campaignId);

        //Check and add segment configuration
        if (!TextUtils.isEmpty(mMediationSegment)) {
            rvExtraParams.put(CUSTOM_SEGMENT, mMediationSegment);
        }

        //Check and add itemName/itemCount/privateKey configurations
        addItemNameCountSignature(rvExtraParams, config);

        //Add RewardedVideo customParams if set
        Map<String, String> customParams = SupersonicConfig.getConfigObj().getRewardedVideoCustomParams();
        if (customParams != null && !customParams.isEmpty())
            rvExtraParams.putAll(customParams);

        return rvExtraParams;
    }

    private HashMap<String, String> getInterstitialExtraParams() {
        return getGenenralExtraParams();
    }

    private HashMap<String, String> getOfferwallExtraParams(JSONObject config) {
        HashMap<String, String> owExtraParams = getGenenralExtraParams();

        //Check language configuration
        String language = config.optString(LANGUAGE);
        if (!TextUtils.isEmpty(language)) {
            owExtraParams.put(LANGUAGE, language);
        }

        //Check client side callbacks configuration
        boolean clientSideCallbacks = SupersonicConfig.getConfigObj().getClientSideCallbacks();
        owExtraParams.put(CLIENT_SIDE_CALLBACKS, String.valueOf(clientSideCallbacks));

        //Add Offerwall customParams if set
        Map<String, String> customParams = SupersonicConfig.getConfigObj().getOfferwallCustomParams();
        if (customParams != null && !customParams.isEmpty())
            owExtraParams.putAll(customParams);

        addItemNameCountSignature(owExtraParams, config);

        return owExtraParams;
    }

    private void addItemNameCountSignature(HashMap<String, String> params, JSONObject config) {
        try {
            String itemName = config.optString(ITEM_NAME);
            int itemCount = config.optInt(ITEM_COUNT, -1);
            String privateKey = config.optString(APPLICATION_PRIVATE_KEY);

            boolean shouldAddSignature = true;

            if (TextUtils.isEmpty(itemName))
                shouldAddSignature = false;
            else
                params.put(ITEM_NAME, itemName);

            if (TextUtils.isEmpty(privateKey))
                shouldAddSignature = false;

            if (itemCount == -1)
                shouldAddSignature = false;
            else
                params.put(ITEM_COUNT, String.valueOf(itemCount));


            if (shouldAddSignature) {
                int timestamp = IronSourceUtils.getCurrentTimestamp();
                params.put(TIMESTAMP, String.valueOf(timestamp));
                params.put(ITEM_SIGNATURE, createItemSig(timestamp, itemName, itemCount, privateKey));
            }

        } catch (Exception e) {
            IronSourceLoggerManager.getLogger().logException(IronSourceTag.ADAPTER_API, " addItemNameCountSignature", e);
        }
    }

    private String createItemSig(int timestamp, String itemName, int itemCount, String privateKey) {
        return IronSourceUtils.getMD5(timestamp + itemName + itemCount + privateKey);
    }

    private String createMinimumOfferCommissionSig(double min, String privateKey) {
        return IronSourceUtils.getMD5(min + privateKey);
    }

    private String createUserCreationDateSig(String userid, String uCreationDate, String privateKey) {
        return IronSourceUtils.getMD5(userid + uCreationDate + privateKey);
    }

    // ********** Supersonic Rewarded Video Callbacks **********

    @Override
    public void onRVNoMoreOffers() {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onRVNoMoreOffers ", IronSourceLogLevel.INFO);

        mIsRVAvailable = false;

        for (RewardedVideoSmashListener smash : mAllRewardedVideoSmashes) {
            if (smash != null)
                smash.onRewardedVideoAvailabilityChanged(false);
        }
    }

    @Override
    public void onRVInitSuccess(AdUnitsReady aur) {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onRVInitSuccess ", IronSourceLogLevel.INFO);

        int numOfAdUnits = 0;

        try {
            numOfAdUnits = Integer.parseInt(aur.getNumOfAdUnits());
        } catch (NumberFormatException e) {
            IronSourceLoggerManager.getLogger().logException(IronSourceTag.NATIVE, "onRVInitSuccess:parseInt()", e);
        }

        boolean availability = numOfAdUnits > 0;
        mIsRVAvailable = availability;

        for (RewardedVideoSmashListener smash : mAllRewardedVideoSmashes) {
            if (smash != null)
                smash.onRewardedVideoAvailabilityChanged(availability);
        }
    }

    @Override
    public void onRVInitFail(String error) {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onRVInitFail ", IronSourceLogLevel.INFO);

        for (RewardedVideoSmashListener smash : mAllRewardedVideoSmashes) {
            if (smash != null)
                smash.onRewardedVideoAvailabilityChanged(false);
        }
    }

    @Override
    public void onRVAdClicked() {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onRVAdClicked ", IronSourceLogLevel.INFO);

        if (mActiveRewardedVideoSmash != null)
            mActiveRewardedVideoSmash.onRewardedVideoAdClicked();
    }

    @Override
    public void onRVEventNotificationReceived(String eventName, JSONObject extData) {
        if (extData != null)
            IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onRVEventNotificationReceived: " + eventName + " extData: " + extData.toString(), IronSourceLogLevel.INFO);

        if (!TextUtils.isEmpty(eventName) && AD_VISIBLE_EVENT_NAME.equals(eventName) && mActiveRewardedVideoSmash != null) {
            mActiveRewardedVideoSmash.onRewardedVideoAdVisible();
        }
    }

    @Override
    public void onRVShowFail(String error) {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onRVShowFail ", IronSourceLogLevel.INFO);

        if (mActiveRewardedVideoSmash != null)
            mActiveRewardedVideoSmash.onRewardedVideoAdShowFailed(new IronSourceError(IronSourceError.ERROR_CODE_NO_ADS_TO_SHOW, error));
    }

    @Override
    public void onRVAdCredited(int amount) {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onRVAdCredited ", IronSourceLogLevel.INFO);

        if (mActiveRewardedVideoSmash != null) {
            mActiveRewardedVideoSmash.onRewardedVideoAdRewarded();
        }
    }

    @Override
    public void onRVAdClosed() {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onRVAdClosed ", IronSourceLogLevel.INFO);

        if (mActiveRewardedVideoSmash != null)
            mActiveRewardedVideoSmash.onRewardedVideoAdClosed();
    }

    @Override
    public void onRVAdOpened() {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onRVAdOpened ", IronSourceLogLevel.INFO);

        if (mActiveRewardedVideoSmash != null)
            mActiveRewardedVideoSmash.onRewardedVideoAdOpened();
    }

    // ********** Supersonic Interstitial Callbacks **********

    @Override
    public void onInterstitialInitSuccess() {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onInterstitialInitSuccess ", IronSourceLogLevel.INFO);

        for (InterstitialSmashListener smash : mAllInterstitialSmashes) {
            if (smash != null)
                smash.onInterstitialInitSuccess();
        }
    }

    @Override
    public void onInterstitialInitFailed(String description) {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onInterstitialInitFailed ", IronSourceLogLevel.INFO);

        for (InterstitialSmashListener smash : mAllInterstitialSmashes) {
            if (smash != null)
                smash.onInterstitialInitFailed(ErrorBuilder.buildInitFailedError(description, IronSourceConstants.INTERSTITIAL_AD_UNIT));
        }
    }

    @Override
    public void onInterstitialLoadSuccess() {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onInterstitialLoadSuccess ", IronSourceLogLevel.INFO);

        for (InterstitialSmashListener smash : mAllInterstitialSmashes) {
            if (smash != null)
                smash.onInterstitialAdReady();
        }
    }

    @Override
    public void onInterstitialLoadFailed(String description) {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onInterstitialAdLoadFailed ", IronSourceLogLevel.INFO);

        for (InterstitialSmashListener smash : mAllInterstitialSmashes) {
            if (smash != null)
                smash.onInterstitialAdLoadFailed(ErrorBuilder.buildLoadFailedError(description));
        }
    }

    @Override
    public void onInterstitialOpen() {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onInterstitialAdOpened ", IronSourceLogLevel.INFO);

        if (mActiveInterstitialSmash != null)
            mActiveInterstitialSmash.onInterstitialAdOpened();
    }

    @Override
    public void onInterstitialClose() {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onInterstitialAdClosed ", IronSourceLogLevel.INFO);

        if (mActiveInterstitialSmash != null)
            mActiveInterstitialSmash.onInterstitialAdClosed();
    }

    @Override
    public void onInterstitialShowSuccess() {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onInterstitialAdShowSucceeded ", IronSourceLogLevel.INFO);

        if (mActiveInterstitialSmash != null)
            mActiveInterstitialSmash.onInterstitialAdShowSucceeded();
    }

    @Override
    public void onInterstitialShowFailed(String description) {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onInterstitialAdShowFailed ", IronSourceLogLevel.INFO);

        if (mActiveInterstitialSmash != null) {
            mActiveInterstitialSmash.onInterstitialAdShowFailed(ErrorBuilder.buildShowFailedError(IronSourceConstants.INTERSTITIAL_AD_UNIT, description));
        }
    }

    @Override
    public void onInterstitialClick() {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onInterstitialAdClicked ", IronSourceLogLevel.INFO);

        if (mActiveInterstitialSmash != null)
            mActiveInterstitialSmash.onInterstitialAdClicked();
    }

    @Override
    public void onInterstitialEventNotificationReceived(String eventName, JSONObject extData) {
        if (extData != null) {
            IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onInterstitialEventNotificationReceived: " + eventName + " extData: " + extData.toString(), IronSourceLogLevel.INFO);
            if (!TextUtils.isEmpty(eventName) && AD_VISIBLE_EVENT_NAME.equals(eventName) && mActiveInterstitialSmash != null) {
                mActiveInterstitialSmash.onInterstitialAdVisible();
            }
        }
    }


    // ********** Supersonic Offerwall Callbacks **********

    @Override
    public void onOfferwallInitSuccess() {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onOfferwallInitSuccess ", IronSourceLogLevel.INFO);

        if (mOfferwallListener != null) {
            mOfferwallListener.onOfferwallAvailable(true);
        }
    }

    @Override
    public void onOfferwallInitFail(String description) {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onOfferwallInitFail ", IronSourceLogLevel.INFO);

        if (mOfferwallListener != null) {
            IronSourceError sse = ErrorBuilder.buildGenericError(description);
            mOfferwallListener.onOfferwallAvailable(false, sse);
        }
    }

    @Override
    public void onOfferwallEventNotificationReceived(String eventName, JSONObject extData) {
        if (extData != null) {
            IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL,
                    getProviderName() + " :onOfferwallEventNotificationReceived ", IronSourceLogLevel.INFO);
        }
    }

    @Override
    public void onOWShowSuccess(String placementId) {
        if (TextUtils.isEmpty(placementId)) {
            log(IronSourceTag.ADAPTER_API, getProviderName() + ":onOWShowSuccess()", IronSourceLogLevel.INFO);
        } else {
            log(IronSourceTag.ADAPTER_API, getProviderName() + ":onOWShowSuccess(placementId:" + placementId + ")", IronSourceLogLevel.INFO);
        }

        if (mOfferwallListener != null) {
            mOfferwallListener.onOfferwallOpened();
        }
    }

    @Override
    public void onOWShowFail(String desc) {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onOWShowFail ", IronSourceLogLevel.INFO);

        if (mOfferwallListener != null) {
            IronSourceError sse = ErrorBuilder.buildGenericError(desc);
            mOfferwallListener.onOfferwallShowFailed(sse);
        }
    }

    @Override
    public void onOWGeneric(String arg0, String arg1) {
        // Deprecated
    }

    @Override
    public boolean onOWAdCredited(int credits, int totalCredits, boolean totalCreditsFlag) {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onOWAdCredited ", IronSourceLogLevel.INFO);

        return mOfferwallListener != null && mOfferwallListener.onOfferwallAdCredited(credits, totalCredits, totalCreditsFlag);
    }

    @Override
    public void onOWAdClosed() {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onOWAdClosed ", IronSourceLogLevel.INFO);

        if (mOfferwallListener != null)
            mOfferwallListener.onOfferwallClosed();
    }

    @Override
    public void onGetOWCreditsFailed(String desc) {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onGetOWCreditsFailed ", IronSourceLogLevel.INFO);

        if (mOfferwallListener != null) {
            IronSourceError sse = ErrorBuilder.buildGenericError(desc);
            mOfferwallListener.onGetOfferwallCreditsFailed(sse);
        }
    }

    protected void setMediationState(AbstractSmash.MEDIATION_STATE state, String adUnit){
        if (mSSAPublisher != null) {
            IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :setMediationState(" + adUnit + " , " + getProviderName() + " , " + state.getValue()+")", IronSourceLogLevel.INFO);
            mSSAPublisher.setMediationState(adUnit, getProviderName(), state.getValue());
        }
    }
}
