package com.ironsource.adapters.supersonicads;

import android.text.TextUtils;

import com.ironsource.mediationsdk.config.AbstractAdapterConfig;
import com.ironsource.mediationsdk.config.ConfigValidationResult;
import com.ironsource.mediationsdk.logger.IronSourceLogger;
import com.ironsource.mediationsdk.logger.IronSourceLoggerManager;
import com.ironsource.mediationsdk.utils.ErrorBuilder;
import com.ironsource.mediationsdk.utils.IronSourceConstants;

import org.json.JSONObject;

import java.util.ArrayList;

public class DemandSourceConfig extends AbstractAdapterConfig {
    private final String TAG = DemandSourceConfig.class.getSimpleName();
    private static final String CUSTOM_PARAM_PREFIX = "custom_";

    //** Mandatory **//
    private final String APPLICATION_KEY = "applicationKey";
    private final String USER_ID = "userId";

    //** Optional **//
    static final String CLIENT_SIDE_CALLBACKS = "useClientSideCallbacks";
    static final String APPLICATION_USER_GENDER = "applicationUserGender";
    static final String APPLICATION_USER_AGE_GROUP = "applicationUserAgeGroup";

    // age/gender of the supersonic object - the age provided by the publisher
    private final String AGE = "age";
    private final String GENDER = "gender";

    static final String LANGUAGE = "language";
    private final String APPLICATION_PRIVATE_KEY = "privateKey";
    static final String MAX_VIDEO_LENGTH = "maxVideoLength";
    static final String ITEM_NAME = "itemName";
    static final String ITEM_COUNT = "itemCount";

    private final String SDK_PLUGIN_TYPE = "SDKPluginType";
    static final String CAMPAIGN_ID = "campaignId";
    static final String CUSTOM_SEGMENT = CUSTOM_PARAM_PREFIX + "Segment";

    private final String DYNAMIC_CONTROLLER_URL = "controllerUrl";
    private final String DYNAMIC_CONTROLLER_DEBUG_MODE = "debugMode";
    private final String DYNAMIC_CONTROLLER_CONFIG = "controllerConfig";

    private String mProviderName;

    public DemandSourceConfig(String providerName) {
        super(providerName);
        mProviderName = providerName;
    }

    /* Getters */
    String getRVUserAgeGroup() {
        return mProviderSettings.getRewardedVideoSettings().optString(APPLICATION_USER_AGE_GROUP);
    }

    String getISUserAgeGroup() {
        return mProviderSettings.getInterstitialSettings().optString(APPLICATION_USER_AGE_GROUP);
    }

    public String getRVDynamicControllerUrl() {
        return mProviderSettings.getRewardedVideoSettings().optString(DYNAMIC_CONTROLLER_URL);
    }

    String getISDynamicControllerUrl() {
        return mProviderSettings.getInterstitialSettings().optString(DYNAMIC_CONTROLLER_URL);
    }

    public int getRVDebugMode() {
        int mode = 0;
        if (mProviderSettings.getRewardedVideoSettings().has(DYNAMIC_CONTROLLER_DEBUG_MODE))
            mode = mProviderSettings.getRewardedVideoSettings().optInt(DYNAMIC_CONTROLLER_DEBUG_MODE);

        return mode;
    }

    public int getISDebugMode() {
        int mode = 0;
        if (mProviderSettings.getInterstitialSettings().has(DYNAMIC_CONTROLLER_DEBUG_MODE))
            mode = mProviderSettings.getInterstitialSettings().optInt(DYNAMIC_CONTROLLER_DEBUG_MODE);

        return mode;
    }

    public String getRVControllerConfig() {
        String config = "";

        if (mProviderSettings != null && mProviderSettings.getRewardedVideoSettings() != null) {
            if (mProviderSettings.getRewardedVideoSettings().has(DYNAMIC_CONTROLLER_CONFIG))
                config = mProviderSettings.getRewardedVideoSettings().optString(DYNAMIC_CONTROLLER_CONFIG);
        }

        return config;
    }

    public String getISControllerConfig() {
        String config = "";

        if (mProviderSettings != null && mProviderSettings.getInterstitialSettings() != null) {
            if (mProviderSettings.getInterstitialSettings().has(DYNAMIC_CONTROLLER_CONFIG))
                config = mProviderSettings.getInterstitialSettings().optString(DYNAMIC_CONTROLLER_CONFIG);
        }

        return config;
    }

    //only for rv
    public String getMaxVideoLength() {
        return mProviderSettings.getRewardedVideoSettings().optString(MAX_VIDEO_LENGTH);
    }

    //only for rv
    public String getLanguage() {
        return mProviderSettings.getRewardedVideoSettings().optString(LANGUAGE);
    }

    //only for rv
    public String getPrivateKey() {
        return mProviderSettings.getRewardedVideoSettings().optString(APPLICATION_PRIVATE_KEY);
    }

    //only for rv
    public String getItemName() {
        return mProviderSettings.getRewardedVideoSettings().optString(ITEM_NAME);
    }

    //only for rv
    public int getItemCount() {
        int itemCount = -1;
        try {
            String itemCountString = mProviderSettings.getRewardedVideoSettings().optString(ITEM_COUNT);
            if (!TextUtils.isEmpty(itemCountString))
                itemCount = Integer.valueOf(itemCountString);
        } catch (NumberFormatException e) {
            IronSourceLoggerManager.getLogger().logException(IronSourceLogger.IronSourceTag.NATIVE,
                    TAG + ":getItemCount()", e);
        }

        return itemCount;
    }

    //only for rv
    String getCampaignId() {
        return mProviderSettings.getRewardedVideoSettings().optString(CAMPAIGN_ID);
    }

    //only for rv
    String getMediationSegment() {
        return mProviderSettings.getRewardedVideoSettings().optString(CUSTOM_SEGMENT);
    }

    int getMaxRVAdsPerIteration() {
        return getMaxRVAdsPerIterationToPresent();
    }

    public int getMaxISAdsPerIteration() {
        return getMaxISAdsPerIterationToPresent();
    }

    String getRVUserGender() {
        return mProviderSettings.getRewardedVideoSettings().optString(APPLICATION_USER_GENDER);
    }

    String getISUserGender() {
        return mProviderSettings.getInterstitialSettings().optString(APPLICATION_USER_GENDER);
    }


    /* Setters */

    public void setMediationSegment(String segment) {
        mProviderSettings.setRewardedVideoSettings(CUSTOM_SEGMENT, segment);
    }

    public void setUserAgeGroup(int age) {

        String ageGroup = "0";

        if (age >= 13 && age <= 17) {
            ageGroup = "1";
        } else if (age >= 18 && age <= 20) {
            ageGroup = "2";
        } else if (age >= 21 && age <= 24) {
            ageGroup = "3";
        } else if (age >= 25 && age <= 34) {
            ageGroup = "4";
        } else if (age >= 35 && age <= 44) {
            ageGroup = "5";
        } else if (age >= 45 && age <= 54) {
            ageGroup = "6";
        } else if (age >= 55 && age <= 64) {
            ageGroup = "7";
        } else if (age > 65 && age <= 120) {
            ageGroup = "8";
        }

        mProviderSettings.setRewardedVideoSettings(APPLICATION_USER_AGE_GROUP, ageGroup);
        mProviderSettings.setInterstitialSettings(APPLICATION_USER_AGE_GROUP, ageGroup);
    }

    public void setUserGender(String gender) {
        mProviderSettings.setRewardedVideoSettings(APPLICATION_USER_GENDER, gender);
        mProviderSettings.setInterstitialSettings(APPLICATION_USER_GENDER, gender);
    }

    //AbstractAdapterConfig
    @Override
    protected ArrayList<String> initializeMandatoryFields() {
        ArrayList<String> result = new ArrayList<>();

        result.add(DYNAMIC_CONTROLLER_URL);

        return result;
    }

    @Override
    protected ArrayList<String> initializeOptionalFields() {
        ArrayList<String> result = new ArrayList<>();

        result.add(CLIENT_SIDE_CALLBACKS);
        result.add(APPLICATION_USER_GENDER);
        result.add(APPLICATION_USER_AGE_GROUP);
        result.add(LANGUAGE);
        result.add(MAX_ADS_KEY);
        result.add(MAX_ADS_PER_ITERATION_KEY);
        result.add(APPLICATION_PRIVATE_KEY);
        result.add(MAX_VIDEO_LENGTH);
        result.add(ITEM_NAME);
        result.add(ITEM_COUNT);
        result.add(SDK_PLUGIN_TYPE);
        result.add(DYNAMIC_CONTROLLER_CONFIG);
        result.add(DYNAMIC_CONTROLLER_DEBUG_MODE);
        result.add(REQUEST_URL_KEY);
        result.add(CUSTOM_SEGMENT);

        return result;
    }

    @Override
    protected void validateOptionalField(JSONObject config, String key, ConfigValidationResult result) {
        try {
            if (MAX_ADS_KEY.equals(key)) {
                int maxVideos = config.optInt(key);
                validateMaxVideos(maxVideos, result);
            } else if (MAX_ADS_PER_ITERATION_KEY.equals(key)) {
            } else if (DYNAMIC_CONTROLLER_DEBUG_MODE.equals(key)) {
            } else if (DYNAMIC_CONTROLLER_CONFIG.equals(key)) {
            } else {
                String value = (String) config.get(key);

                if (CLIENT_SIDE_CALLBACKS.equals(key))
                    validateClientSideCallbacks(value, result);
                else if (APPLICATION_USER_GENDER.equals(key))
                    validateGender(value, result);
                else if (APPLICATION_USER_AGE_GROUP.equals(key))
                    validateAgeGroup(value, result);
                else if (LANGUAGE.equals(key))
                    validateLanguage(value, result);
                else if (MAX_VIDEO_LENGTH.equals(key))
                    validateMaxVideoLength(value, result);
                else if (APPLICATION_PRIVATE_KEY.equals(key))
                    validatePrivateKey(value, result);
                else if (ITEM_NAME.equals(key))
                    validateItemName(value, result);
                else if (ITEM_COUNT.equals(key))
                    validateItemCount(value, result);
            }
        } catch (Throwable e) {
            result.setInvalid(ErrorBuilder.buildInvalidKeyValueError(key, mProviderName, null));
        }
    }

    private void validateItemCount(String value, ConfigValidationResult result) {
        try {
            value = value.trim();
            int itemCount = Integer.parseInt(value);
            if (itemCount < 1 || itemCount > 100000) {
                result.setInvalid(ErrorBuilder.buildInvalidKeyValueError(ITEM_COUNT, mProviderName, ITEM_COUNT + " value should be between 1-100000"));
            }
        } catch (NumberFormatException e) {
            result.setInvalid(ErrorBuilder.buildInvalidKeyValueError(ITEM_COUNT, mProviderName, ITEM_COUNT + " value should be between 1-100000"));
        }
    }

    private void validateItemName(String value, ConfigValidationResult result) {
        if (value != null) {

            if (value.length() < 1 || value.length() > 50)
                result.setInvalid(ErrorBuilder.buildInvalidKeyValueError(ITEM_NAME, mProviderName, ITEM_NAME + "length should be between 1-50 characters"));

        } else {
            result.setInvalid(ErrorBuilder.buildInvalidKeyValueError(ITEM_NAME, mProviderName, ITEM_NAME + "length should be between 1-50 characters"));
        }
    }

    private void validatePrivateKey(String value, ConfigValidationResult result) {
        if (value != null) {
            if (value.length() >= 5 && value.length() <= 30) {
                String pattern = "^[a-zA-Z0-9]*$";
                if (!value.matches(pattern)) {
                    result.setInvalid(ErrorBuilder.buildInvalidKeyValueError(APPLICATION_PRIVATE_KEY, mProviderName,
                            APPLICATION_PRIVATE_KEY + " should contains only characters and numbers"));
                }
            } else {
                result.setInvalid(ErrorBuilder.buildInvalidKeyValueError(APPLICATION_PRIVATE_KEY, mProviderName,
                        APPLICATION_PRIVATE_KEY + " length should be between 5-30 characters"));
            }
        } else {
            result.setInvalid(ErrorBuilder.buildInvalidKeyValueError(APPLICATION_PRIVATE_KEY, mProviderName, APPLICATION_PRIVATE_KEY + " length should be between 5-30 characters"));
        }
    }

    private void validateMaxVideoLength(String value, ConfigValidationResult result) {
        try {
            value = value.trim();
            int age = Integer.parseInt(value);
            if (age < 1 || age > 1000) {
                result.setInvalid(ErrorBuilder.buildInvalidKeyValueError(MAX_VIDEO_LENGTH, mProviderName,
                        MAX_VIDEO_LENGTH + " value should be between 1-1000"));
            }
        } catch (NumberFormatException e) {
            result.setInvalid(ErrorBuilder.buildInvalidKeyValueError(MAX_VIDEO_LENGTH, mProviderName, MAX_VIDEO_LENGTH + " value should be between 1-1000"));
        }
    }

    private void validateLanguage(String value, ConfigValidationResult result) {
        if (value != null) {
            value = value.trim();

            String pattern = "^[a-zA-Z]*$";
            if (!value.matches(pattern) || value.length() != 2)
                result.setInvalid(ErrorBuilder.buildInvalidKeyValueError(LANGUAGE, mProviderName, LANGUAGE + " value should be two letters format."));
        } else {
            result.setInvalid(ErrorBuilder.buildInvalidKeyValueError(LANGUAGE, mProviderName, LANGUAGE + " value should be two letters format."));
        }
    }

    private void validateAgeGroup(String value, ConfigValidationResult result) {
        try {
            value = value.trim();
            int age = Integer.parseInt(value);
            if (age < 0 || age > 8) {
                result.setInvalid(ErrorBuilder.buildInvalidKeyValueError(APPLICATION_USER_AGE_GROUP, mProviderName,
                        APPLICATION_USER_AGE_GROUP + " value should be between 0-8"));
            }
        } catch (NumberFormatException e) {
            result.setInvalid(ErrorBuilder.buildInvalidKeyValueError(APPLICATION_USER_AGE_GROUP, mProviderName, APPLICATION_USER_AGE_GROUP + " value should be between 0-8"));
        }
    }

    private void validateGender(String gender, ConfigValidationResult result) {
        try {
            if (gender != null) {
                gender = gender.toLowerCase().trim();

                if (!IronSourceConstants.Gender.MALE.equals(gender) &&
                        !IronSourceConstants.Gender.FEMALE.equals(gender) &&
                        !IronSourceConstants.Gender.UNKNOWN.equals(gender))
                    result.setInvalid(ErrorBuilder.buildInvalidKeyValueError(GENDER, mProviderName, GENDER + " value should be one of male/female/unknown."));
            }
        } catch (Exception e) {
            result.setInvalid(ErrorBuilder.buildInvalidKeyValueError(GENDER, mProviderName, GENDER + " value should be one of male/female/unknown."));
        }
    }

    private void validateClientSideCallbacks(String value, ConfigValidationResult result) {
        validateBoolean(CLIENT_SIDE_CALLBACKS, value, result);
    }

    @Override
    protected void validateMandatoryField(JSONObject config, String key, ConfigValidationResult result) {
        try {
            //get the value of the key which is under validation
            String value = config.optString(key);

            if (APPLICATION_KEY.equals(key))
                validateApplicationKey(value, result);

            else if (USER_ID.equals(key))
                validateUserId(value, result);

            else if (DYNAMIC_CONTROLLER_URL.equals(key))
                validateDynamicUrl(value, result);
        } catch (Throwable e) {
            result.setInvalid(ErrorBuilder.buildInvalidKeyValueError(key, mProviderName, null));
        }
    }

    @Override
    protected void adapterPostValidation(JSONObject config, ConfigValidationResult result) {
        try {
            validatePrivateKeyItemNameCountCombination(config, result);
        } catch (Exception e) {
            result.setInvalid(ErrorBuilder.buildGenericError(""));
        }
    }

    /**
     * //Validates the combination of PRIVATE_KEY, ITEM_NAME and ITEM_COUNT
     *
     * @param config - JSONObject represents the configurations sections of adapter's configurations.
     * @param result - ConfigValidationResult - represents the result of the validation. If the validation status is changed this object fields will be updated.
     */
    private void validatePrivateKeyItemNameCountCombination(JSONObject config, ConfigValidationResult result) {

        if (config.has(APPLICATION_PRIVATE_KEY) || config.has(ITEM_NAME) || config.has(ITEM_COUNT)) {
            //if not all exist
            if (!(config.has(APPLICATION_PRIVATE_KEY) && config.has(ITEM_NAME) && config.has(ITEM_COUNT))) {
                //if we want to remove the invalid optional keys -> remove the comment below
                //config.remove(ITEM_NAME);
                //config.remove(ITEM_COUNT);
                result.setInvalid(ErrorBuilder.buildInvalidKeyValueError(ITEM_NAME + ", " + ITEM_COUNT + " or " + APPLICATION_PRIVATE_KEY, mProviderName,
                        "configure " + ITEM_NAME + "/" + ITEM_COUNT + " requires the following configurations: " + ITEM_NAME + ", " + ITEM_COUNT + " and " + APPLICATION_PRIVATE_KEY));
            }
        }
    }

    private void validateUserId(String value, ConfigValidationResult result) {
        if (value != null) {
            if (value.length() < 1 || value.length() > 64)
                result.setInvalid(ErrorBuilder.buildInvalidKeyValueError(USER_ID, mProviderName, USER_ID + " value should be between 1-64 characters"));
        } else {
            result.setInvalid(ErrorBuilder.buildInvalidKeyValueError(USER_ID, mProviderName, USER_ID + " is missing"));
        }
    }

    private void validateDynamicUrl(String value, ConfigValidationResult result) {
        if (TextUtils.isEmpty(value)) {
            result.setInvalid(ErrorBuilder.buildInvalidKeyValueError(DYNAMIC_CONTROLLER_URL, mProviderName, DYNAMIC_CONTROLLER_URL + " is missing"));
        }
    }

    private void validateApplicationKey(String value, ConfigValidationResult result) {
        if (value != null) {
            value = value.trim();

            //Check application key length
            if (value.length() >= 5 && value.length() <= 10) {
                //check application key is alpha numeric
                String pattern = "^[a-zA-Z0-9]*$";
                if (!value.matches(pattern)) {
                    result.setInvalid(ErrorBuilder.buildInvalidKeyValueError(APPLICATION_KEY, mProviderName,
                            APPLICATION_KEY + " value should contains only english characters and numbers"));
                }
            } else {
                result.setInvalid(ErrorBuilder.buildInvalidKeyValueError(APPLICATION_KEY, mProviderName,
                        APPLICATION_KEY + " length should be between 5-10 characters"));
            }
        } else {
            result.setInvalid(ErrorBuilder.buildInvalidKeyValueError(APPLICATION_KEY, mProviderName, APPLICATION_KEY + " value is missing"));
        }
    }
}
