package com.ironsource.adapters.supersonicads;

import android.app.Activity;
import android.text.TextUtils;

import com.ironsource.mediationsdk.AbstractAdapter;
import com.ironsource.mediationsdk.IronSourceObject;
import com.ironsource.mediationsdk.config.ConfigValidationResult;
import com.ironsource.mediationsdk.logger.IronSourceError;
import com.ironsource.mediationsdk.logger.IronSourceLogger.IronSourceLogLevel;
import com.ironsource.mediationsdk.logger.IronSourceLogger.IronSourceTag;
import com.ironsource.mediationsdk.logger.IronSourceLoggerManager;
import com.ironsource.mediationsdk.model.Placement;
import com.ironsource.mediationsdk.sdk.InternalOfferwallApi;
import com.ironsource.mediationsdk.sdk.InternalOfferwallListener;
import com.ironsource.mediationsdk.sdk.InterstitialManagerListener;
import com.ironsource.mediationsdk.sdk.OfferwallListener;
import com.ironsource.mediationsdk.sdk.RewardedVideoManagerListener;
import com.ironsource.mediationsdk.utils.ErrorBuilder;
import com.ironsource.mediationsdk.utils.IronSourceConstants;
import com.ironsource.mediationsdk.utils.IronSourceUtils;
import com.ironsource.mediationsdk.utils.RewardedVideoHelper;
import com.ironsource.sdk.SSAFactory;
import com.ironsource.sdk.SSAPublisher;
import com.ironsource.sdk.data.AdUnitsReady;
import com.ironsource.sdk.listeners.OnInterstitialListener;
import com.ironsource.sdk.listeners.OnOfferWallListener;
import com.ironsource.sdk.listeners.OnRewardedVideoListener;
import com.ironsource.sdk.utils.SDKUtils;

import java.util.HashMap;
import java.util.Map;

class SupersonicAdsAdapter extends AbstractAdapter implements InternalOfferwallApi, OnOfferWallListener, OnInterstitialListener, OnRewardedVideoListener {
    private final String VERSION = "6.5.3";

    private final String TIMESTAMP = "timestamp";
    private final String ITEM_SIGNATURE = "itemSignature";
    private final String SDK_PLUGIN_TYPE = "SDKPluginType";
    private final String OW_PLACEMENT_ID = "placementId";

    private SSAPublisher mSSAPublisher;

    private RewardedVideoManagerListener mRewardedVideoManager;
    private InterstitialManagerListener mInterstitialManager;
    private InternalOfferwallListener mOfferwallListener;

    private RewardedVideoHelper mRewardedVideoHelper;

    private DemandSourceConfig mAdapterConfig;

    public static SupersonicAdsAdapter startAdapter(String providerName, String providerUrl) {
        return new SupersonicAdsAdapter(providerName, providerUrl);
    }

    private SupersonicAdsAdapter(String providerName, String providerUrl) {
        super(providerName, providerUrl);
        mRewardedVideoHelper = new RewardedVideoHelper();
        mAdapterConfig = new DemandSourceConfig(providerName);

        SDKUtils.setControllerUrl(mAdapterConfig.getRVDynamicControllerUrl());
        if (isAdaptersDebugEnabled()) {
            SDKUtils.setDebugMode(0);
        } else {
            SDKUtils.setDebugMode(mAdapterConfig.getRVDebugMode());
        }
        SDKUtils.setControllerConfig(mAdapterConfig.getRVControllerConfig());
    }

    @Override
    public int getMaxRVAdsPerIteration() {
        return mAdapterConfig.getMaxRVAdsPerIteration();
    }

    @Override
    public int getMaxISAdsPerIteration() {
        return mAdapterConfig.getMaxISAdsPerIteration();
    }

    @Override
    public String getVersion() {
        return VERSION;
    }

    @Override
    public String getCoreSDKVersion() {
        return SDKUtils.getSDKVersion();
    }

    /**
     * Creates a map and adds all extra parameters that relevant for all AdUnits: RewardedVideo, Interstitial and Offerwall
     *
     * @return Map<String,String> - Containing all extra params which relevant for all AdUnits.
     */
    private HashMap<String, String> getGenenralExtraParams() {
        HashMap<String, String> params = new HashMap<>();
        DemandSourceConfig config = mAdapterConfig;

        //Check and add user age group configuration
        String ageGroup = config.getRVUserAgeGroup();
        if (!TextUtils.isEmpty(ageGroup)) {
            params.put(DemandSourceConfig.APPLICATION_USER_AGE_GROUP, ageGroup);
        }

        //Check and add user creation gender configuration
        String uGender = config.getRVUserGender();
        if (!TextUtils.isEmpty(uGender)) {
            params.put(DemandSourceConfig.APPLICATION_USER_GENDER, uGender);
        }

        String pluginType = getPluginType();
        if (!TextUtils.isEmpty(pluginType))
            params.put(SDK_PLUGIN_TYPE, pluginType);

        return params;
    }

    /**
     * Creates a map and adds all extra parameters that relevant RewardedVideo.
     *
     * @return Map<String,String> - Containing all extra params which relevant for RewardedVideo.
     */
    private HashMap<String, String> getRewardedVideoExtraParams() {

        HashMap<String, String> rvExtraParams = getGenenralExtraParams();

        DemandSourceConfig config = mAdapterConfig;

        //Check and add language configuration
        String language = config.getLanguage();
        if (!TextUtils.isEmpty(language)) {
            rvExtraParams.put(DemandSourceConfig.LANGUAGE, language);
        }

        //Check and add maxVideoLength configuration
        String maxVideoLength = config.getMaxVideoLength();
        if (!TextUtils.isEmpty(maxVideoLength)) {
            rvExtraParams.put(DemandSourceConfig.MAX_VIDEO_LENGTH, maxVideoLength);
        }

        //Check and add campaignId configuration
        String campaignId = config.getCampaignId();
        if (!TextUtils.isEmpty(campaignId))
            rvExtraParams.put(DemandSourceConfig.CAMPAIGN_ID, campaignId);

        //Check and add segment configuration
        String segment = config.getMediationSegment();
        if (!TextUtils.isEmpty(segment)) {
            rvExtraParams.put(DemandSourceConfig.CUSTOM_SEGMENT, segment);
        }

        //Check and add itemName/itemCount/privateKey configurations
        addItemNameCountSignature(rvExtraParams);

        //Add RewardedVideo customParams if set
        Map<String, String> customParams = SupersonicConfig.getConfigObj().getRewardedVideoCustomParams();
        if (customParams != null && !customParams.isEmpty())
            rvExtraParams.putAll(customParams);

        return rvExtraParams;
    }

    /**
     * Creates a map and adds all extra parameters that relevant Interstitial.
     *
     * @return Map<String,String> - Containing all extra params which relevant for Interstitial.
     */
    private HashMap<String, String> getInterstitialExtraParams() {
        return getGenenralExtraParams();
    }

    /**
     * Creates a map and adds all extra parameters that relevant Offerwall.
     *
     * @return Map<String,String> - Containing all extra params which relevant for Offerwall.
     */
    private HashMap<String, String> getOfferwallExtraParams() {
        HashMap<String, String> owExtraParams = getGenenralExtraParams();

        //Check language configuration
        String language = mAdapterConfig.getLanguage();
        if (!TextUtils.isEmpty(language)) {
            owExtraParams.put(DemandSourceConfig.LANGUAGE, language);
        }

        //Check client side callbacks configuration
        boolean clientSideCallbacks = SupersonicConfig.getConfigObj().getClientSideCallbacks();
        owExtraParams.put(DemandSourceConfig.CLIENT_SIDE_CALLBACKS, String.valueOf(clientSideCallbacks));

        //Add Offerwall customParams if set
        Map<String, String> customParams = SupersonicConfig.getConfigObj().getOfferwallCustomParams();
        if (customParams != null && !customParams.isEmpty())
            owExtraParams.putAll(customParams);

        addItemNameCountSignature(owExtraParams);

        return owExtraParams;
    }

    private void addItemNameCountSignature(HashMap<String, String> params) {
        try {
            String itemName = mAdapterConfig.getItemName();
            int itemCount = mAdapterConfig.getItemCount();
            String privateKey = mAdapterConfig.getPrivateKey();

            boolean shouldAddSignature = true;

            if (TextUtils.isEmpty(itemName))
                shouldAddSignature = false;
            else
                params.put(DemandSourceConfig.ITEM_NAME, itemName);

            if (TextUtils.isEmpty(privateKey))
                shouldAddSignature = false;

            if (itemCount == -1)
                shouldAddSignature = false;
            else
                params.put(DemandSourceConfig.ITEM_COUNT, String.valueOf(itemCount));


            if (shouldAddSignature) {
                int timestamp = IronSourceUtils.getCurrentTimestamp();
                params.put(TIMESTAMP, String.valueOf(timestamp));
                params.put(ITEM_SIGNATURE, createItemSig(timestamp, itemName, itemCount, privateKey));
            }

        } catch (Exception e) {
            IronSourceLoggerManager.getLogger().logException(IronSourceTag.ADAPTER_API, " addItemNameCountSignature", e);
        }
    }

    private String createItemSig(int timestamp, String itemName, int itemCount, String privateKey) {
        return IronSourceUtils.getMD5(timestamp + itemName + itemCount + privateKey);
    }

    private String createMinimumOfferCommissionSig(double min, String privateKey) {
        return IronSourceUtils.getMD5(min + privateKey);
    }

    private String createUserCreationDateSig(String userid, String uCreationDate, String privateKey) {
        return IronSourceUtils.getMD5(userid + uCreationDate + privateKey);
    }

    /**
     * RewardedVideoApi
     **/
    @Override
    public void initRewardedVideo(final Activity activity, final String appKey, final String userId) {
        log(IronSourceTag.ADAPTER_API, getProviderName() + ":initRewardedVideo(userId:" + userId + ")", IronSourceLogLevel.INFO);

        mRewardedVideoHelper.reset();                                            //reset the state of RewardedVideoHelper for the next init

        ConfigValidationResult validationResult = validateConfigBeforeInitAndCallAvailabilityChangedForInvalid(mAdapterConfig, mRewardedVideoManager);
        if (!validationResult.isValid()) {
            return;
        }

        mRewardedVideoHelper.setMaxVideo(mAdapterConfig.getMaxVideos());

        startRVTimer(mRewardedVideoManager);
        activity.runOnUiThread(new Runnable() {

            @Override
            public void run() {

                mSSAPublisher = SSAFactory.getPublisherInstance(activity);

                HashMap<String, String> rewardedVideoExtraParams = getRewardedVideoExtraParams();

                log(IronSourceTag.ADAPTER_API, getProviderName() + ":initRewardedVideo(appKey:" + appKey + ", userId:" + userId +
                        ", demandSource: " + getProviderName() + ", extraParams:" + rewardedVideoExtraParams + ")", IronSourceLogLevel.INFO);

                mSSAPublisher.initRewardedVideo(appKey, userId, getProviderName(), rewardedVideoExtraParams, SupersonicAdsAdapter.this);
            }

        });
    }

    @Override
    public void onPause(Activity activity) {
        log(IronSourceTag.ADAPTER_API, getProviderName() + ":onPause()", IronSourceLogLevel.INFO);

        if (mSSAPublisher != null)
            mSSAPublisher.onPause(activity);
    }

    @Override
    public void setAge(int age) {
        mAdapterConfig.setUserAgeGroup(age);
        log(IronSourceTag.ADAPTER_API, getProviderName() + ":setAge(age:" + age + ")", IronSourceLogLevel.INFO);
    }

    @Override
    public void setGender(String gender) {
        mAdapterConfig.setUserGender(gender);
        log(IronSourceTag.ADAPTER_API, getProviderName() + ":setGender(gender:" + gender + ")", IronSourceLogLevel.INFO);
    }

    @Override
    public void setMediationSegment(String segment) {
        mAdapterConfig.setMediationSegment(segment);
        log(IronSourceTag.ADAPTER_API, getProviderName() + ":setMediationSegment(segment:" + segment + ")", IronSourceLogLevel.INFO);
    }

    @Override
    public void onResume(Activity activity) {
        log(IronSourceTag.ADAPTER_API, getProviderName() + ":onResume()", IronSourceLogLevel.INFO);

        if (mSSAPublisher != null) {
            mSSAPublisher.onResume(activity);
        }
    }

    @Override
    public synchronized boolean isRewardedVideoAvailable() {
        boolean availability = mRewardedVideoHelper.isVideoAvailable();

        log(IronSourceTag.ADAPTER_API, getProviderName() + ":isRewardedVideoAvailable():" + availability, IronSourceLogLevel.INFO);

        return availability;
    }

    @Override
    public void setRewardedVideoListener(RewardedVideoManagerListener rewardedVideoManager) {
        mRewardedVideoManager = rewardedVideoManager;
    }

    @Override
    public void showRewardedVideo() {

    }

    @Override
    public void showRewardedVideo(String placementName) {
        log(IronSourceTag.ADAPTER_API, getProviderName() + ":showRewardedVideo(placement:" + placementName + ")", IronSourceLogLevel.INFO);

        boolean shouldNotify;

        if (mSSAPublisher != null) {
            mSSAPublisher.showRewardedVideo(getProviderName());
            mRewardedVideoHelper.setPlacementName(placementName);

            shouldNotify = mRewardedVideoHelper.increaseCurrentVideo();
        } else {
            shouldNotify = mRewardedVideoHelper.setVideoAvailability(false);
            log(IronSourceTag.NATIVE, "Please call init before calling showRewardedVideo", IronSourceLogLevel.WARNING);
            mRewardedVideoManager.onRewardedVideoAdShowFailed(new IronSourceError(IronSourceError.ERROR_CODE_NO_ADS_TO_SHOW, "Please call init before calling showRewardedVideo"), SupersonicAdsAdapter.this);
        }

        if (mRewardedVideoManager != null && shouldNotify) {
            mRewardedVideoManager.onRewardedVideoAvailabilityChanged(mRewardedVideoHelper.isVideoAvailable(), SupersonicAdsAdapter.this);
        }
    }

    @Override
    public void onRVNoMoreOffers() {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onRVNoMoreOffers ", IronSourceLogLevel.INFO);

        cancelRVTimer();
        boolean shouldNotify = mRewardedVideoHelper.setVideoAvailability(false);

        if (mRewardedVideoManager != null && shouldNotify) {
            mRewardedVideoManager.onRewardedVideoAvailabilityChanged(mRewardedVideoHelper.isVideoAvailable(), SupersonicAdsAdapter.this);
        }
    }

    @Override
    public void onRVInitSuccess(AdUnitsReady aur) {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onRVInitSuccess ", IronSourceLogLevel.INFO);

        cancelRVTimer();

        int numOfAdUnits = 0;

        try {
            numOfAdUnits = Integer.parseInt(aur.getNumOfAdUnits());
        } catch (NumberFormatException e) {
            IronSourceLoggerManager.getLogger().logException(IronSourceTag.NATIVE, "onRVInitSuccess:parseInt()", e);
        }
        boolean availability = numOfAdUnits > 0;

        boolean shouldNotify = mRewardedVideoHelper.setVideoAvailability(availability);

        if (mRewardedVideoManager != null) {
            if (shouldNotify)
                mRewardedVideoManager.onRewardedVideoAvailabilityChanged(mRewardedVideoHelper.isVideoAvailable(), SupersonicAdsAdapter.this);
        }
    }

    @Override
    public void onRVInitFail(String error) {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onRVInitFail ", IronSourceLogLevel.INFO);

        cancelRVTimer();

        boolean shouldNotify = mRewardedVideoHelper.setVideoAvailability(false);
        if (shouldNotify && mRewardedVideoManager != null) {
            mRewardedVideoManager.onRewardedVideoAvailabilityChanged(mRewardedVideoHelper.isVideoAvailable(), SupersonicAdsAdapter.this);
        }
    }

    @Override
    public void onRVAdClicked() {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onRVAdClicked ", IronSourceLogLevel.INFO);
        //TODO: need add implementation of the callback
    }

    @Override
    public void onRVShowFail(String error) {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onRVShowFail ", IronSourceLogLevel.INFO);

        if (mRewardedVideoManager != null)
            mRewardedVideoManager.onRewardedVideoAdShowFailed(new IronSourceError(IronSourceError.ERROR_CODE_NO_ADS_TO_SHOW, error), SupersonicAdsAdapter.this);
    }

    @Override
    public void onRVAdCredited(int amount) {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onRVAdCredited ", IronSourceLogLevel.INFO);

        if (mRewardedVideoManager != null) {
            Placement placement = mRewardedVideoConfig.getRewardedVideoPlacement(mRewardedVideoHelper.getPlacementName());
            mRewardedVideoManager.onRewardedVideoAdRewarded(placement, this);
        }
    }

    @Override
    public void onRVAdClosed() {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onRVAdClosed ", IronSourceLogLevel.INFO);

        if (mRewardedVideoManager != null)
            mRewardedVideoManager.onRewardedVideoAdClosed(this);
    }

    @Override
    public void onRVAdOpened() {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onRVAdOpened ", IronSourceLogLevel.INFO);

        if (mRewardedVideoManager != null)
            mRewardedVideoManager.onRewardedVideoAdOpened(SupersonicAdsAdapter.this);
    }

    /**
     * OfferwallApi
     **/
    @Override
    public void getOfferwallCredits() {

        if (mSSAPublisher != null) {
            String appKey = IronSourceObject.getInstance().getIronSourceAppKey();
            String userId = IronSourceObject.getInstance().getIronSourceUserId();

            log(IronSourceTag.ADAPTER_API, getProviderName() + ":getOfferwallCredits(appKey:" + appKey +
                    "userId:" + userId + ")", IronSourceLogLevel.INFO);

            mSSAPublisher.getOfferWallCredits(appKey, userId, this);
        } else
            log(IronSourceTag.NATIVE, "Please call init before calling getOfferwallCredits", IronSourceLogLevel.WARNING);
    }

    @Override
    public void setOfferwallListener(OfferwallListener owListener) {
        // not used here, check setInternalOfferwallListener
    }

    @Override
    public void setInternalOfferwallListener(InternalOfferwallListener listener) {
        mOfferwallListener = listener;
    }

    @Override
    public void initOfferwall(final Activity activity, final String appKey, final String userId) {
        try {
            log(IronSourceTag.ADAPTER_API, getProviderName() + ":initOfferwall(appKey:" + appKey + ", userId:" + userId + ")", IronSourceLogLevel.INFO);
            activity.runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    Map<String, String> offerwallExtraParams = getOfferwallExtraParams();
                    mSSAPublisher = SSAFactory.getPublisherInstance(activity);
                    mSSAPublisher.initOfferWall(appKey, userId, offerwallExtraParams, SupersonicAdsAdapter.this);

                }
            });
        } catch (Exception e) {
            IronSourceLoggerManager.getLogger().logException(IronSourceTag.ADAPTER_API, getProviderName() + ":initOfferwall(userId:" + userId + ")", e);
            mOfferwallListener.onOfferwallAvailable(false,
                    ErrorBuilder.buildInitFailedError("Adapter initialization failure - " + getProviderName() + " - " + e.getMessage(), IronSourceConstants.OFFERWALL_AD_UNIT));
        }
    }

    @Override
    public void showOfferwall() {
    }

    @Override
    public void showOfferwall(String placementId) {
        Map<String, String> offerwallExtraParams = getOfferwallExtraParams();
        if (offerwallExtraParams != null) {
            offerwallExtraParams.put(OW_PLACEMENT_ID, placementId);
        }
        log(IronSourceTag.ADAPTER_API, getProviderName() + ":showOfferwall("
                + "extraParams:" + offerwallExtraParams + ")", IronSourceLogLevel.INFO);

        if (mSSAPublisher != null) {
            mSSAPublisher.showOfferWall(offerwallExtraParams);
        } else {
            log(IronSourceTag.NATIVE, "Please call init before calling showOfferwall", IronSourceLogLevel.WARNING);
        }
    }

    @Override
    public boolean isOfferwallAvailable() {
        return true;
    }

    @Override
    public void onOWShowSuccess(String placementId) {
        if (TextUtils.isEmpty(placementId)) {
            log(IronSourceTag.ADAPTER_API, getProviderName() + ":onOWShowSuccess()", IronSourceLogLevel.INFO);
        } else {
            log(IronSourceTag.ADAPTER_API, getProviderName() + ":onOWShowSuccess(placementId:" + placementId + ")", IronSourceLogLevel.INFO);
        }

        if (mOfferwallListener != null) {
            mOfferwallListener.onOfferwallOpened();
        }
    }

    @Override
    public void onOWShowFail(String desc) {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onOWShowFail ", IronSourceLogLevel.INFO);

        if (mOfferwallListener != null) {
            IronSourceError sse = ErrorBuilder.buildGenericError(desc);
            mOfferwallListener.onOfferwallShowFailed(sse);
        }
    }

    @Override
    public void onOWGeneric(String arg0, String arg1) {
        // Deprecated
    }

    @Override
    public boolean onOWAdCredited(int credits, int totalCredits, boolean totalCreditsFlag) {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onOWAdCredited ", IronSourceLogLevel.INFO);

        return mOfferwallListener != null && mOfferwallListener.onOfferwallAdCredited(credits, totalCredits, totalCreditsFlag);
    }

    @Override
    public void onOWAdClosed() {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onOWAdClosed ", IronSourceLogLevel.INFO);

        if (mOfferwallListener != null)
            mOfferwallListener.onOfferwallClosed();
    }

    @Override
    public void onGetOWCreditsFailed(String desc) {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onGetOWCreditsFailed ", IronSourceLogLevel.INFO);

        if (mOfferwallListener != null) {
            IronSourceError sse = ErrorBuilder.buildGenericError(desc);
            mOfferwallListener.onGetOfferwallCreditsFailed(sse);
        }
    }

    @Override
    public void onOfferwallInitSuccess() {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onOfferwallInitSuccess ", IronSourceLogLevel.INFO);

        if (mOfferwallListener != null) {
            mOfferwallListener.onOfferwallAvailable(true);
        }
    }

    @Override
    public void onOfferwallInitFail(String description) {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onOfferwallInitFail ", IronSourceLogLevel.INFO);

        if (mOfferwallListener != null) {
            IronSourceError sse = ErrorBuilder.buildGenericError(description);
            mOfferwallListener.onOfferwallAvailable(false, sse);
        }
    }
    //End of OfferwallListener

    /**
     * InterstitialAdapterApi
     **/
    @Override
    public void setInterstitialListener(InterstitialManagerListener manager) {
        mInterstitialManager = manager;
    }

    @Override
    public void initInterstitial(final Activity activity, final String appKey, final String userId) {
        ConfigValidationResult validationResult = validateConfigBeforeInitAndCallInitFailForInvalid(mAdapterConfig, mInterstitialManager);
        if (!validationResult.isValid()) {
            return;
        }

        startISInitTimer(mInterstitialManager);

        activity.runOnUiThread(new Runnable() {
            @Override
            public void run() {
                mSSAPublisher = SSAFactory.getPublisherInstance(activity);

                final HashMap<String, String> interstitialExtraParams = getInterstitialExtraParams();

                log(IronSourceTag.ADAPTER_API, getProviderName() + ":initInterstitial(appKey:" + appKey + ", userId:" + userId +
                        ", extraParams:" + interstitialExtraParams + ")", IronSourceLogLevel.INFO);

                mSSAPublisher.initInterstitial(appKey, userId, interstitialExtraParams, SupersonicAdsAdapter.this);
            }
        });
    }

    @Override
    public void loadInterstitial() {
        startISLoadTimer(mInterstitialManager);
        if (mSSAPublisher != null) {
            mSSAPublisher.loadInterstitial();
        } else {
            log(IronSourceTag.NATIVE, "Please call initInterstitial before calling loadInterstitial", IronSourceLogLevel.WARNING);
        }
    }

    @Override
    public void showInterstitial() {

    }

    @Override
    public void showInterstitial(String placementName) {
        if (mSSAPublisher != null) {
            mSSAPublisher.showInterstitial();
        } else
            log(IronSourceTag.NATIVE, "Please call loadInterstitial before calling showInterstitial", IronSourceLogLevel.WARNING);
    }

    @Override
    public boolean isInterstitialReady() {
        return mSSAPublisher != null && mSSAPublisher.isInterstitialAdAvailable();
    }

    @Override
    public void onInterstitialInitSuccess() {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onInterstitialInitSuccess ", IronSourceLogLevel.INFO);

        cancelISInitTimer();
        if (mInterstitialManager != null)
            mInterstitialManager.onInterstitialInitSuccess(SupersonicAdsAdapter.this);
    }

    @Override
    public void onInterstitialInitFailed(String description) {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onInterstitialInitFailed ", IronSourceLogLevel.INFO);

        cancelISInitTimer();

        if (mInterstitialManager != null) {
            mInterstitialManager.onInterstitialInitFailed(ErrorBuilder.buildInitFailedError("Adapter initialization failure - " + getProviderName() + " - " + description, IronSourceConstants.INTERSTITIAL_AD_UNIT), SupersonicAdsAdapter.this);
        }
    }

    @Override
    public void onInterstitialLoadSuccess() {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onInterstitialLoadSuccess ", IronSourceLogLevel.INFO);

        cancelISLoadTimer();
        if (mInterstitialManager != null) {
            mInterstitialManager.onInterstitialAdReady(this);
        }
    }

    @Override
    public void onInterstitialLoadFailed(String description) {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onInterstitialAdLoadFailed ", IronSourceLogLevel.INFO);

        cancelISLoadTimer();
        if (mInterstitialManager != null) {
            String errorString = "" + IronSourceConstants.INTERSTITIAL_AD_UNIT + " Load Fail, "
                    + getProviderName() + " - " + description;
            IronSourceError error = ErrorBuilder.buildLoadFailedError(errorString);
            mInterstitialManager.onInterstitialAdLoadFailed(error, SupersonicAdsAdapter.this);
        }
    }

    @Override
    public void onInterstitialOpen() {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onInterstitialAdOpened ", IronSourceLogLevel.INFO);

        if (mInterstitialManager != null)
            mInterstitialManager.onInterstitialAdOpened(SupersonicAdsAdapter.this);
    }

    @Override
    public void onInterstitialClose() {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onInterstitialAdClosed ", IronSourceLogLevel.INFO);

        if (mInterstitialManager != null)
            mInterstitialManager.onInterstitialAdClosed(SupersonicAdsAdapter.this);
    }

    @Override
    public void onInterstitialShowSuccess() {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onInterstitialAdShowSucceeded ", IronSourceLogLevel.INFO);

        if (mInterstitialManager != null)
            mInterstitialManager.onInterstitialAdShowSucceeded(SupersonicAdsAdapter.this);
    }

    @Override
    public void onInterstitialShowFailed(String description) {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onInterstitialAdShowFailed ", IronSourceLogLevel.INFO);

        if (mInterstitialManager != null) {
            mInterstitialManager.onInterstitialAdShowFailed(ErrorBuilder.buildShowFailedError(IronSourceConstants.INTERSTITIAL_AD_UNIT, description), SupersonicAdsAdapter.this);
        }
    }

    @Override
    public void onInterstitialClick() {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + " :onInterstitialAdClicked ", IronSourceLogLevel.INFO);

        if (mInterstitialManager != null)
            mInterstitialManager.onInterstitialAdClicked(SupersonicAdsAdapter.this);
    }
}
