package com.ironsource.adapters.ironsource;

import android.app.Activity;
import android.content.Context;
import android.text.TextUtils;

import com.ironsource.mediationsdk.AbstractAdapter;
import com.ironsource.mediationsdk.AuctionDataUtils;
import com.ironsource.mediationsdk.BuildConfig;
import com.ironsource.mediationsdk.logger.IronSourceError;
import com.ironsource.mediationsdk.logger.IronSourceLogger;
import com.ironsource.mediationsdk.logger.IronSourceLoggerManager;
import com.ironsource.mediationsdk.sdk.InterstitialSmashListener;
import com.ironsource.mediationsdk.sdk.RewardedVideoSmashListener;
import com.ironsource.mediationsdk.utils.ErrorBuilder;
import com.ironsource.mediationsdk.utils.IronSourceConstants;
import com.ironsource.mediationsdk.utils.IronSourceUtils;
import com.ironsource.mediationsdk.utils.SessionDepthManager;
import com.ironsource.sdk.IronSourceAdInstance;
import com.ironsource.sdk.IronSourceAdInstanceBuilder;
import com.ironsource.sdk.IronSourceNetwork;
import com.ironsource.sdk.constants.Constants;
import com.ironsource.sdk.listeners.OnInterstitialListener;
import com.ironsource.sdk.utils.SDKUtils;
import com.ironsource.mediationsdk.metadata.MetaDataConstants;
import com.ironsource.mediationsdk.metadata.MetaDataUtils;
import com.ironsource.mediationsdk.logger.IronSourceLogger.IronSourceTag;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;

/*
This adapter is responsible for addressing several flows
    - mediation (reward video is loaded automatically)
            -  non bidder - reward video and interstitial
            -  bidder - reward video and interstitial
    - demand only  (reward video is loaded manually)
            - non bidder - reward video and interstitial
            - demand only - bidder - reward video and interstitial
*/
public class IronSourceAdapter extends AbstractAdapter {
    private final static String VERSION = BuildConfig.VERSION_NAME;
    private final static int IS_LOAD_EXCEPTION = 1000;
    private final static int IS_SHOW_EXCEPTION = 1001;
    private final static int RV_LOAD_EXCEPTION = 1002;
    private final static int RV_SHOW_EXCEPTION = 1003;

    private final String DYNAMIC_CONTROLLER_URL = "controllerUrl";
    private final String DYNAMIC_CONTROLLER_DEBUG_MODE = "debugMode";
    private final String DYNAMIC_CONTROLLER_CONFIG = "controllerConfig";

    private final String APPLICATION_USER_GENDER = "applicationUserGender";
    private final String APPLICATION_USER_AGE_GROUP = "applicationUserAgeGroup";
    private final String SDK_PLUGIN_TYPE = "SDKPluginType";
    private final String CUSTOM_SEGMENT = "custom_Segment";
    private final String ADM_KEY = "adm";

    private final String DEMAND_SOURCE_NAME = "demandSourceName";

    private ConcurrentHashMap<String, IronSourceAdInstance> mDemandSourceToISAd;
    private ConcurrentHashMap<String, IronSourceAdInstance> mDemandSourceToRvAd;

    private ConcurrentHashMap<String, RewardedVideoSmashListener> mDemandSourceToRvSmash;
    private ConcurrentHashMap<String, InterstitialSmashListener> mDemandSourceToISSmash;


    private static AtomicBoolean mDidInitSdk = new AtomicBoolean(false);
    private String mUserAgeGroup;
    private String mUserGender;
    private String mMediationSegment;
    private Context mContext;


    public static IronSourceAdapter startAdapter(String providerName) {
        return new IronSourceAdapter(providerName);
    }

    private IronSourceAdapter(String providerName) {
        super(providerName);
        log(IronSourceTag.INTERNAL, providerName + ": new instance");
        mDemandSourceToISAd = new ConcurrentHashMap<>();
        mDemandSourceToRvAd = new ConcurrentHashMap<>();
        mDemandSourceToRvSmash = new ConcurrentHashMap<>();
        mDemandSourceToISSmash = new ConcurrentHashMap<>();
        mUserAgeGroup = null;
        mUserGender = null;
        mMediationSegment = null;
    }

    @Override
    public String getVersion() {
        return VERSION;
    }

    @Override
    public String getCoreSDKVersion() {
        return SDKUtils.getSDKVersion();
    }

    // ********** Base **********

    @Override
    public void onPause(Activity activity) {
        log(IronSourceTag.ADAPTER_API, "IronSourceNetwork.onPause");
        IronSourceNetwork.onPause(activity);
    }

    @Override
    public void onResume(Activity activity) {
        log(IronSourceTag.ADAPTER_API, "IronSourceNetwork.onResume");
        IronSourceNetwork.onResume(activity);
    }

    protected void setConsent(boolean consent) {
        log(IronSourceTag.ADAPTER_API, "setConsent (" + (consent ? "true" : "false") + ")" );

        JSONObject consentParams = new JSONObject();
        try {
            consentParams.put(Constants.RequestParameters.GDPR_CONSENT_STATUS, String.valueOf(consent));
        } catch (JSONException e) {
            logError(IronSourceTag.ADAPTER_API, "setConsent exception " + e.getMessage());
            return;
        }

        IronSourceNetwork.updateConsentInfo(consentParams);
    }

    @Override
    public void setAge(int age) {
        log(IronSourceTag.INTERNAL, "setAge: " + age);
        if (age >= 13 && age <= 17) {
            mUserAgeGroup = "1";
        } else if (age >= 18 && age <= 20) {
            mUserAgeGroup = "2";
        } else if (age >= 21 && age <= 24) {
            mUserAgeGroup = "3";
        } else if (age >= 25 && age <= 34) {
            mUserAgeGroup = "4";
        } else if (age >= 35 && age <= 44) {
            mUserAgeGroup = "5";
        } else if (age >= 45 && age <= 54) {
            mUserAgeGroup = "6";
        } else if (age >= 55 && age <= 64) {
            mUserAgeGroup = "7";
        } else if (age > 65 && age <= 120) {
            mUserAgeGroup = "8";
        } else {
            mUserAgeGroup = "0";
        }
    }

    @Override
    public void setGender(String gender) {
        log(IronSourceTag.INTERNAL, "setGender: " + gender);
        mUserGender = gender;
    }

    @Override
    public void setMediationSegment(String segment) {
        mMediationSegment = segment;
    }

    private HashMap<String, String> getInitParams() {
        HashMap<String, String> params = new HashMap<>();

        if (!TextUtils.isEmpty(mUserAgeGroup)) {
            params.put(APPLICATION_USER_AGE_GROUP, mUserAgeGroup);
        }

        if (!TextUtils.isEmpty(mUserGender)) {
            params.put(APPLICATION_USER_GENDER, mUserGender);
        }

        String pluginType = getPluginType();
        if (!TextUtils.isEmpty(pluginType)) {
            params.put(SDK_PLUGIN_TYPE, pluginType);
        }

        if (!TextUtils.isEmpty(mMediationSegment)) {
            params.put(CUSTOM_SEGMENT, mMediationSegment);
        }

        return params;
    }

    // ********** Init SDK **************

    @Override
    public void earlyInit (final Activity activity, final String appKey, final String userId, final JSONObject config) {
        IronSourceUtils.sendAutomationLog(getDemandSourceName(config) + ": " + "earlyInit");
        initSDK(activity, appKey, userId, config);
    }

    // ********** Interstitial **********


    @Override
    public Map<String, Object> getIsBiddingData(JSONObject config) {
        log(IronSourceTag.ADAPTER_API, "getIsBiddingData");
        Map<String, Object> ret = new HashMap<>();
        String token = IronSourceNetwork.getToken(mContext);

        if (token != null) {
            ret.put(IronSourceConstants.IRONSOURCE_BIDDING_TOKEN_KEY, token);
        }
        else {
            logError(IronSourceTag.ADAPTER_API, "IS bidding token is null");
            ret.put(IronSourceConstants.IRONSOURCE_BIDDING_TOKEN_KEY, "");
        }

        return ret;
    }

    // used for bidders in the mediation
    @Override
    public void initInterstitialForBidding(Activity activity, String appKey, String userId, JSONObject config, InterstitialSmashListener listener) {
        String demandSourceName = getDemandSourceName(config);
        log(IronSourceTag.INTERNAL, config, "initInterstitialForBidding demandSourceName=" + demandSourceName);
        initInterstitialInternal(activity, appKey, userId, config, listener, demandSourceName);
    }

    // used for non bidders in the mediation and bidders in demand only
    @Override
    public void initInterstitial(Activity activity, String appKey, String userId, JSONObject config, InterstitialSmashListener listener) {
        String demandSourceName = getDemandSourceName(config);
        log(IronSourceTag.INTERNAL, config, "initInterstitial demandSourceName=" + demandSourceName);
        initInterstitialInternal(activity, appKey, userId, config, listener, demandSourceName);
    }


    private void initInterstitialInternal(Activity activity, String appKey, String userId, JSONObject config, InterstitialSmashListener listener, String demandSourceName) {
        initSDK(activity, appKey, userId, config);
        mDemandSourceToISSmash.put(demandSourceName,listener);
        listener.onInterstitialInitSuccess();
    }

    // used for bidders in the mediation and demand only
    @Override
    public void loadInterstitial(final JSONObject config, final InterstitialSmashListener listener, final String serverData) {
        log(IronSourceTag.ADAPTER_API, config, "loadInterstitial");
        try {
            loadAdInternal(getDemandSourceName(config), serverData, false, true, false);
        } catch (Exception e) {
            logError(IronSourceTag.ADAPTER_API, "loadInterstitial for bidding exception " + e.getMessage());
            listener.onInterstitialAdLoadFailed(new IronSourceError(IS_LOAD_EXCEPTION, e.getMessage()));
        }

    }

    // used for non bidders in the mediation and demand only
    @Override
    public void loadInterstitial(JSONObject config, InterstitialSmashListener listener) {
        log(IronSourceTag.ADAPTER_API, config, "loadInterstitial");
        try {
            loadAdInternal(getDemandSourceName(config), null, false, false, false);
        } catch (Exception e) {
            logError(IronSourceTag.ADAPTER_API, "loadInterstitial exception " + e.getMessage());
            listener.onInterstitialAdLoadFailed(new IronSourceError(IS_LOAD_EXCEPTION, e.getMessage()));
        }
    }

    @Override
    public void showInterstitial(JSONObject config, InterstitialSmashListener listener) {
        log(IronSourceTag.ADAPTER_API, config, "showInterstitial");
        try {
            IronSourceAdInstance adInstance = mDemandSourceToISAd.get(getDemandSourceName(config));
            showAdInternal(adInstance, SessionDepthManager.INTERSTITIAL);
        } catch (Exception e) {
            logError(IronSourceTag.ADAPTER_API, "showInterstitial exception " + e.getMessage());
            listener.onInterstitialAdShowFailed(new IronSourceError(IS_SHOW_EXCEPTION, e.getMessage()));
        }
    }

    @Override
    public boolean isInterstitialReady(JSONObject config) {
        IronSourceAdInstance adInstance = mDemandSourceToISAd.get(getDemandSourceName(config));
        return adInstance != null && IronSourceNetwork.isAdAvailableForInstance(adInstance);
    }

    // ********** RewardedVideo **********



    @Override
    public Map<String, Object> getRvBiddingData(JSONObject config) {
        log(IronSourceTag.ADAPTER_API, "getRvBiddingData");
        Map<String, Object> biddingData = new HashMap<>();
        String token = IronSourceNetwork.getToken(mContext);

        if (token != null) {
            biddingData.put(IronSourceConstants.IRONSOURCE_BIDDING_TOKEN_KEY, token);
        }
        else {
            logError(IronSourceTag.ADAPTER_API, "RV bidding token is null");
            biddingData.put(IronSourceConstants.IRONSOURCE_BIDDING_TOKEN_KEY, "");
        }
        return biddingData;
    }

    // used for bidders in the mediation
    @Override
    public void initRvForBidding(Activity activity, String appKey, String userId, JSONObject config, RewardedVideoSmashListener listener) {
        String demandSourceName = getDemandSourceName(config);
        log(IronSourceTag.INTERNAL, config, "initRvForBidding demandSourceName=" + demandSourceName);
        initRewardedVideoInternal(activity, appKey, userId, config, listener, demandSourceName);
        listener.onRewardedVideoInitSuccess();
    }

    // used for bidders and non bidders in demand only
    @Override
    public void initRvForDemandOnly(Activity activity, String appKey, String userId, JSONObject config, RewardedVideoSmashListener listener) {
        String demandSourceName = getDemandSourceName(config);
        log(IronSourceTag.INTERNAL, config, "initRvForDemandOnly demandSourceName=" + demandSourceName);
        initRewardedVideoInternal(activity, appKey, userId, config, listener, demandSourceName);
    }

    // used for non bidders in the mediation
    @Override
    public void initRewardedVideo(Activity activity, String appKey, String userId, JSONObject config, RewardedVideoSmashListener listener) {
        String demandSourceName = getDemandSourceName(config);
        log(IronSourceTag.INTERNAL, config, "initRewardedVideo with demandSourceName=" + demandSourceName);
        initRewardedVideoInternal(activity, appKey, userId, config, listener, demandSourceName);
        fetchRewardedVideo(config);
    }

    private void initRewardedVideoInternal(Activity activity, String appKey, String userId, JSONObject config, RewardedVideoSmashListener listener, String demandSourceName) {
        initSDK(activity, appKey, userId, config);
        mDemandSourceToRvSmash.put(demandSourceName,listener);
    }


    // get ad instance for iab, traditional, demand only, for the first time will create the ad
    private IronSourceAdInstance getAdInstance(String demandSourceName, boolean isDemandOnlyForRv, boolean isBidder, boolean isRewarded) {
        IronSourceAdInstance adInstance = isRewarded ?
                mDemandSourceToRvAd.get(demandSourceName):
                mDemandSourceToISAd.get(demandSourceName);

        if (adInstance == null) {
            log(IronSourceTag.ADAPTER_API,  "getAdInstance creating ad instance for " + demandSourceName + " isDemandOnlyForRv=" + isDemandOnlyForRv + " isBidder=" + isBidder + " isRewarded=" + isRewarded);
            IronSourceAdInstanceBuilder builder;
            if (isRewarded) {
                builder = new IronSourceAdInstanceBuilder(demandSourceName, new IronSourceRewardedVideoListener(mDemandSourceToRvSmash.get(demandSourceName), demandSourceName, isDemandOnlyForRv))
                        .setExtraParams(getInitParams());
                builder.setRewarded();
            }
            else {
                builder = new IronSourceAdInstanceBuilder(demandSourceName,new IronSourceInterstitialListener(mDemandSourceToISSmash.get(demandSourceName), demandSourceName))
                        .setExtraParams(getInitParams());
            }

            if (isBidder) {
                builder.setInAppBidding();
            }
            adInstance = builder.build();
            if (isRewarded) {
                mDemandSourceToRvAd.put(demandSourceName,adInstance);
            }
            else {
                mDemandSourceToISAd.put(demandSourceName,adInstance);
            }
        }
        return adInstance;
    }
    // TODO unify API calls

    // used for non bidders in the mediation
    @Override
    public void fetchRewardedVideo(JSONObject config) {
        log(IronSourceTag.ADAPTER_API, config, "fetchRewardedVideo");
        String demandSourceName = getDemandSourceName(config);
        try {
            loadAdInternal(demandSourceName,null, false, false, true);
        } catch (Exception e) {
            logError(IronSourceTag.ADAPTER_API, "fetchRewardedVideo exception " + e.getMessage());
            IronSourceAdInstance adInstance = mDemandSourceToRvAd.get(demandSourceName);
            if (adInstance != null && adInstance.getAdListener() != null) {
                OnInterstitialListener adListener = adInstance.getAdListener();
                adListener.onInterstitialLoadFailed("fetch exception");
            }
        }
    }

    // used for non bidders in demand only
    @Override
    public void loadVideoForDemandOnly(JSONObject config, RewardedVideoSmashListener listener) {
        log(IronSourceTag.ADAPTER_API, config, "loadVideoForDemandOnly");
        try {
            loadAdInternal(getDemandSourceName(config), null, true, false, true);
        } catch (Exception e) {
            logError(IronSourceTag.ADAPTER_API, "loadVideoForDemandOnly exception " + e.getMessage());
            listener.onRewardedVideoLoadFailed(new IronSourceError(RV_LOAD_EXCEPTION, e.getMessage()));
        }
    }

    // used for bidders in demand only
    @Override
    public void loadVideoForDemandOnly(JSONObject config, RewardedVideoSmashListener listener, String serverData) {
        log(IronSourceTag.ADAPTER_API, config, "loadVideoForDemandOnly in bidding mode");
        try {
            loadAdInternal(getDemandSourceName(config), serverData, true, true, true);
        } catch (Exception e) {
            logError(IronSourceTag.ADAPTER_API, "loadVideoForDemandOnly exception " + e.getMessage());
            listener.onRewardedVideoLoadFailed(new IronSourceError(RV_LOAD_EXCEPTION, e.getMessage()));
        }
    }

    // used  bidders in the mediation
    @Override
    public void loadVideo(JSONObject config, final RewardedVideoSmashListener listener, final String serverData) {
        log(IronSourceTag.ADAPTER_API, config, "loadVideo (RV in bidding mode)");
        try {
            loadAdInternal(getDemandSourceName(config), serverData, false, true, true);
        } catch (Exception e) {
            logError(IronSourceTag.ADAPTER_API, "loadVideo exception " + e.getMessage());
            listener.onRewardedVideoLoadFailed(new IronSourceError(RV_LOAD_EXCEPTION, e.getMessage()));
            listener.onRewardedVideoAvailabilityChanged(false);
        }
    }


    private void loadAdInternal(String demandSourceName, String serverData, boolean isDemandOnlyForRv, boolean isBidder, boolean isRewarded) throws Exception {
        HashMap<String, String> loadConfig = new HashMap<>();
        if (serverData != null) {
            loadConfig.put(ADM_KEY, AuctionDataUtils.getInstance().getAdmFromServerData(serverData));
            Map<String, String> auctionResponseServerDataParams = AuctionDataUtils.getInstance().getAuctionResponseServerDataParams(serverData);
            loadConfig.putAll(auctionResponseServerDataParams);
        }
        IronSourceAdInstance adInstance = getAdInstance(demandSourceName, isDemandOnlyForRv, isBidder, isRewarded);
        printInstanceExtraParams(loadConfig);
        log(IronSourceTag.ADAPTER_API, "loadAd demandSourceName=" + adInstance.getName());
        IronSourceNetwork.loadAd(adInstance, loadConfig);
    }

    @Override
    public void showRewardedVideo(JSONObject config, RewardedVideoSmashListener listener) {
        try {
            IronSourceAdInstance adInstance = mDemandSourceToRvAd.get(getDemandSourceName(config));
            showAdInternal(adInstance, SessionDepthManager.REWARDEDVIDEO);
        } catch (Exception e) {
            logError(IronSourceTag.ADAPTER_API, "showRewardedVideo exception " + e.getMessage());
            listener.onRewardedVideoAdShowFailed(new IronSourceError(RV_SHOW_EXCEPTION, e.getMessage()));
        }
    }


    private void showAdInternal(IronSourceAdInstance adInstance, int adUnit) throws Exception {
        int sessionDepth = SessionDepthManager.getInstance().getSessionDepth(adUnit);
        Map<String, String> showParams = new HashMap<>();
        showParams.put(Constants.RequestParameters.SESSION_DEPTH, String.valueOf(sessionDepth));
        log(IronSourceTag.ADAPTER_API, "showAd demandSourceName=" + adInstance.getName() + " showParams=" + showParams);
        IronSourceNetwork.showAd(adInstance, showParams);

    }
    @Override
    public boolean isRewardedVideoAvailable(JSONObject config) {
        IronSourceAdInstance adInstance = mDemandSourceToRvAd.get(getDemandSourceName(config));
        return adInstance != null && IronSourceNetwork.isAdAvailableForInstance(adInstance);
    }

    private void initSDK(Activity activity, String appKey, String userId, JSONObject config) {

        if (activity == null) {
            logError(IronSourceTag.INTERNAL, "initSDK: null activity");
            return;
        }
        mContext = activity.getApplicationContext();
        if (mDidInitSdk.compareAndSet(false,true)) {

            int logLevel = config.optInt(DYNAMIC_CONTROLLER_DEBUG_MODE, IronSourceLogger.IronSourceLogLevel.VERBOSE);
            if (isAdaptersDebugEnabled()) {
                logLevel = IronSourceLogger.IronSourceLogLevel.ERROR;
            }
            log(IronSourceTag.ADAPTER_API, "initSDK setting debug mode to " + logLevel);
            SDKUtils.setDebugMode(logLevel);

            SDKUtils.setControllerUrl(config.optString(DYNAMIC_CONTROLLER_URL));
            log(IronSourceTag.ADAPTER_API, "IronSourceNetwork setting controller url to  "+ config.optString(DYNAMIC_CONTROLLER_URL));

            SDKUtils.setControllerConfig(config.optString(DYNAMIC_CONTROLLER_CONFIG));
            log(IronSourceTag.ADAPTER_API, "IronSourceNetwork setting controller config to  "+ config.optString(DYNAMIC_CONTROLLER_CONFIG));

            HashMap<String, String> initParams = getInitParams();
            log(IronSourceTag.ADAPTER_API, "initSDK with appKey=" + appKey+ " userId=" + userId+ " parameters "+ initParams);
            IronSourceNetwork.initSDK(activity, appKey, userId, initParams);
        }
    }

    // ********** Callbacks **********

    private class IronSourceInterstitialListener implements OnInterstitialListener {
        private String mDemandSourceName;
        private InterstitialSmashListener mListener;

        IronSourceInterstitialListener (InterstitialSmashListener listener, String demandSourceName) {
            mDemandSourceName = demandSourceName;
            mListener = listener;
        }

        @Override
        public void onInterstitialInitSuccess() {
            //Deprecated - Old SuperSonicAds API callback
            log(IronSourceTag.ADAPTER_CALLBACK, mDemandSourceName + " interstitialListener onInterstitialInitSuccess");
        }

        @Override
        public void onInterstitialInitFailed(String description) {
            //Deprecated - Old SuperSonicAds API callback
            log(IronSourceTag.ADAPTER_CALLBACK, mDemandSourceName + " interstitialListener onInterstitialInitFailed");
        }

        @Override
        public void onInterstitialLoadSuccess() {
            log(IronSourceTag.ADAPTER_CALLBACK, mDemandSourceName + " interstitialListener onInterstitialLoadSuccess");
            mListener.onInterstitialAdReady();
        }

        @Override
        public void onInterstitialLoadFailed(String description) {
            log(IronSourceTag.ADAPTER_CALLBACK, mDemandSourceName + " interstitialListener onInterstitialLoadFailed " + description);
            mListener.onInterstitialAdLoadFailed(ErrorBuilder.buildLoadFailedError(description));
        }

        @Override
        public void onInterstitialOpen() {
            log(IronSourceTag.ADAPTER_CALLBACK, mDemandSourceName + " interstitialListener onInterstitialOpen");
            mListener.onInterstitialAdOpened();
        }

        @Override
        public void onInterstitialAdRewarded(String demandSourceId, int amount) {
            log(IronSourceTag.ADAPTER_CALLBACK, mDemandSourceName + " interstitialListener onInterstitialAdRewarded demandSourceId=" + demandSourceId + " amount=" + amount);
        }

        @Override
        public void onInterstitialClose() {
            log(IronSourceTag.ADAPTER_CALLBACK, mDemandSourceName + " interstitialListener onInterstitialClose");
            mListener.onInterstitialAdClosed();
        }

        @Override
        public void onInterstitialShowSuccess() {
            log(IronSourceTag.ADAPTER_CALLBACK, mDemandSourceName + " interstitialListener onInterstitialShowSuccess");
            mListener.onInterstitialAdShowSucceeded();
        }

        @Override
        public void onInterstitialShowFailed(String description) {
            log(IronSourceTag.ADAPTER_CALLBACK, mDemandSourceName + " interstitialListener onInterstitialShowFailed " + description);
            mListener.onInterstitialAdShowFailed(ErrorBuilder.buildShowFailedError(IronSourceConstants.INTERSTITIAL_AD_UNIT, description));
        }

        @Override
        public void onInterstitialClick() {
            log(IronSourceTag.ADAPTER_CALLBACK, mDemandSourceName + " interstitialListener onInterstitialClick");
            mListener.onInterstitialAdClicked();
        }

        @Override
        public void onInterstitialEventNotificationReceived(String eventName, JSONObject extData) {
            log(IronSourceTag.ADAPTER_CALLBACK, mDemandSourceName + " interstitialListener onInterstitialEventNotificationReceived eventName=" + eventName);
            mListener.onInterstitialAdVisible();
        }
    }


    private class IronSourceRewardedVideoListener implements OnInterstitialListener {
        private String mDemandSourceName;
        RewardedVideoSmashListener mListener;
        boolean mIsRvDemandOnly;

        IronSourceRewardedVideoListener (RewardedVideoSmashListener listener, String demandSourceName) {
            mDemandSourceName = demandSourceName;
            mListener = listener;
            mIsRvDemandOnly = false;
        }

        IronSourceRewardedVideoListener (RewardedVideoSmashListener listener, String demandSourceName, boolean isDemandOnly) {
            mDemandSourceName = demandSourceName;
            mListener = listener;
            mIsRvDemandOnly = isDemandOnly;
        }

        @Override
        public void onInterstitialInitSuccess() {
            //Deprecated - Old SuperSonicAds API callback
            log(IronSourceTag.ADAPTER_CALLBACK, mDemandSourceName + " rewardedVideoListener onInterstitialInitSuccess");
        }

        @Override
        public void onInterstitialInitFailed(String description) {
            //Deprecated - Old SuperSonicAds API callback
            log(IronSourceTag.ADAPTER_CALLBACK, mDemandSourceName + " rewardedVideoListener onInterstitialInitFailed");
        }

        @Override
        public void onInterstitialLoadSuccess() {
            log(IronSourceTag.ADAPTER_CALLBACK, mDemandSourceName + " rewardedVideoListener onInterstitialLoadSuccess");
            if (mIsRvDemandOnly) {
                mListener.onRewardedVideoLoadSuccess();
            }
            else {
                mListener.onRewardedVideoAvailabilityChanged(true);
            }
        }

        @Override
        public void onInterstitialLoadFailed(String description) {
            log(IronSourceTag.ADAPTER_CALLBACK, mDemandSourceName + " rewardedVideoListener onInterstitialLoadFailed " + description);
            if (mIsRvDemandOnly) {
                mListener.onRewardedVideoLoadFailed(ErrorBuilder.buildLoadFailedError(description));
            }
            else {
                mListener.onRewardedVideoAvailabilityChanged(false);
            }
        }

        @Override
        public void onInterstitialOpen() {
            log(IronSourceTag.ADAPTER_CALLBACK, mDemandSourceName + " rewardedVideoListener onInterstitialOpen");
            mListener.onRewardedVideoAdOpened();
        }

        @Override
        public void onInterstitialAdRewarded(String demandSourceId, int amount) {
            log(IronSourceTag.ADAPTER_CALLBACK,  mDemandSourceName + " rewardedVideoListener onInterstitialAdRewarded demandSourceId=" + demandSourceId + " amount=" + amount);
            mListener.onRewardedVideoAdRewarded();
        }

        @Override
        public void onInterstitialClose() {
            log(IronSourceTag.ADAPTER_CALLBACK,  mDemandSourceName + " rewardedVideoListener onInterstitialClose");
            mListener.onRewardedVideoAdClosed();
        }

        @Override
        public void onInterstitialShowSuccess() {
            log(IronSourceTag.ADAPTER_CALLBACK, mDemandSourceName + " rewardedVideoListener onInterstitialShowSuccess");
            if (!mIsRvDemandOnly) {
                mListener.onRewardedVideoAvailabilityChanged(false);
            }
        }

        @Override
        public void onInterstitialShowFailed(String description) {
            log(IronSourceTag.ADAPTER_CALLBACK, "rewardedVideoListener onInterstitialShowSuccess " + description);
            mListener.onRewardedVideoAdShowFailed(ErrorBuilder.buildShowFailedError(IronSourceConstants.REWARDED_VIDEO_AD_UNIT, description));
        }

        @Override
        public void onInterstitialClick() {
            log(IronSourceTag.ADAPTER_CALLBACK, mDemandSourceName + " rewardedVideoListener onInterstitialClick");
            mListener.onRewardedVideoAdClicked();
        }

        @Override
        public void onInterstitialEventNotificationReceived(String eventName, JSONObject extData) {
            log(IronSourceTag.ADAPTER_CALLBACK, mDemandSourceName + " rewardedVideoListener onInterstitialEventNotificationReceived eventName=" + eventName);
            mListener.onRewardedVideoAdVisible();
        }
    }


    private String getDemandSourceName(JSONObject config) {

        if (!TextUtils.isEmpty(config.optString(DEMAND_SOURCE_NAME))) {
            return config.optString(DEMAND_SOURCE_NAME);
        }
        else {
            return getProviderName();
        }
    }


    // ********** Helpers **********


    private void logError(IronSourceTag tag, String text) {
        IronSourceLoggerManager.getLogger().log(tag, "IronSourceAdapter: " + text, IronSourceLogger.IronSourceLogLevel.ERROR);
    }

    private void log(IronSourceTag tag, String text) {
        IronSourceLoggerManager.getLogger().log(tag, "IronSourceAdapter: " + text, IronSourceLogger.IronSourceLogLevel.VERBOSE);
    }

    private void log(IronSourceTag tag, JSONObject config, String text) {
        IronSourceLoggerManager.getLogger().log(tag, "IronSourceAdapter " + getDemandSourceName(config) + ": " + text, IronSourceLogger.IronSourceLogLevel.VERBOSE);
    }

    private boolean isValidMetaData(String key, String value) {
        // In case the formatted CCPA Value is empty it means we didn't get a valid boolean string value and therefore we don't want to pass it to the network
        if (key.equals(MetaDataConstants.META_DATA_CCPA_KEY)) {
            return MetaDataUtils.isValidCCPAMetaData(key, value);
        }

        return true;
    }

    @Override
    protected void setMetaData(String key, String value) {
        if (!mDidInitSdk.get()) {
            log(IronSourceTag.ADAPTER_API, "setMetaData: key=" + key +", value=" + value);

            if (!isValidMetaData(key, value)) {
                log(IronSourceTag.ADAPTER_API, "MetaData not valid");
                return;
            }

            JSONObject metaDataJson = new JSONObject();
            try {
                metaDataJson.put(key, value);
                IronSourceNetwork.updateMetadata(metaDataJson);
            } catch (JSONException e) {
                logError(IronSourceTag.ADAPTER_API, "setMetaData error - " + e);
                e.printStackTrace();
            }

        }
    }

    private void printInstanceExtraParams(Map<String, String> params) {
        if (params != null && params.size() > 0) {
            log(IronSourceTag.ADAPTER_API, "instance extra params:" );
            for (String key : params.keySet()) {
                log(IronSourceTag.ADAPTER_API, key+"=" + params.get(key));

            }
        }
    }
}
