package com.ironsource.adapters.ironsource;

import android.app.Activity;
import android.content.Context;
import android.text.TextUtils;

import com.ironsource.mediationsdk.AbstractAdapter;
import com.ironsource.mediationsdk.BuildConfig;
import com.ironsource.mediationsdk.logger.IronSourceError;
import com.ironsource.mediationsdk.logger.IronSourceLogger;
import com.ironsource.mediationsdk.logger.IronSourceLoggerManager;
import com.ironsource.mediationsdk.sdk.InterstitialSmashListener;
import com.ironsource.mediationsdk.sdk.RewardedVideoSmashListener;
import com.ironsource.mediationsdk.utils.ErrorBuilder;
import com.ironsource.mediationsdk.utils.IronSourceConstants;
import com.ironsource.mediationsdk.utils.IronSourceUtils;
import com.ironsource.mediationsdk.utils.SessionDepthManager;
import com.ironsource.sdk.IronSourceAdInstance;
import com.ironsource.sdk.IronSourceAdInstanceBuilder;
import com.ironsource.sdk.IronSourceNetwork;
import com.ironsource.sdk.constants.Constants;
import com.ironsource.sdk.listeners.OnInterstitialListener;
import com.ironsource.sdk.utils.SDKUtils;
import com.ironsource.mediationsdk.logger.IronSourceLogger.IronSourceTag;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;


public class IronSourceAdapter extends AbstractAdapter {
    private final static String VERSION = BuildConfig.VERSION_NAME;
    private final static int IS_LOAD_EXCEPTION = 1000;
    private final static int IS_SHOW_EXCEPTION = 1001;
    private final static int RV_LOAD_EXCEPTION = 1002;
    private final static int RV_SHOW_EXCEPTION = 1003;

    private final String DYNAMIC_CONTROLLER_URL = "controllerUrl";
    private final String DYNAMIC_CONTROLLER_DEBUG_MODE = "debugMode";
    private final String DYNAMIC_CONTROLLER_CONFIG = "controllerConfig";

    private final String APPLICATION_USER_GENDER = "applicationUserGender";
    private final String APPLICATION_USER_AGE_GROUP = "applicationUserAgeGroup";
    private final String SDK_PLUGIN_TYPE = "SDKPluginType";
    private final String CUSTOM_SEGMENT = "custom_Segment";
    private final String ADM_KEY = "adm";

    private final String DEMAND_SOURCE_NAME = "demandSourceName";

    private ConcurrentHashMap<String, IronSourceAdInstance> mDemandSourceToISAd;
    private ConcurrentHashMap<String, IronSourceAdInstance> mDemandSourceToRvAd;

    private static AtomicBoolean mDidInitSdk = new AtomicBoolean(false);
    private String mUserAgeGroup;
    private String mUserGender;
    private String mMediationSegment;
    private Context mContext;


    public static IronSourceAdapter startAdapter(String providerName) {
        return new IronSourceAdapter(providerName);
    }

    private IronSourceAdapter(String providerName) {
        super(providerName);
        log(IronSourceTag.INTERNAL, providerName + ": new instance");
        mDemandSourceToISAd = new ConcurrentHashMap<>();
        mDemandSourceToRvAd = new ConcurrentHashMap<>();
        mUserAgeGroup = null;
        mUserGender = null;
        mMediationSegment = null;
    }

    @Override
    public String getVersion() {
        return VERSION;
    }

    @Override
    public String getCoreSDKVersion() {
        return SDKUtils.getSDKVersion();
    }

    // ********** Base **********

    @Override
    public void onPause(Activity activity) {
        IronSourceNetwork.onPause(activity);
    }

    @Override
    public void onResume(Activity activity) {
        IronSourceNetwork.onResume(activity);
    }

    protected void setConsent(boolean consent) {
        log(IronSourceTag.ADAPTER_API, "setConsent (" + (consent ? "true" : "false") + ")" );

        JSONObject consentParams = new JSONObject();
        try {
            consentParams.put(Constants.RequestParameters.GDPR_CONSENT_STATUS, String.valueOf(consent));
        } catch (JSONException e) {
            logError(IronSourceTag.ADAPTER_API, "setConsent exception " + e.getMessage());
            return;
        }

        IronSourceNetwork.updateConsentInfo(consentParams);
    }

    @Override
    public void setAge(int age) {
        log(IronSourceTag.INTERNAL, "setAge: " + age);
        if (age >= 13 && age <= 17) {
            mUserAgeGroup = "1";
        } else if (age >= 18 && age <= 20) {
            mUserAgeGroup = "2";
        } else if (age >= 21 && age <= 24) {
            mUserAgeGroup = "3";
        } else if (age >= 25 && age <= 34) {
            mUserAgeGroup = "4";
        } else if (age >= 35 && age <= 44) {
            mUserAgeGroup = "5";
        } else if (age >= 45 && age <= 54) {
            mUserAgeGroup = "6";
        } else if (age >= 55 && age <= 64) {
            mUserAgeGroup = "7";
        } else if (age > 65 && age <= 120) {
            mUserAgeGroup = "8";
        } else {
            mUserAgeGroup = "0";
        }
    }

    @Override
    public void setGender(String gender) {
        log(IronSourceTag.INTERNAL, "setGender: " + gender);
        mUserGender = gender;
    }

    @Override
    public void setMediationSegment(String segment) {
        mMediationSegment = segment;
    }

    private HashMap<String, String> getInitParams() {
        HashMap<String, String> params = new HashMap<>();

        if (!TextUtils.isEmpty(mUserAgeGroup)) {
            params.put(APPLICATION_USER_AGE_GROUP, mUserAgeGroup);
        }

        if (!TextUtils.isEmpty(mUserGender)) {
            params.put(APPLICATION_USER_GENDER, mUserGender);
        }

        String pluginType = getPluginType();
        if (!TextUtils.isEmpty(pluginType)) {
            params.put(SDK_PLUGIN_TYPE, pluginType);
        }

        if (!TextUtils.isEmpty(mMediationSegment)) {
            params.put(CUSTOM_SEGMENT, mMediationSegment);
        }

        return params;
    }

    // ********** Init SDK **************

    @Override
    public void earlyInit (final Activity activity, final String appKey, final String userId, final JSONObject config) {
        IronSourceUtils.sendAutomationLog(getDemandSourceName(config) + ": " + "earlyInit");
        initSDK(activity, appKey, userId, config);
    }

    // ********** Interstitial **********


    @Override
    public Map<String, Object> getIsBiddingData(JSONObject config) {
        Map<String, Object> ret = new HashMap<>();
        String token = IronSourceNetwork.getToken(mContext);

        if (token != null) {
            ret.put("token", token);
        }
        else {
            logError(IronSourceTag.ADAPTER_API, "IS bidding token is null");
            ret.put("token", "");
        }

        return ret;
    }

    @Override
    public void initInterstitialForBidding(Activity activity, String appKey, String userId, JSONObject config, InterstitialSmashListener listener) {
        log(IronSourceTag.INTERNAL, config, "initInterstitialForBidding");
        initSDK(activity, appKey, userId, config);
        String demandSourceName = getDemandSourceName(config);
        IronSourceAdInstance adInstance = new IronSourceAdInstanceBuilder(demandSourceName, new IronSourceInterstitialListener(listener, demandSourceName))
                .setExtraParams(getInitParams())
                .setInAppBidding()
                .build();
        mDemandSourceToISAd.put(demandSourceName,adInstance);
        listener.onInterstitialInitSuccess();
    }

    @Override
    public void initInterstitial(Activity activity, String appKey, String userId, JSONObject config, InterstitialSmashListener listener) {
        log(IronSourceTag.INTERNAL, config, "initInterstitial");
        initSDK(activity, appKey, userId, config);
        String demandSourceName = getDemandSourceName(config);
        IronSourceAdInstance adInstance = new IronSourceAdInstanceBuilder(demandSourceName,new IronSourceInterstitialListener(listener, demandSourceName))
                .setExtraParams(getInitParams())
                .build();
        mDemandSourceToISAd.put(demandSourceName,adInstance);
        listener.onInterstitialInitSuccess();
    }

    @Override
    public void loadInterstitial(final JSONObject config, final InterstitialSmashListener listener, final String serverData) {
        log(IronSourceTag.ADAPTER_API, config, "loadInterstitial");
        try {
            HashMap<String, String> loadConfig = new HashMap<>();
            loadConfig.put(ADM_KEY, serverData);
            IronSourceAdInstance adInstance = mDemandSourceToISAd.get(getDemandSourceName(config));
            IronSourceNetwork.loadAd(adInstance, loadConfig);
        } catch (Exception e) {
            logError(IronSourceTag.ADAPTER_API, "loadInterstitial for bidding exception " + e.getMessage());
            listener.onInterstitialAdLoadFailed(new IronSourceError(IS_LOAD_EXCEPTION, e.getMessage()));
        }

    }

    @Override
    public void loadInterstitial(JSONObject config, InterstitialSmashListener listener) {
        log(IronSourceTag.ADAPTER_API, config, "loadInterstitial");
        try {
            IronSourceAdInstance adInstance = mDemandSourceToISAd.get(getDemandSourceName(config));
            IronSourceNetwork.loadAd(adInstance);
        } catch (Exception e) {
            logError(IronSourceTag.ADAPTER_API, "loadInterstitial exception " + e.getMessage());
            listener.onInterstitialAdLoadFailed(new IronSourceError(IS_LOAD_EXCEPTION, e.getMessage()));
        }
    }

    @Override
    public void showInterstitial(JSONObject config, InterstitialSmashListener listener) {
        log(IronSourceTag.ADAPTER_API, config, "showInterstitial");
        try {
            int sessionDepth = SessionDepthManager.getInstance().getSessionDepth(SessionDepthManager.INTERSTITIAL);
            Map<String, String> showParams = new HashMap<>();
            showParams.put(Constants.RequestParameters.SESSION_DEPTH, String.valueOf(sessionDepth));
            IronSourceAdInstance adInstance = mDemandSourceToISAd.get(getDemandSourceName(config));
            IronSourceNetwork.showAd(adInstance, showParams);
        } catch (Exception e) {
            logError(IronSourceTag.ADAPTER_API, "showInterstitial exception " + e.getMessage());
            listener.onInterstitialAdShowFailed(new IronSourceError(IS_SHOW_EXCEPTION, e.getMessage()));
        }
    }

    @Override
    public boolean isInterstitialReady(JSONObject config) {
        IronSourceAdInstance adInstance = mDemandSourceToISAd.get(getDemandSourceName(config));
        return adInstance != null && IronSourceNetwork.isAdAvailableForInstance(adInstance);
    }

    // ********** RewardedVideo **********



    @Override
    public Map<String, Object> getRvBiddingData(JSONObject config) {
        Map<String, Object> biddingData = new HashMap<>();
        String token = IronSourceNetwork.getToken(mContext);

        if (token != null) {
            biddingData.put("token", token);
        }
        else {
            logError(IronSourceTag.ADAPTER_API, "RV bidding token is null");
            biddingData.put("token", "");
        }
        return biddingData;
    }

    @Override
    public void initRvForBidding(Activity activity, String appKey, String userId, JSONObject config, RewardedVideoSmashListener listener) {
        log(IronSourceTag.INTERNAL, config, "initRvForBidding");
        initSDK(activity, appKey, userId, config);
        String demandSourceName = getDemandSourceName(config);
        IronSourceAdInstance adInstance = new IronSourceAdInstanceBuilder(demandSourceName, new IronSourceRewardedVideoListener(listener, demandSourceName))
                .setInAppBidding()
                .setRewarded()
                .setExtraParams(getInitParams())
                .build();
        mDemandSourceToRvAd.put(demandSourceName,adInstance);
        listener.onRewardedVideoInitSuccess();

    }

    @Override
    public void initRvForDemandOnly(Activity activity, String appKey, String userId, JSONObject config, RewardedVideoSmashListener listener) {
        log(IronSourceTag.INTERNAL, config, "initRvForDemandOnly");
        initSDK(activity, appKey, userId, config);
        String demandSourceName = getDemandSourceName(config);
        IronSourceAdInstance adInstance = new IronSourceAdInstanceBuilder(demandSourceName, new IronSourceRewardedVideoListener(listener, demandSourceName,true))
                .setRewarded()
                .setExtraParams(getInitParams())
                .build();
        mDemandSourceToRvAd.put(demandSourceName,adInstance);
    }

    @Override
    public void initRewardedVideo(Activity activity, String appKey, String userId, JSONObject config, RewardedVideoSmashListener listener) {
        log(IronSourceTag.INTERNAL, config, "initRewardedVideo");
        initSDK(activity, appKey, userId, config);
        String demandSourceName = getDemandSourceName(config);
        IronSourceAdInstance adInstance = new IronSourceAdInstanceBuilder(demandSourceName, new IronSourceRewardedVideoListener(listener, demandSourceName))
                .setRewarded()
                .setExtraParams(getInitParams())
                .build();
        mDemandSourceToRvAd.put(demandSourceName,adInstance);
        fetchRewardedVideo(config);
    }

    @Override
    public void fetchRewardedVideo(JSONObject config) {
        log(IronSourceTag.ADAPTER_API, config, "fetchRewardedVideo");
        IronSourceAdInstance adInstance = mDemandSourceToRvAd.get(getDemandSourceName(config));

        if (adInstance  == null) {
            logError(IronSourceTag.ADAPTER_API, "fetchRewardedVideo exception: null adInstance ");
            return;
        }

        try {
            IronSourceNetwork.loadAd(adInstance);
        } catch (Exception e) {
            logError(IronSourceTag.ADAPTER_API, "fetchRewardedVideo exception " + e.getMessage());
            OnInterstitialListener listener = adInstance.getAdListener();
            if (listener != null) {
                listener.onInterstitialLoadFailed("fetch exception");
            }
        }
    }

    @Override
    public void loadVideoForDemandOnly(JSONObject config, RewardedVideoSmashListener listener) {
        log(IronSourceTag.ADAPTER_API, config, "loadVideoForDemandOnly");
        try {
            IronSourceAdInstance adInstance = mDemandSourceToRvAd.get(getDemandSourceName(config));
            IronSourceNetwork.loadAd(adInstance);
        } catch (Exception e) {
            logError(IronSourceTag.ADAPTER_API, "loadVideoForDemandOnly exception " + e.getMessage());
            listener.onRewardedVideoLoadFailed(new IronSourceError(RV_LOAD_EXCEPTION, e.getMessage()));
        }
    }

    @Override
    public void loadVideo(JSONObject config, final RewardedVideoSmashListener listener, final String serverData) {
        log(IronSourceTag.ADAPTER_API, config, "loadVideo (RV in bidding mode)");
        try {
            HashMap<String, String> loadConfig = new HashMap<>();
            loadConfig.put(ADM_KEY, serverData);
            IronSourceAdInstance adInstance = mDemandSourceToRvAd.get(getDemandSourceName(config));
            IronSourceNetwork.loadAd(adInstance, loadConfig);
        } catch (Exception e) {
            logError(IronSourceTag.ADAPTER_API, "loadVideo exception " + e.getMessage());
            listener.onRewardedVideoLoadFailed(new IronSourceError(RV_LOAD_EXCEPTION, e.getMessage()));
            listener.onRewardedVideoAvailabilityChanged(false);
        }
    }

    @Override
    public void showRewardedVideo(JSONObject config, RewardedVideoSmashListener listener) {
        log(IronSourceTag.ADAPTER_API, config, "showRewardedVideo");
        try {
            IronSourceAdInstance adInstance = mDemandSourceToRvAd.get(getDemandSourceName(config));
            int sessionDepth = SessionDepthManager.getInstance().getSessionDepth(SessionDepthManager.REWARDEDVIDEO);
            Map<String, String> showParams = new HashMap<>();
            showParams.put(Constants.RequestParameters.SESSION_DEPTH, String.valueOf(sessionDepth));
            IronSourceNetwork.showAd(adInstance, showParams);
        } catch (Exception e) {
            logError(IronSourceTag.ADAPTER_API, "showRewardedVideo exception " + e.getMessage());
            listener.onRewardedVideoAdShowFailed(new IronSourceError(RV_SHOW_EXCEPTION, e.getMessage()));
        }
    }

    @Override
    public boolean isRewardedVideoAvailable(JSONObject config) {
        IronSourceAdInstance adInstance = mDemandSourceToRvAd.get(getDemandSourceName(config));
        return adInstance != null && IronSourceNetwork.isAdAvailableForInstance(adInstance);
    }

    private void initSDK(Activity activity, String appKey, String userId, JSONObject config) {

        if (activity == null) {
            logError(IronSourceTag.INTERNAL, "initSDK: null activity");
            return;
        }
        mContext = activity.getApplicationContext();
        if (mDidInitSdk.compareAndSet(false,true)) {

            if (isAdaptersDebugEnabled()) {
                log(IronSourceTag.INTERNAL, "IronSourceNetwork.initSDK debug mode enabled");
                SDKUtils.setDebugMode(IronSourceLogger.IronSourceLogLevel.ERROR);
            } else {
                SDKUtils.setDebugMode(config.optInt(DYNAMIC_CONTROLLER_DEBUG_MODE, IronSourceLogger.IronSourceLogLevel.VERBOSE));
            }

            SDKUtils.setControllerUrl(config.optString(DYNAMIC_CONTROLLER_URL));
            SDKUtils.setControllerConfig(config.optString(DYNAMIC_CONTROLLER_CONFIG, ""));
            HashMap<String, String> initParams = getInitParams();
            log(IronSourceTag.ADAPTER_API, "IronSourceNetwork.initSDK with "+ initParams);
            IronSourceNetwork.initSDK(activity, appKey, userId, initParams);
        }
    }

    // ********** Callbacks **********

    private class IronSourceInterstitialListener implements OnInterstitialListener {
        private String mDemandSourceName;
        private InterstitialSmashListener mListener;

        IronSourceInterstitialListener (InterstitialSmashListener listener, String demandSourceName) {
            mDemandSourceName = demandSourceName;
            mListener = listener;
        }

        @Override
        public void onInterstitialInitSuccess() {
            //Deprecated - Old SuperSonicAds API callback
            log(IronSourceTag.ADAPTER_CALLBACK, mDemandSourceName + " interstitialListener onInterstitialInitSuccess");
        }

        @Override
        public void onInterstitialInitFailed(String description) {
            //Deprecated - Old SuperSonicAds API callback
            log(IronSourceTag.ADAPTER_CALLBACK, mDemandSourceName + " interstitialListener onInterstitialInitFailed");
        }

        @Override
        public void onInterstitialLoadSuccess() {
            log(IronSourceTag.ADAPTER_CALLBACK, mDemandSourceName + " interstitialListener onInterstitialLoadSuccess");
            mListener.onInterstitialAdReady();
        }

        @Override
        public void onInterstitialLoadFailed(String description) {
            log(IronSourceTag.ADAPTER_CALLBACK, mDemandSourceName + " interstitialListener onInterstitialLoadFailed " + description);
            mListener.onInterstitialAdLoadFailed(ErrorBuilder.buildLoadFailedError(description));
        }

        @Override
        public void onInterstitialOpen() {
            log(IronSourceTag.ADAPTER_CALLBACK, mDemandSourceName + " interstitialListener onInterstitialOpen");
            mListener.onInterstitialAdOpened();
        }

        @Override
        public void onInterstitialAdRewarded(String demandSourceId, int amount) {
            log(IronSourceTag.ADAPTER_CALLBACK, mDemandSourceName + " interstitialListener onInterstitialAdRewarded demandSourceId=" + demandSourceId + " amount=" + amount);
        }

        @Override
        public void onInterstitialClose() {
            log(IronSourceTag.ADAPTER_CALLBACK, mDemandSourceName + " interstitialListener onInterstitialClose");
            mListener.onInterstitialAdClosed();
        }

        @Override
        public void onInterstitialShowSuccess() {
            log(IronSourceTag.ADAPTER_CALLBACK, mDemandSourceName + " interstitialListener onInterstitialShowSuccess");
            mListener.onInterstitialAdShowSucceeded();
        }

        @Override
        public void onInterstitialShowFailed(String description) {
            log(IronSourceTag.ADAPTER_CALLBACK, mDemandSourceName + " interstitialListener onInterstitialShowFailed " + description);
            mListener.onInterstitialAdShowFailed(ErrorBuilder.buildShowFailedError(IronSourceConstants.INTERSTITIAL_AD_UNIT, description));
        }

        @Override
        public void onInterstitialClick() {
            log(IronSourceTag.ADAPTER_CALLBACK, mDemandSourceName + " interstitialListener onInterstitialClick");
            mListener.onInterstitialAdClicked();
        }

        @Override
        public void onInterstitialEventNotificationReceived(String eventName, JSONObject extData) {
            log(IronSourceTag.ADAPTER_CALLBACK, mDemandSourceName + " interstitialListener onInterstitialEventNotificationReceived eventName=" + eventName);
            mListener.onInterstitialAdVisible();
        }
    }


    private class IronSourceRewardedVideoListener implements OnInterstitialListener {
        private String mDemandSourceName;
        RewardedVideoSmashListener mListener;
        boolean mIsRvDemandOnly;

        IronSourceRewardedVideoListener (RewardedVideoSmashListener listener, String demandSourceName) {
            mDemandSourceName = demandSourceName;
            mListener = listener;
            mIsRvDemandOnly = false;
        }

        IronSourceRewardedVideoListener (RewardedVideoSmashListener listener, String demandSourceName, boolean isDemandOnly) {
            mDemandSourceName = demandSourceName;
            mListener = listener;
            mIsRvDemandOnly = isDemandOnly;
        }

        @Override
        public void onInterstitialInitSuccess() {
            //Deprecated - Old SuperSonicAds API callback
            log(IronSourceTag.ADAPTER_CALLBACK, mDemandSourceName + " rewardedVideoListener onInterstitialInitSuccess");
        }

        @Override
        public void onInterstitialInitFailed(String description) {
            //Deprecated - Old SuperSonicAds API callback
            log(IronSourceTag.ADAPTER_CALLBACK, mDemandSourceName + " rewardedVideoListener onInterstitialInitFailed");
        }

        @Override
        public void onInterstitialLoadSuccess() {
            log(IronSourceTag.ADAPTER_CALLBACK, mDemandSourceName + " rewardedVideoListener onInterstitialLoadSuccess");
            if (mIsRvDemandOnly) {
                mListener.onRewardedVideoLoadSuccess();
            }
            else {
                mListener.onRewardedVideoAvailabilityChanged(true);
            }
        }

        @Override
        public void onInterstitialLoadFailed(String description) {
            log(IronSourceTag.ADAPTER_CALLBACK, mDemandSourceName + " rewardedVideoListener onInterstitialLoadFailed " + description);
            if (mIsRvDemandOnly) {
                mListener.onRewardedVideoLoadFailed(ErrorBuilder.buildLoadFailedError(description));
            }
            else {
                mListener.onRewardedVideoAvailabilityChanged(false);
            }
        }

        @Override
        public void onInterstitialOpen() {
            log(IronSourceTag.ADAPTER_CALLBACK, mDemandSourceName + " rewardedVideoListener onInterstitialOpen");
            mListener.onRewardedVideoAdOpened();
        }

        @Override
        public void onInterstitialAdRewarded(String demandSourceId, int amount) {
            log(IronSourceTag.ADAPTER_CALLBACK,  mDemandSourceName + " rewardedVideoListener onInterstitialAdRewarded demandSourceId=" + demandSourceId + " amount=" + amount);
            mListener.onRewardedVideoAdRewarded();
        }

        @Override
        public void onInterstitialClose() {
            log(IronSourceTag.ADAPTER_CALLBACK,  mDemandSourceName + " rewardedVideoListener onInterstitialClose");
            mListener.onRewardedVideoAdClosed();
        }

        @Override
        public void onInterstitialShowSuccess() {
            log(IronSourceTag.ADAPTER_CALLBACK, mDemandSourceName + " rewardedVideoListener onInterstitialShowSuccess");
        }

        @Override
        public void onInterstitialShowFailed(String description) {
            log(IronSourceTag.ADAPTER_CALLBACK, "rewardedVideoListener onInterstitialShowSuccess " + description);
            mListener.onRewardedVideoAdShowFailed(ErrorBuilder.buildShowFailedError(IronSourceConstants.REWARDED_VIDEO_AD_UNIT, description));
        }

        @Override
        public void onInterstitialClick() {
            log(IronSourceTag.ADAPTER_CALLBACK, mDemandSourceName + " rewardedVideoListener onInterstitialClick");
            mListener.onRewardedVideoAdClicked();
        }

        @Override
        public void onInterstitialEventNotificationReceived(String eventName, JSONObject extData) {
            log(IronSourceTag.ADAPTER_CALLBACK, mDemandSourceName + " rewardedVideoListener onInterstitialEventNotificationReceived eventName=" + eventName);
            mListener.onRewardedVideoAdVisible();
        }
    }


    private String getDemandSourceName(JSONObject config) {

        if (!TextUtils.isEmpty(config.optString(DEMAND_SOURCE_NAME))) {
            return config.optString(DEMAND_SOURCE_NAME);
        }
        else {
            return getProviderName();
        }
    }


    // ********** Helpers **********


    private void logError(IronSourceTag tag, String text) {
        IronSourceLoggerManager.getLogger().log(tag, "IronSourceAdapter: " + text, IronSourceLogger.IronSourceLogLevel.ERROR);
    }

    private void log(IronSourceTag tag, String text) {
        IronSourceLoggerManager.getLogger().log(tag, "IronSourceAdapter: " + text, IronSourceLogger.IronSourceLogLevel.VERBOSE);
    }

    private void log(IronSourceTag tag, JSONObject config, String text) {
        IronSourceLoggerManager.getLogger().log(tag, "IronSourceAdapter " + getDemandSourceName(config) + ": " + text, IronSourceLogger.IronSourceLogLevel.VERBOSE);
    }

    @Override
    protected void setMetaData(String key, String value) {
        if (!mDidInitSdk.get()) {
            log(IronSourceTag.ADAPTER_API, "setMetaData: key=" + key +", value=" + value);
            JSONObject metaDataJson = new JSONObject();
            try {
                metaDataJson.put(key, value);
                IronSourceNetwork.updateMetadata(metaDataJson);
            } catch (JSONException e) {
                logError(IronSourceTag.ADAPTER_API, "setMetaData error - " + e);
                e.printStackTrace();
            }

        }
    }
}
