package com.ironsource.adapters.supersonicads;

import android.app.Activity;
import android.os.Handler;
import android.os.Looper;
import android.text.TextUtils;
import android.view.Gravity;
import android.widget.FrameLayout;

import com.ironsource.mediationsdk.AbstractAdapter;
import com.ironsource.mediationsdk.AbstractSmash;
import com.ironsource.mediationsdk.AdapterUtils;
import com.ironsource.mediationsdk.ISBannerSize;
import com.ironsource.mediationsdk.BuildConfig;
import com.ironsource.mediationsdk.IntegrationData;

import com.ironsource.mediationsdk.IronSourceBannerLayout;
import com.ironsource.mediationsdk.IronSourceObject;
import com.ironsource.mediationsdk.logger.IronSourceError;
import com.ironsource.mediationsdk.logger.IronSourceLogger;
import com.ironsource.mediationsdk.logger.IronSourceLogger.IronSourceLogLevel;
import com.ironsource.mediationsdk.logger.IronSourceLogger.IronSourceTag;
import com.ironsource.mediationsdk.logger.IronSourceLoggerManager;
import com.ironsource.mediationsdk.sdk.BannerSmashListener;
import com.ironsource.mediationsdk.sdk.InternalOfferwallListener;
import com.ironsource.mediationsdk.sdk.InterstitialSmashListener;
import com.ironsource.mediationsdk.sdk.OfferwallAdapterApi;
import com.ironsource.mediationsdk.sdk.RewardedVideoSmashListener;
import com.ironsource.mediationsdk.utils.ErrorBuilder;
import com.ironsource.mediationsdk.utils.IronSourceConstants;
import com.ironsource.mediationsdk.utils.IronSourceUtils;
import com.ironsource.mediationsdk.utils.SessionDepthManager;
import com.ironsource.sdk.ISAdSize;
import com.ironsource.sdk.ISNAdView.ISNAdView;
import com.ironsource.sdk.SSAFactory;
import com.ironsource.sdk.SSAPublisher;
import com.ironsource.sdk.data.AdUnitsReady;
import com.ironsource.sdk.data.SSAEnums;
import com.ironsource.sdk.listeners.OnBannerListener;
import com.ironsource.sdk.listeners.OnInterstitialListener;
import com.ironsource.sdk.listeners.OnOfferWallListener;
import com.ironsource.sdk.listeners.OnRewardedVideoListener;
import com.ironsource.sdk.constants.Constants;
import com.ironsource.sdk.utils.SDKUtils;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;


class SupersonicAdsAdapter extends AbstractAdapter implements OfferwallAdapterApi, OnOfferWallListener, OnInterstitialListener, OnRewardedVideoListener, OnBannerListener {

    private final static String VERSION = BuildConfig.VERSION_NAME;
    private final String TIMESTAMP = "timestamp";
    private final String ITEM_SIGNATURE = "itemSignature";
    private final String SDK_PLUGIN_TYPE = "SDKPluginType";
    private final String OW_PLACEMENT_ID = "placementId";

    private SSAPublisher mSSAPublisher;

    private InternalOfferwallListener mOfferwallListener;

    private boolean mIsRVAvailable = false;
    private AtomicBoolean mDidSetInitParams = new AtomicBoolean(false);

    private String mUserAgeGroup;
    private String mUserGender;
    private String mMediationSegment;
    private boolean mConsent;
    private boolean mDidSetConsent;

    private final String SUPERSONIC_ADS = "SupersonicAds";

    private final String DYNAMIC_CONTROLLER_URL = "controllerUrl";
    private final String DYNAMIC_CONTROLLER_DEBUG_MODE = "debugMode";
    private final String DYNAMIC_CONTROLLER_CONFIG = "controllerConfig";

    private final String APPLICATION_USER_GENDER = "applicationUserGender";
    private final String APPLICATION_USER_AGE_GROUP = "applicationUserAgeGroup";

    private final String LANGUAGE = "language";
    private final String MAX_VIDEO_LENGTH = "maxVideoLength";
    private final String CAMPAIGN_ID = "campaignId";
    private final String CUSTOM_PARAM_PREFIX = "custom_";
    private final String CUSTOM_SEGMENT = CUSTOM_PARAM_PREFIX + "Segment";

    private final String ITEM_NAME = "itemName";
    private final String ITEM_COUNT = "itemCount";
    private final String APPLICATION_PRIVATE_KEY = "privateKey";

    private final String CLIENT_SIDE_CALLBACKS = "useClientSideCallbacks";
    private final String AD_VISIBLE_EVENT_NAME = "impressions";

    private static Handler mUIThreadHandler;
    private ISNAdView mIsnAdView;

    public static SupersonicAdsAdapter startAdapter(String providerName) {
        return new SupersonicAdsAdapter(providerName);
    }

    private SupersonicAdsAdapter(String providerName) {
        super(providerName);
    }

    public static IntegrationData getIntegrationData(Activity activity) {
        IntegrationData ret = new IntegrationData("SupersonicAds", VERSION);
        ret.activities = new String[]{"com.ironsource.sdk.controller.ControllerActivity",
                "com.ironsource.sdk.controller.InterstitialActivity", "com.ironsource.sdk.controller.OpenUrlActivity"};
        return ret;
    }

    @Override
    public String getVersion() {
        return VERSION;
    }

    @Override
    public String getCoreSDKVersion() {
        return SDKUtils.getSDKVersion();
    }

    public static String getAdapterSDKVersion() { return SDKUtils.getSDKVersion(); }


    // ********** Base **********

    @Override
    public void onPause(Activity activity) {
        if (mSSAPublisher != null)
            mSSAPublisher.onPause(activity);
    }

    @Override
    public void onResume(Activity activity) {
        if (mSSAPublisher != null)
            mSSAPublisher.onResume(activity);
    }

    @Override
    public void setAge(int age) {
        if (age >= 13 && age <= 17) {
            mUserAgeGroup = "1";
        } else if (age >= 18 && age <= 20) {
            mUserAgeGroup = "2";
        } else if (age >= 21 && age <= 24) {
            mUserAgeGroup = "3";
        } else if (age >= 25 && age <= 34) {
            mUserAgeGroup = "4";
        } else if (age >= 35 && age <= 44) {
            mUserAgeGroup = "5";
        } else if (age >= 45 && age <= 54) {
            mUserAgeGroup = "6";
        } else if (age >= 55 && age <= 64) {
            mUserAgeGroup = "7";
        } else if (age > 65 && age <= 120) {
            mUserAgeGroup = "8";
        } else {
            mUserAgeGroup = "0";
        }
    }

    @Override
    public void setGender(String gender) {
        mUserGender = gender;
    }

    @Override
    public void setMediationSegment(String segment) {
        mMediationSegment = segment;
    }

    // ********** RewardedVideo **********

    @Override
    public void initRewardedVideo(final Activity activity, final String appKey, final String userId, final JSONObject config, RewardedVideoSmashListener listener) {
        log(IronSourceTag.INTERNAL, getProviderName() + ": initRewardedVideo", IronSourceLogLevel.VERBOSE);

        setParamsBeforeInit(config);
        activity.runOnUiThread(new Runnable() {

            @Override
            public void run() {
                try {
                    mSSAPublisher = SSAFactory.getPublisherInstance(activity);
                    HashMap<String, String> rewardedVideoExtraParams = getRewardedVideoExtraParams(config);

                    if (mDidSetConsent) {
                        applyConsent(mConsent);
                    }

                    mSSAPublisher.initRewardedVideo(
                            appKey,
                            userId,
                            getProviderName(),
                            rewardedVideoExtraParams,
                            SupersonicAdsAdapter.this
                    );

                } catch (Exception e) {
                    e.printStackTrace();
                    onRVInitFail("initRewardedVideo");
                }
            }

        });
    }

    @Override
    public void fetchRewardedVideo(JSONObject config) {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.ADAPTER_API, getProviderName() + ": fetchRewardedVideo", IronSourceLogLevel.INFO);

        for (RewardedVideoSmashListener smash : mAllRewardedVideoSmashes) {
            if (smash != null) {
                smash.onRewardedVideoAvailabilityChanged(mIsRVAvailable);
            }
        }
    }

    @Override
    public void showRewardedVideo(JSONObject config, RewardedVideoSmashListener listener) {
        mActiveRewardedVideoSmash = listener;

        if (mSSAPublisher != null) {
            int sessionDepth = SessionDepthManager.getInstance().getSessionDepth(SessionDepthManager.REWARDEDVIDEO);

            JSONObject showParams = new JSONObject();
            try {
                showParams.put(Constants.RequestParameters.DEMAND_SOURCE_NAME, getProviderName());
                showParams.put(Constants.RequestParameters.SESSION_DEPTH, sessionDepth);
            } catch (JSONException e) {
                e.printStackTrace();
            }

            mSSAPublisher.showRewardedVideo(showParams);
        } else {
            mIsRVAvailable = false;

            if (mActiveRewardedVideoSmash != null)
                mActiveRewardedVideoSmash.onRewardedVideoAdShowFailed(ErrorBuilder.buildNoAdsToShowError(IronSourceConstants.REWARDED_VIDEO_AD_UNIT));

            for (RewardedVideoSmashListener smash : mAllRewardedVideoSmashes) {
                if (smash != null)
                    smash.onRewardedVideoAvailabilityChanged(false);
            }
        }
    }

    @Override
    public boolean isRewardedVideoAvailable(JSONObject config) {
        return mIsRVAvailable;
    }

    // ********** Init SDK **********


    @Override
    public void preInitInterstitial (final Activity activity, final String appKey, final String userId, final JSONObject config, InterstitialSmashListener listener) {
        IronSourceUtils.sendAutomationLog(getProviderName() + ": preInitInterstitial");

        setParamsBeforeInit(config);
        activity.runOnUiThread(new Runnable() {
            @Override
            public void run() {
                try {
                    mSSAPublisher = SSAFactory.getPublisherInstance(activity);

                    final HashMap<String, String> interstitialExtraParams = getInterstitialExtraParams();

                    if (mDidSetConsent) {
                        applyConsent(mConsent);
                    }
                    mSSAPublisher.initInterstitial(appKey, userId, getProviderName(), interstitialExtraParams, null);
                } catch (Exception e) {
                    e.printStackTrace();
                    onInterstitialInitFailed(e.getMessage());
                }
            }
        });
    }



    // ********** Interstitial **********

    @Override
    public void initInterstitial(final Activity activity, final String appKey, final String userId, final JSONObject config, InterstitialSmashListener listener) {
        log(IronSourceTag.INTERNAL, getProviderName() + ": initInterstitial", IronSourceLogLevel.VERBOSE);

        setParamsBeforeInit(config);
        activity.runOnUiThread(new Runnable() {
            @Override
            public void run() {
                try {
                    mSSAPublisher = SSAFactory.getPublisherInstance(activity);

                    final HashMap<String, String> interstitialExtraParams = getInterstitialExtraParams();

                    if (mDidSetConsent) {
                        applyConsent(mConsent);
                    }
                    mSSAPublisher.initInterstitial(appKey, userId, getProviderName(), interstitialExtraParams, SupersonicAdsAdapter.this);
                } catch (Exception e) {
                    e.printStackTrace();
                    onInterstitialInitFailed(e.getMessage());
                }
            }
        });
    }

    @Override
    public void loadInterstitial(JSONObject config, InterstitialSmashListener listener) {
        if (mSSAPublisher != null) {
            JSONObject loadParams = new JSONObject();
            try {
                loadParams.put(Constants.RequestParameters.DEMAND_SOURCE_NAME, getProviderName());
            } catch (JSONException e) {
                e.printStackTrace();
            }
            mSSAPublisher.loadInterstitial(loadParams);
        } else {
            log(IronSourceTag.NATIVE, "Please call initInterstitial before calling loadInterstitial", IronSourceLogLevel.WARNING);

            for (InterstitialSmashListener smash : mAllInterstitialSmashes) {
                if (smash != null)
                    smash.onInterstitialAdLoadFailed(ErrorBuilder.buildLoadFailedError("Load was called before Init"));
            }
        }
    }

    @Override
    public void showInterstitial(JSONObject config, InterstitialSmashListener listener) {
        mActiveInterstitialSmash = listener;

        if (mSSAPublisher != null) {
            int sessionDepth = SessionDepthManager.getInstance().getSessionDepth(SessionDepthManager.INTERSTITIAL);

            JSONObject showParams = new JSONObject();
            try {
                showParams.put(Constants.RequestParameters.DEMAND_SOURCE_NAME, getProviderName());
                showParams.put(Constants.RequestParameters.SESSION_DEPTH, sessionDepth);
            } catch (JSONException e) {
                e.printStackTrace();
            }

            mSSAPublisher.showInterstitial(showParams);
        } else {
            log(IronSourceTag.NATIVE, "Please call loadInterstitial before calling showInterstitial", IronSourceLogLevel.WARNING);

            if (mActiveInterstitialSmash != null)
                mActiveInterstitialSmash.onInterstitialAdShowFailed(ErrorBuilder.buildNoAdsToShowError(IronSourceConstants.INTERSTITIAL_AD_UNIT));
        }
    }

    @Override
    public boolean isInterstitialReady(JSONObject config) {
        return mSSAPublisher != null && mSSAPublisher.isInterstitialAdAvailable(getProviderName());
    }

    // ********** Offerwall **********

    @Override
    public void setInternalOfferwallListener(InternalOfferwallListener listener) {
        mOfferwallListener = listener;
    }

    @Override
    public void initOfferwall(final Activity activity, final String appKey, final String userId, final JSONObject config) {
        log(IronSourceTag.INTERNAL, getProviderName() + ": initOfferwall", IronSourceLogLevel.VERBOSE);

        setParamsBeforeInit(config);
        activity.runOnUiThread(new Runnable() {
            @Override
            public void run() {
                try {
                    Map<String, String> offerwallExtraParams = getOfferwallExtraParams(config);
                    mSSAPublisher = SSAFactory.getPublisherInstance(activity);

                    if (mDidSetConsent) {
                        applyConsent(mConsent);
                    }

                    mSSAPublisher.initOfferWall(appKey, userId, offerwallExtraParams, SupersonicAdsAdapter.this);
                } catch (Exception e) {
                    e.printStackTrace();
                    IronSourceLoggerManager.getLogger().logException(IronSourceTag.ADAPTER_API, getProviderName() + ":initOfferwall(userId:" + userId + ")", e);
                    mOfferwallListener.onOfferwallAvailable(false,
                            ErrorBuilder.buildInitFailedError("Adapter initialization failure - " + getProviderName() + " - " + e.getMessage(), IronSourceConstants.OFFERWALL_AD_UNIT));
                }
            }
        });
    }

    @Override
    public void getOfferwallCredits() {

        if (mSSAPublisher != null) {
            String appKey = IronSourceObject.getInstance().getIronSourceAppKey();
            String userId = IronSourceObject.getInstance().getIronSourceUserId();

            mSSAPublisher.getOfferWallCredits(appKey, userId, this);
        } else
            log(IronSourceTag.NATIVE, "Please call init before calling getOfferwallCredits", IronSourceLogLevel.WARNING);
    }

    @Override
    public void showOfferwall(String placementId, JSONObject config) {
        Map<String, String> offerwallExtraParams = getOfferwallExtraParams(config);
        if (offerwallExtraParams != null) {
            offerwallExtraParams.put(OW_PLACEMENT_ID, placementId);
        }

        if (mSSAPublisher != null) {
            mSSAPublisher.showOfferWall(offerwallExtraParams);
        } else {
            log(IronSourceTag.NATIVE, "Please call init before calling showOfferwall", IronSourceLogLevel.WARNING);
        }
    }

    @Override
    public boolean isOfferwallAvailable() {
        return true;
    }

    // ********** Banners **********

    @Override
    public void initBanners(final Activity activity, final String appKey, final String userId, final JSONObject config, final BannerSmashListener listener) {
        log(IronSourceTag.INTERNAL, getProviderName() + ": initBanners", IronSourceLogLevel.VERBOSE);

        setParamsBeforeInit(config);
        activity.runOnUiThread(new Runnable() {
            @Override
            public void run() {
                try {
                    Map<String, String> bannerExtraParams = getBannerExtraParams(config);
                    mSSAPublisher = SSAFactory.getPublisherInstance(activity);

                    if (mDidSetConsent) {
                        applyConsent(mConsent);
                    }

                    mSSAPublisher.initBanner(appKey, userId, getProviderName(), bannerExtraParams, SupersonicAdsAdapter.this);

                } catch (Exception e) {
                    e.printStackTrace();
                    onBannerInitFailed(e.getMessage());
                }
            }
        });
    }

    @Override
    public void loadBanner(final IronSourceBannerLayout banner, final JSONObject config, final BannerSmashListener listener) {
        try {
            if (mSSAPublisher == null) {
                log(IronSourceTag.NATIVE, "Please call initBanner before calling loadBanner", IronSourceLogLevel.WARNING);
                for (BannerSmashListener smash : mAllBannerSmashes) {
                    if (smash != null) {
                        smash.onBannerAdLoadFailed(ErrorBuilder.buildLoadFailedError("Load was called before Init"));
                    }
                }
            }

            if (banner == null) {
                IronSourceLoggerManager.getLogger().log(IronSourceLogger.IronSourceTag.INTERNAL, "SupersonicAds loadBanner banner == null", IronSourceLogger.IronSourceLogLevel.ERROR);
                return;
            }
            mActiveBannerSmash = listener;
            //Remove ISNAdView instance from memory in order to prevent multiple instances of it
            //in scenarios when there's a failure in SDK6 that we're not aware of
            if (mIsnAdView != null) {
                mIsnAdView.performCleanup();
                mIsnAdView = null;
            }
            final JSONObject loadParams = new JSONObject();
            loadParams.put(Constants.RequestParameters.DEMAND_SOURCE_NAME, getProviderName());
            loadParams.put(Constants.ParametersKeys.PRODUCT_TYPE, SSAEnums.ProductType.Banner);

            if (mUIThreadHandler == null) {
                mUIThreadHandler = new Handler(Looper.getMainLooper());
            }


            mUIThreadHandler.post(new Runnable() {
                @Override
                public void run() {
                    try {
                        mIsnAdView = createBanner(banner.getActivity(), banner.getSize(), mActiveBannerSmash);
                        if (mIsnAdView != null) {
                            mIsnAdView.loadAd(loadParams);
                        }
                    } catch (Exception e) {
                        String errorString = IronSourceConstants.BANNER_AD_UNIT + " Load Fail, "
                                + getProviderName() + " - " + e.getMessage();

                        IronSourceError error = ErrorBuilder.buildLoadFailedError(errorString);

                        if (mActiveBannerSmash != null) {
                            mActiveBannerSmash.onBannerAdLoadFailed(error);
                        }
                    }
                }
            });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    public void reloadBanner(final JSONObject config) {
        try {
            if (mIsnAdView != null) {
                mIsnAdView.loadAd(config);
            }
        } catch (Exception e) {
            e.printStackTrace();
            log(IronSourceTag.NATIVE, getProviderName() + " reloadBanner Failed to reload banner ad", IronSourceLogLevel.WARNING);
        }
    }

    @Override
    public void destroyBanner(final JSONObject config) {
        if (mIsnAdView != null) {
            mIsnAdView.performCleanup();
            mIsnAdView = null;
        }
    }

    // ********** Helpers **********

    private void setParamsBeforeInit(JSONObject config) {
        if (mDidSetInitParams.compareAndSet(false,true )) {

            SDKUtils.setControllerUrl(config.optString(DYNAMIC_CONTROLLER_URL));

            if (isAdaptersDebugEnabled()) {
                SDKUtils.setDebugMode(3);
            } else {
                SDKUtils.setDebugMode(config.optInt(DYNAMIC_CONTROLLER_DEBUG_MODE, 0));
            }

            SDKUtils.setControllerConfig(config.optString(DYNAMIC_CONTROLLER_CONFIG, ""));
        }
    }

    private HashMap<String, String> getGenenralExtraParams() {
        HashMap<String, String> params = new HashMap<>();

        //Check and add user age group configuration
        if (!TextUtils.isEmpty(mUserAgeGroup))
            params.put(APPLICATION_USER_AGE_GROUP, mUserAgeGroup);

        //Check and add user creation gender configuration
        if (!TextUtils.isEmpty(mUserGender))
            params.put(APPLICATION_USER_GENDER, mUserGender);

        String pluginType = getPluginType();
        if (!TextUtils.isEmpty(pluginType))
            params.put(SDK_PLUGIN_TYPE, pluginType);

        return params;
    }

    private HashMap<String, String> getRewardedVideoExtraParams(JSONObject config) {
        HashMap<String, String> rvExtraParams = getGenenralExtraParams();

        //Check and add language configuration
        String language = config.optString(LANGUAGE);
        if (!TextUtils.isEmpty(language)) {
            rvExtraParams.put(LANGUAGE, language);
        }

        //Check and add maxVideoLength configuration
        String maxVideoLength = config.optString(MAX_VIDEO_LENGTH);
        if (!TextUtils.isEmpty(maxVideoLength)) {
            rvExtraParams.put(MAX_VIDEO_LENGTH, maxVideoLength);
        }

        //Check and add campaignId configuration
        String campaignId = config.optString(CAMPAIGN_ID);
        if (!TextUtils.isEmpty(campaignId))
            rvExtraParams.put(CAMPAIGN_ID, campaignId);

        //Check and add segment configuration
        if (!TextUtils.isEmpty(mMediationSegment)) {
            rvExtraParams.put(CUSTOM_SEGMENT, mMediationSegment);
        }

        //Check and add itemName/itemCount/privateKey configurations
        addItemNameCountSignature(rvExtraParams, config);

        //Add RewardedVideo customParams if set
        Map<String, String> customParams = SupersonicConfig.getConfigObj().getRewardedVideoCustomParams();
        if (customParams != null && !customParams.isEmpty())
            rvExtraParams.putAll(customParams);

        return rvExtraParams;
    }

    private HashMap<String, String> getInterstitialExtraParams() {
        return getGenenralExtraParams();
    }

    private HashMap<String, String> getOfferwallExtraParams(JSONObject config) {
        HashMap<String, String> owExtraParams = getGenenralExtraParams();

        //Check language configuration
        String language = config.optString(LANGUAGE);
        if (!TextUtils.isEmpty(language)) {
            owExtraParams.put(LANGUAGE, language);
        }

        //Check client side callbacks configuration
        boolean clientSideCallbacks = SupersonicConfig.getConfigObj().getClientSideCallbacks();
        owExtraParams.put(CLIENT_SIDE_CALLBACKS, String.valueOf(clientSideCallbacks));

        //Add Offerwall customParams if set
        Map<String, String> customParams = SupersonicConfig.getConfigObj().getOfferwallCustomParams();
        if (customParams != null && !customParams.isEmpty())
            owExtraParams.putAll(customParams);

        addItemNameCountSignature(owExtraParams, config);

        return owExtraParams;
    }

    private HashMap<String, String> getBannerExtraParams(JSONObject config) {
        return getGenenralExtraParams();
    }


    private void addItemNameCountSignature(HashMap<String, String> params, JSONObject config) {
        try {
            String itemName = config.optString(ITEM_NAME);
            int itemCount = config.optInt(ITEM_COUNT, -1);
            String privateKey = config.optString(APPLICATION_PRIVATE_KEY);

            boolean shouldAddSignature = true;

            if (TextUtils.isEmpty(itemName))
                shouldAddSignature = false;
            else
                params.put(ITEM_NAME, itemName);

            if (TextUtils.isEmpty(privateKey))
                shouldAddSignature = false;

            if (itemCount == -1)
                shouldAddSignature = false;
            else
                params.put(ITEM_COUNT, String.valueOf(itemCount));


            if (shouldAddSignature) {
                int timestamp = IronSourceUtils.getCurrentTimestamp();
                params.put(TIMESTAMP, String.valueOf(timestamp));
                params.put(ITEM_SIGNATURE, createItemSig(timestamp, itemName, itemCount, privateKey));
            }

        } catch (Exception e) {
            IronSourceLoggerManager.getLogger().logException(IronSourceTag.ADAPTER_API, " addItemNameCountSignature", e);
        }
    }

    private String createItemSig(int timestamp, String itemName, int itemCount, String privateKey) {
        return IronSourceUtils.getMD5(timestamp + itemName + itemCount + privateKey);
    }

    private String createMinimumOfferCommissionSig(double min, String privateKey) {
        return IronSourceUtils.getMD5(min + privateKey);
    }

    private String createUserCreationDateSig(String userid, String uCreationDate, String privateKey) {
        return IronSourceUtils.getMD5(userid + uCreationDate + privateKey);
    }

    private ISNAdView createBanner(final Activity activity, final ISBannerSize bannerSize, BannerSmashListener listener) {
        final String BANNER = "BANNER",
                LARGE = "LARGE",
                RECTANGLE = "RECTANGLE",
                SMART = "SMART",
                CUSTOM = "CUSTOM";

        final int STANDARD_SCREEN_BANNER_WIDTH = 320,
                STANDARD_BANNER_SIZE_HEIGHT = 50,
                LARGE_SCREEN_BANNER_WIDTH = 728,
                LARGE_BANNER_SIZE_HEIGHT = 90;

        String sizeLabel = bannerSize.getDescription();
        int widthDp = STANDARD_SCREEN_BANNER_WIDTH;
        int heightDp = STANDARD_BANNER_SIZE_HEIGHT;

        switch (sizeLabel) {
            case BANNER: {
                widthDp = STANDARD_SCREEN_BANNER_WIDTH;
                heightDp = STANDARD_BANNER_SIZE_HEIGHT;
                break;
            }
            case LARGE: {
                widthDp = STANDARD_SCREEN_BANNER_WIDTH;
                heightDp = LARGE_BANNER_SIZE_HEIGHT;
                break;
            }
            case SMART: {
                boolean isLargeScreen = AdapterUtils.isLargeScreen(activity);
                widthDp = isLargeScreen ? LARGE_SCREEN_BANNER_WIDTH : STANDARD_SCREEN_BANNER_WIDTH;
                heightDp = isLargeScreen ? LARGE_BANNER_SIZE_HEIGHT : STANDARD_BANNER_SIZE_HEIGHT;
                break;
            }
            case CUSTOM: {
                widthDp = bannerSize.getWidth();
                heightDp = bannerSize.getHeight();
                if (widthDp < STANDARD_SCREEN_BANNER_WIDTH || (heightDp != STANDARD_BANNER_SIZE_HEIGHT && heightDp != LARGE_BANNER_SIZE_HEIGHT)) { //Limitation for CUSTOM banner size (JIRA UC-460)
                    if (listener != null) {
                        listener.onBannerAdLoadFailed(ErrorBuilder.unsupportedBannerSize(SUPERSONIC_ADS));
                    }
                    return null;
                }
                break;
            }
            case RECTANGLE:
            default: {
                if (listener != null) {
                    listener.onBannerAdLoadFailed(ErrorBuilder.unsupportedBannerSize(SUPERSONIC_ADS));
                }
                return null;
            }
        }
        int widthPixels = AdapterUtils.dpToPixels(activity, widthDp);
        int heightPixels = AdapterUtils.dpToPixels(activity, heightDp);
        ISAdSize bannerAdSize = new ISAdSize(widthPixels, heightPixels, sizeLabel);

        return mSSAPublisher.createBanner(activity, bannerAdSize);
    }

    // ********** Supersonic Rewarded Video Callbacks **********

    @Override
    public void onRVNoMoreOffers() {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.ADAPTER_CALLBACK, getProviderName() + ": onRVNoMoreOffers ", IronSourceLogLevel.INFO);

        mIsRVAvailable = false;

        for (RewardedVideoSmashListener smash : mAllRewardedVideoSmashes) {
            if (smash != null)
                smash.onRewardedVideoAvailabilityChanged(false);
        }
    }

    @Override
    public void onRVInitSuccess(AdUnitsReady aur) {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.ADAPTER_CALLBACK, getProviderName() + ": onRVInitSuccess ", IronSourceLogLevel.INFO);

        int numOfAdUnits = 0;

        try {
            numOfAdUnits = Integer.parseInt(aur.getNumOfAdUnits());
        } catch (NumberFormatException e) {
            IronSourceLoggerManager.getLogger().logException(IronSourceTag.NATIVE, ": onRVInitSuccess: parseInt()", e);
        }

        boolean availability = numOfAdUnits > 0;
        mIsRVAvailable = availability;

        for (RewardedVideoSmashListener smash : mAllRewardedVideoSmashes) {
            if (smash != null)
                smash.onRewardedVideoAvailabilityChanged(availability);
        }
    }

    @Override
    public void onRVInitFail(String error) {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.ADAPTER_CALLBACK, getProviderName() + ": onRVInitFail ", IronSourceLogLevel.INFO);

        for (RewardedVideoSmashListener smash : mAllRewardedVideoSmashes) {
            if (smash != null)
                smash.onRewardedVideoAvailabilityChanged(false);
        }
    }

    @Override
    public void onRVAdClicked() {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.ADAPTER_CALLBACK, getProviderName() + ": onRVAdClicked ", IronSourceLogLevel.INFO);

        if (mActiveRewardedVideoSmash != null)
            mActiveRewardedVideoSmash.onRewardedVideoAdClicked();
    }

    @Override
    public void onRVEventNotificationReceived(String eventName, JSONObject extData) {
        if (extData != null)
            IronSourceLoggerManager.getLogger().log(IronSourceTag.ADAPTER_CALLBACK, getProviderName() + ": onRVEventNotificationReceived: " + eventName + " extData: " + extData.toString(), IronSourceLogLevel.INFO);

        if (!TextUtils.isEmpty(eventName) && AD_VISIBLE_EVENT_NAME.equals(eventName) && mActiveRewardedVideoSmash != null) {
            mActiveRewardedVideoSmash.onRewardedVideoAdVisible();
        }
    }

    @Override
    public void onRVShowFail(String error) {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.ADAPTER_CALLBACK, getProviderName() + ": onRVShowFail ", IronSourceLogLevel.INFO);

        if (mActiveRewardedVideoSmash != null)
            mActiveRewardedVideoSmash.onRewardedVideoAdShowFailed(new IronSourceError(IronSourceError.ERROR_CODE_NO_ADS_TO_SHOW, error));
    }

    @Override
    public void onRVAdCredited(int amount) {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.ADAPTER_CALLBACK, getProviderName() + ": onRVAdCredited ", IronSourceLogLevel.INFO);

        if (mActiveRewardedVideoSmash != null) {
            mActiveRewardedVideoSmash.onRewardedVideoAdRewarded();
        }
    }

    @Override
    public void onRVAdClosed() {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.ADAPTER_CALLBACK, getProviderName() + ": onRVAdClosed ", IronSourceLogLevel.INFO);

        if (mActiveRewardedVideoSmash != null)
            mActiveRewardedVideoSmash.onRewardedVideoAdClosed();
    }

    @Override
    public void onRVAdOpened() {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.ADAPTER_CALLBACK, getProviderName() + ": onRVAdOpened ", IronSourceLogLevel.INFO);

        if (mActiveRewardedVideoSmash != null)
            mActiveRewardedVideoSmash.onRewardedVideoAdOpened();
    }

    // ********** Supersonic Interstitial Callbacks **********

    @Override
    public void onInterstitialInitSuccess() {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.ADAPTER_CALLBACK, getProviderName() + ": onInterstitialInitSuccess ", IronSourceLogLevel.INFO);
        for (InterstitialSmashListener smash : mAllInterstitialSmashes) {
            if (smash != null)
                smash.onInterstitialInitSuccess();
        }
    }

    @Override
    public void onInterstitialInitFailed(String description) {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.ADAPTER_CALLBACK, getProviderName() + ": onInterstitialInitFailed ", IronSourceLogLevel.INFO);
        for (InterstitialSmashListener smash : mAllInterstitialSmashes) {
            if (smash != null)
                smash.onInterstitialInitFailed(ErrorBuilder.buildInitFailedError(description, IronSourceConstants.INTERSTITIAL_AD_UNIT));
        }
    }

    @Override
    public void onInterstitialLoadSuccess() {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.ADAPTER_CALLBACK, getProviderName() + ": onInterstitialLoadSuccess ", IronSourceLogLevel.INFO);

        for (InterstitialSmashListener smash : mAllInterstitialSmashes) {
            if (smash != null)
                smash.onInterstitialAdReady();
        }
    }

    @Override
    public void onInterstitialLoadFailed(String description) {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.ADAPTER_CALLBACK, getProviderName() + ": onInterstitialAdLoadFailed ", IronSourceLogLevel.INFO);

        for (InterstitialSmashListener smash : mAllInterstitialSmashes) {
            if (smash != null)
                smash.onInterstitialAdLoadFailed(ErrorBuilder.buildLoadFailedError(description));
        }
    }

    @Override
    public void onInterstitialOpen() {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.ADAPTER_CALLBACK, getProviderName() + ": onInterstitialAdOpened ", IronSourceLogLevel.INFO);

        if (mActiveInterstitialSmash != null)
            mActiveInterstitialSmash.onInterstitialAdOpened();
    }

    @Override
    public void onInterstitialAdRewarded(String demandSourceId, int amount) {

    }

    @Override
    public void onInterstitialClose() {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.ADAPTER_CALLBACK, getProviderName() + ": onInterstitialAdClosed ", IronSourceLogLevel.INFO);

        if (mActiveInterstitialSmash != null)
            mActiveInterstitialSmash.onInterstitialAdClosed();
    }

    @Override
    public void onInterstitialShowSuccess() {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.ADAPTER_CALLBACK, getProviderName() + ": onInterstitialAdShowSucceeded ", IronSourceLogLevel.INFO);

        if (mActiveInterstitialSmash != null)
            mActiveInterstitialSmash.onInterstitialAdShowSucceeded();
    }

    @Override
    public void onInterstitialShowFailed(String description) {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.ADAPTER_CALLBACK, getProviderName() + ": onInterstitialAdShowFailed ", IronSourceLogLevel.INFO);

        if (mActiveInterstitialSmash != null) {
            mActiveInterstitialSmash.onInterstitialAdShowFailed(ErrorBuilder.buildShowFailedError(IronSourceConstants.INTERSTITIAL_AD_UNIT, description));
        }
    }

    @Override
    public void onInterstitialClick() {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.ADAPTER_CALLBACK, getProviderName() + ": onInterstitialAdClicked ", IronSourceLogLevel.INFO);

        if (mActiveInterstitialSmash != null)
            mActiveInterstitialSmash.onInterstitialAdClicked();
    }

    @Override
    public void onInterstitialEventNotificationReceived(String eventName, JSONObject extData) {
        if (extData != null) {
            IronSourceLoggerManager.getLogger().log(IronSourceTag.ADAPTER_CALLBACK, getProviderName() + ": onInterstitialEventNotificationReceived: " + eventName + " extData: " + extData.toString(), IronSourceLogLevel.INFO);
            if (!TextUtils.isEmpty(eventName) && AD_VISIBLE_EVENT_NAME.equals(eventName) && mActiveInterstitialSmash != null) {
                mActiveInterstitialSmash.onInterstitialAdVisible();
            }
        }
    }


    // ********** Supersonic Offerwall Callbacks **********

    @Override
    public void onOfferwallInitSuccess() {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.ADAPTER_CALLBACK, getProviderName() + ": onOfferwallInitSuccess ", IronSourceLogLevel.INFO);

        if (mOfferwallListener != null) {
            mOfferwallListener.onOfferwallAvailable(true);
        }
    }

    @Override
    public void onOfferwallInitFail(String description) {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.ADAPTER_CALLBACK, getProviderName() + ": onOfferwallInitFail ", IronSourceLogLevel.INFO);

        if (mOfferwallListener != null) {
            IronSourceError sse = ErrorBuilder.buildGenericError(description);
            mOfferwallListener.onOfferwallAvailable(false, sse);
        }
    }

    @Override
    public void onOfferwallEventNotificationReceived(String eventName, JSONObject extData) {
        if (extData != null) {
            IronSourceLoggerManager.getLogger().log(IronSourceTag.ADAPTER_CALLBACK,
                    getProviderName() + " :onOfferwallEventNotificationReceived ", IronSourceLogLevel.INFO);
        }
    }

    @Override
    public void onOWShowSuccess(String placementId) {
        if (TextUtils.isEmpty(placementId)) {
            log(IronSourceTag.ADAPTER_API, getProviderName() + ":onOWShowSuccess()", IronSourceLogLevel.INFO);
        } else {
            log(IronSourceTag.ADAPTER_API, getProviderName() + ":onOWShowSuccess(placementId:" + placementId + ")", IronSourceLogLevel.INFO);
        }

        if (mOfferwallListener != null) {
            mOfferwallListener.onOfferwallOpened();
        }
    }

    @Override
    public void onOWShowFail(String desc) {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.ADAPTER_CALLBACK, getProviderName() + ": onOWShowFail ", IronSourceLogLevel.INFO);

        if (mOfferwallListener != null) {
            IronSourceError sse = ErrorBuilder.buildGenericError(desc);
            mOfferwallListener.onOfferwallShowFailed(sse);
        }
    }

    @Override
    public void onOWGeneric(String arg0, String arg1) {
        // Deprecated
    }

    @Override
    public boolean onOWAdCredited(int credits, int totalCredits, boolean totalCreditsFlag) {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.ADAPTER_CALLBACK, getProviderName() + ": onOWAdCredited ", IronSourceLogLevel.INFO);

        return mOfferwallListener != null && mOfferwallListener.onOfferwallAdCredited(credits, totalCredits, totalCreditsFlag);
    }

    @Override
    public void onOWAdClosed() {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.ADAPTER_CALLBACK, getProviderName() + ": onOWAdClosed ", IronSourceLogLevel.INFO);

        if (mOfferwallListener != null)
            mOfferwallListener.onOfferwallClosed();
    }

    @Override
    public void onGetOWCreditsFailed(String desc) {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.ADAPTER_CALLBACK, getProviderName() + ": onGetOWCreditsFailed ", IronSourceLogLevel.INFO);

        if (mOfferwallListener != null) {
            IronSourceError sse = ErrorBuilder.buildGenericError(desc);
            mOfferwallListener.onGetOfferwallCreditsFailed(sse);
        }
    }

    // ********** Supersonic Banner Callbacks **********

    @Override
    public void onBannerInitSuccess() {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.ADAPTER_CALLBACK, getProviderName() + ": onBannerInitSuccess ", IronSourceLogLevel.INFO);

        for (BannerSmashListener smash : mAllBannerSmashes) {
            if (smash != null)
                smash.onBannerInitSuccess();
        }
    }

    @Override
    public void onBannerInitFailed(String description) {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.ADAPTER_CALLBACK, getProviderName() + ": onBannerInitFailed ", IronSourceLogLevel.INFO);

        for (BannerSmashListener smash : mAllBannerSmashes) {
            if (smash != null)
                smash.onBannerInitFailed(ErrorBuilder.buildInitFailedError(description, IronSourceConstants.BANNER_AD_UNIT));
        }
    }

    @Override
    public void onBannerClick() {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.ADAPTER_CALLBACK, getProviderName() + ": onBannerAdClicked ", IronSourceLogLevel.INFO);

        if (mActiveBannerSmash != null)
            mActiveBannerSmash.onBannerAdClicked();
    }

    @Override
    public void onBannerLoadSuccess() {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.ADAPTER_CALLBACK, getProviderName() + ": onBannerLoadSuccess ", IronSourceLogLevel.INFO);

        for (BannerSmashListener smash : mAllBannerSmashes) {
            if (smash != null && mIsnAdView != null) {
                FrameLayout.LayoutParams layoutParams = new FrameLayout.LayoutParams(mIsnAdView.getAdViewSize().getWidth(), mIsnAdView.getAdViewSize().getHeight());
                layoutParams.gravity = Gravity.CENTER;
                smash.onBannerAdLoaded(mIsnAdView, layoutParams);
            }
        }
    }

    @Override
    public void onBannerLoadFail(String description) {
        IronSourceLoggerManager.getLogger().log(IronSourceTag.ADAPTER_CALLBACK, getProviderName() + ": onBannerLoadFail", IronSourceLogLevel.INFO);

        for (BannerSmashListener smash : mAllBannerSmashes) {
            if (smash != null)
                smash.onBannerAdLoadFailed(ErrorBuilder.buildInitFailedError(description, IronSourceConstants.BANNER_AD_UNIT));
        }
    }

    @Override
    protected void addBannerListener(BannerSmashListener listener) {
        mAllBannerSmashes.add(listener);
    }

    @Override
    protected void removeBannerListener(BannerSmashListener listener) {
        mAllBannerSmashes.remove(listener);
    }


    protected void setMediationState(AbstractSmash.MEDIATION_STATE state, String adUnit) {
        if (mSSAPublisher != null) {
            IronSourceLoggerManager.getLogger().log(IronSourceTag.INTERNAL, getProviderName() + ": setMediationState(" + adUnit + " , " + getProviderName() + " , " + state.getValue() + ")", IronSourceLogLevel.INFO);
            mSSAPublisher.setMediationState(adUnit, getProviderName(), state.getValue());
        }
    }

    protected void setConsent(boolean consent) {
        mDidSetConsent = true;
        mConsent = consent;
        applyConsent(consent);
    }

    private void applyConsent(boolean consent) {
        if (mSSAPublisher == null) {
            // if there is not instance OR consent was not changed do nothing
            return;
        }

        JSONObject consentParams = new JSONObject();
        try {
            consentParams.put(Constants.RequestParameters.GDPR_CONSENT_STATUS, String.valueOf(consent));
            consentParams.put(Constants.RequestParameters.DEMAND_SOURCE_NAME, getProviderName());
        } catch (JSONException e) {
            e.printStackTrace();
        }
        mSSAPublisher.updateConsentInfo(consentParams);
    }
}
