/*
 * Decompiled with CFR 0.152.
 */
package com.intuit.oauth2.client;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import com.intuit.oauth2.config.OAuth2Config;
import com.intuit.oauth2.data.BearerTokenResponse;
import com.intuit.oauth2.data.PlatformResponse;
import com.intuit.oauth2.data.UserInfoResponse;
import com.intuit.oauth2.exception.ConnectionException;
import com.intuit.oauth2.exception.OAuthException;
import com.intuit.oauth2.exception.OpenIdException;
import com.intuit.oauth2.http.HttpRequestClient;
import com.intuit.oauth2.http.MethodType;
import com.intuit.oauth2.http.Request;
import com.intuit.oauth2.http.Response;
import com.intuit.oauth2.utils.LoggerImpl;
import com.intuit.oauth2.utils.MapperImpl;
import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.security.KeyFactory;
import java.security.PublicKey;
import java.security.Signature;
import java.security.spec.RSAPublicKeySpec;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import org.apache.commons.codec.binary.Base64;
import org.apache.http.NameValuePair;
import org.apache.http.message.BasicNameValuePair;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;

public class OAuth2PlatformClient {
    private OAuth2Config oauth2Config;
    private static final Logger logger = LoggerImpl.getInstance();
    private static final ObjectMapper mapper = MapperImpl.getInstance();

    public OAuth2PlatformClient(OAuth2Config oauth2Config) {
        this.oauth2Config = oauth2Config;
    }

    protected OAuth2PlatformClient() {
    }

    public BearerTokenResponse retrieveBearerTokens(String authCode, String redirectURI) throws OAuthException {
        logger.debug("Enter OAuth2PlatformClient::retrieveBearerTokens");
        try {
            HttpRequestClient client = new HttpRequestClient(this.oauth2Config.getProxyConfig());
            Request request = new Request.RequestBuilder(MethodType.POST, this.oauth2Config.getIntuitBearerTokenEndpoint()).requiresAuthentication(true).authString(this.getAuthHeader()).postParams(this.getUrlParameters(null, authCode, redirectURI)).build();
            Response response = client.makeRequest(request);
            logger.debug("Response Code : " + response.getStatusCode());
            logger.debug("intuit_tid : " + response.getIntuit_tid());
            if (response.getStatusCode() != 200) {
                logger.debug("failed getting access token");
                logger.debug("Response content : " + response.getContent());
                throw new OAuthException("Failed getting access token", response.getStatusCode() + "", response.getIntuit_tid(), response);
            }
            ObjectReader reader = mapper.readerFor(BearerTokenResponse.class);
            BearerTokenResponse bearerTokenResponse = (BearerTokenResponse)reader.readValue(response.getContent());
            bearerTokenResponse.setIntuit_tid(response.getIntuit_tid());
            return bearerTokenResponse;
        }
        catch (OAuthException ex) {
            logger.error("OAuthException while retrieving bearer tokens: " + ex.getResponseContent());
            throw new OAuthException("Failed getting access token", ex.getStatusCode() + "", ex.getIntuit_tid(), ex.getResponse());
        }
        catch (Exception ex) {
            logger.error("Exception while retrieving bearer tokens", ex);
            throw new OAuthException(ex.getMessage(), ex);
        }
    }

    public BearerTokenResponse refreshToken(String refreshToken) throws OAuthException {
        logger.debug("Enter OAuth2PlatformClient::refreshToken");
        try {
            HttpRequestClient client = new HttpRequestClient(this.oauth2Config.getProxyConfig());
            Request request = new Request.RequestBuilder(MethodType.POST, this.oauth2Config.getIntuitBearerTokenEndpoint()).requiresAuthentication(true).authString(this.getAuthHeader()).postParams(this.getUrlParameters("refresh", refreshToken, null)).build();
            Response response = client.makeRequest(request);
            logger.debug("Response Code : " + response.getStatusCode());
            logger.debug("intuit_tid : " + response.getIntuit_tid());
            if (response.getStatusCode() != 200) {
                logger.debug("Failed to refresh token");
                logger.debug("Response content : " + response.getContent());
                throw new OAuthException("Failed to refresh token", response.getStatusCode() + "", response.getIntuit_tid(), response);
            }
            ObjectReader reader = mapper.readerFor(BearerTokenResponse.class);
            BearerTokenResponse bearerTokenResponse = (BearerTokenResponse)reader.readValue(response.getContent());
            bearerTokenResponse.setIntuit_tid(response.getIntuit_tid());
            return bearerTokenResponse;
        }
        catch (OAuthException ex) {
            logger.error("OAuthException while calling refreshToken:  " + ex.getResponseContent());
            throw new OAuthException(ex.getMessage(), ex.getStatusCode() + "", ex.getIntuit_tid(), ex.getResponse());
        }
        catch (Exception ex) {
            logger.error("Exception while calling refreshToken ");
            throw new OAuthException(ex.getMessage(), ex);
        }
    }

    private List<NameValuePair> getUrlParameters(String action, String token, String redirectUri) {
        ArrayList<NameValuePair> urlParameters = new ArrayList<NameValuePair>();
        if (action == "revoke") {
            urlParameters.add(new BasicNameValuePair("token", token));
        } else if (action == "refresh") {
            urlParameters.add(new BasicNameValuePair("refresh_token", token));
            urlParameters.add(new BasicNameValuePair("grant_type", "refresh_token"));
        } else {
            urlParameters.add(new BasicNameValuePair("code", token));
            urlParameters.add(new BasicNameValuePair("redirect_uri", redirectUri));
            urlParameters.add(new BasicNameValuePair("grant_type", "authorization_code"));
        }
        return urlParameters;
    }

    public PlatformResponse revokeToken(String token) throws ConnectionException {
        logger.debug("Enter OAuth2PlatformClient::revokeToken");
        PlatformResponse platformResponse = new PlatformResponse();
        try {
            HttpRequestClient client = new HttpRequestClient(this.oauth2Config.getProxyConfig());
            Request request = new Request.RequestBuilder(MethodType.POST, this.oauth2Config.getIntuitRevokeTokenEndpoint()).requiresAuthentication(true).authString(this.getAuthHeader()).postParams(this.getUrlParameters("revoke", token, null)).build();
            Response response = client.makeRequest(request);
            logger.debug("Response Code : " + response.getStatusCode());
            logger.debug("intuit_tid: " + response.getIntuit_tid());
            if (response.getStatusCode() != 200) {
                logger.debug("failed to revoke token");
                platformResponse.setStatus("ERROR");
                platformResponse.setErrorCode(response.getStatusCode() + "");
                platformResponse.setErrorMessage("Failed to revoke token");
                return platformResponse;
            }
            platformResponse.setStatus("SUCCESS");
            platformResponse.setIntuit_tid(response.getIntuit_tid());
            return platformResponse;
        }
        catch (Exception ex) {
            logger.error("Exception while calling revokeToken ", ex);
            throw new ConnectionException(ex.getMessage(), ex);
        }
    }

    private String getAuthHeader() {
        byte[] bytesEncoded = Base64.encodeBase64((this.oauth2Config.getClientId() + ":" + this.oauth2Config.getClientSecret()).getBytes());
        String base64ClientIdSec = new String(bytesEncoded);
        return "Basic " + base64ClientIdSec;
    }

    public UserInfoResponse getUserInfo(String accessToken) throws OpenIdException {
        logger.debug("Enter OAuth2PlatformClient::getUserInfo");
        try {
            HttpRequestClient client = new HttpRequestClient(this.oauth2Config.getProxyConfig());
            Request request = new Request.RequestBuilder(MethodType.GET, this.oauth2Config.getUserProfileEndpoint()).requiresAuthentication(true).authString("Bearer " + accessToken).build();
            Response response = client.makeRequest(request);
            logger.debug("Response Code : " + response.getStatusCode());
            logger.debug("intuit_tid: " + response.getIntuit_tid());
            if (response.getStatusCode() == 200) {
                ObjectReader reader = mapper.readerFor(UserInfoResponse.class);
                UserInfoResponse userInfoResponse = (UserInfoResponse)reader.readValue(response.getContent());
                userInfoResponse.setIntuit_tid(response.getIntuit_tid());
                return userInfoResponse;
            }
            logger.debug("failed getting user info");
            throw new OpenIdException("failed getting user info", response.getStatusCode() + "", response.getIntuit_tid(), response);
        }
        catch (OpenIdException ex) {
            logger.error("OpenIdException while retrieving user info: " + ex.getResponseContent());
            throw new OpenIdException("failed getting user info", ex.getStatusCode() + "", ex.getIntuit_tid(), ex.getResponse());
        }
        catch (Exception ex) {
            logger.error("Exception while retrieving user info ", ex);
            throw new OpenIdException(ex.getMessage(), ex);
        }
    }

    public boolean validateIDToken(String idToken) throws OpenIdException {
        logger.debug("Enter OAuth2PlatformClient::validateIDToken");
        String[] idTokenParts = idToken.split("\\.");
        if (idTokenParts.length < 3) {
            logger.debug("invalid idTokenParts length");
            return false;
        }
        String idTokenHeader = this.base64UrlDecode(idTokenParts[0]);
        String idTokenPayload = this.base64UrlDecode(idTokenParts[1]);
        byte[] idTokenSignature = this.base64UrlDecodeToBytes(idTokenParts[2]);
        JSONObject idTokenHeaderJson = new JSONObject(idTokenHeader);
        JSONObject idTokenHeaderPayload = new JSONObject(idTokenPayload);
        String issuer = idTokenHeaderPayload.getString("iss");
        if (!issuer.equalsIgnoreCase(this.oauth2Config.getIntuitIdTokenIssuer())) {
            logger.debug("issuer value mismtach");
            return false;
        }
        JSONArray jsonaud = idTokenHeaderPayload.getJSONArray("aud");
        String aud = jsonaud.getString(0);
        if (!aud.equalsIgnoreCase(this.oauth2Config.getClientId())) {
            logger.debug("incorrect client id");
            return false;
        }
        Long expirationTimestamp = idTokenHeaderPayload.getLong("exp");
        Long currentTime = System.currentTimeMillis() / 1000L;
        if (expirationTimestamp - currentTime <= 0L) {
            logger.debug("expirationTimestamp has elapsed");
            return false;
        }
        HashMap<String, JSONObject> keyMap = this.getKeyMapFromJWKSUri();
        if (keyMap == null || keyMap.isEmpty()) {
            logger.debug("unable to retrive keyMap from JWKS url");
            return false;
        }
        String keyId = idTokenHeaderJson.getString("kid");
        JSONObject keyDetails = keyMap.get(keyId);
        String exponent = keyDetails.getString("e");
        String modulo = keyDetails.getString("n");
        PublicKey publicKey = this.getPublicKey(modulo, exponent);
        byte[] data = (idTokenParts[0] + "." + idTokenParts[1]).getBytes(StandardCharsets.UTF_8);
        try {
            boolean isSignatureValid = this.verifyUsingPublicKey(data, idTokenSignature, publicKey);
            logger.debug("isSignatureValid: " + isSignatureValid);
            return isSignatureValid;
        }
        catch (GeneralSecurityException e) {
            logger.error("Exception while validating ID token ", e);
            throw new OpenIdException(e.getMessage(), e);
        }
    }

    private HashMap<String, JSONObject> getKeyMapFromJWKSUri() throws OpenIdException {
        logger.debug("Enter OAuth2PlatformClient::getKeyMapFromJWKSUri");
        try {
            HttpRequestClient client = new HttpRequestClient(this.oauth2Config.getProxyConfig());
            Request request = new Request.RequestBuilder(MethodType.GET, this.oauth2Config.getIntuitJwksURI()).requiresAuthentication(false).build();
            Response response = client.makeRequest(request);
            logger.debug("Response Code : " + response.getStatusCode());
            if (response.getStatusCode() != 200) {
                logger.debug("failed JWKS URI");
                throw new OpenIdException("failed JWKS URI", response.getStatusCode() + "");
            }
            return this.buildKeyMap(response.getContent());
        }
        catch (Exception ex) {
            logger.error("Exception while retrieving jwks ", ex);
            throw new OpenIdException(ex.getMessage(), ex);
        }
    }

    private PublicKey getPublicKey(String MODULUS, String EXPONENT) {
        byte[] nb = this.base64UrlDecodeToBytes(MODULUS);
        byte[] eb = this.base64UrlDecodeToBytes(EXPONENT);
        BigInteger n = new BigInteger(1, nb);
        BigInteger e = new BigInteger(1, eb);
        RSAPublicKeySpec rsaPublicKeySpec = new RSAPublicKeySpec(n, e);
        try {
            PublicKey publicKey = KeyFactory.getInstance("RSA").generatePublic(rsaPublicKeySpec);
            return publicKey;
        }
        catch (Exception ex) {
            logger.error("Exception while getting public key ", ex);
            throw new RuntimeException("Cant create public key", ex);
        }
    }

    private boolean verifyUsingPublicKey(byte[] data, byte[] signature, PublicKey pubKey) throws GeneralSecurityException {
        Signature sig = Signature.getInstance("SHA256withRSA");
        sig.initVerify(pubKey);
        sig.update(data);
        return sig.verify(signature);
    }

    private String base64UrlDecode(String input) {
        byte[] decodedBytes = this.base64UrlDecodeToBytes(input);
        String result = new String(decodedBytes, StandardCharsets.UTF_8);
        return result;
    }

    private byte[] base64UrlDecodeToBytes(String input) {
        Base64 decoder = new Base64(-1, null, true);
        byte[] decodedBytes = decoder.decode(input);
        return decodedBytes;
    }

    private HashMap<String, JSONObject> buildKeyMap(String content) throws ConnectionException {
        HashMap<String, JSONObject> retMap = new HashMap<String, JSONObject>();
        JSONObject jwksPayload = new JSONObject(content);
        JSONArray keysArray = jwksPayload.getJSONArray("keys");
        for (int i = 0; i < keysArray.length(); ++i) {
            JSONObject object = keysArray.getJSONObject(i);
            String keyId = object.getString("kid");
            retMap.put(keyId, object);
        }
        return retMap;
    }
}

