package com.instabug.featuresrequest.ui.featuredetails;

import android.annotation.SuppressLint;
import android.graphics.drawable.GradientDrawable;
import android.os.Build;
import android.os.Bundle;
import android.text.TextUtils;
import android.view.View;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.ListView;
import android.widget.RelativeLayout;
import android.widget.TextView;

import com.instabug.featuresrequest.R;
import com.instabug.featuresrequest.models.FeatureRequest;
import com.instabug.featuresrequest.models.StatusChange;
import com.instabug.featuresrequest.models.TimelineObject;
import com.instabug.featuresrequest.models.TimelineResponse;
import com.instabug.featuresrequest.ui.addcomment.AddCommentFragment;
import com.instabug.featuresrequest.ui.base.featureslist.OnVoteChangeListener;
import com.instabug.featuresrequest.ui.custom.DynamicToolbarFragment;
import com.instabug.featuresrequest.ui.custom.ToolbarActionButton;
import com.instabug.featuresrequest.utils.DateUtils;
import com.instabug.featuresrequest.utils.ListviewUtils;
import com.instabug.featuresrequest.utils.StatusBinder;
import com.instabug.featuresrequest.utils.TextViewUtils;
import com.instabug.featuresrequest.utils.ViewUtilsKt;
import com.instabug.library.Instabug;
import com.instabug.library.InstabugColorTheme;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.AttrResolver;
import com.instabug.library.view.ViewUtils;

import java.util.ArrayList;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.content.ContextCompat;
import androidx.core.graphics.drawable.DrawableCompat;

import static com.instabug.featuresrequest.ui.custom.ToolbarActionButton.ViewType.ICON;
import static com.instabug.featuresrequest.ui.custom.ToolbarActionButton.ViewType.VOTE;

/**
 * Created by mohamedzakaria on 9/20/17.
 */

// no place for error here
@SuppressLint("ERADICATE_FIELD_NOT_INITIALIZED")
public class FeatureRequestsDetailsFragment extends DynamicToolbarFragment<FeaturesRequestsDetailsPresenter>
        implements FeaturesRequestsDetailsContract.View {
    private static final String KEY_FEATURE = "key_feature";

    @Nullable
    private LinearLayout voteActionLayout;
    @Nullable
    private FeatureRequest featureRequest;
    @Nullable
    private TextView voteTextCount;
    @Nullable
    private TextView featureTitle;
    @Nullable
    private TextView featureStatus;
    @Nullable
    private TextView ownerName;
    @Nullable
    private TextView date;
    @Nullable
    private TextView commentsCount;
    @Nullable
    private ImageView voteIcon;
    @Nullable
    private TextView featureDescription;
    @Nullable
    private LinearLayout noCommentsLayout;
    @Nullable
    private LinearLayout addCommentLayoutContainer;
    @Nullable
    private ListView commentsList;
    private boolean isDescriptionExpanded = false;
    @Nullable
    private TimelineAdapter adapter;
    @NonNull
    private ArrayList<TimelineObject> comments = new ArrayList<>();
    private OnVoteChangeListener onVoteChangeListener;

    private boolean isVoteChanged = false;

    public static FeatureRequestsDetailsFragment newInstance(FeatureRequest featureRequest, OnVoteChangeListener onVoteChangeListener) {
        Bundle args = new Bundle();
        args.putSerializable(KEY_FEATURE, featureRequest);
        FeatureRequestsDetailsFragment fragment = new FeatureRequestsDetailsFragment();
        fragment.setVoteChangeListener(onVoteChangeListener);
        fragment.setArguments(args);
        return fragment;
    }

    private void setVoteChangeListener(OnVoteChangeListener onVoteChangeListener) {
        this.onVoteChangeListener = onVoteChangeListener;
    }

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        if (getArguments() != null) {
            featureRequest = (FeatureRequest) getArguments().getSerializable(KEY_FEATURE);
        }
        presenter = new FeaturesRequestsDetailsPresenter(this);
    }

    @Override
    protected ToolbarActionButton getToolbarCloseActionButton() {
        return new ToolbarActionButton(R.drawable.ibg_core_ic_back, R.string.feature_request_go_back,
                () -> {
                    if (presenter != null) {
                        presenter.onBackBtnClicked();
                    }
                }, ICON);
    }

    @Override
    protected void addToolbarActionButtons() {
        toolbarActionButtons.add(new ToolbarActionButton(-1, R.string.ib_feature_rq_str_votes, () -> {
            isVoteChanged = true;
            if (presenter != null && featureRequest != null) {
                presenter.onVoteButtonClick(featureRequest);
            }
        }, VOTE));
    }

    @Override
    protected String getTitle() {
        return getLocalizedString(R.string.feature_requests_details);
    }

    @Override
    protected int getContentLayout() {
        return R.layout.ib_fr_features_details_fragment;
    }

    @Override
    protected void initContentViews(View rootView, @Nullable Bundle savedInstanceState) {
        // Use a local allocation to overcome SpotBugs false-positive possible NPE warnings
        RelativeLayout toolbar = this.toolbar;
        FeaturesRequestsDetailsPresenter presenter = this.presenter;
        if (toolbar != null) {
            voteActionLayout = toolbar.findViewById(R.id.ib_feature_request_toolbar_vote_action_layout);
            voteTextCount = toolbar.findViewById(R.id.ib_toolbar_vote_count);
            voteIcon = toolbar.findViewById(R.id.ib_toolbar_vote_icon);
        }
        featureDescription = rootView.findViewById(R.id.ib_fr_tv_feature_details_desc);
        featureTitle = rootView.findViewById(R.id.ib_fr_details_title);
        featureStatus = rootView.findViewById(R.id.instabug_txt_feature_request_status);
        date = rootView.findViewById(R.id.ib_txt_feature_request_date);
        ownerName = rootView.findViewById(R.id.ib_txt_feature_request_owner);
        TextView addFeature = rootView.findViewById(R.id.tv_add_comment);
        commentsCount = rootView.findViewById(R.id.ib_features_request_comment_count);
        noCommentsLayout = rootView.findViewById(R.id.ib_fr_details_no_comments_layout);
        ImageView noCommentsIcon = rootView.findViewById(R.id.ib_fr_details_no_comments_icon);
        commentsList = rootView.findViewById(R.id.instabug_feature_details_comments_list);
        addCommentLayoutContainer = rootView.findViewById(R.id.addCommentLayoutContainer);

        this.toolbar = toolbar;

        if (Build.VERSION.SDK_INT > Build.VERSION_CODES.UPSIDE_DOWN_CAKE) {
            LinearLayout mainLayout = findViewById(R.id.ib_fr_tv_feature_details_main_layout);
            if(mainLayout != null) {
                ViewUtilsKt.addSystemWindowInsetToPadding(mainLayout, true, false, true, true);
            }
        }


        // styles
        if (noCommentsIcon != null && getContext() != null) {
            noCommentsIcon.setColorFilter(AttrResolver.getColor(getContext(), R.attr.ib_fr_no_comments_icon_color));
        }

        if (addFeature != null) {
            addFeature.setOnClickListener(this);
        }

        adapter = new TimelineAdapter(comments, this);
        if (commentsList != null)
            commentsList.setAdapter(adapter);

        if (presenter == null || featureRequest == null) return;

        fillData(featureRequest);

        presenter.getFeatureTimeline(featureRequest.getFeatureId());
        this.presenter = presenter;
    }

    @Override
    public void fillData(final FeatureRequest featureRequest) {
        this.featureRequest = featureRequest;
        if (featureTitle != null)
            featureTitle.setText(featureRequest.getTitle());
        if (featureDescription != null) {
            if (featureRequest.getDescription() != null
                    && !featureRequest.getDescription().equalsIgnoreCase("null")
                    && !TextUtils.isEmpty(featureRequest.getDescription())) {
                featureDescription.setVisibility(View.VISIBLE);
                TextViewUtils.makeTextViewResizable(
                        featureDescription,
                        featureRequest.getDescription(),
                        getLocalizedString(R.string.feature_request_str_more),
                        getLocalizedString(R.string.feature_request_str_less),
                        !isDescriptionExpanded, new Runnable() {
                            @Override
                            public void run() {
                                isDescriptionExpanded = !isDescriptionExpanded;
                            }
                        });
            } else {
                featureDescription.setVisibility(View.GONE);
            }
        }

        //disable adding comments and voting ability if the feature has been marked as completed
        if (addCommentLayoutContainer != null && voteActionLayout != null) {
            if (featureRequest.isCompleted()) {
                addCommentLayoutContainer.setVisibility(View.GONE);
                voteActionLayout.setEnabled(false);
            } else {
                addCommentLayoutContainer.setVisibility(View.VISIBLE);
                voteActionLayout.setEnabled(true);
            }
        }
        if (getContext() == null) return;

        if (ownerName != null) {
            ownerName.setText((featureRequest.getCreatorName() != null
                    && !featureRequest.getCreatorName().equalsIgnoreCase("null")
                    && !TextUtils.isEmpty(featureRequest.getCreatorName()))
                    ? getLocalizedString(R.string.feature_request_owner, featureRequest.getCreatorName())
                    : getLocalizedString(R.string.feature_request_owner_anonymous));
        }
        if (commentsCount != null) {
            commentsCount.setText(getLocalizedString(R.string.feature_request_comments_count, featureRequest.getCommentsCount()));
        }

        StatusBinder.bind(featureRequest.getStatus(), featureRequest.getColorCode(), featureStatus, getContext());
        if (date != null) {
            date.setText(DateUtils.getTimeAgo(getContext(), featureRequest.getDate()));
        }
        handleVote(featureRequest);
    }

    @Override
    public void handleBackBtnClicked() {
        if (getActivity() != null) {
            getActivity().onBackPressed();
        }
    }


    private void handleVote(final FeatureRequest featureRequest) {
        if (voteActionLayout != null) {
            voteActionLayout.post(new Runnable() {
                @Override
                public void run() {
                    if (isAdded() && !isRemoving() && getContext() != null && voteActionLayout != null) {
                        // Use a local allocation to overcome SpotBugs false-positive possible NPE warnings
                        TextView localVoteTextCount = voteTextCount;
                        if (voteIcon != null && localVoteTextCount != null) {
                            voteIcon.setImageResource(R.drawable.ibg_fr_ic_vote_arrow_white);
                            GradientDrawable gradientDrawable = (GradientDrawable) voteActionLayout.getBackground();
                            localVoteTextCount.setText(getLocalizedString(R.string.feature_request_votes_count, featureRequest.getLikesCount()));
                            if (Instabug.getTheme() == InstabugColorTheme.InstabugColorThemeLight) {
                                if (featureRequest.isLiked()) {
                                    gradientDrawable.setStroke(ViewUtils.convertDpToPx(getContext(), 2), ContextCompat.getColor(getContext(), android.R.color.white));
                                    gradientDrawable.setColor(ContextCompat.getColor(getContext(), android.R.color.white));

                                    localVoteTextCount.setTextColor(SettingsManager.getInstance().getPrimaryColor());
                                    DrawableCompat.setTint(voteIcon.getDrawable(), SettingsManager.getInstance().getPrimaryColor());
                                } else {
                                    gradientDrawable.setStroke(ViewUtils.convertDpToPx(getContext(), 2), ContextCompat.getColor(getContext(), R.color.ib_fr_toolbar_vote_btn_stroke_color));
                                    gradientDrawable.setColor(ContextCompat.getColor(getContext(), android.R.color.transparent));

                                    localVoteTextCount.setTextColor(ContextCompat.getColor(getContext(), android.R.color.white));
                                    DrawableCompat.setTint(voteIcon.getDrawable(), ContextCompat.getColor(getContext(), android.R.color.white));
                                }
                            } else {
                                if (featureRequest.isLiked()) {
                                    gradientDrawable.setStroke(ViewUtils.convertDpToPx(getContext(), 2), SettingsManager.getInstance().getPrimaryColor());
                                    gradientDrawable.setColor(SettingsManager.getInstance().getPrimaryColor());

                                    localVoteTextCount.setTextColor(ContextCompat.getColor(getContext(), android.R.color.white));
                                    DrawableCompat.setTint(voteIcon.getDrawable(), ContextCompat.getColor(getContext(), android.R.color.white));
                                } else {
                                    gradientDrawable.setStroke(ViewUtils.convertDpToPx(getContext(), 2), SettingsManager.getInstance().getPrimaryColor());
                                    gradientDrawable.setColor(ContextCompat.getColor(getContext(), android.R.color.transparent));

                                    localVoteTextCount.setTextColor(SettingsManager.getInstance().getPrimaryColor());
                                    DrawableCompat.setTint(voteIcon.getDrawable(), SettingsManager.getInstance().getPrimaryColor());
                                }
                            }
                            voteTextCount = localVoteTextCount;
                            if (voteActionLayout != null) {
                                if (Build.VERSION.SDK_INT >= 16) {
                                    voteActionLayout.setBackground(gradientDrawable);
                                } else {
                                    voteActionLayout.setBackgroundDrawable(gradientDrawable);
                                }
                            }
                        }
                    }
                }
            });
        }
    }


    @Override
    public void onClick(View v) {
        if (v.getId() == R.id.tv_add_comment) {
            // TODO: 2/19/18 handle add feature action
            if (getActivity() != null && featureRequest != null) {
                getActivity().getSupportFragmentManager()
                        .beginTransaction()
                        .add(R.id.instabug_fragment_container, AddCommentFragment.newInstance(featureRequest.getFeatureId()))
                        .addToBackStack("add_comment")
                        .commit();
            }
        }
    }

    @Override
    public void showNoCommentsLayout() {
        if (noCommentsLayout != null)
            noCommentsLayout.setVisibility(View.VISIBLE);
    }

    @Override
    public void updateVote(FeatureRequest featureRequest) {
        handleVote(featureRequest);
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        if (onVoteChangeListener != null && isVoteChanged) {
            onVoteChangeListener.onVoteChange();
        }
    }

    @Override
    public void showComments(TimelineResponse timeline) {
        // Use a local allocation to overcome SpotBugs false-positive possible NPE warnings
        ListView localCommentsList = commentsList;
        if (localCommentsList != null) {
            comments = new ArrayList<>();
            adapter = null;
            adapter = new TimelineAdapter(comments, this);
            localCommentsList.setAdapter(adapter);
            comments.addAll(timeline.getTimelineObjects());
            adapter.notifyDataSetChanged();
            if (noCommentsLayout != null)
                noCommentsLayout.setVisibility(View.GONE);
            localCommentsList.invalidate();
            ListviewUtils.setListViewHeightBasedOnChildren(localCommentsList);
        }
        commentsList = localCommentsList;
    }

    @Override
    public void handelCompletedFeaturesActions() {
        if (comments.size() > 0) {
            for (int i = 0; i < comments.size() - 1; i++) {
                TimelineObject timelineObject = comments.get(i);
                if (timelineObject instanceof StatusChange) {
                    if (addCommentLayoutContainer != null && voteActionLayout != null) {
                        if (((StatusChange) timelineObject).getNewStatus() == FeatureRequest.Status.Completed) {
                            addCommentLayoutContainer.setVisibility(View.GONE);
                            voteActionLayout.setEnabled(false);
                        } else {
                            addCommentLayoutContainer.setVisibility(View.VISIBLE);
                            voteActionLayout.setEnabled(true);
                        }
                        break;
                    }
                }
            }
        }
    }

    @Override
    public void updateListHeight() {
        ListviewUtils.setListViewHeightBasedOnChildren(commentsList);
    }

    public void onNewCommentAdded() {
        if (featureRequest == null || presenter == null) return;

        // Use a local allocation to overcome SpotBugs false-positive possible NPE warnings
        FeaturesRequestsDetailsPresenter presenter = this.presenter;
        featureRequest.setCommentsCount(featureRequest.getCommentsCount() + 1);
        fillData(featureRequest);
        presenter.getFeatureTimeline(featureRequest.getFeatureId());
        this.presenter = presenter;
    }
}
