package com.instabug.featuresrequest.ui.base.featureslist;

import android.content.Context;
import android.os.Looper;

import androidx.annotation.Nullable;

import com.instabug.featuresrequest.Constants;
import com.instabug.featuresrequest.R;
import com.instabug.featuresrequest.cache.FeatureRequestsDBHelper;
import com.instabug.featuresrequest.eventbus.FeatureRequestsEventBus;
import com.instabug.featuresrequest.models.FeatureRequest;
import com.instabug.featuresrequest.models.FeatureRequestResponse;
import com.instabug.featuresrequest.network.service.FeaturesRequestService;
import com.instabug.featuresrequest.network.service.FeaturesRequestVoteJob;
import com.instabug.featuresrequest.settings.FeaturesRequestSettings;
import com.instabug.featuresrequest.ui.base.FeaturesListBo;
import com.instabug.featuresrequest.utils.FeaturesRequestUtils;
import com.instabug.library.Instabug;
import com.instabug.library.core.ui.BasePresenter;
import com.instabug.library.networkv2.NetworkManager;
import com.instabug.library.networkv2.request.Request;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.threading.PoolProvider;

import org.json.JSONException;
import org.json.JSONObject;

import io.reactivexport.disposables.CompositeDisposable;
import io.reactivexport.functions.Consumer;

/**
 * @author hossam.
 */

public class FeaturesListPresenter extends BasePresenter<FeaturesListContract.View> implements
        FeaturesListContract.Presenter {

    @Nullable
    private FeaturesListContract.View viewInstance;
    FeaturesListBo featuresListBo;
    private CompositeDisposable disposables;

    public FeaturesListPresenter(FeaturesListContract.View view, FeaturesListBo featuresListBo,
                                 boolean myPosts) {
        super(view);

        if (this.view != null) {
            viewInstance = this.view.get();
        }
        this.featuresListBo = featuresListBo;
        addDataToBo(featuresListBo, featuresListBo.getPage(),
                false, FeaturesRequestSettings.shouldSortByTopRated(), myPosts, true);
        subscribeToVoteEvents();
    }

    public void addDataToBo(final FeaturesListBo featuresListBo,
                            int page,
                            boolean completed,
                            boolean sortByTopVotedFeatures,
                            boolean myPosts,
                            final boolean shouldRefresh) {
        PoolProvider.postIOTask(() -> {
                    if (FeaturesRequestUtils.isFeaturesRequestFeatureEnabled()
                            && Instabug.getApplicationContext() != null) {

                        if (page == 1)
                            PoolProvider.postMainThreadTask(() -> {
                                if (viewInstance != null)
                                    viewInstance.setRefreshing(true);
                            });

                        FeaturesRequestService.getInstance().fetchFeaturesRequests(
                                page,
                                completed,
                                sortByTopVotedFeatures,
                                myPosts,
                                new Request.Callbacks<JSONObject, Throwable>() {
                                    @Override
                                    public void onSucceeded(@Nullable JSONObject response) {

                                        if (response == null) {
                                            onFailed(new JSONException("response json is null"));
                                            return;
                                        }

                                        try {
                                            FeatureRequestResponse featureRequestResponse =
                                                    FeatureRequestResponse.fromJson(response);
                                            PoolProvider.postMainThreadTask(new Runnable() {
                                                @Override
                                                public void run() {
                                                    if (featureRequestResponse.getFeatureRequestList() != null
                                                            && featureRequestResponse.getFeatureRequestList().size() > 0) {
                                                        if (shouldRefresh) {
                                                            featuresListBo.destroy();
                                                        }
                                                        featuresListBo.addFeatures(
                                                                featureRequestResponse.getFeatureRequestList());
                                                        if (featureRequestResponse.hasNextPage()) {
                                                            featuresListBo.incrementPageNumber();
                                                        } else {
                                                            featuresListBo.setHasNextPage(false);
                                                        }
                                                    }
                                                    onRequestFinish();
                                                }
                                            });
                                        } catch (
                                                JSONException e) {
                                            onFailed(e);
                                        }


                                    }

                                    @Override
                                    public void onFailed(Throwable error) {
                                        InstabugSDKLogger.e(Constants.LOG_TAG, error.getMessage() != null ? error.getMessage() :
                                                "Error While fetching features Requests", error);
                                        PoolProvider.postMainThreadTask(new Runnable() {
                                            @Override
                                            public void run() {
                                                onRequestFinishedWithError();
                                            }
                                        });
                                    }
                                });

                    } else {
                        PoolProvider.postMainThreadTask(() -> {

                            if (viewInstance == null) return;

                            if (featuresListBo.getFeaturesCount() == 0) {
                                viewInstance.showErrorView();
                            } else {
                                viewInstance.hideLoadMore();
                            }
                        });
                    }
                }
        );
    }

    protected void onRequestFinish() {
        if (viewInstance == null || !viewInstance.getViewContext().isAdded()
                || viewInstance.getViewContext().getContext() == null)
            return;

        viewInstance.setRefreshing(false);
        if (getFeaturesCount() == 0) {
            if (NetworkManager.isOnline()) {
                viewInstance.showEmptyView();
            } else {
                viewInstance.showErrorView();
            }
        } else {
            viewInstance.showFeatures();
        }
    }

    protected void onRequestFinishedWithError() {
        if (viewInstance == null) {
            return;
        }
        viewInstance.setRefreshing(false);
        if (getFeaturesCount() == 0) {
            viewInstance.showErrorView();
        } else {
            viewInstance.showToast(R.string.feature_requests_error_state_title);
            viewInstance.hideLoadMore();
        }
    }

    @Override
    public void onFeatureClicked(int position) {
        if (viewInstance != null) {
            viewInstance.navigateToTheDetailsScreen((featuresListBo.getFeatureByIndex(position)));
        }
    }

    @Override
    public void onUpVoteFeatureClicked(FeatureRequest featureRequest) {
        featureRequest.setUserVoteStatus(FeatureRequest.UserVoteStatus.USER_VOTED_UP);
        try {
            FeatureRequestsDBHelper.insert(featureRequest);
        } catch (JSONException e) {
        }
        submitVote();
        FeatureRequestsEventBus.getInstance().post(featureRequest);
        if (viewInstance != null) {
            viewInstance.notifyDataSetChanged();
        }
    }

    private void submitVote() {
        if (viewInstance != null) {
            Context context = viewInstance.getViewContext().getContext();
            if (context != null) {
                FeaturesRequestVoteJob.getInstance().start();
            }
        }
    }

    @Override
    public void onUnVoteFeatureClicked(FeatureRequest featureRequest) {
        featureRequest.setUserVoteStatus(FeatureRequest.UserVoteStatus.USER_UN_VOTED);
        try {
            FeatureRequestsDBHelper.insert(featureRequest);
        } catch (JSONException e) {
        }
        submitVote();
        FeatureRequestsEventBus.getInstance().post(featureRequest);
        if (viewInstance != null) {
            viewInstance.notifyDataSetChanged();
        }
    }

    @Override
    public void onLoadMoreTriggered() {
        if (viewInstance != null && featuresListBo.getPage() != 1) {
            if (featuresListBo.hasNextPage()) {
                viewInstance.showFooterView();
                addDataToBo(featuresListBo, featuresListBo.getPage(), false,
                        FeaturesRequestSettings.shouldSortByTopRated(), viewInstance.shouldShowMyPosts(), false);
            } else {
                viewInstance.hideFooterView();
            }
        }
    }

    @Override
    public void handlePbiFooter() {
        if (viewInstance != null)
            viewInstance.handlePoweredByFooter();
    }

    @Override
    public int getFeaturesCount() {
        return featuresListBo.getFeaturesCount();
    }

    @Override
    public void onPullToRefreshTriggered() {
        featuresListBo.setHasNextPage(true);
        if (viewInstance != null && Instabug.getApplicationContext() != null) {
            if (NetworkManager.isOnline()) {
                viewInstance.hideEmptyView();
                viewInstance.hideErrorView();
                addDataToBo(featuresListBo, 1, false,
                        FeaturesRequestSettings.shouldSortByTopRated(), viewInstance.shouldShowMyPosts(), true);
            } else {
                if (featuresListBo.getFeaturesCount() == 0) {
                    if (NetworkManager.isOnline()) {
                        viewInstance.showEmptyView();
                    } else {
                        viewInstance.showErrorView();
                    }
                } else {
                    viewInstance.showErrorToastMessage();
                    viewInstance.hideFooterView();
                }
            }
        }
    }

    @Override
    public void onBindFeatureViewItemAtPosition(int position, FeatureHolder holder) {
        FeatureRequest featureRequest = featuresListBo.getFeatureByIndex(position);
        holder.setFeatureTitle(featureRequest.getTitle());
        holder.setFeatureStateColor(featureRequest);
        holder.setCommentsCount(featureRequest.getCommentsCount());
        holder.setVotesCount(featureRequest.getLikesCount());
        holder.setCreationDate(featureRequest.getDate());
        holder.setVoteButtonStyle(featureRequest.isLiked());
        holder.setVoteButtonListener(featureRequest);
    }

    @Override
    public void onEmptyViewPositiveButtonClicked() {
        if (viewInstance != null) {
            viewInstance.navigateToAddNewFeatureScreen();
        }
    }

    @Override
    public void onDestroy() {
        if (disposables != null) {
            disposables.dispose();
        }
        featuresListBo.destroy();
    }

    @Override
    public void onSortActionChanged() {
        onPullToRefreshTriggered();
    }

    @Override
    public void onErrorStateClicked() {
        if (viewInstance != null) {
            viewInstance.hideErrorView();
            onPullToRefreshTriggered();
        }
    }

    public boolean hasNext() {
        return featuresListBo.hasNextPage();
    }

    private void subscribeToVoteEvents() {
        if (disposables == null || disposables.isDisposed()) {
            disposables = new CompositeDisposable();
        }
        disposables.add(FeatureRequestsEventBus.getInstance()
                .subscribe(new Consumer<FeatureRequest>() {
                    @Override
                    public void accept(FeatureRequest newFeatureRequest) {
                        if (featuresListBo.getFeatures() == null
                                || featuresListBo.getFeatures().isEmpty()) {
                            return;
                        }
                        new android.os.Handler(Looper.getMainLooper()).postDelayed(new Runnable() {
                            @Override
                            public void run() {
                                if (viewInstance != null) {
                                    viewInstance.updateItemsOrder();
                                }
                            }
                        }, 1000);
                    }
                }));

    }
}
