package com.instabug.featuresrequest.ui.addcomment;

import static com.instabug.featuresrequest.ui.custom.ToolbarActionButton.ViewType.ICON;
import static com.instabug.featuresrequest.ui.custom.ToolbarActionButton.ViewType.TEXT;

import android.annotation.SuppressLint;
import android.os.Build;
import android.os.Bundle;
import android.text.Editable;
import android.text.TextUtils;
import android.util.Patterns;
import android.view.View;
import android.widget.RelativeLayout;
import android.widget.TextView;
import android.widget.Toast;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.content.ContextCompat;

import com.google.android.material.textfield.TextInputEditText;
import com.google.android.material.textfield.TextInputLayout;
import com.instabug.featuresrequest.R;
import com.instabug.featuresrequest.models.NewComment;
import com.instabug.featuresrequest.settings.FeaturesRequestSettings;
import com.instabug.featuresrequest.ui.FeaturesRequestActivity;
import com.instabug.featuresrequest.ui.custom.DynamicToolbarFragment;
import com.instabug.featuresrequest.ui.custom.ToolbarActionButton;
import com.instabug.featuresrequest.utils.TextInputLayoutHelper;
import com.instabug.featuresrequest.utils.ViewUtilsKt;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.AttrResolver;
import com.instabug.library.util.KeyboardUtils;
import com.instabug.library.util.SimpleTextWatcher;
import com.instabug.library.view.IBGProgressDialog;
import com.instabug.library.view.ViewUtils;

/**
 * Created by mzelzoghbi on 3/8/18.
 */
@SuppressLint("ERADICATE_FIELD_NOT_INITIALIZED")
public class AddCommentFragment extends DynamicToolbarFragment<AddCommentPresenter> implements AddCommentContract.View {
    private static final String KEY_FEATURE_ID = "featureId";

    private AddCommentPresenter presenter;
    private long featureId;

    @Nullable
    private TextInputLayout commentInputLayout;
    @Nullable
    private TextInputLayout nameInputLayout;
    @Nullable
    private TextInputLayout emailInputLayout;

    @Nullable
    private TextInputEditText commentEditTextLayout;
    @Nullable
    private TextInputEditText nameEditTextLayout;
    @Nullable
    private TextInputEditText emailEditTextLayout;

    @Nullable
    private View underLineComment;
    @Nullable
    private View underLineName;
    @Nullable
    private View underLineEmail;

    @Nullable
    private IBGProgressDialog preparingProgressDialog;
    @Nullable
    private TextView emailDisclaimer;

    @Nullable
    private TextView toolbarPostTextView;

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        presenter = new AddCommentPresenter(this);
        if (getArguments() != null)
            featureId = getArguments().getLong(KEY_FEATURE_ID);
    }

    @Override
    protected ToolbarActionButton getToolbarCloseActionButton() {
        return new ToolbarActionButton(R.drawable.ibg_core_ic_close, R.string.close,
                new ToolbarActionButton.OnToolbarButtonClick() {
                    @Override
                    public void onClick() {
                        presenter.onCloseButtonClicked();
                    }
                }, ICON);
    }

    @Override
    protected void addToolbarActionButtons() {
        toolbarActionButtons.add(new ToolbarActionButton(-1,
                R.string.feature_request_str_post_comment, new
                ToolbarActionButton.OnToolbarButtonClick() {
                    @Override
                    public void onClick() {
                        presenter.onAddCommentButtonClicked();
                    }
                }, TEXT));
    }

    @Override
    protected String getTitle() {
        return getLocalizedString(R.string.feature_request_comments);
    }

    @Override
    protected int getContentLayout() {
        return R.layout.ib_fr_add_comment_fragment;
    }

    @Override
    protected void initContentViews(View rootView, @Nullable Bundle savedInstanceState) {
        commentInputLayout = rootView.findViewById(R.id.feature_request_comment_text_input_layout);
        nameInputLayout = rootView.findViewById(R.id.feature_request_name_text_input_layout);
        emailInputLayout = rootView.findViewById(R.id.feature_request_email_text_input_layout);

        commentEditTextLayout = rootView.findViewById(R.id.feature_request_comment_edittext_layout);
        if (commentInputLayout != null)
            commentInputLayout.setHint(getLocalizedString(R.string.add_feature) + "*");
        nameEditTextLayout = rootView.findViewById(R.id.feature_request_name_edittext_layout);
        emailEditTextLayout = rootView.findViewById(R.id.feature_request_email_edittext_layout);

        underLineComment = rootView.findViewById(R.id.feature_requests_comment_text_underline);
        underLineName = rootView.findViewById(R.id.feature_requests_name_text_underline);
        underLineEmail = rootView.findViewById(R.id.feature_requests_email_text_underline);

        emailDisclaimer = rootView.findViewById(R.id.feature_request_email_disclaimer);

        if (Build.VERSION.SDK_INT > Build.VERSION_CODES.UPSIDE_DOWN_CAKE) {
            RelativeLayout mainLayout = findViewById(R.id.ib_fr_comment_fragment_main_layout);
            if(mainLayout != null) {
                ViewUtilsKt.addSystemWindowInsetToPadding(mainLayout, true, false, true, true);
            }
        }

        TextInputLayoutHelper.setUpperHintColor(commentInputLayout, SettingsManager.getInstance().getPrimaryColor());
        TextInputLayoutHelper.setUpperHintColor(nameInputLayout, SettingsManager.getInstance().getPrimaryColor());
        TextInputLayoutHelper.setUpperHintColor(emailInputLayout, SettingsManager.getInstance().getPrimaryColor());

        handleEditTextFocusChange();
        if (FeaturesRequestSettings.getInstance().isUserIdentificationStateEnabled())
            presenter.fillEmailAndName();
        else {
            hideEmail();
            hideUserName();
        }
        presenter.handelRequiredFieldsHints();
        toolbarPostTextView = (TextView) findTextViewByTitle(R.string.feature_request_str_post_comment);
        setToolbarPostTextViewEnabled(false);
    }

    private void hideUserName() {
        if (nameInputLayout != null) nameInputLayout.setVisibility(View.GONE);

        if (underLineName != null) underLineName.setVisibility(View.GONE);
    }

    private void hideEmail() {
        if (emailInputLayout != null) emailInputLayout.setVisibility(View.GONE);

        if (underLineEmail != null) underLineEmail.setVisibility(View.GONE);

        if (emailDisclaimer != null) emailDisclaimer.setVisibility(View.GONE);

    }

    @Override
    public void onStop() {
        super.onStop();
        if (getActivity() != null) {
            KeyboardUtils.hide(getActivity());
        }
    }

    private void handleEditTextFocusChange() {
        // Use a local allocation to overcome SpotBugs false-positive possible NPE warnings

        if (commentEditTextLayout == null) return;
        TextInputEditText commentsEditTextLayout = this.commentEditTextLayout;
        commentsEditTextLayout.setOnFocusChangeListener((v, hasFocus) -> {
            View underLineComment = AddCommentFragment.this.underLineComment;
            TextInputLayout commentInputLayout = AddCommentFragment.this.commentInputLayout;
            if (getContext() == null || underLineComment == null) return;
            if (hasFocus) {
                underLineComment.getLayoutParams().height = ViewUtils.convertDpToPx(getContext(), 2);
                if (commentInputLayout != null && commentInputLayout.isErrorEnabled()) {
                    TextInputLayoutHelper.setUpperHintColor(commentInputLayout,
                            ContextCompat.getColor(getContext(), R.color.ib_fr_add_comment_error));
                    underLineComment.setBackgroundColor(ContextCompat.getColor(getContext(), R.color.ib_fr_add_comment_error));
                } else {
                    TextInputLayoutHelper.setUpperHintColor(commentInputLayout, SettingsManager.getInstance().getPrimaryColor());
                    underLineComment.setBackgroundColor(SettingsManager.getInstance().getPrimaryColor());
                }
            } else {
                TextInputLayoutHelper.setUpperHintColor(commentInputLayout, SettingsManager.getInstance().getPrimaryColor());
                underLineComment.setBackgroundColor(AttrResolver.getColor(getContext(), R.attr.ib_fr_add_comment_edit_text_underline_color));
                underLineComment.getLayoutParams().height = ViewUtils.convertDpToPx(getContext(), 1);
            }
            underLineComment.requestLayout();
            AddCommentFragment.this.underLineComment = underLineComment;
            AddCommentFragment.this.commentInputLayout = commentInputLayout;
        });

        if (nameEditTextLayout == null) return;
        nameEditTextLayout.setOnFocusChangeListener((v, hasFocus) -> {
            //localUnderLineName
            View underLineName = AddCommentFragment.this.underLineName;
            if (AddCommentFragment.this.getContext() == null || underLineName == null)
                return;
            if (hasFocus) {
                underLineName.getLayoutParams().height = ViewUtils.convertDpToPx(AddCommentFragment.this.getContext(), 2);
                underLineName.setBackgroundColor(SettingsManager.getInstance().getPrimaryColor());
            } else {
                underLineName.setBackgroundColor(AttrResolver.getColor(AddCommentFragment.this.getContext(), R.attr.ib_fr_add_comment_edit_text_underline_color));
                underLineName.getLayoutParams().height = ViewUtils.convertDpToPx(AddCommentFragment.this.getContext(), 1);
            }
            underLineName.requestLayout();
            AddCommentFragment.this.underLineName = underLineName;
        });

        if (emailEditTextLayout == null) return;
        TextInputEditText localEmailEditTextLayout = emailEditTextLayout;
        localEmailEditTextLayout.setOnFocusChangeListener((v, hasFocus) -> {
            View underLineEmail = AddCommentFragment.this.underLineEmail;
            if (getContext() == null || underLineEmail == null
                    || emailInputLayout == null || nameInputLayout == null) return;
            if (hasFocus) {

                underLineEmail.getLayoutParams().height = ViewUtils.convertDpToPx(getContext(), 2);
                if (emailInputLayout.isErrorEnabled()) {
                    nameInputLayout.setErrorEnabled(true);
                    TextInputLayoutHelper.setUpperHintColor(emailInputLayout,
                            ContextCompat.getColor(getContext(), R.color.ib_fr_add_comment_error));
                    underLineEmail.setBackgroundColor(ContextCompat.getColor(getContext(), R.color.ib_fr_add_comment_error));
                } else {
                    nameInputLayout.setErrorEnabled(false);
                    TextInputLayoutHelper.setUpperHintColor(emailInputLayout, SettingsManager.getInstance().getPrimaryColor());
                    underLineEmail.setBackgroundColor(SettingsManager.getInstance().getPrimaryColor());
                }
            } else {
                TextInputLayoutHelper.setUpperHintColor(emailInputLayout, SettingsManager.getInstance().getPrimaryColor());
                underLineEmail.setBackgroundColor(AttrResolver.getColor(getContext(), R.attr.ib_fr_add_comment_edit_text_underline_color));
                underLineEmail.getLayoutParams().height = ViewUtils.convertDpToPx(getContext(), 1);
            }
            underLineEmail.requestLayout();
            AddCommentFragment.this.underLineEmail = underLineEmail;
        });

        localEmailEditTextLayout.addTextChangedListener(new SimpleTextWatcher() {
            @Override
            public void afterTextChanged(Editable s) {
                if (presenter == null) return;
                TextInputEditText commentsEditTextLayout = AddCommentFragment.this.commentEditTextLayout;
                if (presenter.isCommenterEmailRequired()) {
                    if (!(s.toString().equals(presenter.getEnteredEmail()))) {

                        if (validateEmail()) {
                            if (commentsEditTextLayout != null
                                    && commentsEditTextLayout.getText() != null
                                    && !commentsEditTextLayout.getText().toString().trim().isEmpty()) {
                                setToolbarPostTextViewEnabled(true);
                            }
                        } else {
                            setToolbarPostTextViewEnabled(false);
                        }
                    }
                }
                if (emailDisclaimer == null) return;
                if (!TextUtils.isEmpty(s.toString())) {
                    emailDisclaimer.setVisibility(View.GONE);
                } else {
                    emailDisclaimer.setVisibility(View.VISIBLE);
                }
            }
        });
        commentsEditTextLayout.addTextChangedListener(new SimpleTextWatcher() {
            @Override
            public void afterTextChanged(Editable s) {
                super.afterTextChanged(s);
                View underLineComment = AddCommentFragment.this.underLineComment;
                TextInputEditText emailEditTextLayout = AddCommentFragment.this.emailEditTextLayout;
                TextInputLayout commentInputLayout = AddCommentFragment.this.commentInputLayout;

                if (underLineComment == null) return;

                if (commentsEditTextLayout.getText() != null && commentsEditTextLayout.getText().toString().trim().isEmpty()) {
                    setFieldError(true, commentInputLayout, underLineComment, getLocalizedString(R.string
                            .feature_request_str_add_comment_comment_empty));
                    setToolbarPostTextViewEnabled(false);
                } else {
                    setFieldError(false, commentInputLayout, underLineComment, getLocalizedString(R.string
                            .feature_request_str_add_comment_comment_empty));
                    if (emailEditTextLayout != null && presenter.isCommenterEmailRequired()) {
                        Editable editable = emailEditTextLayout.getText();
                        setToolbarPostTextViewEnabled(
                                editable != null
                                        && !editable.toString().trim().isEmpty()
                                        && Patterns.EMAIL_ADDRESS.matcher(editable.toString()).matches()
                        );
                    } else {
                        setToolbarPostTextViewEnabled(true);
                    }
                }
                AddCommentFragment.this.emailEditTextLayout = emailEditTextLayout;
                AddCommentFragment.this.commentInputLayout = commentInputLayout;
            }
        });
    }

    public static AddCommentFragment newInstance(@NonNull long featureId) {
        AddCommentFragment addCommentFragment = new AddCommentFragment();
        Bundle bundle = new Bundle();
        bundle.putLong(KEY_FEATURE_ID, featureId);
        addCommentFragment.setArguments(bundle);
        return addCommentFragment;
    }

    @Override
    public void onClick(View v) {

    }

    @Override
    public void onCloseButtonClicked() {
        if (getActivity() == null) return;
        getActivity().onBackPressed();
    }

    @Override
    public void validate() {
        if (!validateCommentBody()) {
            return;
        }
        if (presenter.isCommenterEmailRequired()) {
            if (!validateEmail()) {
                return;
            }
        }
        if (commentEditTextLayout == null || nameEditTextLayout == null || emailEditTextLayout == null)
            return;

        if (commentEditTextLayout.getText() != null
                && nameEditTextLayout.getText() != null
                && emailEditTextLayout.getText() != null) {

            presenter.addComment(new NewComment(featureId, commentEditTextLayout.getText().toString(),
                    nameEditTextLayout.getText().toString(),
                    emailEditTextLayout.getText().toString()));
        }
    }

    private boolean validateCommentBody() {
        View localUnderLineComment = underLineComment;
        if (getContext() == null || commentEditTextLayout == null || localUnderLineComment == null)
            return false;

        if (commentEditTextLayout.getText() != null && TextUtils.isEmpty(commentEditTextLayout.getText().toString())) {
            setFieldError(true, commentInputLayout, localUnderLineComment,
                    getLocalizedString(R.string.feature_request_str_add_comment_comment_empty));
            if (commentInputLayout != null)
                commentInputLayout.requestFocus();
            localUnderLineComment.setBackgroundColor(ContextCompat.getColor(getContext(), R.color.ib_fr_add_comment_error));
            return false;
        } else {
            setFieldError(false, commentInputLayout, localUnderLineComment, null);
        }
        underLineComment = localUnderLineComment;
        return true;
    }

    private boolean validateEmail() {
        if (emailInputLayout == null || underLineEmail == null
                || emailEditTextLayout == null || emailEditTextLayout.getText() == null)
            return false;

        if (TextUtils.isEmpty(emailEditTextLayout.getText().toString())
                || !Patterns.EMAIL_ADDRESS.matcher(emailEditTextLayout.getText().toString()).matches()) {
            setFieldError(true, emailInputLayout, underLineEmail,
                    getLocalizedString(R.string.feature_request_str_add_comment_valid_email));
            emailEditTextLayout.requestFocus();
            return false;
        } else {
            setFieldError(false, emailInputLayout, underLineEmail, null);
        }
        return true;
    }

    @Override
    public void setUserEmail(String userEmail) {
        if (emailEditTextLayout != null) {
            emailEditTextLayout.setText(userEmail);
        }
    }

    @Override
    public void setUserName(String username) {
        if (nameEditTextLayout != null) {
            nameEditTextLayout.setText(username);
        }
    }

    @Override
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public void showLoadingDialog() {
        // Use a local allocation to overcome SpotBugs false-positive possible NPE warnings
        IBGProgressDialog localPreparingProgressDialog = preparingProgressDialog;
        if (localPreparingProgressDialog != null) {
            if (!localPreparingProgressDialog.isShowing()) {
                localPreparingProgressDialog.show();
            }
        } else {
            if (getActivity() == null) return;
            localPreparingProgressDialog = new IBGProgressDialog.Builder()
                    .setProgressColor(SettingsManager.getInstance().getPrimaryColor())
                    .setMessage(getLocalizedString(R.string
                            .feature_request_str_adding_your_comment))
                    .build(getActivity());

            preparingProgressDialog = localPreparingProgressDialog;
            preparingProgressDialog.show();
        }
    }

    @Override
    public void dismissLoadingDialog() {
        if (preparingProgressDialog != null) {
            if (preparingProgressDialog.isShowing()) {
                preparingProgressDialog.dismiss();
            }
        }
    }

    @Override
    public void showError() {
        if (getActivity() == null) return;
        Toast.makeText(getActivity(), R.string.feature_request_str_add_comment_error, Toast.LENGTH_LONG).show();
    }

    @Override
    public String getUsername() {
        if (nameEditTextLayout == null || nameEditTextLayout.getText() == null) return "";
        return nameEditTextLayout.getText().toString();
    }

    @Override
    public String getEmail() {
        if (emailEditTextLayout == null || emailEditTextLayout.getText() == null) return "";
        return emailEditTextLayout.getText().toString();
    }

    @Override
    public void setEmailHintStringWithAsterisk(boolean withAsterisk) {
        if (emailInputLayout == null) return;

        if (withAsterisk) {
            emailInputLayout.setHint(getLocalizedString(R.string.ib_email_label) + "*");
        } else {
            emailInputLayout.setHint(getLocalizedString(R.string.ib_email_label));
        }
    }

    @Override
    public void onCommentAddedSuccessfully() {
        if (getActivity() == null) return;
        if (getActivity() instanceof FeaturesRequestActivity) {
            ((FeaturesRequestActivity) getActivity()).notifyCommentsChanged();
            getActivity().onBackPressed();
        }
    }

    private void setFieldError(boolean showError,
                               @Nullable TextInputLayout textInputLayout,
                               View underlineView,
                               @Nullable String errorString) {
        if (getContext() == null) return;

        if (textInputLayout != null) {
            if (showError) {
                textInputLayout.setErrorEnabled(true);
                textInputLayout.setError(errorString);
                TextInputLayoutHelper.setUpperHintColor(textInputLayout, ContextCompat.getColor(getContext(),
                        R.color.ib_fr_add_comment_error));
                underlineView.setBackgroundColor(ContextCompat.getColor(getContext(), R.color.ib_fr_add_comment_error));
            } else {
                TextInputLayoutHelper.setUpperHintColor(textInputLayout, SettingsManager.getInstance().getPrimaryColor());
                textInputLayout.setError(null);
                if (textInputLayout.getEditText() != null && textInputLayout.getEditText().isFocused()) {
                    underlineView.setBackgroundColor(SettingsManager.getInstance().getPrimaryColor());
                } else {
                    underlineView.setBackgroundColor(AttrResolver.getColor(getContext(), R.attr.ib_fr_add_comment_edit_text_underline_color));
                }
                textInputLayout.setErrorEnabled(false);
            }
        }
    }

    private void setToolbarPostTextViewEnabled(Boolean enabled) {
        if (toolbarPostTextView != null) {
            if (enabled) {
                toolbarPostTextView.setEnabled(true);
                toolbarPostTextView.setTextColor(getResources().getColor(android.R.color.white));

            } else {
                toolbarPostTextView.setEnabled(false);
                toolbarPostTextView.setTextColor(getResources().getColor(android.R.color
                        .darker_gray));
            }
        }
    }
}
