package com.instabug.featuresrequest.cache;

import com.instabug.featuresrequest.Constants;
import com.instabug.featuresrequest.models.FeatureRequest;
import com.instabug.featuresrequest.models.FeatureRequestResponse;
import com.instabug.library.internal.storage.cache.Cache;
import com.instabug.library.internal.storage.cache.CacheManager;
import com.instabug.library.internal.storage.cache.InMemoryCache;
import com.instabug.library.util.InstabugSDKLogger;

import java.util.ArrayList;
import java.util.List;

import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

/**
 * Created by mohamedzakaria on 9/6/17.
 */

public class FeatureRequestCacheManager {

    public static final String FEATURES_REQUEST_DISK_CACHE_FILE_NAME = "/features_request.cache";
    public static final String FEATURES_REQUEST_DISK_CACHE_KEY = "features_request_disk_cache";
    public static final String FEATURES_REQUEST_MEMORY_CACHE_KEY = "features_request_memory_cache";
    public static final String FEATURES_REQUESTS_EXTRAS_DISK_CACHE_FILE_NAME =
            "/features_request_extras.cache";
    public static final String FEATURES_REQUESTS_EXTRAS_DISK_CACHE_KEY =
            "features_request_extras_disk_cache";
    public static final String FEATURES_REQUESTS_EXTRAS_MEMORY_CACHE_KEY =
            "features_request_extras_memory_cache";

    /**
     * Loads feature request cache from disk if it's not in memory
     *
     * @return in-memory cache for feature request
     * @throws IllegalArgumentException if the from cache is not found
     */
    @Nullable
    public static InMemoryCache<Long, FeatureRequest> getCache() throws IllegalArgumentException {
        if (!CacheManager.getInstance().cacheExists(FEATURES_REQUEST_MEMORY_CACHE_KEY)) {

            CacheManager.getInstance().migrateCache(FEATURES_REQUEST_DISK_CACHE_KEY,
                    FEATURES_REQUEST_MEMORY_CACHE_KEY, new CacheManager.KeyExtractor<Long,
                            FeatureRequest>() {
                        @Override
                        public Long extractKey(FeatureRequest value) {
                            return value.getFeatureId();
                        }
                    });
            Cache featuresRequestsMemoryCache =
                    CacheManager.getInstance().getCache(FEATURES_REQUEST_MEMORY_CACHE_KEY);
            if (featuresRequestsMemoryCache != null) {
                InstabugSDKLogger.v(Constants.LOG_TAG,
                        "In-memory Feature Request cache restored from disk, "
                                + featuresRequestsMemoryCache.size()
                                + " elements restored");
            }
        }
        InstabugSDKLogger.v(Constants.LOG_TAG, "In-memory features " +
                "request cache found");
        return (InMemoryCache<Long, FeatureRequest>) CacheManager.getInstance().getCache
                (FEATURES_REQUEST_MEMORY_CACHE_KEY);
    }


    @VisibleForTesting
    static void tearDown() {
        CacheManager.getInstance().deleteCache(FEATURES_REQUEST_MEMORY_CACHE_KEY);
        CacheManager.getInstance().deleteCache(FEATURES_REQUEST_DISK_CACHE_KEY);
    }

    /**
     * Saves all cached feature requests from
     * {@link FeatureRequestCacheManager#FEATURES_REQUEST_MEMORY_CACHE_KEY }to disk cache
     */
    public static void saveCacheToDisk() {
        Cache featuresRequestDiskCache = CacheManager.getInstance().getCache
                (FEATURES_REQUEST_DISK_CACHE_KEY);

        Cache featuresRequestMemoryCache = CacheManager.getInstance().getCache
                (FEATURES_REQUEST_MEMORY_CACHE_KEY);

        int featuresRequestDiskCacheElementsCount;
        if (featuresRequestDiskCache != null && featuresRequestMemoryCache != null) {
            InstabugSDKLogger.v(Constants.LOG_TAG,
                    "Checking old values cached " + featuresRequestDiskCache.getValues());

            InstabugSDKLogger.v(Constants.LOG_TAG,
                    "Saving In-memory feature requests cache to disk, no. of feature requests to " +
                            "save is "
                            + featuresRequestMemoryCache.size());
            CacheManager.getInstance()
                    .migrateCache(featuresRequestMemoryCache, featuresRequestDiskCache,
                            new CacheManager.KeyExtractor<String, FeatureRequest>() {
                                @Override
                                public String extractKey(FeatureRequest value) {
                                    return String.valueOf(value.getFeatureId());
                                }
                            });

            featuresRequestDiskCacheElementsCount = featuresRequestDiskCache.getValues().size();
        } else {
            featuresRequestDiskCacheElementsCount = 0;
        }

        InstabugSDKLogger.v(Constants.LOG_TAG,
                "In-memory feature requests cache had been persisted on-disk, "
                        + featuresRequestDiskCacheElementsCount + " feautre reuests saved");

    }

    public static void addFeatures(List<FeatureRequest> newFeatureRequests) {
        for (FeatureRequest feature : newFeatureRequests) {
            addFeature(feature);
        }
    }

    public static void addFeature(FeatureRequest feature) {
        InMemoryCache<Long, FeatureRequest> cache = getCache();
        if (cache != null) {
            cache.put(feature.getFeatureId(), feature);
        }
    }


    public static List<FeatureRequest> getFeatures() {
        InMemoryCache<Long, FeatureRequest> cache = getCache();
        if (cache != null) {
            return cache.getValues();
        }
        return new ArrayList<>();
    }

    @Nullable
    public static FeatureRequest getFeature(long featureId) {
        InMemoryCache<Long, FeatureRequest> cache = getCache();
        if (cache != null) {
            for (FeatureRequest featureRequest : cache.getValues()) {
                if (featureRequest.getFeatureId() == featureId)
                    return featureRequest;
            }
        }
        return null;
    }

    public static List<FeatureRequest> getVotesToUpload() {
        List<FeatureRequest> featureRequests = new ArrayList<>();
        InMemoryCache<Long, FeatureRequest> cache = getCache();
        if (cache != null) {
            List<FeatureRequest> values = cache.getValues();
            InstabugSDKLogger.v(Constants.LOG_TAG, "Votes size: " + values.size());
            featureRequests.addAll(values);
        }
        return featureRequests;
    }


    /**
     * // todo not the best please revisit this solution
     * Loads feature request extras cache from disk if it's not in memory
     * Feature request extras are the has_next_page & completed_features_count that are returned
     * with fetching features list
     *
     * @return in-memory cache for feature request
     * @throws IllegalArgumentException if the from cache is not found
     */
    @Nullable
    public static InMemoryCache<String, Object> getExtrasCache() throws IllegalArgumentException {
        if (!CacheManager.getInstance().cacheExists(FEATURES_REQUESTS_EXTRAS_MEMORY_CACHE_KEY)) {
            InstabugSDKLogger.v(Constants.LOG_TAG, "In-memory features request " +
                    "extas cache not found, loading it from disk "
                    + CacheManager.getInstance().getCache
                    (FEATURES_REQUESTS_EXTRAS_MEMORY_CACHE_KEY));
            CacheManager.getInstance().migrateCache(FEATURES_REQUESTS_EXTRAS_DISK_CACHE_KEY,
                    FEATURES_REQUESTS_EXTRAS_MEMORY_CACHE_KEY, new CacheManager
                            .KeyExtractor<String, Object>() {
                        @Override
                        public String extractKey(Object value) {
                            return value instanceof Boolean ? FeatureRequestResponse.KEY_NEXT_PAGE :
                                    FeatureRequestResponse.KEY_COMPLETED_FEATURES_COUNT;
                        }
                    });

            Cache featuresRequestsMemoryCache =
                    CacheManager.getInstance().getCache(FEATURES_REQUESTS_EXTRAS_MEMORY_CACHE_KEY);
            if (featuresRequestsMemoryCache != null) {
                InstabugSDKLogger.v(Constants.LOG_TAG,
                        "In-memory Feature Request extras cache restored from disk, "
                                + featuresRequestsMemoryCache.size()
                                + " elements restored");
            }
        }
        InstabugSDKLogger.v(Constants.LOG_TAG, "In-memory features request extras " +
                "cache found");
        return (InMemoryCache<String, Object>) CacheManager.getInstance().getCache
                (FEATURES_REQUESTS_EXTRAS_MEMORY_CACHE_KEY);
    }

    /**
     * Saves all cached feature requests from
     * {@link FeatureRequestCacheManager#FEATURES_REQUEST_MEMORY_CACHE_KEY }to disk cache
     */
    public static void saveExtrasCacheToDisk() {
        Cache featuresRequestExtrasDiskCache = CacheManager.getInstance().getCache
                (FEATURES_REQUESTS_EXTRAS_DISK_CACHE_KEY);

        Cache featuresRequestExtrasMemoryCache = CacheManager.getInstance().getCache
                (FEATURES_REQUESTS_EXTRAS_MEMORY_CACHE_KEY);

        int featuresRequestDiskCacheElementsCount;
        if (featuresRequestExtrasDiskCache != null && featuresRequestExtrasMemoryCache != null) {
            InstabugSDKLogger.v(Constants.LOG_TAG,
                    "Checking old values cached " + featuresRequestExtrasDiskCache.getValues());

            InstabugSDKLogger.v(Constants.LOG_TAG,
                    "Saving In-memory feature requests extras cache to disk, no. of feature " +
                            "requests to save is "
                            + featuresRequestExtrasMemoryCache.size());
            CacheManager.getInstance()
                    .migrateCache(featuresRequestExtrasMemoryCache, featuresRequestExtrasDiskCache,
                            new CacheManager.KeyExtractor<String, Object>() {
                                @Override
                                public String extractKey(Object value) {
                                    return value instanceof Boolean ? FeatureRequestResponse
                                            .KEY_NEXT_PAGE : FeatureRequestResponse
                                            .KEY_COMPLETED_FEATURES_COUNT;
                                }
                            });

            featuresRequestDiskCacheElementsCount = featuresRequestExtrasDiskCache.getValues()
                    .size();
        } else {
            featuresRequestDiskCacheElementsCount = 0;
        }

        InstabugSDKLogger.v(Constants.LOG_TAG,
                "In-memory feature requests extras cache had been persisted on-disk, "
                        + featuresRequestDiskCacheElementsCount + " feauture requests extras " +
                        "saved");

    }

    public static void setHasNextPage(boolean hasNextPage) {
        InMemoryCache<String, Object> cache = getExtrasCache();
        if (cache != null) {
            cache.put(FeatureRequestResponse.KEY_NEXT_PAGE, hasNextPage);
        }
    }

    public static boolean hasNextPage() {
        InMemoryCache<String, Object> cache = getExtrasCache();
        if (cache != null) {
            if (cache.get(FeatureRequestResponse.KEY_NEXT_PAGE) != null) {
                return cache.get(FeatureRequestResponse.KEY_NEXT_PAGE) != null && (boolean) cache.get(FeatureRequestResponse.KEY_NEXT_PAGE);
            }
        }
        return false;
    }

    public static int getCompletedFeaturesCount() {
        InMemoryCache<String, Object> cache = getExtrasCache();
        if (cache != null) {
            if (cache.get(FeatureRequestResponse.KEY_COMPLETED_FEATURES_COUNT) != null) {
                return cache.get(FeatureRequestResponse.KEY_COMPLETED_FEATURES_COUNT) == null ? 0 : (int) cache.get(FeatureRequestResponse.KEY_COMPLETED_FEATURES_COUNT);
            }
        }
        return 0;
    }

    public static void setCompletedFeaturesCount(int count) {
        InMemoryCache<String, Object> cache = getExtrasCache();
        if (cache != null) {
            cache.put(FeatureRequestResponse.KEY_COMPLETED_FEATURES_COUNT, count);
        }
    }

}
