package com.instabug.featuresrequest.models;

import android.annotation.SuppressLint;

import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.featuresrequest.Constants;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.internal.storage.cache.Cacheable;
import com.instabug.library.user.UserManagerWrapper;
import com.instabug.library.util.InstabugSDKLogger;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

/**
 * Created by mohamedzakaria on 8/28/17.
 */

public class FeatureRequest implements Cacheable, Serializable {

    @VisibleForTesting
    static final String KEY_ID = "id";
    @VisibleForTesting
    static final String KEY_TITLE = "title";
    @VisibleForTesting
    static final String KEY_DESCRIPTION = "description";
    @VisibleForTesting
    static final String KEY_STATUS = "status";
    @VisibleForTesting
    static final String KEY_LIKES_COUNT = "likes_count";
    @VisibleForTesting
    static final String KEY_COMMENTS_COUNT = "comments_count";
    @VisibleForTesting
    static final String KEY_IS_LIKED = "liked";
    @VisibleForTesting
    static final String KEY_DATE = "date";
    private static final String KEY_COLOR_CODE = "color_code";
    private static final String KEY_CREATOR_NAME = "creator_name";
    // internal use only not available in the object returned from backend
    private static final String KEY_VOTE_UPDATED = "ib_user_vote_status";
    private long featureId;
    private String title;
    private String description;
    private Status status;
    private String colorCode;
    private String creatorName;
    private long date;
    private int likesCount;
    private int commentsCount;
    private boolean isLiked;

    @Nullable
    private String requesterName;
    private String requesterEmail;
    private String deviceToken;

    private UserVoteStatus userVoteStatus = UserVoteStatus.NOTHING;

    public static List<FeatureRequest> fromJson(JSONArray featuresJsonArray) throws JSONException {
        List<FeatureRequest> featuresRequestList = new ArrayList<>();
        for (int i = 0; i < featuresJsonArray.length(); i++) {
            JSONObject featureObject = featuresJsonArray.getJSONObject(i);
            FeatureRequest featureRequest = new FeatureRequest(UserManagerWrapper.getUserName(), UserManagerWrapper.getUserEmail(), InstabugCore.getPushNotificationToken());
            featureRequest.fromJson(featureObject.toString());
            featuresRequestList.add(featureRequest);
        }
        return featuresRequestList;
    }

    public static FeatureRequest fromJson(JSONObject featureRequestJsobObject) throws JSONException {
        FeatureRequest featureRequest = new FeatureRequest(UserManagerWrapper.getUserName(), UserManagerWrapper.getUserEmail(), InstabugCore.getPushNotificationToken());
        featureRequest.fromJson(featureRequestJsobObject.toString());
        return featureRequest;
    }

    @SuppressLint("ERADICATE_FIELD_NOT_INITIALIZED")
    public FeatureRequest(@Nullable String requesterName, String requesterEmail, String deviceToken) {
        this.status = Status.Open;
        this.featureId = System.currentTimeMillis() / 1000L;
        this.requesterName = requesterName;
        this.requesterEmail = requesterEmail;
        this.deviceToken = deviceToken;
    }

    public long getFeatureId() {
        return featureId;
    }

    public void setFeatureId(long featureId) {
        this.featureId = featureId;
    }

    public String getTitle() {
        return title;
    }

    public void setTitle(String title) {
        this.title = title;
    }

    public Status getStatus() {
        return status;
    }

    public void setStatus(Status status) {
        this.status = status;
    }

    public int getLikesCount() {
        return likesCount;
    }

    public void setLikesCount(int likesCount) {
        this.likesCount = likesCount;
    }

    public int getCommentsCount() {
        return commentsCount;
    }

    public void setCommentsCount(int commentsCount) {
        this.commentsCount = commentsCount;
    }

    public boolean isLiked() {
        return isLiked;
    }

    public void setLiked(boolean liked) {
        isLiked = liked;
    }

    public UserVoteStatus getUserVoteStatus() {
        return userVoteStatus;
    }

    public void setUserVoteStatus(UserVoteStatus userVoteStatus) {
        this.userVoteStatus = userVoteStatus;
    }

    public String getColorCode() {
        return colorCode;
    }

    public void setColorCode(String colorCode) {
        this.colorCode = colorCode;
    }

    public long getDate() {
        return date;
    }

    public void setDate(long date) {
        this.date = date;
    }

    public String getCreatorName() {
        return creatorName;
    }

    public void setCreatorName(String creatorName) {
        this.creatorName = creatorName;
    }

    @Nullable
    public String getRequesterName() {
        return requesterName;
    }

    public String getRequesterEmail() {
        return requesterEmail;
    }

    public String getDeviceToken() {
        return deviceToken;
    }

    public JSONObject getFeatureJsonObject() throws JSONException {
        JSONObject featureJsonObject = new JSONObject();
        featureJsonObject
                .put(KEY_TITLE, getTitle())
                .put(KEY_DESCRIPTION, getDescription());
        return featureJsonObject;
    }

    @Override
    public String toJson() throws JSONException {
        JSONObject featureJsonObject = new JSONObject();
        featureJsonObject
                .put(KEY_ID, getFeatureId())
                .put(KEY_TITLE, getTitle())
                .put(KEY_DESCRIPTION, getDescription())
                .put(KEY_STATUS, getStatus().toInt())
                .put(KEY_DATE, getDate())
                .put(KEY_LIKES_COUNT, getLikesCount())
                .put(KEY_COMMENTS_COUNT, getCommentsCount())
                .put(KEY_IS_LIKED, isLiked())
                .put(KEY_VOTE_UPDATED, getUserVoteStatus().toInt())
                .put(KEY_COLOR_CODE, getColorCode())
                .put(KEY_CREATOR_NAME, getCreatorName());
        return featureJsonObject.toString();
    }

    @Override
    public void fromJson(String modelAsJson) throws JSONException {
        InstabugSDKLogger.v(Constants.LOG_TAG, "Parsing feature request: " + modelAsJson);

        JSONObject featureJsonObject = new JSONObject(modelAsJson);

        if (featureJsonObject.has(KEY_ID)) {
            setFeatureId(featureJsonObject.getLong(KEY_ID));
        }

        if (featureJsonObject.has(KEY_TITLE)) {
            setTitle(featureJsonObject.getString(KEY_TITLE));
        }

        if (featureJsonObject.has(KEY_DESCRIPTION)) {
            setDescription(featureJsonObject.getString(KEY_DESCRIPTION));
        }

        if (featureJsonObject.has(KEY_CREATOR_NAME)) {
            setCreatorName(featureJsonObject.getString(KEY_CREATOR_NAME));
        }

        handleFeatureStatus(featureJsonObject);

        if (featureJsonObject.has(KEY_COLOR_CODE)) {
            setColorCode(featureJsonObject.getString(KEY_COLOR_CODE));
        }
        if (featureJsonObject.has(KEY_LIKES_COUNT)) {
            setLikesCount(featureJsonObject.getInt(KEY_LIKES_COUNT));
        }

        if (featureJsonObject.has(KEY_DATE)) {
            setDate(featureJsonObject.getLong(KEY_DATE));
        }

        if (featureJsonObject.has(KEY_COMMENTS_COUNT)) {
            setCommentsCount(featureJsonObject.getInt(KEY_COMMENTS_COUNT));
        }
        if (featureJsonObject.has(KEY_IS_LIKED)) {
            setLiked(featureJsonObject.getBoolean(KEY_IS_LIKED));
        }

        updateVoteStatus(featureJsonObject);
    }

    private void updateVoteStatus(JSONObject featureJsonObject) throws JSONException {
        if (featureJsonObject.has(KEY_VOTE_UPDATED)) {
            int userVoteStatus = featureJsonObject.getInt(KEY_VOTE_UPDATED);
            switch (userVoteStatus) {
                case 0:
                    setUserVoteStatus(UserVoteStatus.NOTHING);
                    break;
                case 1:
                    setUserVoteStatus(UserVoteStatus.UPLOADED);
                    break;
                case 2:
                    setUserVoteStatus(UserVoteStatus.USER_VOTED_UP);
                    break;
                case 3:
                    setUserVoteStatus(UserVoteStatus.USER_UN_VOTED);
                    break;
                default:
                    setUserVoteStatus(UserVoteStatus.NOTHING);
                    break;

            }
        }
    }

    private void handleFeatureStatus(JSONObject featureJsonObject) throws JSONException {
        if (featureJsonObject.has(KEY_STATUS)) {
            int type = featureJsonObject.getInt(KEY_STATUS);
            switch (type) {
                case 0:
                    setStatus(Status.Open);
                    break;
                case 1:
                    setStatus(Status.Planned);
                    break;
                case 2:
                    setStatus(Status.InProgress);
                    break;
                case 3:
                    setStatus(Status.Completed);
                    break;
                case 4:
                    setStatus(Status.MaybeLater);
                    break;
                default:
                    break;
            }
        }
    }

    public String getDescription() {
        return description;
    }

    public void setDescription(String description) {
        this.description = description;
    }

    public boolean isCompleted() {
        return getStatus() == Status.Completed;
    }

    public enum Status {
        Open(0), Planned(1), InProgress(2), Completed(3), MaybeLater(4);
        private int status;

        Status(int type) {
            this.status = type;
        }

        public int toInt() {
            return status;
        }
    }

    public enum UserVoteStatus {
        NOTHING(0), UPLOADED(1), USER_VOTED_UP(2), USER_UN_VOTED(3);
        private int status;

        UserVoteStatus(int status) {
            this.status = status;
        }

        public int toInt() {
            return status;
        }

    }
}
