package com.instabug.fatalhangs.di

import com.instabug.commons.configurations.ConfigurationsHandler
import com.instabug.commons.di.CommonsLocator
import com.instabug.commons.session.SessionLinker
import com.instabug.fatalhangs.IBGFatalHangDetector
import com.instabug.fatalhangs.cache.FatalHangsCacheManager
import com.instabug.fatalhangs.cache.FatalHangsCacheManagerImpl
import com.instabug.fatalhangs.configuration.FatalHangsConfigurationHandlerImpl
import com.instabug.fatalhangs.configuration.FatalHangsConfigurationProvider
import com.instabug.fatalhangs.configuration.FatalHangsConfigurationProviderImpl
import com.instabug.fatalhangs.constants.Constants
import com.instabug.fatalhangs.model.FatalHang
import com.instabug.fatalhangs.sync.FatalHangsSyncManager
import com.instabug.fatalhangs.sync.FatalHangsSyncManagerImpl
import com.instabug.library.Instabug
import com.instabug.library.SpansCacheDirectory
import com.instabug.library.networkv2.NetworkManager
import com.instabug.library.tracking.InstabugInternalTrackingDelegate
import com.instabug.library.util.threading.PoolProvider
import com.instabug.library.visualusersteps.ReproConfigurationsProvider
import java.io.File
import java.lang.ref.WeakReference
import java.util.concurrent.ThreadPoolExecutor

object FatalHangsServiceLocator {

    private val objectsMap = mutableMapOf<String, WeakReference<Any?>?>()

    private fun getIfAvailable(key: String): Any? {
        return if (objectsMap.containsKey(key)
            && objectsMap[key] != null && objectsMap[key]!!.get() != null
        ) {
            objectsMap[key]!!.get()
        } else null
    }

    private fun saveInMap(key: String, value: Any) {
        objectsMap[key] = WeakReference(value)
    }

    fun getContext() = Instabug.getApplicationContext()

    fun getIOExecutor(): ThreadPoolExecutor? = PoolProvider.getInstance().ioExecutor

    fun getFatalHangDetectorThread(callback: (FatalHang) -> Unit): Thread =
        IBGFatalHangDetector(callback)


    fun getFatalHangsCacheManager(): FatalHangsCacheManager {
        val key = FatalHangsCacheManager::class.toString()
        var fatalHangsCacheManager = getIfAvailable(key)
        if (fatalHangsCacheManager == null) {
            fatalHangsCacheManager = FatalHangsCacheManagerImpl()
            saveInMap(key, fatalHangsCacheManager)
        }
        return fatalHangsCacheManager as FatalHangsCacheManager
    }

    @Synchronized
    fun getFatalHangsSyncManager(): FatalHangsSyncManager {
        val key = FatalHangsSyncManager::class.toString()
        var fatalHangsSyncManager = getIfAvailable(key)
        if (fatalHangsSyncManager == null) {
            fatalHangsSyncManager = FatalHangsSyncManagerImpl()
            saveInMap(key, fatalHangsSyncManager)
        }
        return fatalHangsSyncManager as FatalHangsSyncManager
    }

    val trackingDelegate: InstabugInternalTrackingDelegate
        get() = InstabugInternalTrackingDelegate.getInstance()

    fun getStoreLimit() = Constants.FATAL_HANGS_STORE_LIMIT

    fun getNetworkManager() = NetworkManager()

    fun getFileFromPath(path: String) = File(path)

    val fatalHangsConfigurationProvider: FatalHangsConfigurationProvider
            by lazy { FatalHangsConfigurationProviderImpl() }

    val fatalHangsConfigurationHandler: ConfigurationsHandler
            by lazy { FatalHangsConfigurationHandlerImpl() }

    val reproConfigurationsProvider: ReproConfigurationsProvider
        get() = fatalHangsConfigurationProvider

    val reproScreenshotsCacheDir: SpansCacheDirectory
        get() = CommonsLocator.reproScreenshotsCacheDir

    val sessionLinker: SessionLinker
        get() = CommonsLocator.sessionLinker
}