package com.instabug.crash.models

import com.instabug.crash.CrashReporting

/**
 * Presents a reportable non-fatal exception that can be used for manual reporting.
 * @see CrashReporting.report
 */
class IBGNonFatalException private constructor(
    val throwable: Throwable,
    val userAttributes: Map<String, String>,
    val fingerprint: String?,
    val level: Level
) {

    /**
     * Must be used to build an object of [IBGNonFatalException], which can be later used to report
     * a manually handled exception
     *
     * @param throwable the original [Throwable] that should be reported.
     */
    class Builder(private val throwable: Throwable) {
        private var userAttributes: Map<String, String> = emptyMap()
        private var fingerprint: String? = null
        private var level = Level.ERROR

        /**
         * To be used to set attributes to be added to the reported
         *
         * @param userAttributes The user attributes to be added to the report.
         * @return [Builder] instance
         */
        fun setUserAttributes(userAttributes: Map<String, String>): Builder =
            apply { this.userAttributes = userAttributes }

        /**
         * To be used to set a fingerprint which overrides Instabug's default grouping algorithm.
         *
         * @param fingerprint The fingerprint that'll be used to override Instabug's default grouping algorithm
         * @return [Builder] instance
         */
        fun setFingerprint(fingerprint: String): Builder = apply { this.fingerprint = fingerprint }

        /**
         * To be used to set the severity level to be assigned to the report. Severity level
         * is also used for grouping. Default is [Level.ERROR].
         *
         * @param level The severity [Level] to be assigned to the report.
         * @return [Builder] instance
         */
        fun setLevel(level: Level): Builder = apply { this.level = level }

        /**
         * To be used to build the [IBGNonFatalException] object to be used for reporting.
         *
         * @return The built object of [IBGNonFatalException]
         * @see CrashReporting.report
         */
        fun build(): IBGNonFatalException =
            IBGNonFatalException(throwable, userAttributes, fingerprint, level)
    }

    /**
     * The severity levels that can be assigned to reports.
     *
     * @see Builder.setLevel
     */
    enum class Level(val severity: Int) {
        INFO(0),
        WARNING(1),
        ERROR(2),
        CRITICAL(3);

        companion object {
            @JvmStatic
            fun parse(severity: Int) = values().find { level -> level.severity == severity }
        }
    }
}