package com.instabug.crash.models;

import android.annotation.SuppressLint;
import android.content.Context;
import android.net.Uri;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.WorkerThread;

import com.instabug.commons.AttachmentsHolder;
import com.instabug.commons.BasicAttachmentsHolder;
import com.instabug.commons.caching.DiskHelper;
import com.instabug.commons.configurations.ConfigurationsProvider;
import com.instabug.commons.models.Incident;
import com.instabug.commons.models.IncidentMetadata;
import com.instabug.crash.Constants;
import com.instabug.crash.di.CrashesServiceLocator;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.internal.storage.cache.Cacheable;
import com.instabug.library.model.Attachment;
import com.instabug.library.model.State;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.visualusersteps.ReproConfigurationsProvider;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.File;
import java.util.List;

import kotlin.Pair;


/**
 * @author mesbah.
 */
public class Crash implements Incident, Cacheable, AttachmentsHolder {
    private static final String TAG = "Crash";

    // keys used by toJson() & fromJson() methods
    public static final String KEY_ID = "id";
    public static final String KEY_TEMPORARY_SERVER_TOKEN = "temporary_server_token";
    public static final String KEY_CRASH_MESSAGE = "crash_message";
    public static final String KEY_ATTACHMENTS = "attachments";
    public static final String KEY_STATE = "state";
    public static final String KEY_CRASH_STATE = "crash_state";
    public static final String KEY_HANDLED = "handled";
    public static final String KEY_RETRY_COUNT = "retry_count";
    public static final String KEY_THREADS_DETAILS = "threads_details";
    public static final String KEY_FINGERPRINT = "fingerprint";
    public static final String KEY_LEVEL = "level";

    private final String id;
    @Nullable
    private String temporaryServerToken;
    @Nullable
    private String crashMessage;
    private final AttachmentsHolder attachmentsHolder;
    @Nullable
    private State state;
    private CrashState crashState;
    private boolean handled;
    private int retryCount;
    @Nullable
    private String threadsDetails;
    @Nullable
    private String fingerprint;
    @Nullable
    private IBGNonFatalException.Level level;

    @NonNull
    private IncidentMetadata metadata;

    public Crash(@NonNull String id, @NonNull IncidentMetadata metadata) {
        this.id = id;
        this.metadata = metadata;
        this.crashState = CrashState.NOT_AVAILABLE;
        this.attachmentsHolder = new BasicAttachmentsHolder();
    }

    public Crash(@NonNull String id, @Nullable State state, @NonNull IncidentMetadata metadata) {
        this(id, metadata);
        this.state = state;
        this.retryCount = 0;
    }

    @NonNull
    public String getId() {
        return id;
    }

    @Nullable
    public String getTemporaryServerToken() {
        return temporaryServerToken;
    }

    public Crash setTemporaryServerToken(String temporaryServerToken) {
        this.temporaryServerToken = temporaryServerToken;
        return this;
    }

    @Nullable
    public String getCrashMessage() {
        return crashMessage;
    }

    public Crash setCrashMessage(String crashMessage) {
        this.crashMessage = crashMessage;
        return this;
    }

    public Crash setThreadsDetails(@Nullable String threadsDetails) {
        this.threadsDetails = threadsDetails;
        return this;
    }

    @Nullable
    public String getThreadsDetails() {
        return threadsDetails;
    }

    public Crash addAttachment(Uri attachmentUri) {
        addAttachment(attachmentUri, Attachment.Type.ATTACHMENT_FILE, false);
        return this;
    }

    @Nullable
    public State getState() {
        return state;
    }

    public Crash setState(State state) {
        this.state = state;
        return this;
    }

    public CrashState getCrashState() {
        return crashState;
    }

    public Crash setCrashState(CrashState crashState) {
        this.crashState = crashState;
        return this;
    }

    public boolean isHandled() {
        return handled;
    }

    public Crash setHandled(boolean handled) {
        this.handled = handled;
        return this;
    }

    public int getRetryCount() {
        return retryCount;
    }

    public void setRetryCount(int retryCount) {
        this.retryCount = retryCount;
    }

    @Nullable
    public String getFingerprint() {
        return fingerprint;
    }

    public void setFingerprint(@Nullable String fingerprint) {
        this.fingerprint = fingerprint;
    }

    @Nullable
    public IBGNonFatalException.Level getLevel() {
        return level;
    }

    public void setLevel(@Nullable IBGNonFatalException.Level level) {
        this.level = level;
    }

    public void setLevel(int severity) {
        this.level = IBGNonFatalException.Level.parse(severity);
    }

    public void setMetadata(@NonNull IncidentMetadata metadata) {
        this.metadata = metadata;
    }

    @Override
    public String toJson() throws JSONException {
        JSONObject crash = new JSONObject();
        crash.put(KEY_ID, getId())
                .put(KEY_TEMPORARY_SERVER_TOKEN, getTemporaryServerToken())
                .put(KEY_CRASH_MESSAGE, getCrashMessage())
                .put(KEY_CRASH_STATE, getCrashState().toString())
                .put(KEY_ATTACHMENTS, Attachment.toJson(getAttachments()))
                .put(KEY_HANDLED, isHandled())
                .put(KEY_RETRY_COUNT, getRetryCount())
                .put(KEY_THREADS_DETAILS, getThreadsDetails())
                .put(KEY_FINGERPRINT, getFingerprint());

        IBGNonFatalException.Level level = getLevel();
        if (level != null) {
            crash.put(KEY_LEVEL, level.getSeverity());
        }

        if (getState() != null) {
            crash.put(KEY_STATE, getState().toJson());
        } else {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Error parsing crash: state is null");
        }
        return crash.toString();
    }

    @Override
    public void fromJson(String crashASJson) throws JSONException {
        JSONObject crashJsonObject = new JSONObject(crashASJson);
        if (crashJsonObject.has(KEY_TEMPORARY_SERVER_TOKEN))
            setTemporaryServerToken(crashJsonObject.getString(KEY_TEMPORARY_SERVER_TOKEN));
        if (crashJsonObject.has(KEY_CRASH_MESSAGE))
            setCrashMessage(crashJsonObject.getString(KEY_CRASH_MESSAGE));
        if (crashJsonObject.has(KEY_CRASH_STATE))
            setCrashState(CrashState.valueOf(crashJsonObject.getString(KEY_CRASH_STATE)));
        if (crashJsonObject.has(KEY_STATE)) {
            State state = new State();
            state.fromJson(crashJsonObject.getString(KEY_STATE));
            setState(state);
        }
        if (crashJsonObject.has(KEY_ATTACHMENTS))
            setAttachments(Attachment.fromJson(crashJsonObject.getJSONArray(KEY_ATTACHMENTS)));
        if (crashJsonObject.has(KEY_HANDLED))
            setHandled(crashJsonObject.getBoolean(KEY_HANDLED));
        if (crashJsonObject.has(KEY_RETRY_COUNT))
            setRetryCount(crashJsonObject.getInt(KEY_RETRY_COUNT));
        if (crashJsonObject.has(KEY_THREADS_DETAILS))
            setThreadsDetails(crashJsonObject.getString(KEY_THREADS_DETAILS));
        if (crashJsonObject.has(KEY_FINGERPRINT))
            setFingerprint(crashJsonObject.getString(KEY_FINGERPRINT));
        if (crashJsonObject.has(KEY_LEVEL))
            setLevel(crashJsonObject.getInt(KEY_LEVEL));
    }

    @NonNull
    @Override
    public IncidentMetadata getMetadata() {
        return metadata;
    }

    @NonNull
    @Override
    public Type getType() {
        if (handled) return Type.NonFatalCrash;
        return Type.FatalCrash;
    }

    @Override
    public String toString() {
        return "Internal Id: " + id + ", TemporaryServerToken:" + temporaryServerToken
                + ", crashMessage:" + crashMessage + ", handled:" + handled + ", retryCount:" + retryCount
                + ", threadsDetails:" + threadsDetails + ", fingerprint:" + fingerprint + ", level:" + level;
    }

    @Override
    @SuppressLint("NP_METHOD_PARAMETER_TIGHTENS_ANNOTATION")
    public boolean equals(@Nullable Object crash) {
        if (crash == null) return false;
        if (crash instanceof Crash) {
            Crash comparedCrash = (Crash) crash;
            if (String.valueOf(comparedCrash.getId()).equals(String.valueOf(getId()))
                    && String.valueOf(comparedCrash.getCrashMessage()).equals(String.valueOf
                    (getCrashMessage()))
                    && String.valueOf(comparedCrash.getTemporaryServerToken()).equals(String
                    .valueOf(getTemporaryServerToken()))
                    && comparedCrash.getCrashState() == getCrashState()
                    && comparedCrash.getState() != null && comparedCrash.getState().equals(getState())
                    && comparedCrash.isHandled() == isHandled()
                    && comparedCrash.getRetryCount() == getRetryCount()
                    && comparedCrash.getAttachments() != null
                    && comparedCrash.getAttachments().size() == getAttachments().size()
                    && ((comparedCrash.getThreadsDetails() == null && getThreadsDetails() == null)
                    || (comparedCrash.getThreadsDetails() != null && comparedCrash.getThreadsDetails().equals(getThreadsDetails())))
                    && ((comparedCrash.getFingerprint() == null && getFingerprint() == null)
                    || (comparedCrash.getFingerprint() != null && comparedCrash.getFingerprint().equals(getFingerprint())))
                    && ((comparedCrash.getLevel() == null && getLevel() == null)
                    || (comparedCrash.getLevel() != null && comparedCrash.getLevel().equals(getLevel())))) {
                for (int i = 0; i < comparedCrash.getAttachments().size(); i++) {
                    if (!(comparedCrash.getAttachments().get(i).equals(getAttachments().get(i))))
                        return false;
                }
                return true;
            } else {
                return false;
            }
        } else {
            return false;
        }
    }

    @Override
    public int hashCode() {
        if (getId() != null) {
            return getId().hashCode();
        } else {
            return -1;
        }
    }

    @NonNull
    @Override
    public File getSavingDirOnDisk(@NonNull Context ctx) {
        return DiskHelper.getIncidentSavingDirectory(ctx, getType().name(), id);
    }

    @NonNull
    @Override
    public List<Attachment> getAttachments() {
        return attachmentsHolder.getAttachments();
    }

    @Override
    public void addAttachment(@Nullable Uri uri, @NonNull Attachment.Type type, boolean isEncrypted) {
        attachmentsHolder.addAttachment(uri, type, isEncrypted);
    }

    @Override
    public void setAttachments(@NonNull List<? extends Attachment> attachments) {
        attachmentsHolder.setAttachments(attachments);
    }

    public enum CrashState {
        READY_TO_BE_SENT, LOGS_READY_TO_BE_UPLOADED, ATTACHMENTS_READY_TO_BE_UPLOADED, NOT_AVAILABLE,
        WAITING_FOR_SCREEN_RECORDING_TO_BE_TRIMMED
    }

    public static class Factory {

        public static final boolean STATE_LOG_AND_ATTACHMENTS_DEFAULT_PARAM = true;

        @SuppressLint("CheckResult")
        public Crash create(@Nullable State state, @NonNull Context context, boolean handled) {
            return create(state, context, handled, STATE_LOG_AND_ATTACHMENTS_DEFAULT_PARAM);
        }

        @NonNull
        public Crash create(
                @Nullable State state,
                @NonNull Context context,
                boolean handled,
                boolean withStateLogsAndAttachments
        ) {
            return create(
                    System.currentTimeMillis() + "",
                    state,
                    IncidentMetadata.Factory.create(),
                    context,
                    handled,
                    withStateLogsAndAttachments
            );
        }

        public Crash create(@NonNull String id, @NonNull IncidentMetadata metadata) {
            return new Crash(id, metadata);
        }

        public Crash create(
                @NonNull String id,
                @Nullable State state,
                @NonNull IncidentMetadata metadata,
                @NonNull Context context,
                boolean handled
        ) {
            return create(id, state, metadata, context, handled, STATE_LOG_AND_ATTACHMENTS_DEFAULT_PARAM);
        }

        @NonNull
        public Crash create(
                @NonNull String id,
                @Nullable State state,
                @NonNull IncidentMetadata metadata,
                @NonNull Context context,
                boolean handled,
                boolean withStateLogsAndAttachments
        ) {
            final Crash crash = new Crash(id, state, metadata);
            crash.setHandled(handled);
            if (withStateLogsAndAttachments) {
                updateReproInteractionsIfApplicable(crash);
                addReproScreenshotsAttachmentIfApplicable(crash, context);
            }
            return crash;
        }


        public static void updateReproInteractionsIfApplicable(@NonNull Crash crash) {
            try {
                State crashState = crash.getState();
                if (crashState == null || !shouldUpdateReproInteractions(crash))
                    return;
                crashState.updateVisualUserSteps();
            } catch (Throwable t) {
                IBGDiagnostics.reportNonFatal(t, "Error while updating Repro interactions in crash incident: " + t.getMessage());
            }
        }

        @WorkerThread
        public static void addReproScreenshotsAttachmentIfApplicable(@NonNull Crash crash, @NonNull Context context) {
            try {
                if (!shouldAddReproScreenshotAttachmentToCrash(crash))
                    return;
                File reproScreenshotsDir = CrashesServiceLocator.getReproScreenshotsCacheDir().getCurrentSpanDirectory();
                if (reproScreenshotsDir == null) return;
                Pair<String, Boolean> zippingResult = DiskHelper.getReproScreenshotsZipPath(context, crash.getId(), crash.getSavingDirOnDisk(context), reproScreenshotsDir);
                if (zippingResult.getFirst() == null) return;
                crash.addAttachment(Uri.parse(zippingResult.getFirst()), Attachment.Type.VISUAL_USER_STEPS, zippingResult.getSecond());
            } catch (Throwable t) {
                IBGDiagnostics.reportNonFatal(t, "Error while adding Repro screenshots attachment to crash incident: " + t.getMessage());
            }
        }


        private static boolean shouldAddReproScreenshotAttachmentToCrash(@NonNull Crash crash) {
            boolean isHandled = crash.isHandled();
            ReproConfigurationsProvider configurationsProvider;

            if(isHandled) {
                configurationsProvider = CrashesServiceLocator.getNonFatalsConfigurationsProvider();
            } else {
                configurationsProvider = CrashesServiceLocator.getReproConfigurationsProvider();
            }

            return configurationsProvider.isReproScreenshotsEnabled();
        }


        private static boolean shouldUpdateReproInteractions(@NonNull Crash crash) {
            boolean isHandled = crash.isHandled();
            ReproConfigurationsProvider configurationsProvider;

            if(isHandled) {
                configurationsProvider = CrashesServiceLocator.getNonFatalsConfigurationsProvider();
            } else {
                configurationsProvider = CrashesServiceLocator.getReproConfigurationsProvider();
            }

            return configurationsProvider.isReproStepsEnabled();
        }
    }
}
