package com.instabug.crash;

import android.content.Context;
import android.net.Uri;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.commons.caching.DiskHelper;
import com.instabug.commons.threading.CrashDetailsParser;
import com.instabug.commons.utils.StateExtKt;
import com.instabug.crash.cache.CrashReportsDbHelper;
import com.instabug.crash.di.CrashesServiceLocator;
import com.instabug.crash.models.Crash;
import com.instabug.crash.settings.PerSessionSettings;
import com.instabug.crash.utils.CrashReportingUtility;
import com.instabug.crash.utils.ExceptionFormatter;
import com.instabug.early_crash.EarlyUncaughtExceptionHandlingDelegate;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.internal.storage.DiskUtils;
import com.instabug.library.internal.storage.operation.WriteStateToFileDiskOperation;
import com.instabug.library.model.State;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.InstabugSDKLogger;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.File;

public class InstabugUncaughtExceptionHandler implements Thread.UncaughtExceptionHandler {

    public static Boolean isRegistered = false;

    @VisibleForTesting
    public static boolean isInEarlyCapturingMode = false;
    @Nullable
    private final Thread.UncaughtExceptionHandler defaultUncaughtExceptionHandler;
    @Nullable
    private final Context context;

    public InstabugUncaughtExceptionHandler(@Nullable Context context) {
        this.context = context;
        defaultUncaughtExceptionHandler = Thread.getDefaultUncaughtExceptionHandler();
        setRegistered();
    }

    public InstabugUncaughtExceptionHandler(@Nullable Context context, boolean isInEarlyCapturingMode) {
        this(context);
        setEarlyCapturingMode(isInEarlyCapturingMode);
    }

    private void setRegistered() {
        InstabugUncaughtExceptionHandler.isRegistered = true;
    }

    public static void setEarlyCapturingMode(boolean isInEarlyCapturingMode) {
        InstabugUncaughtExceptionHandler.isInEarlyCapturingMode = isInEarlyCapturingMode;
    }

    @Override
    public void uncaughtException(@NonNull final Thread thread, @NonNull final Throwable throwable) {

        InstabugSDKLogger.e(Constants.LOG_TAG, "InstabugUncaughtExceptionHandler Caught an Unhandled Exception: " +
                throwable.getClass().getCanonicalName(), throwable);

        logOnEarlyCapturingMode("Is in early capture mode: " + isInEarlyCapturingMode);

        if (!isIsInEarlyCapturingMode() && !CrashReportingUtility.isCrashReportingEnabled()) {
            logOnEarlyCapturingMode("Crash reporting is disabled, skipping...");
            if (defaultUncaughtExceptionHandler != null)
                defaultUncaughtExceptionHandler.uncaughtException(thread, throwable);
            return;
        }
        try {
            logOnEarlyCapturingMode("Creating a crash report ...");
            CrashDetailsParser parser = ReportCreationHelper.parseCrashDetails(thread, throwable);
            if (parser == null) return;
            if (isIsInEarlyCapturingMode()) {
                handleEarlyException(parser);
                return;
            }
            if (isImmediateMetadataSyncEnabled()) {
                handleImmediateSyncCrash(parser);
                return;
            }
            ReportCreationHelper.performPreReportActivities();

            if (context == null || isRNCrashedSession()) return;

            State state = ReportCreationHelper.prepareReportState(context);
            ReportCreationHelper.modifyReportStateWithUserInput(state);
            Crash crash = new Crash.Factory().create(state, context, false);
            crash = updateCrash(crash, parser.getCrashDetails(), parser.getThreadsDetails());
            ReportCreationHelper.handleReportAttachments(crash, context);
            cacheCrash(context, crash);
            ReportCreationHelper.performPostReportActivities(crash);
            logOnEarlyCapturingMode("Crash report created");
        } catch (Exception e) {
            reportNonFatalAndFailGracefully(e, "Error while capturing crash report: " + e.getMessage());
            InstabugSDKLogger.e(Constants.LOG_TAG, "Error: " + e.getMessage() + " while capturing crash report");
        } catch (OutOfMemoryError outOfMemoryError) {
            reportNonFatalAndFailGracefully(outOfMemoryError, "OOM in uncaughtExceptionHandler");
        } finally {
            if (defaultUncaughtExceptionHandler != null)
                defaultUncaughtExceptionHandler.uncaughtException(thread, throwable);
        }
    }

    private boolean isIsInEarlyCapturingMode() {
        return isInEarlyCapturingMode &&
                CrashesServiceLocator.getCrashConfigurationProvider().isCrashReportingEnabled();
    }

    private boolean isRNCrashedSession() {
        return PerSessionSettings.getInstance().isIsRNCrashedSession();
    }

    @NonNull
    public Crash updateCrash(Crash crash, JSONObject crashJsonObject, @Nullable JSONArray threadsDetails) {
        InstabugSDKLogger.v(Constants.LOG_TAG, "Updating crash before persisting to disk");
        crash.setCrashMessage(crashJsonObject.toString())
                .setThreadsDetails(threadsDetails != null ? threadsDetails.toString() : null)
                .setCrashState(Crash.CrashState.READY_TO_BE_SENT)
                .setHandled(false);
        return crash;
    }

    public static void cacheCrash(Context context, Crash crash) {
        try {
            State crashState = crash.getState();
            if (crashState != null) {
                StateExtKt.updateScreenShotAnalytics(crashState);
                InstabugSDKLogger.v(Constants.LOG_TAG, "caching crash " + crash.getId());
                File stateFile = DiskHelper.getIncidentStateFile(crash.getSavingDirOnDisk(context), CrashReporting.CRASH_STATE);
                Uri uri = DiskUtils.with(context)
                        .writeOperation(new WriteStateToFileDiskOperation(stateFile, crashState.toJson()))
                        .execute();
                crashState.setUri(uri);
            }
        } catch (Throwable throwable) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Error " + throwable.getMessage() + " while caching crash state file.");
            IBGDiagnostics.reportNonFatal(throwable, "Error while caching fatal crash report state file: " + throwable.getMessage());
        }
        CrashReportsDbHelper.trimAndInsert(crash);
    }

    public JSONObject createJSONObject(Thread thread, Throwable throwable) throws JSONException {
        JSONObject crashJsonObject = new JSONObject();
        JSONObject t = new JSONObject();
        t.put("threadName", thread.getName());
        t.put("threadId", thread.getId());
        t.put("threadPriority", thread.getPriority());
        t.put("threadState", thread.getState().toString());
        ThreadGroup group = thread.getThreadGroup();
        if (group != null) {
            JSONObject threadGroup = new JSONObject();
            threadGroup.put("name", group.getName());
            threadGroup.put("maxPriority", group.getMaxPriority());
            threadGroup.put("activeCount", group.activeCount());
            t.put("threadGroup", threadGroup);
        }

        crashJsonObject.put("thread", t);
        crashJsonObject.put("error", ExceptionFormatter.createExceptionJson(throwable, null));
        return crashJsonObject;
    }

    @VisibleForTesting
    public static void reset() {
        isRegistered = false;
        Thread.setDefaultUncaughtExceptionHandler(null);
    }

    private void logOnEarlyCapturingMode(String message) {
        if (isInEarlyCapturingMode) {
            InstabugSDKLogger.w(Constants.LOG_TAG, message);
            return;
        }
        InstabugSDKLogger.d(Constants.LOG_TAG, message);
    }

    private void handleEarlyException(CrashDetailsParser parser) {
        if (isRNCrashedSession()) return;
        new EarlyUncaughtExceptionHandlingDelegate.Factory()
                .buildWithDefaults()
                .invoke(parser, context);
        SettingsManager.getInstance().setCrashedSession(true);
    }

    private void handleImmediateSyncCrash(CrashDetailsParser parser) {
        if (context == null || isRNCrashedSession()) return;

        new ImmediateSyncUncaughtExceptionHandlerDelegate.Factory()
                .create()
                .invoke(parser, context);
    }

    private void reportNonFatalAndFailGracefully(Throwable throwable, String message) {
        try {
            IBGDiagnostics.reportNonFatalWithPromise(throwable, message).get();
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        } catch (Throwable t) {
            // Swallow
        }
    }

    private boolean isImmediateMetadataSyncEnabled() {
        return CrashesServiceLocator.getCrashConfigurationProvider().isMetadataImmediateSyncAvailable();
    }
}
